/* 
 * $Id: local-id3.c,v 1.17 2004/03/18 14:08:22 jylefort Exp $
 *
 * Copyright (c) 2004 Jean-Yves Lefort <jylefort@brutele.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <errno.h>
#include <id3tag.h>
#include "gettext.h"
#include "local.h"

/*** cpp *********************************************************************/

#define UCS4_TO_UTF8(ucs4, err) \
  (g_ucs4_to_utf8((const gunichar *) (ucs4), -1, NULL, NULL, (err)))

/*** frame read/write callbacks **********************************************/

static gboolean local_id3_frame_read_cb (LocalStream *stream,
					 const struct id3_tag *tag,
					 const char *frame_name,
					 gpointer data,
					 GError **err);
static gboolean local_id3_frame_read_genre_cb (LocalStream *stream,
					       const struct id3_tag *tag,
					       const char *frame_name,
					       gpointer data,
					       GError **err);
static gboolean local_id3_frame_read_duration_cb (LocalStream *stream,
						  const struct id3_tag *tag,
						  const char *frame_name,
						  gpointer data,
						  GError **err);

static gboolean local_id3_frame_write_cb (LocalStream *stream,
					  const struct id3_tag *tag,
					  const char *frame_name,
					  const GValue *value,
					  gpointer data,
					  GError **err);

/*** constant definitions ****************************************************/

static const struct
{
  const char	*name;
  int		field_id;

  gboolean	(*read_cb)	(LocalStream		*stream,
				 const struct id3_tag	*tag,
				 const char		*frame_name,
				 gpointer		data,
				 GError			**err);
  gboolean	(*write_cb)	(LocalStream		*stream,
				 const struct id3_tag	*tag,
				 const char		*frame_name,
				 const GValue		*value,
				 gpointer		data,
				 GError			**err);

  gpointer	data;
} frames[] = {
  /* ID3v1 frames */
  {
    ID3_FRAME_TITLE,			LOCAL_FIELD_TITLE,
    local_id3_frame_read_cb,		local_id3_frame_write_cb,
    GUINT_TO_POINTER(G_STRUCT_OFFSET(LocalStream, title))
  },
  {
    ID3_FRAME_ARTIST,			LOCAL_FIELD_ARTIST,
    local_id3_frame_read_cb,		local_id3_frame_write_cb,
    GUINT_TO_POINTER(G_STRUCT_OFFSET(LocalStream, artist))
  },
  {
    ID3_FRAME_ALBUM,			LOCAL_FIELD_ALBUM,
    local_id3_frame_read_cb,		local_id3_frame_write_cb,
    GUINT_TO_POINTER(G_STRUCT_OFFSET(LocalStream, album))
  },
  {
    ID3_FRAME_YEAR,			LOCAL_FIELD_YEAR,
    local_id3_frame_read_cb,		local_id3_frame_write_cb,
    GUINT_TO_POINTER(G_STRUCT_OFFSET(LocalStream, year))
  },
  {
    ID3_FRAME_GENRE,			LOCAL_FIELD_GENRE,
    local_id3_frame_read_genre_cb,	local_id3_frame_write_cb,
    GUINT_TO_POINTER(G_STRUCT_OFFSET(LocalStream, genre))
  },
  {
    ID3_FRAME_COMMENT,			LOCAL_FIELD_COMMENT,
    local_id3_frame_read_cb,		local_id3_frame_write_cb,
    GUINT_TO_POINTER(G_STRUCT_OFFSET(LocalStream, comment))
  },

  /* ID3v2 frames */
  {
    "TLEN",				LOCAL_FIELD_DURATION,
    local_id3_frame_read_duration_cb,	NULL, /* read-only */
    NULL
  }
};

/*** function declarations ***************************************************/

static void local_id3_read_frames (LocalStream *stream,
				   const struct id3_tag *tag);
static gboolean local_id3_write_frames (LocalStream *stream,
					const struct id3_tag *tag,
					GSList *fields,
					GSList *values,
					GError **err);

static const id3_ucs4_t *local_id3_frame_get_ucs4 (const struct id3_tag *tag,
						   const char *frame_name);

/*** implementation **********************************************************/

static void
local_id3_read_frames (LocalStream *stream, const struct id3_tag *tag)
{
  int i;

  g_return_if_fail(stream != NULL);

  for (i = 0; i < G_N_ELEMENTS(frames); i++)
    if (frames[i].read_cb)
      {
	GError *err = NULL;
	
	if (! frames[i].read_cb(stream, tag, frames[i].name, frames[i].data, &err))
	  {
	    st_notice(_("Local: %s: unable to read frame %s: %s"), stream->pathname, frames[i].name, err->message);
	    g_error_free(err);
	  }
      }
}

static gboolean
local_id3_write_frames (LocalStream *stream,
			const struct id3_tag *tag,
			GSList *fields,
			GSList *values,
			GError **err)
{
  GString *errors = NULL;
  GSList *f = fields;
  GSList *v = values;

  g_return_val_if_fail(stream != NULL, FALSE);

  while (f && v)
    {
      STHandlerField *field = f->data;
      const GValue *value = v->data;
      int i;

      for (i = 0; i < G_N_ELEMENTS(frames); i++)
	if (frames[i].field_id == field->id && frames[i].write_cb)
	  {
	    GError *tmp_err = NULL;

	    if (! frames[i].write_cb(stream, tag, frames[i].name, value, frames[i].data, &tmp_err))
	      {
		char *msg;

		msg = g_strdup_printf(_("unable to set %s field (%s)"), st_handler_field_get_label(field), tmp_err->message);
		g_error_free(tmp_err);

		if (errors)
		  g_string_append_printf(errors, "; %s", msg);
		else
		  errors = g_string_new(msg);

		g_free(msg);
	      }
	  }

      f = f->next;
      v = v->next;
    }

  if (errors)
    {
      g_set_error(err, 0, 0, "%s", errors->str);
      g_string_free(errors, TRUE);

      return FALSE;
    }
  else
    return TRUE;
}

static gboolean
local_id3_frame_read_cb (LocalStream *stream,
			 const struct id3_tag *tag,
			 const char *frame_name,
			 gpointer data,
			 GError **err)
{
  const id3_ucs4_t *ucs4;

  ucs4 = local_id3_frame_get_ucs4(tag, frame_name);
  if (ucs4)
    {
      char **str = (void *) stream + GPOINTER_TO_UINT(data);
      return (*str = UCS4_TO_UTF8(ucs4, err)) != NULL;
    }
  else
    return TRUE;		/* missing frame isn't an error */
}

static gboolean
local_id3_frame_read_genre_cb (LocalStream *stream,
			       const struct id3_tag *tag,
			       const char *frame_name,
			       gpointer data,
			       GError **err)
{
  const id3_ucs4_t *genre;

  genre = local_id3_frame_get_ucs4(tag, frame_name);
  if (genre)
    {
      const id3_ucs4_t *genre_name = id3_genre_name(genre);
      
      if (genre_name)
	return (stream->genre = UCS4_TO_UTF8(genre_name, err)) != NULL;
    }
  
  return TRUE;			/* missing frame isn't an error */
}

static gboolean
local_id3_frame_read_duration_cb (LocalStream *stream,
				  const struct id3_tag *tag,
				  const char *frame_name,
				  gpointer data,
				  GError **err)
{
  const id3_ucs4_t *tlen;

  tlen = local_id3_frame_get_ucs4(tag, frame_name);
  if (tlen)
    {
      char *utf8 = UCS4_TO_UTF8(tlen, err);
      unsigned int total_time;

      if (! utf8)
	return FALSE;		/* conversion error */

      total_time = atoi(utf8) / 1000;
      if (total_time > 0)	/* avoid bogus durations */
	stream->duration = LOCAL_DURATION(total_time);

      g_free(utf8);
    }

  return TRUE;			/* missing frame isn't an error */
}

static gboolean
local_id3_frame_write_cb (LocalStream *stream,
			  const struct id3_tag *tag,
			  const char *frame_name,
			  const GValue *value,
			  gpointer data,
			  GError **err)
{
  char **str = (void *) stream + GPOINTER_TO_UINT(data);
  const struct id3_frame *frame;
  union id3_field *field;
  id3_ucs4_t *ucs4;
  gboolean status;
  
  frame = id3_tag_findframe(tag, frame_name, 0);
  if (! frame)
    {
      g_set_error(err, 0, 0, _("%s frame does not exist in file"), frame_name);
      return FALSE;
    }

  field = id3_frame_field(frame, 1);
  if (! field)
    {
      g_set_error(err, 0, 0, _("unable to get ID3 field of %s frame"), frame_name);
      return FALSE;
    }
  
  ucs4 = (id3_ucs4_t *) g_utf8_to_ucs4_fast(g_value_get_string(value) ? g_value_get_string(value) : "", -1, NULL);
  status = id3_field_setstrings(field, 1, &ucs4) >= 0;
  g_free(ucs4);

  if (! status)
    {
      g_set_error(err, 0, 0, _("unable to put string into %s frame"), frame_name);
      return FALSE;
    }

  g_free(*str);
  *str = g_value_dup_string(value);

  return TRUE;
}

static const id3_ucs4_t *
local_id3_frame_get_ucs4 (const struct id3_tag *tag, const char *frame_name)
{
  const struct id3_frame *frame;
  const id3_ucs4_t *ucs4 = NULL;

  g_return_val_if_fail(tag != NULL, NULL);
  g_return_val_if_fail(frame_name != NULL, NULL);

  frame = id3_tag_findframe(tag, frame_name, 0);
  if (frame)
    {
      const union id3_field *field;

      field = id3_frame_field(frame, 1);
      if (field)
	ucs4 = id3_field_getstrings(field, 0);
    }

  return ucs4;
}

void
local_id3_read (LocalStream *stream)
{
  struct id3_file *file;
  const struct id3_tag *tag;

  g_return_if_fail(stream != NULL);

  /* open the file */

  file = id3_file_open(stream->pathname, ID3_FILE_MODE_READONLY);
  if (! file)
    {
      st_notice(_("Local: unable to open %s: %s"), stream->pathname, g_strerror(errno));
      return;
    }

  /* read the frames we know */

  tag = id3_file_tag(file);
  local_id3_read_frames(stream, tag);

  /* close the file */

  if (id3_file_close(file) < 0)
    st_notice(_("Local: unable to close %s: %s"), stream->pathname, g_strerror(errno));
}

gboolean
local_id3_modify (LocalStream *stream,
		  GSList *fields,
		  GSList *values,
		  GError **err)
{
  struct id3_file *file;
  const struct id3_tag *tag;
  gboolean status = TRUE;

  g_return_val_if_fail(stream != NULL, FALSE);

  file = id3_file_open(stream->pathname, ID3_FILE_MODE_READWRITE);
  if (! file)
    {
      g_set_error(err, 0, 0, _("unable to open file: %s"), g_strerror(errno));
      return FALSE;
    }

  tag = id3_file_tag(file);
  status = local_id3_write_frames(stream, tag, fields, values, err);

  if (id3_file_update(file) < 0 && status)
    {
      g_set_error(err, 0, 0, _("unable to update file"));
      status = FALSE;
    }

  if (id3_file_close(file) < 0 && status)
    {
      g_set_error(err, 0, 0, _("unable to close file: %s"), g_strerror(errno));
      status = FALSE;
    }

  return status;
}
