/*
 * $Id: local.c,v 1.40 2004/01/31 17:46:43 jylefort Exp $
 *
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#ifdef WITH_ID3TAG
#include <id3tag.h>
#endif
#ifdef WITH_VORBIS
#include <vorbis/vorbisfile.h>
#endif
#include <streamtuner/streamtuner.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include "art/icon.h"
#include "gettext.h"

/*** cpp *********************************************************************/

#define COPYRIGHT		"Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort"

#define DURATION(seconds)				\
  g_strdup_printf("%02u:%02u",				\
		  (unsigned int) (seconds) / 60,	\
		  (unsigned int) (seconds) % 60)

/*** type definitions ********************************************************/

typedef struct
{
  STStream	stream;

  char		*track;

  char		*title;
  char		*artist;
  char		*album;
  char		*year;
  char		*duration;

  char		*filename;
} LocalStream;

enum {
  FIELD_TRACK,
  FIELD_TITLE,
  FIELD_ARTIST,
  FIELD_ALBUM,
  FIELD_YEAR,
  FIELD_FILENAME,
  FIELD_DURATION,
};

/*** variable declarations ***************************************************/

static const char *local_root;
static GSList *local_extensions = NULL;

/*** function declarations ***************************************************/

static LocalStream *stream_new_cb (gpointer data);
static void stream_field_get_cb (LocalStream *stream,
				 STHandlerField *field,
				 GValue *value,
				 gpointer data);
static void stream_field_set_cb (LocalStream *stream,
				 STHandlerField *field,
				 const GValue *value,
				 gpointer data);
static void stream_free_cb (LocalStream *stream, gpointer data);

static gboolean stream_tune_in_multiple_cb (GSList *streams,
					    gpointer data,
					    GError **err);

static gboolean refresh_cb (STCategory *category,
			    GNode **categories,
			    GList **streams,
			    gpointer data,
			    GError **err);

static gboolean refresh_categories (GNode *root, GError **err);
static gboolean refresh_streams (STCategory *category,
				 GList **streams,
				 GError **err);

static gboolean extension_is_handled (const char *extension);

#ifdef WITH_ID3TAG
static void mp3_stream_read (LocalStream *stream);
static void id3_read_tag (const struct id3_tag *tag, const char *frame_name, char **string);
#endif

#ifdef WITH_VORBIS
static void ogg_stream_read (LocalStream *stream);
static void stream_field_append (char **field, const char *str);
#endif

/*** implementation **********************************************************/

static LocalStream *
stream_new_cb (gpointer data)
{
  return g_new0(LocalStream, 1);
}

static void
stream_field_get_cb (LocalStream *stream,
		     STHandlerField *field,
		     GValue *value,
		     gpointer data)
{
  switch (field->id)
    {
    case FIELD_TRACK:
      g_value_set_string(value, stream->track);
      break;

    case FIELD_TITLE:
      g_value_set_string(value, stream->title);
      break;

    case FIELD_ARTIST:
      g_value_set_string(value, stream->artist);
      break;

    case FIELD_ALBUM:
      g_value_set_string(value, stream->album);
      break;
      
    case FIELD_YEAR:
      g_value_set_string(value, stream->year);
      break;

    case FIELD_FILENAME:
      g_value_set_string(value, stream->filename);
      break;

    case FIELD_DURATION:
      g_value_set_string(value, stream->duration);
      break;

    default:
      g_assert_not_reached();
    }
}

static void
stream_field_set_cb (LocalStream *stream,
		     STHandlerField *field,
		     const GValue *value,
		     gpointer data)
{
  switch (field->id)
    {
    case FIELD_TRACK:
      stream->track = g_value_dup_string(value);
      break;

    case FIELD_TITLE:
      stream->title = g_value_dup_string(value);
      break;

    case FIELD_ARTIST:
      stream->artist = g_value_dup_string(value);
      break;

    case FIELD_ALBUM:
      stream->album = g_value_dup_string(value);
      break;
      
    case FIELD_YEAR:
      stream->year = g_value_dup_string(value);
      break;

    case FIELD_FILENAME:
      stream->filename = g_value_dup_string(value);
      break;

    case FIELD_DURATION:
      stream->duration = g_value_dup_string(value);
      break;

    default:
      g_assert_not_reached();
    }
}

static void
stream_free_cb (LocalStream *stream, gpointer data)
{
  g_free(stream->track);
  g_free(stream->title);
  g_free(stream->artist);
  g_free(stream->album);
  g_free(stream->year);
  g_free(stream->duration);
  g_free(stream->filename);

  st_stream_free((STStream *) stream);
}

static gboolean
stream_tune_in_multiple_cb (GSList *streams, gpointer data, GError **err)
{
  char *m3uname;
  GSList *filenames = NULL;
  GSList *l;
  gboolean status;

  /* create a list of filenames from STREAMS */

  for (l = streams; l; l = l->next)
    filenames = g_slist_append(filenames, ((LocalStream *) l->data)->filename);

  /* write the .m3u */

  m3uname = st_m3u_mktemp("streamtuner.local.XXXXXX", filenames, err);
  g_slist_free(filenames);

  if (! m3uname)
    return FALSE;

  /* open the .m3u */

  status = st_action_run("play-m3u", m3uname, err);
  g_free(m3uname);

  return status;
}

static gboolean
refresh_cb (STCategory *category,
	    GNode **categories,
	    GList **streams,
	    gpointer data,
	    GError **err)
{
  *categories = g_node_new(NULL);

  if (! refresh_categories(*categories, err))
    return FALSE;

  if (! refresh_streams(category, streams, err))
    return FALSE;

  return TRUE;
}

static gboolean
refresh_categories (GNode *root, GError **err)
{
  GDir *dir;
  char *rootdir;
  const char *filename;
  gboolean status = TRUE;

  GError *tmp_err = NULL;

  g_return_val_if_fail(root != NULL, FALSE);

  if (root->data)
    rootdir = g_build_filename(local_root,
			       ((STCategory *) root->data)->url_postfix,
			       NULL);
  else
    rootdir = g_strdup(local_root);
  
  if (! (dir = g_dir_open(rootdir, 0, &tmp_err)))
    {
      g_set_error(err, 0, 0, _("unable to open directory %s: %s"),
		  rootdir, tmp_err->message);
      g_error_free(tmp_err);

      status = FALSE;
      goto end;
    }
  
  while ((filename = g_dir_read_name(dir)))
    {
      GNode *node;
      char *pathname;
      
      if (st_is_aborted())
	{
	  status = FALSE;
	  goto end;
	}
      
      if (filename[0] == '.')
	continue;
      
      pathname = g_build_filename(rootdir, filename, NULL);
      
      if (g_file_test(pathname, G_FILE_TEST_IS_DIR)) {
	STCategory *category;
	
	category = st_category_new();
	
	category->name = g_strdup(filename);
	if (! (category->label = g_filename_to_utf8(filename, -1, NULL, NULL,
						    &tmp_err)))
	  {
	    st_notice(_("Local: unable to convert directory name to UTF-8: %s"),
		      tmp_err->message);
	    g_clear_error(&tmp_err);
	  }
	
	if (root->data)
	  category->url_postfix = g_build_filename(((STCategory *) root->data)->url_postfix,
						   filename,
						   NULL);
	else
	  category->url_postfix = g_strdup(filename);
	
	node = g_node_new(category);
	g_node_append(root, node);
	
	if (! refresh_categories(node, err))
	  {
	    status = FALSE;
	    goto end;
	  }
      }
      
      g_free(pathname);
    }

 end:
  g_free(rootdir);
  
  return status;
}

static gboolean
refresh_streams (STCategory *category, GList **streams, GError **err)
{
  GDir *dir;
  char *dirname;
  const char *filename;
  gboolean status = TRUE;
  
  GError *tmp_err = NULL;

  g_return_val_if_fail(category != NULL, FALSE);

  if (category->url_postfix)
    dirname = g_build_filename(local_root, category->url_postfix, NULL);
  else
    dirname = g_strdup(local_root);

  if (! (dir = g_dir_open(dirname, 0, &tmp_err)))
    {
      g_set_error(err, 0, 0, _("unable to open directory %s: %s"),
		  dirname, tmp_err->message);
      g_error_free(tmp_err);

      status = FALSE;
      goto end;
    }

  while ((filename = g_dir_read_name(dir)))
    {
      LocalStream *stream;
      char *extension;
      
      if (st_is_aborted())
	{
	  status = FALSE;
	  goto end;
	}
      
      if (filename[0] == '.')
	continue;
      
      extension = strrchr(filename, '.');
      if (! extension++ || ! extension_is_handled(extension))
	continue;

      stream = stream_new_cb(NULL);
      
      ((STStream *) stream)->name = g_strdup(filename);
      if (! (stream->track = g_filename_to_utf8(filename, -1, NULL, NULL,
						&tmp_err)))
	{
	  st_notice(_("Local: unable to convert filename to UTF-8: %s"),
		    tmp_err->message);
	  g_clear_error(&tmp_err);
	}
      
      stream->filename = g_build_filename(dirname, filename, NULL);
      
      if (! strcasecmp(extension, "mp3"))
	{
#ifdef WITH_ID3TAG
	  mp3_stream_read(stream);
#endif
	}
      else if (! strcasecmp(extension, "ogg"))
	{
#ifdef WITH_VORBIS
	ogg_stream_read(stream);
#endif
	}
      
      *streams = g_list_append(*streams, stream);
    }
  
 end:
  g_free(dirname);
  
  return status;
}

static gboolean
extension_is_handled (const char *extension)
{
  GSList *l;

  g_return_val_if_fail(extension != NULL, FALSE);

  for (l = local_extensions; l; l = l->next)
    if (! strcasecmp(extension, l->data))
      return TRUE;

  return FALSE;
}

#ifdef WITH_ID3TAG
static void
mp3_stream_read (LocalStream *stream)
{
  struct id3_file *file;
  const struct id3_tag *tag;
  char *tlen = NULL;

  file = id3_file_open(stream->filename, ID3_FILE_MODE_READONLY);
  if (! file)
    {
      st_notice(_("Local: unable to open %s: %s"), stream->filename, strerror(errno));
      return;
    }

  tag = id3_file_tag(file);

  id3_read_tag(tag, ID3_FRAME_TITLE, &stream->title);
  id3_read_tag(tag, ID3_FRAME_ARTIST, &stream->artist);
  id3_read_tag(tag, ID3_FRAME_ALBUM, &stream->album);
  id3_read_tag(tag, ID3_FRAME_YEAR, &stream->year);

  id3_read_tag(tag, "TLEN", &tlen);
  if (tlen)
    {
      unsigned int total_time;

      total_time = atoi(tlen) / 1000;
      if (total_time > 0)	/* avoid bogus durations */
	stream->duration = DURATION(total_time);

      g_free(tlen);
    }
  
  if (id3_file_close(file) < 0)
    {
      st_notice(_("Local: unable to close %s: %s"), stream->filename, strerror(errno));
      return;
    }
}

static void
id3_read_tag (const struct id3_tag *tag, const char *frame_name, char **string)
{
  const struct id3_frame *frame;

  frame = id3_tag_findframe(tag, frame_name, 0);
  if (frame)
    {
      const union id3_field *field;
      const id3_ucs4_t *ucs4;
      GError *err = NULL;
      char *utf8;

      field = id3_frame_field(frame, 1);

      ucs4 = id3_field_getstrings(field, 0);
      if (! ucs4)
	return;

      utf8 = g_ucs4_to_utf8((const gunichar *) ucs4, -1, NULL, NULL, &err);
      if (! utf8)
	{
	  st_notice(_("Local: unable to convert ID3 field to UTF-8: %s"), err->message);
	  g_error_free(err);

	  return;
	}

      *string = utf8;
    }
}
#endif /* WITH_ID3TAG */

#ifdef WITH_VORBIS
static void
ogg_stream_read (LocalStream *stream)
{
  FILE *file;
  OggVorbis_File vf;
  int status;
  const vorbis_comment *comments;
  double total_time;
  int i;

  file = fopen(stream->filename, "r");
  if (! file)
    {
      st_notice(_("Local: unable to open %s: %s"), stream->filename, strerror(errno));
      return;
    }

  status = ov_open(file, &vf, NULL, 0);
  if (status < 0)
    {
      const char *err;

      switch (status)
	{
	case OV_EREAD:		err = _("a read from media returned an error"); break;
	case OV_ENOTVORBIS:	err = _("bitstream is not Vorbis data"); break;
	case OV_EVERSION:	err = _("Vorbis version mismatch"); break;
	case OV_EBADHEADER:	err = _("invalid Vorbis bitstream header"); break;
	case OV_EFAULT:		err = _("internal logic fault"); break;
	default:		err = _("unknown error"); break;
	}
      
      st_notice(_("Local: unable to ov_open() %s: %s"), stream->filename, err);
      fclose(file);

      return;
    }

  comments = ov_comment(&vf, -1);
  g_return_if_fail(comments != NULL);

  for (i = 0; i < comments->comments; i++)
    {
      char *equal_sign;

      equal_sign = strchr(comments->user_comments[i], '=');
      if (equal_sign)
	{
	  char *name;
	  char *value;

	  name = g_strndup(comments->user_comments[i], equal_sign - comments->user_comments[i]);
	  value = equal_sign + 1;

	  if (! strcasecmp(name, "title"))
	    stream_field_append(&stream->title, value);
	  else if (! strcasecmp(name, "artist"))
	    stream_field_append(&stream->artist, value);
	  else if (! strcasecmp(name, "album"))
	    stream_field_append(&stream->album, value);
	  else if (! strcasecmp(name, "date"))
	    stream_field_append(&stream->year, value);

	  g_free(name);
	}
    }

  total_time = ov_time_total(&vf, -1);
  if (total_time != OV_EINVAL)
    stream->duration = DURATION(total_time);
  else
    st_notice(_("Local: unable to read duration of %s"), stream->filename);
  
  ov_clear(&vf);
}

static void
stream_field_append (char **field, const char *str)
{
  char *new;

  g_return_if_fail(field != NULL);

  new = *field ? g_strdup_printf("%s, %s", *field, str) : g_strdup(str);

  g_free(*field);
  *field = new;
}
#endif /* WITH_VORBIS */

static void
init_handler (void)
{
  STHandler *handler;
  char *home;

  GNode *stock_categories;
  STCategory *category;

  handler = st_handler_new("local");

  st_handler_set_label(handler, _("Local"));
  st_handler_set_copyright(handler, COPYRIGHT);
  st_handler_set_description(handler, _("Local Music Collection"));

  home = g_strconcat("file://", local_root, NULL);
  st_handler_set_home(handler, home);
  g_free(home);

  st_handler_set_icon_from_inline(handler, sizeof(art_icon), art_icon);

  stock_categories = g_node_new(NULL);

  category = st_category_new();
  category->name = "__main";
  category->label = _("Root");
  
  g_node_append_data(stock_categories, category);
  
  st_handler_set_stock_categories(handler, stock_categories);

  st_handler_bind(handler, ST_HANDLER_EVENT_REFRESH, refresh_cb, NULL);

  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_NEW, stream_new_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FIELD_GET, stream_field_get_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FIELD_SET, stream_field_set_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FREE, stream_free_cb, NULL);

  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_TUNE_IN_MULTIPLE, stream_tune_in_multiple_cb, NULL);

  st_handler_add_field(handler, st_handler_field_new(FIELD_TRACK,
						     _("Track"),
						     G_TYPE_STRING,
						     TRUE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_TITLE,
						     _("Title"),
						     G_TYPE_STRING,
						     TRUE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_ARTIST,
						     _("Artist"),
						     G_TYPE_STRING,
						     TRUE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_ALBUM,
						     _("Album"),
						     G_TYPE_STRING,
						     TRUE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_YEAR,
						     _("Year"),
						     G_TYPE_STRING,
						     TRUE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_FILENAME,
						     _("Filename"),
						     G_TYPE_STRING,
						     FALSE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_DURATION,
						     _("Duration"),
						     G_TYPE_STRING,
						     TRUE));

  st_handlers_add(handler);
}

static void
init_extensions (const char *extensions)
{
  char **v;
  int i;

  v = g_strsplit(extensions, ":", 0);

  for (i = 0; v[i]; i++)
    local_extensions = g_slist_append(local_extensions, g_strdup(v[i]));

  g_strfreev(v);
}

gboolean
plugin_init (GError **err)
{
  const char *extensions;

  bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");

  if (! st_check_api_version(5, 4))
    {
      g_set_error(err, 0, 0, _("API version mismatch"));
      return FALSE;
    }

  if (! (local_root = g_getenv("STREAMTUNER_LOCAL_ROOT")))
    {
      g_set_error(err, 0, 0, _("You must point the STREAMTUNER_LOCAL_ROOT environment variable to your discotheque before using the Local plugin."));
      return FALSE;
    }
  
  if (! (extensions = g_getenv("STREAMTUNER_LOCAL_EXTENSIONS")))
    extensions = "mp3:ogg";

  init_handler();
  init_extensions(extensions);

  st_action_register("play-m3u", _("Listen to a .m3u file"), "xmms %q");

  return TRUE;
}
