/*
 * Copyright (c) 2002, 2003 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <glib.h>
#include <sys/types.h>
#include <regex.h>
#include <stdarg.h>

/*** API implementation ******************************************************/

/*
 * If STR matches REGEXP, returns TRUE and stores newly-allocated substrings
 * into the ... list. Otherwise, returns FALSE.
 *
 * You should make sure the number of variable arguments is equal to
 * REGEXP->re_nsub.
 */
gboolean
st_re_parse (regex_t *regexp, const char *str, ...)
{
  regmatch_t pmatch[regexp->re_nsub + 1];

  g_return_val_if_fail(regexp != NULL, FALSE);
  g_return_val_if_fail(str != NULL, FALSE);

  if (regexec(regexp, str, regexp->re_nsub + 1, pmatch, 0) == 0)
    {
      va_list ap;

      char **sub;
      int i;

      va_start(ap, str);
      
      for (i = 0; i < regexp->re_nsub; i++)
	{
	  int index = i + 1;

	  sub = va_arg(ap, char **);

	  g_return_val_if_fail(sub != NULL, FALSE);

	  *sub = g_strndup(&str[pmatch[index].rm_so],
			   pmatch[index].rm_eo - pmatch[index].rm_so);
	}
	  
      va_end(ap);
      
      return TRUE;
    }

  return FALSE;
}
