/*
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <string.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include "art/plugins_preferences.h"
#include "sg-util.h"
#include "sgtk-hig.h"
#include "st-plugins-tab.h"
#include "st-preferences-tab.h"
#include "st-plugins.h"
#include "st-settings.h"

/*** type definitions ********************************************************/

enum {
  /* invisible columns */
  COLUMN_FILENAME,

  /* visible columns */
  COLUMN_ENABLED, COLUMN_PLUGIN,
  COLUMN_PATH,

  N_COLUMNS
};

/*** function declarations ***************************************************/

static void st_plugins_tab_class_init (STPluginsTabClass *class);
static void st_plugins_tab_init (STPluginsTab *tab);

static gboolean st_plugins_tab_search_equal_func (GtkTreeModel *model,
						  int column,
						  const char *key,
						  GtkTreeIter *iter,
						  gpointer search_data);
static void st_plugins_tab_toggled_h (GtkCellRendererToggle *renderer,
				      const char *path,
				      gpointer user_data);

/*** implementation **********************************************************/

GType
st_plugins_tab_get_type (void)
{
  static GType plugins_tab_type = 0;
  
  if (! plugins_tab_type)
    {
      static const GTypeInfo plugins_tab_info = {
	sizeof(STPluginsTabClass),
	NULL,
	NULL,
	(GClassInitFunc) st_plugins_tab_class_init,
	NULL,
	NULL,
	sizeof(STPluginsTab),
	0,
	(GInstanceInitFunc) st_plugins_tab_init,
      };

      plugins_tab_type = g_type_register_static(ST_TYPE_PREFERENCES_TAB,
						"STPluginsTab",
						&plugins_tab_info,
						0);
    }

  return plugins_tab_type;
}

static void
st_plugins_tab_class_init (STPluginsTabClass *class)
{
  STPreferencesTabClass *preferences_tab_class = ST_PREFERENCES_TAB_CLASS(class);

  preferences_tab_class->link_id = "preferences-plugins";
}

static void
st_plugins_tab_init (STPluginsTab *tab)
{
  GdkPixbuf *pixbuf;
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;
  GtkWidget *scrolled;
  GtkListStore *store;
  GtkWidget *view;
  GSList *l;

  pixbuf = gdk_pixbuf_new_from_inline(sizeof(art_plugins_preferences),
				      art_plugins_preferences,
				      FALSE,
				      NULL);
  st_preferences_tab_construct_label(ST_PREFERENCES_TAB(tab), pixbuf, _("Plugins"));
  g_object_unref(pixbuf);

  gtk_container_set_border_width(GTK_CONTAINER(tab), SGTK_HIG_CONTROLS_BORDER_WIDTH);

  /* construct containers */

  scrolled = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);

  /* construct the view */
  
  store = gtk_list_store_new(N_COLUMNS,
			     G_TYPE_STRING,
			     G_TYPE_BOOLEAN,
			     G_TYPE_STRING,
			     G_TYPE_STRING);
  view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));

  /* construct the plugin column */

  column = gtk_tree_view_column_new();
  gtk_tree_view_column_set_title(column, _("Plugin"));
  gtk_tree_view_column_set_sort_column_id(column, COLUMN_PLUGIN);

  renderer = gtk_cell_renderer_toggle_new();
  g_signal_connect(renderer, "toggled", G_CALLBACK(st_plugins_tab_toggled_h), store);
  
  gtk_tree_view_column_pack_start(column, renderer, FALSE);
  gtk_tree_view_column_set_attributes(column, renderer, "active", COLUMN_ENABLED, NULL);

  renderer = gtk_cell_renderer_text_new();

  gtk_tree_view_column_pack_start(column, renderer, TRUE);
  gtk_tree_view_column_set_attributes(column, renderer, "text", COLUMN_PLUGIN, NULL);

  gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);

  /* construct the path column */

  column = gtk_tree_view_column_new_with_attributes(_("Path"),
						    gtk_cell_renderer_text_new(),
						    "text", COLUMN_PATH,
						    NULL);
  gtk_tree_view_column_set_sort_column_id(column, COLUMN_PATH);
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);

  /* widget setup and packing */

  gtk_tree_view_set_enable_search(GTK_TREE_VIEW(view), TRUE);
  gtk_tree_view_set_search_equal_func(GTK_TREE_VIEW(view),
				      st_plugins_tab_search_equal_func,
				      NULL,
				      NULL);

  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(store),
				       COLUMN_PLUGIN,
				       GTK_SORT_ASCENDING);

  gtk_container_add(GTK_CONTAINER(scrolled), view);
  gtk_container_add(GTK_CONTAINER(tab), scrolled);
  gtk_widget_show_all(scrolled);

  /* fill it */

  SG_LIST_FOREACH(l, st_plugins_list)
    {
      const char *filename = l->data;
      char *plugin;
      char *path;
      GtkTreeIter iter;

      plugin = g_path_get_basename(filename);
      path = g_path_get_dirname(filename);

      gtk_list_store_append(store, &iter);
      gtk_list_store_set(store, &iter,
			 COLUMN_FILENAME, filename,
			 COLUMN_ENABLED, ! sg_str_slist_contains(st_settings.disabled_plugins, filename),
			 COLUMN_PLUGIN, plugin,
			 COLUMN_PATH, path,
			 -1);

      g_free(plugin);
      g_free(path);
    }

  g_object_unref(store);
}

static gboolean
st_plugins_tab_search_equal_func (GtkTreeModel *model,
				  int column,
				  const char *key,
				  GtkTreeIter *iter,
				  gpointer search_data)
{
  char *plugin;
  char *path;
  gboolean equal = FALSE;

  gtk_tree_model_get(model, iter,
		     COLUMN_PLUGIN, &plugin,
		     COLUMN_PATH, &path,
		     -1);

  equal = sg_utf8_strcasecontains(plugin, key) || sg_utf8_strcasecontains(path, key);

  g_free(plugin);
  g_free(path);

  return ! equal;
}

static void
st_plugins_tab_toggled_h (GtkCellRendererToggle *renderer,
			  const char *path,
			  gpointer user_data)
{
  GtkTreeModel *model = user_data;
  char *filename;
  GtkTreeIter iter;
  gboolean status;
  GSList *l;
  gboolean disabled = FALSE;
  
  status = gtk_tree_model_get_iter_from_string(model, &iter, path);
  g_return_if_fail(status == TRUE);

  gtk_tree_model_get(model, &iter, COLUMN_FILENAME, &filename, -1);

  SG_LIST_FOREACH(l, st_settings.disabled_plugins)
    {
      char *disabled_filename = l->data;

      if (! strcmp(disabled_filename, filename))
	{
	  disabled = TRUE;
	  g_free(disabled_filename);
	  st_settings.disabled_plugins = g_slist_delete_link(st_settings.disabled_plugins, l);
	  break;
	}
    }

  if (! disabled)
    st_settings.disabled_plugins = g_slist_append(st_settings.disabled_plugins, filename);
  else
    g_free(filename);

  gtk_list_store_set(GTK_LIST_STORE(model), &iter, COLUMN_ENABLED, disabled, -1);
}

GtkWidget *
st_plugins_tab_new (void)
{
  return g_object_new(ST_TYPE_PLUGINS_TAB, NULL);
}
