/*
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <string.h>
#include <stdlib.h>
#include <glib/gi18n.h>
#include "streamtuner.h"
#include "art/shoutcast.h"

#define STREAMS_PER_PAGE		100

#define SHOUTCAST_ROOT			"http://www.shoutcast.com/"

/*** type definitions ********************************************************/

typedef struct
{
  STStream	stream;

  char		*genre;
  char		*description;
  char		*now_playing;
  int		listeners;
  int		max;
  int		bitrate;
  char		*url_postfix;
  char		*homepage;

  GSList	*url_list;       /* will be read and written in threads... */
  GMutex	*url_list_mutex; /* ...so we protect it                    */
  /*
   * We don't protect the other stream members because they are
   * guaranteed to never be accessed concurrently
   * (ST_HANDLER_EVENT_STREAM_FIELD_SET isn't a problem because it is
   * only called once, at streamtuner initialization time).
   */
} SHOUTcastStream;

enum {
  FIELD_GENRE,
  FIELD_DESCRIPTION,
  FIELD_NOW_PLAYING,
  FIELD_LISTENERS,
  FIELD_MAX,
  FIELD_BITRATE,
  FIELD_URL_POSTFIX,
  FIELD_HOMEPAGE,
  FIELD_URL_LIST
};

typedef struct
{
  STCategory		*category;
  GNode			**categories;
  GList			**streams;

  int			page;
  int			npages;

  char			*charset;
  GNode			*parent_node;
  SHOUTcastStream	*stream;

  STTransferSession	*session;
} ReloadInfo;

/*** variable declarations ***************************************************/

static STHandler *handler = NULL;

static regex_t re_header_charset;
static regex_t re_body_charset;
static regex_t re_genre;
static regex_t re_playing;
static regex_t re_listeners;
static regex_t re_bitrate;
static regex_t re_stream;
static regex_t re_category;
static regex_t re_page;

/*** function declarations ***************************************************/

static gboolean	reload_cb	(STCategory	*category,
				 GNode		**categories,
				 GList		**streams,
				 gpointer	data,
				 GError		**err);

static gboolean	reload_page		(const char	*url,
					 ReloadInfo	*info,
					 GError		**err);
static void	reload_header_cb	(const char	*line,
					 gpointer	data);
static void	reload_body_cb		(const char	*line,
					 gpointer	data);

static SHOUTcastStream *stream_new_cb (gpointer data);
static void stream_field_get_cb (SHOUTcastStream	*stream,
				 STHandlerField		*field,
				 GValue			*value,
				 gpointer		data);
static void stream_field_set_cb (SHOUTcastStream	*stream,
				 STHandlerField		*field,
				 const GValue		*value,
				 gpointer		data);
static void stream_stock_field_get_cb (SHOUTcastStream		*stream,
				       STHandlerStockField	stock_field,
				       GValue			*value,
				       gpointer			data);
static void stream_free_cb	(SHOUTcastStream	*stream,
				 gpointer		data);

static gboolean stream_resolve		(SHOUTcastStream	*stream,
					 GError			**err);

static gboolean stream_resolve_cb	(SHOUTcastStream	*stream,
					 gpointer		data,
					 GError			**err);
static gboolean	stream_tune_in_cb	(SHOUTcastStream	*stream,
					 gpointer		data,
					 GError			**err);
static gboolean	stream_record_cb	(SHOUTcastStream	*stream,
					 gpointer		data,
					 GError			**err);
static gboolean	stream_browse_cb	(SHOUTcastStream	*stream,
					 gpointer		data,
					 GError			**err);

static int	search_url_cb		(STCategory		*category);

static gboolean	init_re		(void);
static void	init_handler	(void);

/*** implementation **********************************************************/

static SHOUTcastStream *
stream_new_cb (gpointer data)
{
  SHOUTcastStream *stream;

  stream = g_new0(SHOUTcastStream, 1);
  stream->url_list_mutex = g_mutex_new();

  return stream;
}

static void
stream_field_get_cb (SHOUTcastStream *stream,
		     STHandlerField *field,
		     GValue *value,
		     gpointer data)
{
  switch (field->id)
    {
    case FIELD_GENRE:
      g_value_set_string(value, stream->genre);
      break;

    case FIELD_DESCRIPTION:
      g_value_set_string(value, stream->description);
      break;

    case FIELD_NOW_PLAYING:
      g_value_set_string(value, stream->now_playing);
      break;

    case FIELD_LISTENERS:
      g_value_set_int(value, stream->listeners);
      break;

    case FIELD_MAX:
      g_value_set_int(value, stream->max);
      break;

    case FIELD_BITRATE:
      g_value_set_int(value, stream->bitrate);
      break;

    case FIELD_URL_POSTFIX:
      g_value_set_string(value, stream->url_postfix);
      break;

    case FIELD_HOMEPAGE:
      g_value_set_string(value, stream->homepage);
      break;

    case FIELD_URL_LIST:
      {
	GValueArray *value_array = g_value_array_new(0);
	GSList *l;

	g_mutex_lock(stream->url_list_mutex);
	for (l = stream->url_list; l; l = l->next)
	  {
	    GValue url_value = { 0, };

	    g_value_init(&url_value, G_TYPE_STRING);
	    g_value_set_string(&url_value, l->data);
	    g_value_array_append(value_array, &url_value);
	    g_value_unset(&url_value);
	  }
	g_mutex_unlock(stream->url_list_mutex);

	g_value_set_boxed_take_ownership(value, value_array);
	break;
      }
      
    default:
      g_return_if_reached();
    }
}

static void
stream_field_set_cb (SHOUTcastStream *stream,
		     STHandlerField *field,
		     const GValue *value,
		     gpointer data)
{
  switch (field->id)
    {
    case FIELD_GENRE:
      stream->genre = g_value_dup_string(value);
      break;

    case FIELD_DESCRIPTION:
      stream->description = g_value_dup_string(value);
      break;

    case FIELD_NOW_PLAYING:
      stream->now_playing = g_value_dup_string(value);
      break;

    case FIELD_LISTENERS:
      stream->listeners = g_value_get_int(value);
      break;

    case FIELD_MAX:
      stream->max = g_value_get_int(value);
      break;

    case FIELD_BITRATE:
      stream->bitrate = g_value_get_int(value);
      break;

    case FIELD_URL_POSTFIX:
      stream->url_postfix = g_value_dup_string(value);
      break;

    case FIELD_HOMEPAGE:
      stream->homepage = g_value_dup_string(value);
      break;

    case FIELD_URL_LIST:
      {
	GValueArray *value_array;
	int i;

	value_array = g_value_get_boxed(value);

	g_mutex_lock(stream->url_list_mutex);
	for (i = 0; i < value_array->n_values; i++)
	  {
	    GValue *url_value = g_value_array_get_nth(value_array, i);
	    stream->url_list = g_slist_append(stream->url_list, g_value_dup_string(url_value));
	  }
	g_mutex_unlock(stream->url_list_mutex);

	break;
      }
      
    default:
      g_return_if_reached();
    }
}

static void
stream_stock_field_get_cb (SHOUTcastStream *stream,
			   STHandlerStockField stock_field,
			   GValue *value,
			   gpointer data)
{
  switch (stock_field)
    {
    case ST_HANDLER_STOCK_FIELD_NAME:
      g_value_set_string(value, stream->description);
      break;

    case ST_HANDLER_STOCK_FIELD_GENRE:
      g_value_set_string(value, stream->genre);
      break;

    default:
      /*
       * This can not happen.
       *
       * However, handlers not shipped with streamtuner MUST ignore
       * unknown stock fields, to stay compatible with future versions
       * of streamtuner.
       */
      g_return_if_reached();
    }
}

static void
stream_free_cb (SHOUTcastStream *stream, gpointer data)
{
  GSList *l;

  g_free(stream->genre);
  g_free(stream->description);
  g_free(stream->now_playing);
  g_free(stream->url_postfix);
  g_free(stream->homepage);

  for (l = stream->url_list; l; l = l->next)
    g_free(l->data);

  g_slist_free(stream->url_list);
  g_mutex_free(stream->url_list_mutex);

  st_stream_free((STStream *) stream);
}

static gboolean
stream_resolve (SHOUTcastStream *stream, GError **err)
{
  gboolean already_resolved;
  STTransferSession *session;
  char *url;
  char *playlist;
  gboolean status;

  g_return_val_if_fail(stream != NULL, FALSE);

  g_mutex_lock(stream->url_list_mutex);
  already_resolved = stream->url_list != NULL;
  g_mutex_unlock(stream->url_list_mutex);

  if (already_resolved)
    return TRUE;		/* already resolved */

  url = g_strconcat(SHOUTCAST_ROOT, stream->url_postfix, NULL);
  session = st_transfer_session_new();
  status = st_transfer_session_get(session, url, 0, NULL, &playlist, err);
  st_transfer_session_free(session);
  g_free(url);

  if (status)
    {
      gboolean empty;

      g_mutex_lock(stream->url_list_mutex);
      stream->url_list = st_pls_parse(playlist);
      empty = stream->url_list == NULL;
      g_mutex_unlock(stream->url_list_mutex);

      g_free(playlist);

      if (empty)
	{
	  g_set_error(err, 0, 0, _("stream is empty"));
	  return FALSE;
	}
    }
  
  return status;
}

static gboolean
stream_resolve_cb (SHOUTcastStream *stream, gpointer data, GError **err)
{
  return stream_resolve(stream, err);
}

static gboolean
stream_tune_in_cb (SHOUTcastStream *stream, gpointer data, GError **err)
{
  char *m3uname;
  gboolean status;
  
  if (! stream_resolve(stream, err))
    return FALSE;

  g_mutex_lock(stream->url_list_mutex);
  m3uname = st_m3u_mktemp("streamtuner.shoutcast.XXXXXX", stream->url_list, err);
  g_mutex_unlock(stream->url_list_mutex);

  if (! m3uname)
    return FALSE;
  
  status = st_action_run("play-m3u", m3uname, err);
  g_free(m3uname);

  return status;
}

static gboolean
stream_record_cb (SHOUTcastStream *stream, gpointer data, GError **err)
{
  gboolean status;

  if (! stream_resolve(stream, err))
    return FALSE;

  g_mutex_lock(stream->url_list_mutex);
  status = st_action_run("record-stream", stream->url_list->data, err);
  g_mutex_unlock(stream->url_list_mutex);

  return status;
}

static gboolean
stream_browse_cb (SHOUTcastStream *stream, gpointer data, GError **err)
{
  if (! stream->homepage) /* older versions of the plugin didn't have this field */
    {
      g_set_error(err, 0, 0, _("the stream is too old, please reload"));
      return FALSE;
    }

  return st_action_run("view-web", stream->homepage, err);
}

static gboolean
reload_cb (STCategory *category,
	   GNode **categories,
	   GList **streams,
	   gpointer data,
	   GError **err)
{
  ReloadInfo info;
  char *url;
  gboolean status;

  g_return_val_if_fail(category != NULL, FALSE);
  g_return_val_if_fail(category->url_postfix != NULL, FALSE);

  *categories = g_node_new(NULL);
  *streams = NULL;

  info.category = category;
  info.categories = categories;
  info.streams = streams;

  info.session = st_transfer_session_new();

  url = g_strdup_printf(SHOUTCAST_ROOT "directory/?numresult=%i%s",
			STREAMS_PER_PAGE, category->url_postfix);
  status = reload_page(url, &info, err);
  g_free(url);

  if (! status)
    goto end;

  /*
   * The main category contains _all_ SHOUTcast streams (a lot), so we
   * just load the first page.
   */
  if (strcmp(category->name, "__main") != 0)
    while (info.page > 0 && info.page < info.npages)
      {
	if (st_is_aborted())
	  {
	    status = FALSE;
	    goto end;
	  }

	url = g_strdup_printf(SHOUTCAST_ROOT "directory/index.phtml?startat=%i",
			      info.page * STREAMS_PER_PAGE);
	status = reload_page(url, &info, err);
	g_free(url);
	
	if (! status)
	  goto end;
      }
  
 end:
  st_transfer_session_free(info.session);
  
  return status;
}

static gboolean
reload_page (const char *url, ReloadInfo *info, GError **err)
{
  gboolean status;
  GError *tmp_err = NULL;

  info->page = 0;
  info->npages = 0;

  info->charset = NULL;
  info->parent_node = NULL;
  info->stream = NULL;

  status = st_transfer_session_get_by_line(info->session,
					   url,
					   0,
					   reload_header_cb,
					   info,
					   reload_body_cb,
					   info,
					   &tmp_err);

  g_free(info->charset);

  if (info->stream)
    {
      stream_free_cb(info->stream, NULL);
      if (status) /* only display warning if the transfer was otherwise correct */
	st_handler_notice(handler, _("EOF: found unterminated stream"));
    }

  if (! status)
    {
      if (tmp_err)
	{
	  g_set_error(err, 0, 0, _("unable to transfer: %s"), tmp_err->message);
	  g_error_free(tmp_err);
	}
      
      return FALSE;
    }
  
  return TRUE;
}

static void
reload_header_cb (const char *line, gpointer data)
{
  ReloadInfo *info = data;
  char *charset;

  if (st_re_parse(&re_header_charset, line, &charset))
    {
      g_free(info->charset);
      info->charset = charset;
    }
}

static void
reload_body_cb (const char *line, gpointer data)
{
  ReloadInfo *info = data;
  char *sub1, *sub2, *sub3;
  char *converted = NULL;
  char *expanded;

  if (! info->charset && ! strncasecmp(line, "</head>", 6))
    info->charset = g_strdup("ISO8859-1"); /* we got no charset info, use a fallback */
  else
    {
      char *charset;

      if (st_re_parse(&re_body_charset, line, &charset))
	{			/* body charset overrides header charset */
	  g_free(info->charset);
	  info->charset = charset;
	}
    }
  
  if (info->charset)
    {
      GError *convert_err = NULL;
      
      if ((converted = g_convert(line,
				 strlen(line),
				 "UTF-8",
				 info->charset,
				 NULL,
				 NULL,
				 &convert_err)))
	line = converted;
      else
	{
	  st_handler_notice(handler, _("unable to convert line to UTF-8 encoding: %s"),
			    convert_err->message);
	  
	  g_error_free(convert_err);
	  return;
	}
    }
  
  expanded = st_sgml_ref_expand(line);
  line = expanded;

  if (st_re_parse(&re_stream, line, &sub1))
    {
      if (info->stream)	/* a malformed stream remains, free it */
	{
	  st_handler_notice(handler, _("found unterminated stream"));
	  stream_free_cb(info->stream, NULL);
	}
	  
      info->stream = stream_new_cb(NULL);
      info->stream->url_postfix = sub1;
    }
  else if (st_re_parse(&re_bitrate, line, &sub1))
    {
      if (info->stream)
	{
	  info->stream->bitrate = atoi(sub1);
	  
	  if (info->stream->genre && info->stream->description && info->stream->homepage)
	    {
	      ((STStream *) info->stream)->name =
		g_strdup_printf("%s%s%i",
				info->stream->genre,
				info->stream->description,
				info->stream->bitrate);
	      
	      *(info->streams) = g_list_append(*(info->streams), info->stream);
	    }
	  else
	    {
	      st_handler_notice(handler, _("found incomplete stream"));
	      stream_free_cb(info->stream, NULL);
	    }

	  info->stream = NULL;
	}
      else
	{
	  st_handler_notice(handler, _("found misplaced bitrate"));
	  g_free(sub1);
	}
    }
  else if (info->page < 2
	   && st_re_parse(&re_category, line, &sub1, &sub2)
	   && strcmp(sub1, "TopTen") != 0)
    {
      STCategory *category;
      GNode *node;
      char *escaped;
      
      category = st_category_new();
      
      category->name = sub1;
      category->label = sub2;
      
      escaped = st_transfer_escape(sub1);
      category->url_postfix = g_strconcat("&sgenre=", escaped, NULL);
      g_free(escaped);
      
      node = g_node_new(category);
      
      if (! strncmp(category->label, " - ", 3))
	{
	  if (info->parent_node)
	    {
	      category->label = g_strdup(&sub2[3]);
	      g_free(sub2);
	      
	      g_node_append(info->parent_node, node);
	    }
	  else
	    {
	      st_handler_notice(handler, _("parent category not found"));
	      st_category_free(category);
	      g_node_destroy(node);
	    }
	}
      else
	{
	  g_node_append(*(info->categories), node);
	  info->parent_node = node;
	}
    }
  else if (st_re_parse(&re_genre, line, &sub1, &sub2, &sub3))
    {
      if (info->stream)
	{
	  info->stream->genre = sub1;
	  info->stream->homepage = sub2;
	  info->stream->description = sub3;
	}
      else
	{
	  st_handler_notice(handler, _("found misplaced genre, homepage and description"));
	  g_free(sub1);
	  g_free(sub2);
	  g_free(sub3);
	}
    }
  else if (st_re_parse(&re_playing, line, &sub1))
    {
      if (info->stream)
	info->stream->now_playing = sub1;
      else
	{
	  st_handler_notice(handler, _("found misplaced now_playing"));
	  g_free(sub1);
	}
    }
  else if (st_re_parse(&re_listeners, line, &sub1, &sub2))
    {
      if (info->stream)
	{
	  info->stream->listeners = atoi(sub1);
	  info->stream->max = atoi(sub2);
	}
      else
	{
	  st_handler_notice(handler, _("found misplaced listeners and max"));
	  g_free(sub1);
	  g_free(sub2);
	}
    }
  else if (st_re_parse(&re_page, line, &sub1, &sub2))
    {
      info->page = atoi(sub1);
      info->npages = atoi(sub2);

      g_free(sub1);
      g_free(sub2);
    }
  
  g_free(converted);
  g_free(expanded);
}

static gboolean
search_url_cb (STCategory *category)
{
  char *str;

  str = st_search_dialog();
  if (str)
    {
      char *escaped;

      g_free(category->label);
      category->label = g_strdup_printf(_("Search results for \"%s\""), str);

      escaped = st_transfer_escape(str);
      g_free(str);

      g_free(category->url_postfix);
      category->url_postfix = g_strconcat("&s=", escaped, NULL);
      g_free(escaped);

      return TRUE;
    }
  else
    return FALSE;
}

static gboolean
init_re (void)
{
  int status;

  status = regcomp(&re_header_charset, "^Content-Type: .*charset=(.*)", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_body_charset, "<meta http-equiv=.* content=.*charset=(.*)\"", REG_EXTENDED | REG_ICASE);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_genre, "^      <td .*<b>\\[(.*)\\]</font>.*<a .*href=\"(.*)\">(.*)</a>", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_playing, "^<font .*>Now Playing:</font>(.*)</font></font></td>", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_listeners, "      <td .*>([0-9]+)/([0-9]+)</font>", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_bitrate, "^      <td .*>([0-9]+)</font>", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_stream, "<a href=\"/(sbin/shoutcast-playlist.pls.*filename.pls)\"", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_category, "^\t\t?<OPTION VALUE=\"(.*)\">(.*)$", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_page, "^Page ([0-9]+) of ([0-9]+)", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);

  return TRUE;
}

static void
init_handler (void)
{
  GNode *stock_categories;
  STCategory *category;

  stock_categories = g_node_new(NULL);

  category = st_category_new();
  category->name = "__main";
  category->label = _("Top streams");
  category->url_postfix = "&sgenre=TopTen";
  
  g_node_append_data(stock_categories, category);
  
  category = st_category_new();
  category->name = "__search";
  category->label = g_strdup(_("Search"));
  category->url_cb = search_url_cb;

  g_node_append_data(stock_categories, category);

  handler = st_handler_new("shoutcast");

  st_handler_set_label(handler, "SHOUTcast");
  st_handler_set_copyright(handler, "Copyright \302\251 2002, 2003, 2004 Jean-Yves Lefort");
  st_handler_set_description(handler, _("SHOUTcast Yellow Pages"));
  st_handler_set_home(handler, "http://www.shoutcast.com/");
  st_handler_set_icon_from_inline(handler, sizeof(art_shoutcast), art_shoutcast);
  st_handler_set_stock_categories(handler, stock_categories);
  
  st_handler_bind(handler, ST_HANDLER_EVENT_RELOAD, reload_cb, NULL);

  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_NEW, stream_new_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FIELD_GET, stream_field_get_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FIELD_SET, stream_field_set_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_STOCK_FIELD_GET, stream_stock_field_get_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FREE, stream_free_cb, NULL);

  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_RESOLVE, stream_resolve_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_TUNE_IN, stream_tune_in_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_RECORD, stream_record_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_BROWSE, stream_browse_cb, NULL);

  st_handler_add_field(handler, st_handler_field_new(FIELD_GENRE,
						     _("Genre"),
						     G_TYPE_STRING,
						     ST_HANDLER_FIELD_VISIBLE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_DESCRIPTION,
						     _("Description"),
						     G_TYPE_STRING,
						     ST_HANDLER_FIELD_VISIBLE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_NOW_PLAYING,
						     _("Now playing"),
						     G_TYPE_STRING,
						     ST_HANDLER_FIELD_VISIBLE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_LISTENERS,
						     _("Listeners"),
						     G_TYPE_INT,
						     ST_HANDLER_FIELD_VISIBLE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_MAX,
						     _("Max"),
						     G_TYPE_INT,
						     ST_HANDLER_FIELD_VISIBLE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_BITRATE,
						     _("Bitrate"),
						     G_TYPE_INT,
						     ST_HANDLER_FIELD_VISIBLE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_URL_POSTFIX,
						     _("URL postfix"),
						     G_TYPE_STRING,
						     0));
  st_handler_add_field(handler, st_handler_field_new(FIELD_HOMEPAGE,
						     _("Homepage"),
						     G_TYPE_STRING,
						     ST_HANDLER_FIELD_VISIBLE
						     | ST_HANDLER_FIELD_START_HIDDEN));
  st_handler_add_field(handler, st_handler_field_new(FIELD_URL_LIST,
						     _("URL list"),
						     G_TYPE_VALUE_ARRAY,
						     ST_HANDLER_FIELD_VISIBLE
						     | ST_HANDLER_FIELD_START_HIDDEN));

  st_handlers_add(handler);
}

gboolean
plugin_init (GError **err)
{
  gboolean status;
  
  if (! st_check_api_version(5, 7))
    {
      g_set_error(err, 0, 0, _("API version mismatch"));
      return FALSE;
    }
  
  status = init_re();
  g_return_val_if_fail(status == TRUE, FALSE);
  
  init_handler();
  
  st_action_register("play-m3u", _("Listen to a .m3u file"), "xmms %q");
  st_action_register("record-stream", _("Record a stream"), "xterm -hold -e streamripper %q");
  st_action_register("view-web", _("Open a web page"), "epiphany %q");

  return TRUE;
}
