/*
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <Python.h>
#include <glib/gi18n.h>
#include "streamtuner.h"
#include "pst-stream.h"
#include "pst-handler-field.h"
#include "pst-helpers.h"
#include "pst-transfer-session.h"

struct _PythonStream
{
  STStream	stream;
  PSTStream	*p;
};

/*** attributes **************************************************************/

static PyObject *pst_stream_get_name (PSTStream *self, void *closure);
static int pst_stream_set_name (PSTStream *self, PyObject *value, void *closure);

static PyGetSetDef getsetters[] = {
  { "name", (getter) pst_stream_get_name, (setter) pst_stream_set_name },
  
  { NULL }
};

/*** methods *****************************************************************/

static PyObject *pst_stream_set_field (PSTStream *self, PyObject *args);
static PyObject *pst_stream_get_field (PSTStream *self, PyObject *args);

static PyMethodDef methods[] = {
  { "set_field", (PyCFunction) pst_stream_set_field, METH_VARARGS },
  { "get_field", (PyCFunction) pst_stream_get_field, METH_VARARGS },
  
  { NULL }
};

/*** type object *************************************************************/

static PyObject *pst_stream_new (PyTypeObject *type, PyObject *args, PyObject *keywords);
static void pst_stream_dealloc (PSTStream *self);

PyTypeObject PSTStream_Type = {
  PyObject_HEAD_INIT(NULL)
  0,				/* ob_size */
  "ST.Stream",			/* tp_name */
  sizeof(PSTStream),		/* tp_basicsize */
  0,				/* tp_itemsize */
  (destructor) pst_stream_dealloc, /* tp_dealloc */
  NULL,				/* tp_print */
  NULL,				/* tp_getattr */
  NULL,				/* tp_setattr */
  NULL,				/* tp_compare */
  NULL,				/* tp_repr */
  NULL,				/* tp_as_number */
  NULL,				/* tp_as_sequence */
  NULL,				/* tp_as_mapping */
  NULL,				/* tp_hash */
  NULL,				/* tp_call */
  NULL,				/* tp_str */
  NULL,				/* tp_getattro */
  NULL,				/* tp_setattro */
  NULL,				/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /* tp_flags */
  NULL,				/* tp_doc */
  NULL,				/* tp_traverse */
  NULL,				/* tp_clear */
  NULL,				/* tp_richcompare */
  0,				/* tp_weaklistoffset */
  NULL,				/* tp_iter */
  NULL,				/* tp_iternext */
  methods,			/* tp_methods */
  NULL,				/* tp_members */
  getsetters,			/* tp_getset */
  NULL,				/* tp_base */
  NULL,				/* tp_dict */
  NULL,				/* tp_descr_get */
  NULL,				/* tp_descr_set */
  0,				/* tp_dictoffset */
  NULL,				/* tp_init */
  NULL,				/* tp_alloc */
  pst_stream_new		/* tp_new */
};

/*** function declarations ***************************************************/

static void pst_stream_construct (PSTStream *self);

/*** type methods ************************************************************/

static PyObject *
pst_stream_new (PyTypeObject *type, PyObject *args, PyObject *keywords)
{
  static char *keyword_list[] = { NULL };
  PSTStream *self;

  if (! PyArg_ParseTupleAndKeywords(args, keywords, "", keyword_list))
    return NULL;

  self = (PSTStream *) type->tp_alloc(type, 0);
  pst_stream_construct(self);

  return (PyObject *) self;
}

static void
pst_stream_construct (PSTStream *self)
{
  self->stream = g_new0(PythonStream, 1);
  self->stream->p = self;
  self->fields = PyDict_New();
}

static void
pst_stream_dealloc (PSTStream *self)
{
  st_stream_free((STStream *) self->stream);
  Py_DECREF(self->fields);

  self->ob_type->tp_free((PyObject *) self);
}

static PyObject *
pst_stream_get_name (PSTStream *self, void *closure)
{
  return pst_string_getter(((STStream *) self->stream)->name);
}

static int
pst_stream_set_name (PSTStream *self, PyObject *value, void *closure)
{
  return pst_string_setter(&((STStream *) self->stream)->name, value);
}

/*** object methods **********************************************************/

static PyObject *
pst_stream_set_field (PSTStream *self, PyObject *args)
{
  PyObject *pid;
  PyObject *pvalue;

  if (! PyArg_ParseTuple(args, "OO", &pid, &pvalue))
    return NULL;

  PyDict_SetItem(self->fields, pid, pvalue);

  return pst_none();
}

static PyObject *
pst_stream_get_field (PSTStream *self, PyObject *args)
{
  PyObject *pid;
  PyObject *pvalue;

  if (! PyArg_ParseTuple(args, "O", &pid))
    return NULL;

  pvalue = PyDict_GetItem(self->fields, pid);
  if (! pvalue)
    {
      PyErr_SetString(PyExc_LookupError, _("no such field"));
      return NULL;
    }

  Py_INCREF(pvalue);
  return pvalue;
}

/*** streamtuner callbacks ***************************************************/

PythonStream *
pst_stream_new_cb (gpointer data)
{
  PSTStream *p;

  p = PyObject_New(PSTStream, &PSTStream_Type);
  pst_stream_construct(p);

  return p->stream;
}

void
pst_stream_field_set_cb (PythonStream *stream,
			 STHandlerField *field,
			 const GValue *value,
			 gpointer data)
{
  PyObject *pid;
  PyObject *pvalue;

  pid = PyInt_FromLong(field->id);
  pvalue = pst_object_from_gvalue(value);

  PyDict_SetItem(stream->p->fields, pid, pvalue);

  Py_DECREF(pid);
  Py_DECREF(pvalue);
}

void
pst_stream_field_get_cb (PythonStream *stream,
			 STHandlerField *field,
			 GValue *value,
			 gpointer data)
{
  PyObject *pid;
  PyObject *pvalue;

  pid = PyInt_FromLong(field->id);
  pvalue = PyDict_GetItem(stream->p->fields, pid);
  Py_DECREF(pid);

  if (pvalue)
    {
      GValue item = { 0, };
      
      Py_INCREF(pvalue);
      pst_object_as_gvalue(pvalue, &item);
      Py_DECREF(pvalue);

      g_value_copy(&item, value);
      g_value_unset(&item);
    }
}

void
pst_stream_free_cb (PythonStream *stream, gpointer data)
{
  Py_DECREF(stream->p);
}

gboolean
pst_stream_cb (PythonStream *stream, PSTCallbackInfo *info, GError **err)
{
  PyObject *presult;

  /* might have been aborted while we were waiting for the GIL */
  if (st_is_aborted())
    return FALSE;

  presult = PyObject_CallMethod(info->object, info->method, "O", stream->p);
  if (! presult)
    {
      if (! PyErr_ExceptionMatches(PyExc_AbortError))
	pst_set_error(err);
      return FALSE;
    }
  Py_DECREF(presult);
  
  return TRUE;
}

gboolean
pst_stream_tune_in_multiple_cb (GSList *streams, PSTCallbackInfo *info, GError **err)
{
  PyObject *tuple;
  PyObject *presult;
  GSList *l;
  int i = 0;

  /* might have been aborted while we were waiting for the GIL */
  if (st_is_aborted())
    return FALSE;

  tuple = PyTuple_New(g_slist_length(streams));
  for (l = streams; l; l = l->next)
    {
      PythonStream *stream = l->data;
      
      Py_INCREF(stream->p); /* because PyTuple_SetItem() steals a ref */
      PyTuple_SetItem(tuple, i++, (PyObject *) stream->p);
    }
  
  presult = PyObject_CallMethod(info->object, info->method, "O", tuple);
  Py_DECREF(tuple);

  if (! presult)
    {
      if (! PyErr_ExceptionMatches(PyExc_AbortError))
	pst_set_error(err);
      return FALSE;
    }
  Py_DECREF(presult);
  
  return TRUE;
}

void
pst_stream_stock_field_get_cb (PythonStream *stream,
			       STHandlerStockField stock_field,
			       GValue *value,
			       PSTCallbackInfo *info)
{
  PyObject *presult;

  presult = PyObject_CallMethod(info->object, info->method,
				"Oi", stream->p, stock_field);
  if (! presult)
    {
      PyErr_Print();
      return;
    }

  if (presult != Py_None)
    switch (stock_field)
      {
      case ST_HANDLER_STOCK_FIELD_NAME:
      case ST_HANDLER_STOCK_FIELD_GENRE:
	g_value_set_string(value, PyString_AsString(presult));
	break;
      }

  Py_DECREF(presult);
}

gboolean
pst_stream_modify_cb (PythonStream *stream,
		      GSList *fields,
		      GSList *values,
		      PSTCallbackInfo *info,
		      GError **err)
{
  GSList *f;
  GSList *v;
  PyObject *tuple;
  int i = 0;
  PyObject *presult;

  tuple = PyTuple_New(MIN(g_slist_length(fields), g_slist_length(values)));

  for (f = fields, v = values; f && v; f = f->next, v = v->next)
    {
      PyObject *field = PSTHandlerField_FromField(f->data);
      PyObject *value = pst_object_from_gvalue(v->data);
      PyObject *item = PyTuple_New(2);

      PyTuple_SetItem(item, 0, field);
      PyTuple_SetItem(item, 1, value);

      PyTuple_SetItem(tuple, i++, item);
    }

  presult = PyObject_CallMethod(info->object, info->method,
				"OO", stream->p, tuple);
  Py_DECREF(tuple);

  if (! presult)
    {
      pst_set_error(err);
      return FALSE;
    }

  Py_DECREF(presult);
  return TRUE;
}

/*** C API *******************************************************************/

gboolean
pst_stream_register (PyObject *module)
{
  g_return_val_if_fail(module != NULL, FALSE);

  if (PyType_Ready(&PSTStream_Type) < 0)
    return FALSE;

  Py_INCREF(&PSTStream_Type);
  PyModule_AddObject(module, "Stream", (PyObject *) &PSTStream_Type);

  return TRUE;
}

PythonStream *
pst_stream_copy (PythonStream *stream)
{
  PSTStream *p;

  g_return_val_if_fail(stream != NULL, NULL);

  p = PyObject_New(PSTStream, &PSTStream_Type);
  pst_stream_construct(p);

  ((STStream *) p->stream)->name = g_strdup(((STStream *) stream)->name);
  PyDict_Update(p->fields, stream->p->fields);

  return p->stream;
}
