/*
 * $Id: st-stream-task.c,v 1.10.2.3 2004/08/05 12:46:36 jylefort Exp $
 *
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "gettext.h"
#include "sg-util.h"
#include "st-dialog-api.h"
#include "st-shell.h"
#include "st-bookmarks.h"
#include "st-handler.h"
#include "st-stream-task.h"
#include "st-thread.h"

/*** type definitions ********************************************************/

typedef struct
{
  STShell		*shell;
  STHandler		*handler;
  STStreamTask		task;
  GSList		*streams;
  STThread		*thread;
} TaskInfo;

/*** function declarations ***************************************************/

static void st_stream_task_thread (gpointer data);
static void st_stream_task_add_bookmarks_thread (gpointer data);
static void st_stream_task_abort_cb (gpointer data);
static void st_stream_task_finish (TaskInfo *info);
static void st_stream_task_free_info (TaskInfo *info);

/*** implementation **********************************************************/

static void
st_stream_task_thread (gpointer data)
{
  TaskInfo *info = data;
  GSList *l;
  gboolean status;
  GError *err = NULL;

  /* if there's no multiple callback, only keep the first stream */

  if (info->task != ST_STREAM_TASK_TUNE_IN || ! st_handler_event_is_bound(info->handler, ST_HANDLER_EVENT_STREAM_TUNE_IN_MULTIPLE))
    {
      sg_objects_free(info->streams->next);
      info->streams->next = NULL;
    }

  /* set running */

  GDK_THREADS_ENTER();
  SG_LIST_FOREACH(l, info->streams)
    st_stream_bag_set_running(l->data, TRUE);
  gdk_flush();
  GDK_THREADS_LEAVE();

  /* run the appropriate callback */

  if (info->task == ST_STREAM_TASK_TUNE_IN)
    {
      if (st_handler_event_is_bound(info->handler, ST_HANDLER_EVENT_STREAM_TUNE_IN_MULTIPLE))
	status = st_stream_bag_tune_in_multiple(info->streams, &err);
      else
	status = st_stream_bag_tune_in(info->streams->data, &err);
    }
  else if (info->task == ST_STREAM_TASK_RECORD)
    status = st_stream_bag_record(info->streams->data, &err);
  else if (info->task == ST_STREAM_TASK_BROWSE)
    status = st_stream_bag_browse(info->streams->data, &err);
  else
    g_return_if_reached();

  /* finish */

  st_thread_validate_callback_return(info->thread, status, err);
  if (! st_thread_is_aborted(info->thread))
    {
      GDK_THREADS_ENTER();
      st_stream_task_finish(info);
      gdk_flush();
      GDK_THREADS_LEAVE();
    }

  if (err)
    {
      static const char *primary[] = {
	N_("Unable to tune in"),
	N_("Unable to record"),
	N_("Unable to browse")
      };
      char *normalized;
  
      normalized = st_dialog_normalize(err->message);
      st_error_dialog(_(primary[info->task]), "%s", normalized);
      
      g_free(normalized);
      g_error_free(err);
    }
  
  st_stream_task_free_info(info);
}

static void
st_stream_task_add_bookmarks_thread (gpointer data)
{
  TaskInfo *info = data;
  GSList *l;
  
  SG_LIST_FOREACH(l, info->streams)
    {
      STStreamBag *bag = l->data;
      GError *err = NULL;
      gboolean status = TRUE;
      
      GDK_THREADS_ENTER();
      st_stream_bag_set_running(bag, TRUE);
      gdk_flush();
      GDK_THREADS_LEAVE();

      if (st_handler_event_is_bound(info->handler, ST_HANDLER_EVENT_STREAM_RESOLVE))
	{
	  status = st_stream_bag_resolve(bag, &err);
	  st_thread_validate_callback_return(info->thread, status, err);

	  if (err)
	    g_error_free(err);	/* do not display error */
	}
      
      if (st_thread_is_aborted(info->thread))
	break;
      else
	{
	  GDK_THREADS_ENTER();
	  st_stream_bag_set_running(bag, FALSE);
	  st_bookmarks_add(bag);
	  gdk_flush();
	  GDK_THREADS_LEAVE();
	}
    }

  if (! st_thread_is_aborted(info->thread))
    {
      GDK_THREADS_ENTER();
      st_shell_remove_stream_task(info->shell, info->thread);
      gdk_flush();
      GDK_THREADS_LEAVE();
    }

  st_stream_task_free_info(info);
}

static void
st_stream_task_abort_cb (gpointer data)
{
  TaskInfo *info = data;

  st_stream_task_finish(info);
}

/*
 * Must be called with the GDK lock held.
 */
static void
st_stream_task_finish (TaskInfo *info)
{
  GSList *l;

  g_return_if_fail(info != NULL);

  SG_LIST_FOREACH(l, info->streams)
    st_stream_bag_set_running(l->data, FALSE);

  st_shell_remove_stream_task(info->shell, info->thread);
}

void
st_stream_task_launch (STShell *shell,
		       STHandler *handler,
		       STStreamTask task,
		       GSList *streams)
{
  TaskInfo *info;

  g_return_if_fail(ST_IS_SHELL(shell));
  g_return_if_fail(ST_IS_HANDLER(handler));
  g_return_if_fail(streams != NULL);

  info = g_new(TaskInfo, 1);

  info->shell = shell;
  info->handler = handler;
  info->task = task;
  info->streams = streams;
  info->thread = st_thread_new(handler);
  info->thread->thread_cb = info->task == ST_STREAM_TASK_ADD_BOOKMARKS
    ? st_stream_task_add_bookmarks_thread
    : st_stream_task_thread;
  info->thread->abort_cb = st_stream_task_abort_cb;
  info->thread->cb_data = info;

  st_shell_add_stream_task(info->shell, info->thread, task == ST_STREAM_TASK_TUNE_IN);
  st_thread_run(info->thread);
}

static void
st_stream_task_free_info (TaskInfo *info)
{
  g_return_if_fail(info != NULL);

  sg_objects_free(info->streams);
  g_free(info);
}
