/*
 * $Id: st-transfer.c,v 1.73.2.7 2004/07/26 14:38:27 jylefort Exp $
 *
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <string.h>
#include <curl/curl.h>
#ifndef ST_REGRESSION_TEST
#include <gtk/gtk.h>
#include "gettext.h"
#include "sg-util.h"
#include "sgtk-auth-dialog.h"
#include "st-dialog-api.h"
#include "st-settings.h"
#include "st-thread.h"
#include "st-main.h"
#endif /* ST_REGRESSION_TEST */
#include "st-transfer.h"

#define AGENT_STRING			"streamtuner"

/* maintain compatibility with old libcurl versions */

#ifndef CURLOPT_WRITEDATA
#define CURLOPT_WRITEDATA		CURLOPT_FILE
#endif

#ifndef CURLOPT_HEADERDATA
#define CURLOPT_HEADERDATA		CURLOPT_WRITEHEADER
#endif

/*** type definitions ********************************************************/

/*
 * libcurl already provides this prototype as curl_write_callback, but
 * we are not gonna rely on it as it's undocumented
 */
typedef size_t (CurlCallback) (const char *buffer,
			       size_t size,
			       size_t nitems,
			       gpointer data);

typedef struct
{
#ifndef ST_REGRESSION_TEST
  STThread			*thread;
#endif /* ST_REGRESSION_TEST */
  double			downloaded;

  const char			*url;
  unsigned int			flags;

  STTransferSession		*session;
  char				error[CURL_ERROR_SIZE];

  CurlCallback			*header_cb;
  gpointer			header_data;

  CurlCallback			*body_cb;
  gpointer			body_data;

  curl_proxytype		proxy_type;
  char				*proxy_server;
  int				proxy_port;
  char				*proxy_userpwd;
} STTransfer;

struct _STTransferSession
{
  CURL				*handle;
};

typedef struct
{
  STTransfer			*transfer;

  GString			*line;
  char				terminator[2]; /* \n\0, \r\n or \r\0 */

  STTransferLineCallback	*line_cb;
  gpointer			line_cb_data;
} STTransferLineData;

/*** function declarations ***************************************************/

static gboolean st_transfer_perform (STTransfer *transfer, GError **err);

static gboolean st_transfer_set_proxy_settings (STTransfer *transfer);
static void st_transfer_free_proxy_settings (STTransfer *transfer);

#ifndef ST_REGRESSION_TEST
static char *st_transfer_password_prompt (const char *username);
#endif /* ST_REGRESSION_TEST */

static gboolean st_transfer_progress_cb (gpointer data,
					 double dltotal,
					 double dlnow,
					 double ultotal,
					 double ulnow);

static gboolean st_transfer_session_get_internal (STTransferSession *session,
						  const char *url,
						  unsigned int flags,
						  char **headers,
						  char **body,
						  GError **err);

static void st_transfer_line_data_init (STTransferLineData *data,
					STTransfer *transfer,
					STTransferLineCallback *cb,
					gpointer cb_data);
static void st_transfer_line_data_flush (STTransferLineData *data);
static void st_transfer_line_data_free (STTransferLineData *data,
					gboolean flush);

static gboolean st_transfer_session_get_by_line_internal (STTransferSession *session,
							  const char *url,
							  unsigned int flags,
							  STTransferLineCallback *header_cb,
							  gpointer header_data,
							  STTransferLineCallback *body_cb,
							  gpointer body_data,
							  GError **err);

static size_t st_transfer_session_get_cb (const char *buffer,
					  size_t size,
					  size_t nitems,
					  gpointer data);
static size_t st_transfer_session_get_binary_cb (const char *buffer,
						 size_t size,
						 size_t nitems,
						 gpointer data);
static size_t st_transfer_session_get_by_line_cb (const char *buffer,
						  size_t size,
						  size_t nitems,
						  gpointer data);

static void st_transfer_zero_and_free (char *buffer);

/*** API implementation ******************************************************/

STTransferSession *
st_transfer_session_new (void)
{
  STTransferSession *session;

  session = g_new(STTransferSession, 1);
  session->handle = curl_easy_init();

  return session;
}

void
st_transfer_session_free (STTransferSession *session)
{
  g_return_if_fail(session != NULL);

  curl_easy_cleanup(session->handle);
  g_free(session);
}

/*
 * Deprecated.
 */
char *
st_transfer_get_full (const char *url, GError **err)
{
  STTransferSession *session;
  gboolean status;
  char *body;

  g_return_val_if_fail(url != NULL, FALSE);

  session = st_transfer_session_new();
  status = st_transfer_session_get_internal(session, url, 0, NULL, &body, err);
  st_transfer_session_free(session);

  return status ? body : NULL;
}

/*
 * Fetches URL.
 *
 * If HEADERS is not null, write a newly-allocated string containing
 * the HTTP headers at its address.
 *
 * A newly-allocated string containing the body will be written at the
 * address pointed to by BODY.
 *
 * Returns TRUE in case of success, FALSE in case of abort or error
 * (if there was an error, ERR will be set).
 */
gboolean
st_transfer_session_get (STTransferSession *session,
			 const char *url,
			 unsigned int flags,
			 char **headers,
			 char **body,
			 GError **err)
{
  gboolean status;

  g_return_val_if_fail(url != NULL, FALSE);

  status = st_transfer_session_get_internal(session, url, flags, headers, body, err);

  return status;
}

/*
 * Fetches URL as binary data.
 *
 * If HEADERS is not null, write a newly-allocated string containing
 * the HTTP headers at its address.
 *
 * A newly-allocated buffer containing the body will be written at the
 * address pointed to by BODY.
 *
 * The length of the body will be written at the address pointed to by
 * BODY_LEN.
 *
 * Returns TRUE in case of success, FALSE in case of abort or error
 * (if there was an error, ERR will be set).
 */
gboolean
st_transfer_session_get_binary (STTransferSession *session,
				const char *url,
				unsigned int flags,
				char **headers,
				guint8 **body,
				unsigned int *body_len,
				GError **err)
{
  STTransfer transfer;
  GString *header_string;
  GByteArray *body_array;
  gboolean status;

  g_return_val_if_fail(session != NULL, FALSE);
  g_return_val_if_fail(url != NULL, FALSE);
  
  transfer.url = url;
  transfer.flags = flags;
  transfer.session = session;

  if (headers)
    {
      header_string = g_string_new(NULL);
      transfer.header_cb = st_transfer_session_get_cb;
      transfer.header_data = header_string;
    }
  else
    {
      transfer.header_cb = NULL;
      transfer.header_data = NULL;
    }

  if (body)
    {
      body_array = g_byte_array_new();
      transfer.body_cb = st_transfer_session_get_binary_cb;
      transfer.body_data = body_array;
    }
  else
    {
      transfer.body_cb = NULL;
      transfer.body_data = NULL;
    }
  
  status = st_transfer_perform(&transfer, err);
  if (status)
    {
      if (headers)
	*headers = header_string->str;
      if (body)
	{
	  *body = body_array->data;
	  *body_len = body_array->len;
	}
    }

  if (headers)
    g_string_free(header_string, ! status);
  if (body)
    g_byte_array_free(body_array, ! status);

  return status;
}

/*
 * Deprecated
 */
char *
st_transfer_get_full_with_session (STTransferSession *session,
				   const char *url,
				   GError **err)
{
  gboolean status;
  char *body;

  g_return_val_if_fail(session != NULL, FALSE);
  g_return_val_if_fail(url != NULL, FALSE);

  status = st_transfer_session_get_internal(session, url, 0, NULL, &body, err);
  
  return status ? body : NULL;
}

static gboolean
st_transfer_session_get_internal (STTransferSession *session,
				  const char *url,
				  unsigned int flags,
				  char **headers,
				  char **body,
				  GError **err)
{
  STTransfer transfer;
  GString *header_string;
  GString *body_string;
  gboolean status;

  g_return_val_if_fail(session != NULL, FALSE);
  g_return_val_if_fail(url != NULL, FALSE);
  
  transfer.url = url;
  transfer.flags = flags;
  transfer.session = session;

  if (headers)
    {
      header_string = g_string_new(NULL);
      transfer.header_cb = st_transfer_session_get_cb;
      transfer.header_data = header_string;
    }
  else
    {
      transfer.header_cb = NULL;
      transfer.header_data = NULL;
    }

  if (body)
    {
      body_string = g_string_new(NULL);
      transfer.body_cb = st_transfer_session_get_cb;
      transfer.body_data = body_string;
    }
  else
    {
      transfer.body_cb = NULL;
      transfer.body_data = NULL;
    }
  
  status = st_transfer_perform(&transfer, err);
  if (status)
    {
      if (headers)
	*headers = header_string->str;
      if (body)
	*body = body_string->str;
    }

  if (headers)
    g_string_free(header_string, ! status);
  if (body)
    g_string_free(body_string, ! status);

  return status;
}

/*
 * Deprecated.
 */
gboolean
st_transfer_get_lines (const char *url,
		       STTransferLineCallback *cb,
		       gpointer data,
		       GError **err)
{
  STTransferSession *session;
  gboolean status;

  g_return_val_if_fail(url != NULL, FALSE);
  g_return_val_if_fail(cb != NULL, FALSE);

  session = st_transfer_session_new();
  status = st_transfer_session_get_by_line_internal(session, url, 0, NULL, NULL, cb, data, err);
  st_transfer_session_free(session);

  return status;
}

gboolean
st_transfer_session_get_by_line (STTransferSession *session,
				 const char *url,
				 unsigned int flags,
				 STTransferLineCallback *header_cb,
				 gpointer header_data,
				 STTransferLineCallback *body_cb,
				 gpointer body_data,
				 GError **err)
{
  gboolean status;

  g_return_val_if_fail(url != NULL, FALSE);

  status = st_transfer_session_get_by_line_internal(session, url, flags, header_cb, header_data, body_cb, body_data, err);

  return status;
}

/*
 * Deprecated.
 */
gboolean
st_transfer_get_lines_with_session (STTransferSession *session,
				    const char *url,
				    STTransferLineCallback *cb,
				    gpointer data,
				    GError **err)
{
  g_return_val_if_fail(session != NULL, FALSE);
  g_return_val_if_fail(url != NULL, FALSE);
  g_return_val_if_fail(cb != NULL, FALSE);

  return st_transfer_session_get_by_line_internal(session, url, 0, NULL, NULL, cb, data, err);
}

static void
st_transfer_line_data_init (STTransferLineData *data,
			    STTransfer *transfer,
			    STTransferLineCallback *cb,
			    gpointer cb_data)
{
  g_return_if_fail(data != NULL);
  g_return_if_fail(transfer != NULL);

  data->transfer = transfer;
  data->line = g_string_new(NULL);
  *data->terminator = 0;
  data->line_cb = cb;
  data->line_cb_data = cb_data;
}

static void
st_transfer_line_data_flush (STTransferLineData *data)
{
  g_return_if_fail(data != NULL);

  if (data->transfer->flags & ST_TRANSFER_PASS_NEWLINE && *data->terminator)
    g_string_append_len(data->line, data->terminator, sizeof(data->terminator));

  data->line_cb(data->line->str, data->line_cb_data);

  g_string_truncate(data->line, 0);
  *data->terminator = 0;
}

static void
st_transfer_line_data_free (STTransferLineData *data, gboolean flush)
{
  g_return_if_fail(data != NULL);

  if (flush && *data->line->str)
    st_transfer_line_data_flush(data);

  g_string_free(data->line, TRUE);
}

static gboolean
st_transfer_session_get_by_line_internal (STTransferSession *session,
					  const char *url,
					  unsigned int flags,
					  STTransferLineCallback *header_cb,
					  gpointer header_data,
					  STTransferLineCallback *body_cb,
					  gpointer body_data,
					  GError **err)
{
  STTransfer transfer;
  STTransferLineData header_line_data;
  STTransferLineData body_line_data;
  gboolean status;

  g_return_val_if_fail(session != NULL, FALSE);
  g_return_val_if_fail(url != NULL, FALSE);
  
  transfer.url = url;
  transfer.flags = flags;
  transfer.session = session;

  if (header_cb)
    {
      st_transfer_line_data_init(&header_line_data, &transfer, header_cb, header_data);
      transfer.header_cb = st_transfer_session_get_by_line_cb;
      transfer.header_data = &header_line_data;
    }
  else
    {
      transfer.header_cb = NULL;
      transfer.header_data = NULL;
    }
   
  if (body_cb)
    {
      st_transfer_line_data_init(&body_line_data, &transfer, body_cb, body_data);
      transfer.body_cb = st_transfer_session_get_by_line_cb;
      transfer.body_data = &body_line_data;
    }
  else
    {
      transfer.body_cb = NULL;
      transfer.body_data = NULL;
    }

  status = st_transfer_perform(&transfer, err);
  
  if (header_cb)
    st_transfer_line_data_free(&header_line_data, status);
  if (body_cb)
    st_transfer_line_data_free(&body_line_data, status);
  
  return status;
}

static gboolean
st_transfer_perform (STTransfer *transfer, GError **err)
{
  int status;

  g_return_val_if_fail(transfer != NULL, FALSE);
  g_return_val_if_fail(transfer->url != NULL, FALSE);
  g_return_val_if_fail(transfer->session != NULL, FALSE);

#ifndef ST_REGRESSION_TEST
  transfer->thread = st_thread_get();
  g_return_val_if_fail(transfer->thread != NULL, FALSE);

  if (st_thread_is_aborted(transfer->thread))
    return FALSE;
#endif /* ST_REGRESSION_TEST */

  transfer->downloaded = 0;

#ifndef ST_REGRESSION_TEST
  GDK_THREADS_ENTER();
  st_thread_set_progress(transfer->thread, 0);
  gdk_flush();
  GDK_THREADS_LEAVE();
#endif /* ST_REGRESSION_TEST */

  if (! st_transfer_set_proxy_settings(transfer))
    {
#ifndef ST_REGRESSION_TEST
      GDK_THREADS_ENTER();
      st_thread_abort(transfer->thread);
      gdk_flush();
      GDK_THREADS_LEAVE();
#endif /* ST_REGRESSION_TEST */
      
      return FALSE;		/* aborted */
    }

#ifndef ST_REGRESSION_TEST
  GDK_THREADS_ENTER();
  st_thread_printf(transfer->thread, _("Connecting to %s..."), transfer->url);
  gdk_flush();
  GDK_THREADS_LEAVE();
#endif /* ST_REGRESSION_TEST */
  
  curl_easy_setopt(transfer->session->handle, CURLOPT_USERAGENT, AGENT_STRING);
  curl_easy_setopt(transfer->session->handle, CURLOPT_URL, transfer->url);
  curl_easy_setopt(transfer->session->handle, CURLOPT_HEADERFUNCTION, transfer->header_cb);
  curl_easy_setopt(transfer->session->handle, CURLOPT_HEADERDATA, transfer->header_data);
  curl_easy_setopt(transfer->session->handle, CURLOPT_WRITEFUNCTION, transfer->body_cb);
  curl_easy_setopt(transfer->session->handle, CURLOPT_WRITEDATA, transfer->body_data);
  curl_easy_setopt(transfer->session->handle, CURLOPT_NOPROGRESS, FALSE);
  curl_easy_setopt(transfer->session->handle, CURLOPT_PROGRESSFUNCTION, st_transfer_progress_cb);
  curl_easy_setopt(transfer->session->handle, CURLOPT_PROGRESSDATA, transfer);
  curl_easy_setopt(transfer->session->handle, CURLOPT_ERRORBUFFER, transfer->error);
  curl_easy_setopt(transfer->session->handle, CURLOPT_PROXYTYPE, transfer->proxy_type);
  curl_easy_setopt(transfer->session->handle, CURLOPT_PROXY, transfer->proxy_server);
  curl_easy_setopt(transfer->session->handle, CURLOPT_PROXYPORT, transfer->proxy_port);
  curl_easy_setopt(transfer->session->handle, CURLOPT_PROXYUSERPWD, transfer->proxy_userpwd);
  curl_easy_setopt(transfer->session->handle, CURLOPT_COOKIEFILE, "");
  curl_easy_setopt(transfer->session->handle, CURLOPT_FOLLOWLOCATION, TRUE);
  curl_easy_setopt(transfer->session->handle, CURLOPT_NOBODY, transfer->body_cb == NULL);
  curl_easy_setopt(transfer->session->handle, CURLOPT_ENCODING, "");
  curl_easy_setopt(transfer->session->handle, CURLOPT_NOSIGNAL, TRUE);
  
  status = curl_easy_perform(transfer->session->handle);
  
  st_transfer_free_proxy_settings(transfer);

  if (status != CURLE_OK
      && status != CURLE_ABORTED_BY_CALLBACK
#ifndef ST_REGRESSION_TEST
      && ! st_thread_is_aborted(transfer->thread)
#endif /* ST_REGRESSION_TEST */
      )
    g_set_error(err, 0, 0, "%s", transfer->error); /* failure which isn't abort */

  return status == CURLE_OK;
}

static gboolean
st_transfer_set_proxy_settings (STTransfer *transfer)
{
  gboolean status = TRUE;

  g_return_val_if_fail(transfer != NULL, FALSE);

  transfer->proxy_type = 0;
  transfer->proxy_server = NULL;
  transfer->proxy_port = 0;
  transfer->proxy_userpwd = NULL;

#ifndef ST_REGRESSION_TEST
  G_LOCK(st_settings);

  /*
   * We have to set transfer->proxy_userpwd, because even if the proxy
   * is disabled in streamtuner, libcurl might still use http_proxy,
   * ftp_proxy, etc...
   */
  if (st_settings.proxy_auth_enabled && st_settings.proxy_auth_name)
    {
      char *password;

      if (st_settings.proxy_auth_password)
	password = g_strdup(st_settings.proxy_auth_password);
      else
	{
	  GDK_THREADS_ENTER();
	  st_thread_printf(transfer->thread, _("Waiting for proxy password..."));
	  password = st_transfer_password_prompt(st_settings.proxy_auth_name);
	  gdk_flush();
	  GDK_THREADS_LEAVE();
	}

      if (password)
	{
	  transfer->proxy_userpwd = g_strconcat(st_settings.proxy_auth_name, ":", password, NULL);
	  st_transfer_zero_and_free(password);
	}
      else			/* cancelled by user */
	status = FALSE;
    }
  else
    transfer->proxy_userpwd = g_strdup(""); /* libcurl crashes in handleSock5Proxy() if this is NULL */

  if (status && st_settings.proxy_enabled && st_settings.proxy_server)
    { 
      transfer->proxy_type = st_settings.proxy_type;
      transfer->proxy_server = g_strdup(st_settings.proxy_server);
      transfer->proxy_port = st_settings.proxy_port;
    }

  G_UNLOCK(st_settings);
#endif /* ST_REGRESSION_TEST */

  return status;
}

static void
st_transfer_free_proxy_settings (STTransfer *transfer)
{
  g_return_if_fail(transfer != NULL);
  
  g_free(transfer->proxy_server);
  st_transfer_zero_and_free(transfer->proxy_userpwd);
}

char *
st_transfer_escape (const char *url)
{
  return curl_escape(url, 0);
}

static gboolean
st_transfer_progress_cb (gpointer data,
			 double dltotal,
			 double dlnow,
			 double ultotal,
			 double ulnow)
{
#ifndef ST_REGRESSION_TEST
  STTransfer *transfer = data;
#endif /* ST_REGRESSION_TEST */
  gboolean aborted = FALSE;

#ifndef ST_REGRESSION_TEST
  aborted = st_thread_is_aborted(transfer->thread);
  if (! aborted)
    {
      GDK_THREADS_ENTER();

      if (dltotal == 0)
	st_thread_printf(transfer->thread,
			 ngettext("Receiving (%u byte so far)...",
				  "Receiving (%u bytes so far)...",
				  (unsigned int) dlnow),
			 (unsigned int) dlnow);
      else
	st_thread_printf(transfer->thread,
			 ngettext("Receiving (%u byte out of %u)...",
				  "Receiving (%u bytes out of %u)...",
				  (unsigned int) dlnow),
			 (unsigned int) dlnow,
			 (unsigned int) dltotal);
	  
      if (dlnow != transfer->downloaded)
	{
	  st_thread_set_progress(transfer->thread, dltotal == 0 ? -1 : dlnow / dltotal);
	  transfer->downloaded = dlnow;
	}

      gdk_flush();
      GDK_THREADS_LEAVE();
    }
#endif /* ST_REGRESSION_TEST */

  return aborted;
}

#ifndef ST_REGRESSION_TEST
static char *
st_transfer_password_prompt (const char *username)
{
  GtkWidget *dialog;
  char *password = NULL;

  g_return_val_if_fail(username != NULL, NULL);

  dialog = sgtk_auth_dialog_new(st_main_get_transient(),
				_("Please enter your proxy password."));

  sgtk_auth_dialog_set_name(SGTK_AUTH_DIALOG(dialog), username);
  sgtk_auth_dialog_set_name_sensitive(SGTK_AUTH_DIALOG(dialog), FALSE);

  if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK)
    password = g_strdup(sgtk_auth_dialog_get_password(SGTK_AUTH_DIALOG(dialog)));

  gtk_widget_destroy(dialog);

  return password;
}
#endif /* ST_REGRESSION_TEST */

static size_t
st_transfer_session_get_cb (const char *buffer,
			    size_t size,
			    size_t nitems,
			    gpointer data)
{
  GString *string = data;
  size_t len;

  len = size * nitems;
  g_string_append_len(string, buffer, len);
  
  return len;
}

static size_t
st_transfer_session_get_binary_cb (const char *buffer,
				   size_t size,
				   size_t nitems,
				   gpointer data)
{
  GByteArray *array = data;
  size_t len;

  len = size * nitems;
  g_byte_array_append(array, buffer, len);
  
  return len;
}

static size_t
st_transfer_session_get_by_line_cb (const char *buffer,
				    size_t size,
				    size_t nitems,
				    gpointer data)
{
  STTransferLineData *line_data = data;
  size_t len;
  int start = 0;
  int i;

  len = size * nitems;

  /* handles lf (UNIX), crlf (DOS) and cr (Mac) terminators */
  for (i = 0; i < len; i++)
    {
      gboolean processed = FALSE;

      if (*line_data->terminator)
	{
	  if (buffer[i] == '\n' && *line_data->terminator == '\r')
	    {
	      line_data->terminator[1] = '\n';
	      processed = TRUE;
	      start = i + 1;
	    }
	  else
	    start = i;

	  st_transfer_line_data_flush(line_data);
	}

      if (! processed && (buffer[i] == '\n' || buffer[i] == '\r'))
	{
	  g_string_append_len(line_data->line, buffer + start, i - start);
	  line_data->terminator[0] = buffer[i];
	  line_data->terminator[1] = 0;
	}
    }
  if (start < len)
    g_string_append_len(line_data->line, buffer + start, len - start);
  
  return len;
}

static void
st_transfer_zero_and_free (char *buffer)
{
  if (buffer)
    {
      int i;

      for (i = 0; buffer[i]; i++)
	buffer[i] = 0;
      
      g_free(buffer);
    }
}

/*** private implementation **************************************************/

void
st_transfer_init (void)
{
  if (curl_global_init(CURL_GLOBAL_ALL) != CURLE_OK)
    {
#ifdef ST_REGRESSION_TEST
      g_critical("unable to initialize libcurl");
#else
      st_error_dialog(_("A fatal error has occurred."),
		      _("Unable to initialize the curl library."));
      exit(1);
#endif ST_REGRESSION_TEST
    }
}
