/*
 * $Id: st-main.c,v 1.10.2.3 2004/07/25 17:47:24 jylefort Exp $
 *
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <stdlib.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include "gettext.h"
#include "sg-util.h"
#include "sgtk-message-dialog.h"
#include "sgtk-stock.h"
#include "sgtk-util.h"
#include "st-thread.h"
#include "st-stock.h"
#include "st-transfer.h"
#include "st-settings.h"
#include "st-session.h"
#include "st-action.h"
#include "st-handler.h"
#include "st-handlers.h"
#include "st-plugins.h"
#include "st-shell.h"
#include "st-splash.h"
#include "st-util.h"

/*** variable declarations ***************************************************/

static char *custom_private_dir = NULL;
static STShell *shell = NULL;

/*** function declarations ***************************************************/

static STSplash   *st_main_splash_create	(void);
static void       st_main_splash_set_text	(STSplash	*splash,
						 const char	*str);
static void       st_main_splash_set_progress	(STSplash	*splash,
						 double		progress);

static void       st_main_parse_args		(int		argc,
						 char		**argv);
static void       st_main_print_help		(void);
static void       st_main_print_version		(void);

/*** implementation **********************************************************/

int
main (int argc, char **argv)
{
  STSplash *splash = NULL;
  GSList *l;

#ifdef ENABLE_NLS
  bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
  textdomain(GETTEXT_PACKAGE);
#endif

  st_main_parse_args(argc, argv);

  st_thread_init();		/* must be called before gtk_init() */
  gtk_init(&argc, &argv);

  st_settings_init(custom_private_dir);
  if (st_settings.splash_enabled)
    splash = st_main_splash_create();

  st_main_splash_set_text(splash, _("Initializing subsystems..."));
  sgtk_stock_init();		st_main_splash_set_progress(splash, (double) 1 / 8);
  st_stock_init();		st_main_splash_set_progress(splash, (double) 2 / 8);
  st_transfer_init();		st_main_splash_set_progress(splash, (double) 3 / 8);
  st_action_init();		st_main_splash_set_progress(splash, (double) 4 / 8);
  
  st_main_splash_set_text(splash, _("Loading plugins..."));
  st_handlers_load_static();	st_main_splash_set_progress(splash, (double) 5 / 8);
  st_plugins_load();		st_main_splash_set_progress(splash, (double) 6 / 8);

  st_main_splash_set_text(splash, _("Loading session..."));
  st_session_load();		st_main_splash_set_progress(splash, (double) 7 / 8);

  SG_LIST_FOREACH(l, st_handlers_list)
    st_handler_select_category(l->data);

  st_main_splash_set_progress(splash, (double) 8 / 8);

  GDK_THREADS_ENTER();

  shell = st_shell_new();
  g_object_add_weak_pointer(G_OBJECT(shell), (gpointer *) &shell);

  if (splash)
    gtk_widget_destroy(GTK_WIDGET(splash));
  st_shell_show(shell);

  gtk_main();

  GDK_THREADS_LEAVE();

  return 0;
}

static STSplash *
st_main_splash_create (void)
{
  GtkWidget *splash;

  GDK_THREADS_ENTER();

  gtk_window_set_auto_startup_notification(FALSE);
  splash = st_splash_new();
  gtk_widget_show(splash);

  sgtk_flush();
  gtk_window_set_auto_startup_notification(TRUE);
  GDK_THREADS_LEAVE();

  return ST_SPLASH(splash);
}

static void
st_main_splash_set_text (STSplash *splash, const char *str)
{
  g_return_if_fail(str != NULL);

  if (splash)
    {
      GDK_THREADS_ENTER();
      st_splash_set_text(splash, str);
      sgtk_flush();
      GDK_THREADS_LEAVE();
    }
}

static void
st_main_splash_set_progress (STSplash *splash, double progress)
{
  if (splash)
    {
      GDK_THREADS_ENTER();
      st_splash_set_progress(splash, progress);
      sgtk_flush();
      GDK_THREADS_LEAVE();
    }
}

GtkWindow *
st_main_get_transient (void)
{
  return shell ? st_shell_get_transient(shell) : NULL;
}

void
st_main_quit (void)
{
  gboolean save = st_settings.save_automatic;
  
  if (! save)
    {
      GtkWidget *dialog;
      GtkWidget *yes;
      int response;
      
      dialog = sgtk_message_dialog_new(st_main_get_transient(),
				       GTK_STOCK_DIALOG_WARNING,
				       _("Save session before quitting?"),
				       _("If you quit without saving, changes to your session will be discarded."));
				       
      gtk_dialog_add_button(GTK_DIALOG(dialog), ST_STOCK_QUIT_WITHOUT_SAVING, GTK_RESPONSE_NO);
      gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
      yes = gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_SAVE, GTK_RESPONSE_YES);

      gtk_window_set_default(GTK_WINDOW(dialog), yes);

      response = gtk_dialog_run(GTK_DIALOG(dialog));
      gtk_widget_destroy(dialog);

      if (response == GTK_RESPONSE_CANCEL)
	return;
      save = response == GTK_RESPONSE_YES;
    }
  
  g_object_unref(shell);
  if (save)
    st_session_save();
  
  gtk_main_quit();
}

static void
st_main_parse_args (int argc, char **argv)
{
  int c;

  while ((c = getopt(argc, argv, "d:thv")) != -1)
    switch (c)
      {
      case 'd':
	custom_private_dir = g_strdup(optarg);
	break;
      case 'h':
	st_main_print_help();
	exit(0);
      case 'v':
	st_main_print_version();
	exit(0);
      case '?':
	exit(1);
      default:
	g_assert_not_reached();
      }
}

static void
st_main_print_help (void)
{
  g_print(_("Usage: streamtuner [OPTIONS]\n"));
  g_print("\n");
  g_print(_("Options:\n"));
  g_print(_("  -d DIR   Specify a configuration directory (default is ~/.streamtuner)\n"));
  g_print(_("  -h       Show this help and exit\n"));
  g_print(_("  -v       Show version information and exit\n"));
}

static void
st_main_print_version (void)
{
  g_print(_("%s version %s\n"), PACKAGE, VERSION);
  g_print(ST_COPYRIGHT "\n");

#ifdef WITH_STATIC_SHOUTCAST
  g_print("\n");
  g_print(_("The SHOUTcast plugin has been built into streamtuner.\n"));
#endif	/* WITH_STATIC_SHOUTCAST */
}
