/*
 * $Id: sg-util.c,v 1.21.2.2 2004/07/24 16:17:26 jylefort Exp $
 *
 * Copyright (c) 2003 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <glib-object.h>
#include <string.h>
#include "sg-util.h"

/*** function declarations ***************************************************/

static gboolean sg_objects_free_node_cb (GNode *node, gpointer data);

/*** implementation **********************************************************/

char *
sg_str_sentencize (const char *str)
{
  char *sentencized;
  int len;

  g_return_val_if_fail(str != NULL, NULL);

  len = strlen(str);

  if (len > 0
      && str[len - 1] != '.'
      && str[len - 1] != '!'
      && str[len - 1] != '?')
    sentencized = g_strconcat(str, ".", NULL);
  else
    sentencized = g_strdup(str);
  
  if (sentencized[0])
    sentencized[0] = g_ascii_toupper(sentencized[0]);

  return sentencized;
}

void
sg_pointers_free (GSList *pointers)
{
  GSList *l;

  SG_LIST_FOREACH(l, pointers)
    g_free(l->data);

  g_slist_free(pointers);
}

void
sg_objects_free (GSList *objects)
{
  GSList *l;

  SG_LIST_FOREACH(l, objects)
    g_object_unref(l->data);

  g_slist_free(objects);
}

void
sg_objects_free_node (GNode *objects)
{
  g_return_if_fail(objects != NULL);

  g_node_traverse(objects,
		  G_PRE_ORDER,
		  G_TRAVERSE_ALL,
		  -1,
		  sg_objects_free_node_cb,
		  NULL);
  g_node_destroy(objects);
}

static gboolean
sg_objects_free_node_cb (GNode *node, gpointer data)
{
  if (node->data)
    g_object_unref(node->data);

  return FALSE;			/* continue */
}

GSList *
sg_objects_copy (GSList *objects)
{
  GSList *copy;
  GSList *l;

  copy = g_slist_copy(objects);
  SG_LIST_FOREACH(l, copy)
    g_object_ref(l->data);

  return copy;
}

gboolean
sg_value_equal (const GValue *value1, const GValue *value2)
{
  gboolean equal;

  g_return_val_if_fail(G_IS_VALUE(value1), FALSE);
  g_return_val_if_fail(G_IS_VALUE(value2), FALSE);
  g_return_val_if_fail(G_VALUE_TYPE(value1) == G_VALUE_TYPE(value2), FALSE);

  switch (G_VALUE_TYPE(value1))
    {
    case G_TYPE_BOOLEAN:
    case G_TYPE_INT:
      equal = g_value_get_int(value1) == g_value_get_int(value2);
      break;

    case G_TYPE_STRING:
      {
	const char *str1 = g_value_get_string(value1);
	const char *str2 = g_value_get_string(value2);

	equal = (str1 == NULL && str2 == NULL) || (str1 && str2 && ! strcmp(str1, str2));
	break;
      }

    default:
      g_assert_not_reached();	/* unsupported type */
    }

  return equal;
}

int
sg_utf8_strcmp (const char *s1, const char *s2)
{
  int cmp;
  char *normalized_s1;
  char *normalized_s2;

  g_return_val_if_fail(s1 != NULL, 0);
  g_return_val_if_fail(s2 != NULL, 0);

  normalized_s1 = g_utf8_normalize(s1, -1, G_NORMALIZE_ALL);
  normalized_s2 = g_utf8_normalize(s2, -1, G_NORMALIZE_ALL);
  
  cmp = strcmp(normalized_s1, normalized_s2);

  g_free(normalized_s1);
  g_free(normalized_s2);

  return cmp;
}

gboolean
sg_utf8_strcontains (const char *big, const char *little)
{
  gboolean contains;
  char *normalized_big;
  char *normalized_little;

  g_return_val_if_fail(big != NULL, FALSE);
  g_return_val_if_fail(little != NULL, FALSE);

  normalized_big = g_utf8_normalize(big, -1, G_NORMALIZE_ALL);
  normalized_little = g_utf8_normalize(little, -1, G_NORMALIZE_ALL);

  contains = strstr(normalized_big, normalized_little) != NULL;

  g_free(normalized_big);
  g_free(normalized_little);

  return contains;
}

gboolean
sg_utf8_strcasecontains (const char *big, const char *little)
{
  gboolean contains;
  char *normalized_big;
  char *normalized_little;
  char *case_normalized_big;
  char *case_normalized_little;

  g_return_val_if_fail(big != NULL, FALSE);
  g_return_val_if_fail(little != NULL, FALSE);

  normalized_big = g_utf8_normalize(big, -1, G_NORMALIZE_ALL);
  normalized_little = g_utf8_normalize(little, -1, G_NORMALIZE_ALL);
  case_normalized_big = g_utf8_casefold(normalized_big, -1);
  case_normalized_little = g_utf8_casefold(normalized_little, -1);

  contains = strstr(case_normalized_big, case_normalized_little) != NULL;

  g_free(normalized_big);
  g_free(normalized_little);
  g_free(case_normalized_big);
  g_free(case_normalized_little);

  return contains;
}

char *
sg_value_array_get_string (GValueArray *value_array)
{
  GString *string;
  int i;

  g_return_val_if_fail(value_array != NULL, NULL);

  string = g_string_new(NULL);
  for (i = 0; i < value_array->n_values; i++)
    {
      GValue *value = g_value_array_get_nth(value_array, i);
      GValue transformed = { 0, };

      g_value_init(&transformed, G_TYPE_STRING);
      if (g_value_transform(value, &transformed))
	{
	  if (*string->str)
	    g_string_append_printf(string, ", %s", g_value_get_string(&transformed));
	  else
	    g_string_append(string, g_value_get_string(&transformed));
	}
      g_value_unset(&transformed);
    }

  return g_string_free(string, FALSE);
}
