/*
 * $Id: st-network-tab.c,v 1.40.2.1 2004/05/11 10:54:35 jylefort Exp $
 *
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>
#include "art/network_preferences.h"
#include "sgtk-hig.h"
#include "sgtk-util.h"
#include "st-network-tab.h"
#include "st-preferences.h"
#include "st-settings.h"
#include "st-preferences-tab.h"

/*** type definitions ********************************************************/

struct _STNetworkTabPrivate
{
  GtkTooltips	*tooltips;
  GtkWidget	*proxy_check;
  GtkWidget	*type_label;
  GtkWidget	*http_radio;
  GtkWidget	*socks5_radio;
  GtkWidget	*server_label;
  GtkWidget	*server_entry;
  GtkWidget	*port_label;
  GtkWidget	*port_spin;
  GtkWidget	*auth_check;
  GtkWidget	*name_label;
  GtkWidget	*name_entry;
  GtkWidget	*password_label;
  GtkWidget	*password_entry;
};
  
/*** variable declarations ***************************************************/

static GObjectClass *parent_class = NULL;

/*** function declarations ***************************************************/

static void st_network_tab_class_init		(STNetworkTabClass *class);
static void st_network_tab_init			(STNetworkTab	  *tab);
static void st_network_tab_finalize		(GObject          *object);

static void st_network_tab_check_toggled_h	(GtkToggleButton  *button,
						 gpointer	  data);
static void st_network_tab_update_sensitivity	(STNetworkTab	  *tab);

/*** implementation **********************************************************/

GType
st_network_tab_get_type (void)
{
  static GType network_tab_type = 0;
  
  if (! network_tab_type)
    {
      static const GTypeInfo network_tab_info = {
	sizeof(STNetworkTabClass),
	NULL,
	NULL,
	(GClassInitFunc) st_network_tab_class_init,
	NULL,
	NULL,
	sizeof(STNetworkTab),
	0,
	(GInstanceInitFunc) st_network_tab_init,
      };

      network_tab_type = g_type_register_static(ST_TYPE_PREFERENCES_TAB,
						"STNetworkTab",
						&network_tab_info,
						0);
    }

  return network_tab_type;
}

static void
st_network_tab_class_init (STNetworkTabClass *class)
{
  GObjectClass *object_class = G_OBJECT_CLASS(class);
  STPreferencesTabClass *preferences_tab_class = ST_PREFERENCES_TAB_CLASS(class);

  parent_class = g_type_class_peek_parent(class);
  
  object_class->finalize = st_network_tab_finalize;

  preferences_tab_class->link_id = "preferences-network";
}

static void
st_network_tab_init (STNetworkTab *tab)
{
  GdkPixbuf *pixbuf;
  GtkWidget *table;
  GtkWidget *type_box;
  GtkWidget *server_box;

  tab->priv = g_new0(STNetworkTabPrivate, 1);

  tab->priv->tooltips = gtk_tooltips_new();

  pixbuf = gdk_pixbuf_new_from_inline(sizeof(art_network_preferences),
				      art_network_preferences,
				      FALSE,
				      NULL);

  st_preferences_tab_construct_label(ST_PREFERENCES_TAB(tab), pixbuf, _("Network"));
  g_object_unref(pixbuf);

  gtk_container_set_border_width(GTK_CONTAINER(tab), SGTK_HIG_CONTROLS_BORDER_WIDTH);

  tab->priv->proxy_check = gtk_check_button_new_with_mnemonic(_("_Use a proxy server"));
  gtk_tooltips_set_tip(tab->priv->tooltips, tab->priv->proxy_check,
		       _("If this option is enabled and the server field is not empty, streamtuner will use that proxy server for data transfers."),
		       NULL);
  
  tab->priv->type_label = gtk_label_new(_("Type:"));

  type_box = gtk_hbox_new(FALSE, SGTK_HIG_CONTROL_SPACING);
  tab->priv->http_radio = gtk_radio_button_new_with_mnemonic(NULL, _("_HTTP"));
  tab->priv->socks5_radio = gtk_radio_button_new_with_mnemonic_from_widget(GTK_RADIO_BUTTON(tab->priv->http_radio), _("_Socks 5"));
  gtk_box_pack_start(GTK_BOX(type_box), tab->priv->http_radio, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(type_box), tab->priv->socks5_radio, FALSE, FALSE, 0);

  gtk_tooltips_set_tip(tab->priv->tooltips, tab->priv->http_radio,
		       _("Select this option if your proxy server is a HTTP proxy."),
		       NULL);
  gtk_tooltips_set_tip(tab->priv->tooltips, tab->priv->socks5_radio,
		       _("Select this option if your proxy server is a Socks 5 proxy."),
		       NULL);

  tab->priv->server_label = gtk_label_new_with_mnemonic(_("S_erver:"));

  server_box = gtk_hbox_new(FALSE, SGTK_HIG_CONTROL_LABEL_SPACING);
  tab->priv->server_entry = gtk_entry_new();
  tab->priv->port_label = gtk_label_new_with_mnemonic(_("P_ort:"));
  tab->priv->port_spin = gtk_spin_button_new_with_range(0, 65535, 1);
  gtk_box_pack_start(GTK_BOX(server_box), tab->priv->server_entry, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(server_box), tab->priv->port_label, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(server_box), tab->priv->port_spin, FALSE, FALSE, 0);

  gtk_label_set_mnemonic_widget(GTK_LABEL(tab->priv->server_label), tab->priv->server_entry);
  gtk_label_set_mnemonic_widget(GTK_LABEL(tab->priv->port_label), tab->priv->port_spin);

  gtk_tooltips_set_tip(tab->priv->tooltips, tab->priv->server_entry,
		       _("The hostname or IP address of the proxy server"),
		       NULL);
  gtk_tooltips_set_tip(tab->priv->tooltips, tab->priv->port_spin,
		       _("The port number of the proxy server.\n"
			 "\n"
			 "HTTP proxies commonly use port 8080 or 3128.\n"
			 "Socks 5 proxies commonly use port 1080.\n"
			 "\n"
			 "Contact your network administrator for more details."),
		       NULL);

  tab->priv->auth_check = gtk_check_button_new_with_mnemonic(_("Use proxy _authentication"));
  gtk_tooltips_set_tip(tab->priv->tooltips, tab->priv->auth_check,
		       _("Enable this option if your proxy server requires user authentication."),
		       NULL);

  tab->priv->name_label = gtk_label_new_with_mnemonic(_("_Name:"));
  tab->priv->name_entry = gtk_entry_new();
  gtk_label_set_mnemonic_widget(GTK_LABEL(tab->priv->name_label), tab->priv->name_entry);

  gtk_tooltips_set_tip(tab->priv->tooltips, tab->priv->name_entry,
		       _("The login name used to authenticate yourself"),
		       NULL);

  tab->priv->password_label = gtk_label_new_with_mnemonic(_("_Password:"));
  tab->priv->password_entry = gtk_entry_new();
  gtk_label_set_mnemonic_widget(GTK_LABEL(tab->priv->password_label), tab->priv->password_entry);

  gtk_entry_set_visibility(GTK_ENTRY(tab->priv->password_entry), FALSE);
  gtk_tooltips_set_tip(tab->priv->tooltips, tab->priv->password_entry,
		       _("The password used to authenticate yourself.\n"
			 "\n"
			 "If left blank, you'll be prompted for a password upon each transfer."),
		       NULL);

  gtk_misc_set_alignment(GTK_MISC(tab->priv->type_label), 1, 0.5);
  gtk_misc_set_alignment(GTK_MISC(tab->priv->server_label), 1, 0.5);
  gtk_misc_set_alignment(GTK_MISC(tab->priv->name_label), 1, 0.5);
  gtk_misc_set_alignment(GTK_MISC(tab->priv->password_label), 1, 0.5);

  st_preferences_bind_boolean(GTK_TOGGLE_BUTTON(tab->priv->proxy_check),
			      &st_settings.proxy_enabled);
  st_preferences_bind_int_radio(GTK_RADIO_BUTTON(tab->priv->http_radio),
				(int *) &st_settings.proxy_type,
				ST_TRANSFER_PROXY_HTTP);
  st_preferences_bind_int_radio(GTK_RADIO_BUTTON(tab->priv->socks5_radio),
				(int *) &st_settings.proxy_type,
				ST_TRANSFER_PROXY_SOCKS5);
  st_preferences_bind_string(GTK_ENTRY(tab->priv->server_entry),
			     &st_settings.proxy_server);
  st_preferences_bind_int_spin(GTK_SPIN_BUTTON(tab->priv->port_spin),
			       &st_settings.proxy_port);
  st_preferences_bind_boolean(GTK_TOGGLE_BUTTON(tab->priv->auth_check),
			      &st_settings.proxy_auth_enabled);
  st_preferences_bind_string(GTK_ENTRY(tab->priv->name_entry),
			     &st_settings.proxy_auth_name);
  st_preferences_bind_string(GTK_ENTRY(tab->priv->password_entry),
			     &st_settings.proxy_auth_password);

  g_signal_connect(G_OBJECT(tab->priv->proxy_check), "toggled",
		   G_CALLBACK(st_network_tab_check_toggled_h), tab);
  g_signal_connect(G_OBJECT(tab->priv->auth_check), "toggled",
		   G_CALLBACK(st_network_tab_check_toggled_h), tab);

  sgtk_widget_set_next_widget(tab->priv->server_entry, tab->priv->port_spin);
  sgtk_widget_set_next_widget(tab->priv->port_spin, tab->priv->name_entry);
  sgtk_widget_set_next_widget(tab->priv->name_entry, tab->priv->password_entry);
  sgtk_widget_set_next_widget(tab->priv->password_entry, tab->priv->server_entry);

  table = gtk_table_new(6, 2, FALSE);

  gtk_table_set_row_spacings(GTK_TABLE(table), SGTK_HIG_CONTROL_SPACING);
  gtk_table_set_col_spacings(GTK_TABLE(table), SGTK_HIG_CONTROL_LABEL_SPACING);

  gtk_table_attach(GTK_TABLE(table), tab->priv->proxy_check, 0, 2, 0, 1,
		   GTK_EXPAND | GTK_FILL,
		   0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(table), tab->priv->type_label, 0, 1, 1, 2,
		   GTK_FILL,
		   0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(table), type_box, 1, 2, 1, 2,
		   GTK_FILL,
		   0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(table), tab->priv->server_label, 0, 1, 2, 3,
		   GTK_FILL,
		   0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(table), server_box, 1, 2, 2, 3,
		   GTK_EXPAND | GTK_FILL,
		   0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(table), tab->priv->auth_check, 0, 2, 3, 4,
		   GTK_EXPAND | GTK_FILL,
		   0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(table), tab->priv->name_label, 0, 1, 4, 5,
		   GTK_FILL,
		   0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(table), tab->priv->name_entry, 1, 2, 4, 5,
		   GTK_EXPAND | GTK_FILL,
		   0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(table), tab->priv->password_label, 0, 1, 5, 6,
		   GTK_FILL,
		   0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(table), tab->priv->password_entry, 1, 2, 5, 6,
		   GTK_EXPAND | GTK_FILL,
		   0,
		   0,
		   0);

  gtk_widget_show_all(table);
  gtk_container_add(GTK_CONTAINER(tab), table);

  st_network_tab_update_sensitivity(tab);
}

static void
st_network_tab_finalize (GObject *object)
{
  STNetworkTab *tab = ST_NETWORK_TAB(object);

  gtk_object_destroy(GTK_OBJECT(tab->priv->tooltips));
  g_free(tab->priv);

  G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
st_network_tab_check_toggled_h (GtkToggleButton *button, gpointer data)
{
  STNetworkTab *tab = data;

  g_return_if_fail(ST_IS_NETWORK_TAB(tab));

  st_network_tab_update_sensitivity(tab);
}

static void
st_network_tab_update_sensitivity (STNetworkTab *tab)
{
  gboolean proxy, auth;

  g_return_if_fail(ST_IS_NETWORK_TAB(tab));

  proxy = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tab->priv->proxy_check));
  auth = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tab->priv->auth_check));

  gtk_widget_set_sensitive(tab->priv->type_label, proxy); 
  gtk_widget_set_sensitive(tab->priv->http_radio, proxy); 
  gtk_widget_set_sensitive(tab->priv->socks5_radio, proxy); 
  gtk_widget_set_sensitive(tab->priv->server_label, proxy); 
  gtk_widget_set_sensitive(tab->priv->server_entry, proxy);
  gtk_widget_set_sensitive(tab->priv->port_label, proxy); 
  gtk_widget_set_sensitive(tab->priv->port_spin, proxy);
  gtk_widget_set_sensitive(tab->priv->auth_check, proxy);
  gtk_widget_set_sensitive(tab->priv->name_label, proxy && auth);
  gtk_widget_set_sensitive(tab->priv->name_entry, proxy && auth);
  gtk_widget_set_sensitive(tab->priv->password_label, proxy && auth);
  gtk_widget_set_sensitive(tab->priv->password_entry, proxy && auth);
}

GtkWidget *
st_network_tab_new (void)
{
  return g_object_new(ST_TYPE_NETWORK_TAB, NULL);
}
