/*
 * $Id: st-browser-tab-label.c,v 1.8 2004/03/24 21:04:09 jylefort Exp $
 *
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <gtk/gtk.h>
#include "gettext.h"
#include "sgtk-blinking-image.h"
#include "sgtk-hig.h"
#include "sgtk-util.h"
#include "st-browser-tab-label.h"

/*** cpp *********************************************************************/

#define BOX_SPACING			(SGTK_HIG_UNIT / 2)

/*** type definitions ********************************************************/

enum {
  PROP_0,
  PROP_TAB,
  PROP_USE_EVENT_BOX
};

struct _STBrowserTabLabelPrivate
{
  gboolean	use_event_box;
  GtkWidget	*image;
};

/*** variable declarations ***************************************************/

static GObjectClass *parent_class = NULL;

/*** function declarations ***************************************************/

static void st_browser_tab_label_class_init (STBrowserTabLabelClass *class);
static void st_browser_tab_label_init (STBrowserTabLabel *tab_label);
static void st_browser_tab_label_finalize (GObject *object);

static GObject *st_browser_tab_label_constructor (GType type,
						  unsigned int n_construct_properties,
						  GObjectConstructParam *construct_params);
static void st_browser_tab_label_set_property (GObject *object,
					       unsigned int prop_id,
					       const GValue *value,
					       GParamSpec *pspec);

/*** implementation **********************************************************/

GType
st_browser_tab_label_get_type (void)
{
  static GType browser_tab_label_type = 0;
  
  if (! browser_tab_label_type)
    {
      static const GTypeInfo browser_tab_label_info = {
	sizeof(STBrowserTabLabelClass),
	NULL,
	NULL,
	(GClassInitFunc) st_browser_tab_label_class_init,
	NULL,
	NULL,
	sizeof(STBrowserTabLabel),
	0,
	(GInstanceInitFunc) st_browser_tab_label_init,
      };
      
      browser_tab_label_type = g_type_register_static(GTK_TYPE_HBOX,
						      "STBrowserTabLabel",
						      &browser_tab_label_info,
						      0);
    }

  return browser_tab_label_type;
}

static void
st_browser_tab_label_class_init (STBrowserTabLabelClass *class)
{
  GObjectClass *object_class = G_OBJECT_CLASS(class);

  parent_class = g_type_class_peek_parent(class);

  object_class->finalize = st_browser_tab_label_finalize;
  object_class->constructor = st_browser_tab_label_constructor;
  object_class->set_property = st_browser_tab_label_set_property;

  g_object_class_install_property(object_class,
				  PROP_TAB,
				  g_param_spec_object("tab",
						      _("Tab"),
						      _("The tab label's STBrowserTab object"),
						      ST_TYPE_BROWSER_TAB,
						      G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
  g_object_class_install_property(object_class,
				  PROP_USE_EVENT_BOX,
				  g_param_spec_boolean("use-event-box",
						       _("Use Event Box"),
						       _("Whether the tab label should use a GtkEventBox or not"),
						       FALSE,
						       G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}

static void
st_browser_tab_label_init (STBrowserTabLabel *tab_label)
{
  tab_label->priv = g_new0(STBrowserTabLabelPrivate, 1);
}

static void
st_browser_tab_label_finalize (GObject *object)
{
  STBrowserTabLabel *tab_label = ST_BROWSER_TAB_LABEL(object);

  g_free(tab_label->priv);
  
  G_OBJECT_CLASS(parent_class)->finalize(object);
}

static GObject *
st_browser_tab_label_constructor (GType type,
				  unsigned int n_construct_properties,
				  GObjectConstructParam *construct_params)
{
  GObject *object;
  STBrowserTabLabel *tab_label;
  GtkWidget *label;
  GtkWidget *hbox;
  GdkPixbuf *pixbuf;

  object = G_OBJECT_CLASS(parent_class)->constructor(type,
						     n_construct_properties,
						     construct_params);

  tab_label = ST_BROWSER_TAB_LABEL(object);

  /* event box */

  if (tab_label->priv->use_event_box)
    {
      GtkWidget *event_box;

      event_box = gtk_event_box_new();
      hbox = gtk_hbox_new(FALSE, BOX_SPACING);

      gtk_container_add(GTK_CONTAINER(tab_label), event_box);
      gtk_container_add(GTK_CONTAINER(event_box), hbox);

      gtk_widget_show(event_box);
      gtk_widget_show(hbox);
    }
  else
    hbox = GTK_WIDGET(tab_label);

  /* image */

  pixbuf = st_handler_get_pixbuf(tab_label->tab->handler);
  if (pixbuf)
    {
      GdkPixbuf *scaled;

      tab_label->priv->image = sgtk_blinking_image_new();
      gtk_misc_set_alignment(GTK_MISC(tab_label->priv->image), 0, 0.5);

      scaled = sgtk_pixbuf_scale(pixbuf, GTK_ICON_SIZE_MENU);
      g_object_unref(pixbuf);

      sgtk_blinking_image_set_from_pixbuf(SGTK_BLINKING_IMAGE(tab_label->priv->image), scaled);
      g_object_unref(scaled);

      gtk_box_pack_start(GTK_BOX(hbox), tab_label->priv->image, FALSE, FALSE, 0);
      gtk_widget_show(tab_label->priv->image);
    }

  /* label */

  label = gtk_label_new(st_handler_get_label(tab_label->tab->handler));
  gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);

  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
  gtk_widget_show(label);

  return object;
}

static void
st_browser_tab_label_set_property (GObject *object,
				   unsigned int prop_id,
				   const GValue *value,
				   GParamSpec *pspec)
{
  STBrowserTabLabel *tab_label = ST_BROWSER_TAB_LABEL(object);

  switch (prop_id)
    {
    case PROP_TAB:
      tab_label->tab = g_value_get_object(value);
      break;

    case PROP_USE_EVENT_BOX:
      tab_label->priv->use_event_box = g_value_get_boolean(value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
    }
}

GtkWidget *
st_browser_tab_label_new (STBrowserTab *tab, gboolean use_event_box)
{
  return g_object_new(ST_TYPE_BROWSER_TAB_LABEL,
		      "spacing", BOX_SPACING,
		      "tab", tab,
		      "use-event-box", use_event_box,
		      NULL);
}

void
st_browser_tab_label_set_blinking (STBrowserTabLabel *tab_label,
				   gboolean blinking)
{
  g_return_if_fail(ST_IS_BROWSER_TAB_LABEL(tab_label));

  if (tab_label->priv->image)
    sgtk_blinking_image_set_blinking(SGTK_BLINKING_IMAGE(tab_label->priv->image), blinking);
}

void
st_browser_tab_label_set_icon_visible (STBrowserTabLabel *tab_label,
				       gboolean visible)
{
  g_return_if_fail(ST_IS_BROWSER_TAB_LABEL(tab_label));

  if (tab_label->priv->image)
    g_object_set(tab_label->priv->image,
		 "visible", visible,
		 NULL);
}
