/*
 * $Id: st-thread.c,v 1.22 2004/01/27 14:16:42 jylefort Exp $
 *
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include "gettext.h"
#include "sg-util.h"
#include "sg-printable.h"
#include "st-dialog.h"
#include "st-browser-tab.h"
#include "st-statusbar.h"
#include "st-thread.h"
#include "st-handler-event.h"

/*** function declarations ***************************************************/

static gpointer	st_thread_cb (gpointer data);

/*** implementation **********************************************************/

void
st_thread_init (void)
{
  g_thread_init(NULL);

  if (! g_thread_supported())
    {
      st_error_dialog(_("A fatal error has occurred."),
		      _("The GLib thread system is unavailable."));
      exit(1);
    }

  gdk_threads_init();
}

STThread *
st_thread_new (STThreadType type,
	       STHandler *handler,
	       STThreadCallback *cb,
	       gpointer data)
{
  STThread *thread = NULL;

  thread = g_new(STThread, 1);
  
  thread->type = type;
  thread->handler = handler;
  thread->tab = NULL;
  thread->printable = NULL;
  thread->progress_bar = NULL;
  thread->aborted = FALSE;
  thread->aborted_mutex = g_mutex_new();
  thread->cb = cb;
  thread->data = data;
  thread->downloaded = 0;
  thread->ui_timer = g_timer_new();
  thread->ui_first = TRUE;
  
  return thread;
}

void
st_thread_run (STThread *thread)
{
  GError *err = NULL;

  g_return_if_fail(thread != NULL);

  if (! g_thread_create(st_thread_cb, thread, FALSE, &err))
    {
      char *secondary;
      char *normalized;

      secondary = g_strdup_printf(_("Unable to create a thread: %s"),
				  err->message);
      normalized = st_dialog_normalize(secondary);

      st_error_dialog(_("A fatal error has occurred."), "%s", normalized);

      g_free(secondary);
      g_free(normalized);
      g_error_free(err);

      exit(1);
    }
}

static gpointer
st_thread_cb (gpointer data)
{
  STThread *thread = data;
  gpointer user_data = NULL;

  g_return_val_if_fail(thread != NULL, NULL);

  if (st_handler_event_is_bound(thread->handler, ST_HANDLER_EVENT_THREAD_BEGIN))
    user_data = st_handler_event_thread_begin(thread->handler);

  thread->cb(thread->data);

  if (st_handler_event_is_bound(thread->handler, ST_HANDLER_EVENT_THREAD_END))
    st_handler_event_thread_end(thread->handler, user_data);

/* free ourselves */
  g_mutex_free(thread->aborted_mutex);
  g_timer_destroy(thread->ui_timer);
  g_free(thread);

  return NULL;
}

void
st_thread_abort (STThread *thread)
{
  g_return_if_fail(thread != NULL);

  g_mutex_lock(thread->aborted_mutex);
  thread->aborted = TRUE;
  g_mutex_unlock(thread->aborted_mutex);
}

gboolean
st_thread_is_aborted (STThread *thread)
{
  gboolean aborted;

  g_return_val_if_fail(thread != NULL, FALSE);
  
  g_mutex_lock(thread->aborted_mutex);
  aborted = thread->aborted;
  g_mutex_unlock(thread->aborted_mutex);

  return aborted;
}

STThread *
st_thread_get (void)
{
  GThread *g_thread;

  g_thread = g_thread_self();
  return g_thread ? g_thread->data : NULL;
}
