/*
 * $Id: st-stream-view.c,v 1.40 2004/01/30 16:30:43 jylefort Exp $
 *
 * Copyright (c) 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <string.h>
#include <gdk/gdkkeysyms.h>
#include "gettext.h"
#include "sgtk-handlers.h"
#include "sg-util.h"
#include "st-browser-window.h"
#include "st-handler.h"
#include "st-handler-field-api.h"
#include "st-handler-event.h"
#include "st-stream-api.h"
#include "st-stream-view.h"
#include "st-stream-menu-items.h"
#include "st-stock.h"
#include "st-shell.h"

#define DEFAULT_COLUMN_SIZE		100

/*** type definitions ********************************************************/

enum {
  PROP_0,
  PROP_TAB
};
    
enum {
  COLUMN_PTR,
  N_COLUMNS
};

/*** variable declarations ***************************************************/

static GObjectClass *parent_class = NULL;

/*** function declarations ***************************************************/

static void st_stream_view_init		(STStreamView		*view);
static void st_stream_view_class_init	(STStreamViewClass	*class);

static GObject *st_stream_view_constructor (GType type,
					    guint n_construct_properties,
					    GObjectConstructParam *construct_params);
static void st_stream_view_set_property (GObject	*object,
					 guint		prop_id,
					 const GValue	*value,
					 GParamSpec	*pspec);

static void st_stream_view_construct_stream_menu	(STStreamView *view);
static void st_stream_view_construct_columns_menu	(STStreamView *view);
static void st_stream_view_construct_store		(STStreamView *view);
static void st_stream_view_construct_columns		(STStreamView *view);
static void st_stream_view_update_columns_menu		(STStreamView *view);
static void st_stream_view_bind				(STStreamView *view);
static void st_stream_view_save_columns_state		(STStreamView *view);

static void st_stream_view_construct_column_menu  (STStreamView       *view,
						   GtkTreeViewColumn  *column);

static void st_stream_view_column_visible_toggled_h (GtkCheckMenuItem *item,
						     gpointer	      data);
static void st_stream_view_column_hide_activate_h	(GtkMenuItem  *item,
							 gpointer     data);
static void st_stream_view_column_show_all_activate_h	(GtkMenuItem  *item,
							 gpointer     data);
static void st_stream_view_destroy_h		(GtkObject	      *object,
						 gpointer	      data);
static void st_stream_view_row_activated_h	(GtkTreeView	      *gtk_view,
						 GtkTreePath	      *arg1,
						 GtkTreeViewColumn    *arg2,
						 gpointer	      data);
static void st_stream_view_selection_changed_h	(GtkTreeSelection     *selection,
						 gpointer	      data);
static void st_stream_view_selection_foreach_cb	(GtkTreeModel         *model,
						 GtkTreePath          *path,
						 GtkTreeIter          *iter,
						 gpointer             data);

static void st_stream_view_append_stream	(STStreamView	    *view,
						 STStream	    *stream);

static void st_stream_view_restore_selection		(STStreamView *view);
static gboolean st_stream_view_restore_selection_cb	(GtkTreeModel *model,
							 GtkTreePath  *path,
							 GtkTreeIter  *iter,
							 gpointer     data);

/*** implementation **********************************************************/

GType
st_stream_view_get_type (void)
{
  static GType stream_view_type = 0;
  
  if (! stream_view_type)
    {
      static const GTypeInfo stream_view_info = {
	sizeof(STStreamViewClass),
	NULL,
	NULL,
	(GClassInitFunc) st_stream_view_class_init,
	NULL,
	NULL,
	sizeof(STStreamView),
	0,
	(GInstanceInitFunc) st_stream_view_init,
      };
      
      stream_view_type = g_type_register_static(GTK_TYPE_TREE_VIEW,
						"STStreamView",
						&stream_view_info,
						0);
    }

  return stream_view_type;
}

static void
st_stream_view_class_init (STStreamViewClass *class)
{
  GObjectClass *object_class;

  parent_class = g_type_class_peek_parent(class);

  object_class = G_OBJECT_CLASS(class);
  object_class->constructor = st_stream_view_constructor;
  object_class->set_property = st_stream_view_set_property;

  g_object_class_install_property(object_class,
				  PROP_TAB,
				  g_param_spec_object("tab",
						      "Tab",
						      _("The view's parent STBrowserTab"),
						      ST_TYPE_BROWSER_TAB,
						      G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}

static void
st_stream_view_init (STStreamView *view)
{
  view->tab = NULL;
  view->store = NULL;

  view->column_menu = NULL;
  view->columns_menu = NULL;

  view->selection_changed_hid = -1;

  view->menu_items = NULL;
}

static GObject *
st_stream_view_constructor (GType type,
			    guint n_construct_properties,
			    GObjectConstructParam *construct_params)
{
  GObject *object; 
  STStreamView *view;
  GtkTreeSelection *selection;

  object = G_OBJECT_CLASS(parent_class)->constructor(type,
						     n_construct_properties,
						     construct_params);
  view = ST_STREAM_VIEW(object);

  g_return_val_if_fail(view->tab != NULL, NULL);

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(view));
  if (st_handler_event_is_bound(view->tab->handler, ST_HANDLER_EVENT_STREAM_TUNE_IN_MULTIPLE))
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);

  st_stream_view_bind(view);
  st_stream_view_construct_stream_menu(view);

  st_stream_view_construct_store(view);
  st_stream_view_construct_columns(view);

  /* the following should be called after the columns have been created */
  st_stream_view_construct_columns_menu(view);
  st_stream_view_update_columns_menu(view);

  return object;
}

static void
st_stream_view_set_property (GObject *object,
			     guint prop_id,
			     const GValue *value,
			     GParamSpec *pspec)
{
  STStreamView *view;

  view = ST_STREAM_VIEW(object);

  switch (prop_id)
    {
    case PROP_TAB:
      view->tab = g_value_get_object(value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
    }
}

static void
st_stream_view_construct_stream_menu (STStreamView *view)
{
  GtkWidget *menu;

  g_return_if_fail(ST_IS_STREAM_VIEW(view));

  menu = gtk_menu_new();
  view->menu_items = st_stream_menu_items_new(NULL);

  st_stream_menu_items_append_to_shell(view->menu_items, GTK_MENU_SHELL(menu));

  g_signal_connect_swapped(G_OBJECT(view),
			   "popup-menu",
			   G_CALLBACK(sgtk_popup_menu_h),
			   G_OBJECT(menu));
  g_signal_connect_swapped(G_OBJECT(view),
			   "button_press_event",
			   G_CALLBACK(sgtk_popup_menu_mouse_h),
			   G_OBJECT(menu));
}

static void
st_stream_view_construct_columns_menu (STStreamView *view)
{
  GSList *l;
  int vi = 0;

  g_return_if_fail(ST_IS_STREAM_VIEW(view));
  
  view->columns_menu = gtk_menu_new();
  g_object_ref(view->columns_menu);

  SG_LIST_FOREACH(l, view->tab->handler->fields)
    {
      STHandlerField *field = l->data;
      GtkWidget *item;
      GtkTreeViewColumn *column;
      guint hid;

      if (! field->visible)
	continue;

      item = gtk_check_menu_item_new_with_label(field->label);
      gtk_widget_show(item);

      gtk_menu_shell_append(GTK_MENU_SHELL(view->columns_menu), item);

      column = gtk_tree_view_get_column(GTK_TREE_VIEW(view), vi++);
      g_return_if_fail(column != NULL);

      hid = g_signal_connect(G_OBJECT(item),
			     "toggled",
			     G_CALLBACK(st_stream_view_column_visible_toggled_h),
			     NULL);

      g_object_set_data(G_OBJECT(item), "hid", GINT_TO_POINTER(hid));
      g_object_set_data(G_OBJECT(item), "field", field);
      g_object_set_data(G_OBJECT(item), "column", column);
    }
}

static void
st_stream_view_column_visible_toggled_h (GtkCheckMenuItem *item, gpointer data)
{
  STHandlerField *field;
  GtkTreeViewColumn *column;

  field = g_object_get_data(G_OBJECT(item), "field");
  g_return_if_fail(field != NULL);

  column = g_object_get_data(G_OBJECT(item), "column");
  g_return_if_fail(column != NULL);

  field->user_visible = gtk_check_menu_item_get_active(item);
  gtk_tree_view_column_set_visible(column, field->user_visible);
}

static void
st_stream_view_update_columns_menu (STStreamView *view)
{
  GList *l;

  g_return_if_fail(ST_IS_STREAM_VIEW(view));
  g_return_if_fail(GTK_IS_MENU_SHELL(view->columns_menu));

  SG_LIST_FOREACH(l, GTK_MENU_SHELL(view->columns_menu)->children)
    {
      GtkWidget *item = l->data;
      STHandlerField *field;
      guint hid;

      field = g_object_get_data(G_OBJECT(item), "field");
      g_return_if_fail(field != NULL);

      hid = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(item), "hid"));

      g_signal_handler_block(item, hid);
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), field->user_visible);
      g_signal_handler_unblock(item, hid);
    }
}

static void
st_stream_view_construct_store (STStreamView *view)
{
  GSList *l;

  GType *types;
  int vi = 1;

  g_return_if_fail(ST_IS_STREAM_VIEW(view));

  /* create the store */

  types = g_new(GType, g_slist_length(view->tab->handler->fields) + 1);
  types[0] = G_TYPE_POINTER;

  SG_LIST_FOREACH(l, view->tab->handler->fields)
    {
      STHandlerField *field = l->data;

      if (field->visible)
	types[vi++] = field->type;
    }
  
  view->store = gtk_list_store_newv(vi, types);
  g_free(types);

  /* set the sort column if any */

  if (view->tab->handler->fields_sort_index)
    gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(view->store),
					 view->tab->handler->fields_sort_index,
					 (GtkSortType) view->tab->handler->fields_sort_order);

  /* set the store */

  gtk_tree_view_set_model(GTK_TREE_VIEW(view), GTK_TREE_MODEL(view->store));
}

static void
st_stream_view_construct_columns (STStreamView *view)
{
  GSList *l;
  int vi = 1;

  g_return_if_fail(ST_IS_STREAM_VIEW(view));

  SG_LIST_FOREACH(l, view->tab->handler->fields)
    {
      STHandlerField *field = l->data;
      GtkTreeViewColumn *column;
      
      if (! field->visible)
	continue;
      
      column = gtk_tree_view_column_new_with_attributes(field->label, gtk_cell_renderer_text_new(),
							"text", vi,
							NULL);
      
      g_object_set_data(G_OBJECT(column), "field", field);

      gtk_tree_view_column_set_visible(column, field->user_visible);
      gtk_tree_view_column_set_sort_column_id(column, vi++);
      gtk_tree_view_column_set_resizable(column, TRUE);
      gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_FIXED);
      
      gtk_tree_view_column_set_fixed_width(column, field->width > 0
					   ? field->width
					   : DEFAULT_COLUMN_SIZE);
      
      gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);
      st_stream_view_construct_column_menu(view, column);
    }
}

static void
st_stream_view_construct_column_menu (STStreamView *view,
				      GtkTreeViewColumn *column)
{
  GtkWidget *menu;

  GtkWidget *hide;
  GtkWidget *image;
  GtkWidget *separator;
  GtkWidget *show_all;
  
  g_return_if_fail(ST_IS_STREAM_VIEW(view));
  g_return_if_fail(GTK_IS_TREE_VIEW_COLUMN(column));
  g_return_if_fail(GTK_IS_WIDGET(column->button));

  menu = gtk_menu_new();

  hide = gtk_image_menu_item_new_with_mnemonic(_("_Hide this column"));
  image = gtk_image_new_from_stock(GTK_STOCK_CLOSE, GTK_ICON_SIZE_MENU);
  gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(hide), image);
  
  separator = gtk_separator_menu_item_new();
  show_all = gtk_menu_item_new_with_mnemonic(_("_Show all columns"));

  gtk_widget_show(hide);
  gtk_widget_show(image);
  gtk_widget_show(separator);
  gtk_widget_show(show_all);
  
  gtk_menu_shell_append(GTK_MENU_SHELL(menu), hide);
  gtk_menu_shell_append(GTK_MENU_SHELL(menu), separator);
  gtk_menu_shell_append(GTK_MENU_SHELL(menu), show_all);

  g_signal_connect(G_OBJECT(hide),
		   "activate",
		   G_CALLBACK(st_stream_view_column_hide_activate_h),
		   column);
  g_signal_connect(G_OBJECT(show_all),
		   "activate",
		   G_CALLBACK(st_stream_view_column_show_all_activate_h),
		   view);

  g_signal_connect_swapped(G_OBJECT(column->button),
			   "popup-menu",
			   G_CALLBACK(sgtk_popup_menu_h),
			   G_OBJECT(menu));
  g_signal_connect_swapped(G_OBJECT(column->button),
			   "button-press-event",
			   G_CALLBACK(sgtk_popup_menu_mouse_h),
			   G_OBJECT(menu));
}

static void
st_stream_view_column_hide_activate_h (GtkMenuItem *item, gpointer data)
{
  GtkTreeViewColumn *column = data;
  STHandlerField *field;

  g_return_if_fail(GTK_IS_TREE_VIEW_COLUMN(column));

  field = g_object_get_data(G_OBJECT(column), "field");
  g_return_if_fail(field != NULL);

  field->user_visible = FALSE;
  gtk_tree_view_column_set_visible(column, field->user_visible);

  st_stream_view_update_columns_menu(ST_STREAM_VIEW(column->tree_view));
}

static void
st_stream_view_column_show_all_activate_h (GtkMenuItem *item, gpointer data)
{
  STStreamView *view = data;
  GSList *l;
  int vi = 0;

  g_return_if_fail(ST_IS_STREAM_VIEW(view));

  SG_LIST_FOREACH(l, view->tab->handler->fields)
    {
      STHandlerField *field = l->data;
      GtkTreeViewColumn *column;

      if (! field->visible)
	continue;

      column = gtk_tree_view_get_column(GTK_TREE_VIEW(view), vi++);
      g_return_if_fail(column != NULL);

      field->user_visible = TRUE;
      gtk_tree_view_column_set_visible(column, field->user_visible);
    }

  st_stream_view_update_columns_menu(view);
}

static void
st_stream_view_bind (STStreamView *view)
{
  GtkTreeSelection *selection;

  g_return_if_fail(ST_IS_STREAM_VIEW(view));

  g_signal_connect(G_OBJECT(view), "destroy",
		   G_CALLBACK(st_stream_view_destroy_h), NULL);
  g_signal_connect(G_OBJECT(view), "row-activated",
		   G_CALLBACK(st_stream_view_row_activated_h), NULL);

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(view));
  view->selection_changed_hid = g_signal_connect(G_OBJECT(selection),
						 "changed",
						 G_CALLBACK(st_stream_view_selection_changed_h),
						 view);
}

static void
st_stream_view_destroy_h (GtkObject *object,
				      gpointer data)
{
  st_stream_view_save_columns_state(ST_STREAM_VIEW(object));
}

static void
st_stream_view_row_activated_h (GtkTreeView *gtk_view,
				GtkTreePath *arg1,
				GtkTreeViewColumn *arg2,
				gpointer data)
{
  STStreamView *view = (STStreamView *) gtk_view;

  g_return_if_fail(ST_IS_STREAM_VIEW(view));

  if (st_shell_can_tune_in())
    st_shell_tune_in();
  else if (st_shell_can_browse())
    st_shell_browse();
}

static void
st_stream_view_selection_changed_h (GtkTreeSelection *selection, gpointer data)
{
  STStreamView *view = data;
  GSList *selected_streams = NULL;
  
  g_return_if_fail(ST_IS_STREAM_VIEW(view));
  
  gtk_tree_selection_selected_foreach(selection,
				      st_stream_view_selection_foreach_cb,
				      &selected_streams);

  if (selected_streams)
    {
      g_mutex_lock(view->tab->handler->mutex);
      g_slist_free(view->tab->handler->selected_streams);
      view->tab->handler->selected_streams = selected_streams;
      g_mutex_unlock(view->tab->handler->mutex);
  
      st_browser_tab_streams_selected(view->tab);
    }
}

static void
st_stream_view_selection_foreach_cb (GtkTreeModel *model,
				     GtkTreePath *path,
				     GtkTreeIter *iter,
				     gpointer data)
{
  GSList **selected_streams = data;
  STStream *stream;

  gtk_tree_model_get(model, iter,
		     COLUMN_PTR, &stream,
		     -1);
  
  *selected_streams = g_slist_append(*selected_streams, stream);
}

GtkWidget *
st_stream_view_new (STBrowserTab *tab)
{
  return g_object_new(ST_TYPE_STREAM_VIEW,
		      "tab", tab,
		      NULL);
}

static void
st_stream_view_save_columns_state (STStreamView *view)
{
  int sort_column_id;
  GtkSortType order;
  
  GSList *l;
  int vi = 0;			/* visible iterator */

  g_return_if_fail(ST_IS_STREAM_VIEW(view));

  if (gtk_tree_sortable_get_sort_column_id(GTK_TREE_SORTABLE(view->store),
					   &sort_column_id,
					   &order)
      && sort_column_id > 0)	/* column 0 is owned by COLUMN_PTR */
    {
      view->tab->handler->fields_sort_index = sort_column_id;
      view->tab->handler->fields_sort_order = order;
    }

  SG_LIST_FOREACH(l, view->tab->handler->fields)
    {
      STHandlerField *field = l->data;
      GtkTreeViewColumn *column;

      if (field->visible)
	{
	  column = gtk_tree_view_get_column(GTK_TREE_VIEW(view), vi++);
	  g_return_if_fail(column);

	  field->width = gtk_tree_view_column_get_width(column);
	}
    }
}

void
st_stream_view_clear (STStreamView *view)
{
  g_return_if_fail(ST_IS_STREAM_VIEW(view));

  gtk_list_store_clear(view->store);
}

void
st_stream_view_append_streams (STStreamView *view, GList *streams)
{
  GList *l;

  g_return_if_fail(ST_IS_STREAM_VIEW(view));

  SG_LIST_FOREACH(l, streams)
    st_stream_view_append_stream(view, l->data);

  st_stream_view_restore_selection(view);
}

static void
st_stream_view_append_stream (STStreamView *view, STStream *stream)
{
  GtkTreeIter tree_iter;
  GValue value = { 0, };

  GSList *l;
  
  int vi = 1;			/* visible iterator */

  g_return_if_fail(ST_IS_STREAM_VIEW(view));
  
  gtk_list_store_append(view->store, &tree_iter);

  g_value_init(&value, G_TYPE_POINTER);
  g_value_set_pointer(&value, stream);
  gtk_list_store_set_value(view->store, &tree_iter, COLUMN_PTR, &value);
  g_value_unset(&value);

  SG_LIST_FOREACH(l, view->tab->handler->fields)
    {
      STHandlerField *field = l->data;

      if (field->visible)
	{
	  st_handler_stream_field_get(view->tab->handler, stream, field, &value);
	  gtk_list_store_set_value(view->store, &tree_iter, vi++, &value);
	  g_value_unset(&value);
	}
    }
}

static void
st_stream_view_restore_selection (STStreamView *view)
{
  g_return_if_fail(ST_IS_STREAM_VIEW(view));

  if (view->tab->handler->selected_streams)
    gtk_tree_model_foreach(GTK_TREE_MODEL(view->store),
			   st_stream_view_restore_selection_cb,
			   view);
}

static gboolean
st_stream_view_restore_selection_cb (GtkTreeModel *model,
				     GtkTreePath *path,
				     GtkTreeIter *iter,
				     gpointer data)
{
  GSList *l;
  STStreamView *view = data;
  STStream *stream;

  gtk_tree_model_get(model, iter, COLUMN_PTR, &stream, -1);

  g_return_val_if_fail(stream != NULL, FALSE);
  g_return_val_if_fail(stream->name != NULL, FALSE);

  SG_LIST_FOREACH(l, view->tab->handler->selected_streams)
    {
      STStream *selected_stream = l->data;
  
      /*
       * Here we must compare names, because we don't want to
       * differentiate two different stream pointers representing the
       * same logical stream (even if the path of the stream is
       * different).
       *
       * That is, if multiple categories contain the same stream, we
       * want the stream to be visually (but not internally) selected
       * in all of them.
       */

      g_return_val_if_fail(selected_stream != NULL, FALSE);
      g_return_val_if_fail(selected_stream->name != NULL, FALSE);

      if (! strcmp(stream->name, selected_stream->name))
	{
	  GtkTreeSelection *selection;

	  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(view));

	  g_signal_handler_block(selection, view->selection_changed_hid);
	  gtk_tree_selection_select_iter(selection, iter);
	  g_signal_handler_unblock(selection, view->selection_changed_hid);
	}
    }

  return FALSE;
}

void
st_stream_view_update_sensitivity (STStreamView *view)
{
  g_return_if_fail(ST_IS_STREAM_VIEW(view));

  st_stream_menu_items_update_sensitivity(view->menu_items);
}
