/*
 * $Id: st-session.c,v 1.9 2004/01/05 19:44:10 jylefort Exp $
 *
 * Copyright (c) 2002, 2003 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <errno.h>
#include <glib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include "gettext.h"
#include "sg-util.h"
#include "st-cache-load.h"
#include "st-cache-save.h"
#include "st-config-load.h"
#include "st-config-save.h"
#include "st-dialog.h"
#include "st-handler.h"
#include "st-handlers.h"
#include "st-settings.h"

/*** variable declarations ***************************************************/

static char *config_filename;
static char *cache_filename;
static char *accels_filename;

/*** implementation **********************************************************/

void
st_session_init (void)
{
  config_filename = g_build_filename(st_settings.private_dir, "config", NULL);
  cache_filename = g_build_filename(st_settings.private_dir, "cache", NULL);
  accels_filename = g_build_filename(st_settings.private_dir, "accels", NULL);
}

gboolean
st_session_load (GError **err)
{
  GSList *l;
  GError *load_err = NULL;
  
  if (g_file_test(accels_filename, G_FILE_TEST_IS_REGULAR))
    gtk_accel_map_load(accels_filename);

  if (g_file_test(config_filename, G_FILE_TEST_IS_REGULAR))
    {
      if (! st_config_load(config_filename, &load_err))
	{
	  char *normalized;

	  normalized = st_dialog_normalize(load_err->message);

	  st_error_dialog(_("Unable to load configuration."), "%s", normalized);

	  g_free(normalized);
	  g_clear_error(&load_err);
	}
    }
  
  if (g_file_test(cache_filename, G_FILE_TEST_IS_REGULAR))
    {
      if (! st_cache_load(cache_filename, &load_err))
	{
	  char *normalized;

	  normalized = st_dialog_normalize(load_err->message);
	  
	  st_error_dialog(_("Unable to load cache."), "%s", normalized);

	  g_free(normalized);
	  g_clear_error(&load_err);
	}
    }

  SG_LIST_FOREACH(l, st_handlers_list)
    st_handler_selection_restore(l->data);

  return TRUE;
}

gboolean
st_session_save (GError **err)
{
  GError *save_err = NULL;

  if (! g_file_test(st_settings.private_dir, G_FILE_TEST_IS_DIR)
      && mkdir(st_settings.private_dir, 0755) == -1)
    {
      g_set_error(err, 0, 0, _("unable to create directory %s: %s"),
		  st_settings.private_dir, g_strerror(errno));

      return FALSE;
    }
  
  gtk_accel_map_save(accels_filename);

  if (! st_config_save(config_filename, &save_err))
    {
      char *normalized;
      
      normalized = st_dialog_normalize(save_err->message);
      
      st_error_dialog(_("Unable to save configuration."), "%s", normalized);
      
      g_free(normalized);
      g_clear_error(&save_err);
    }
  
  if (! st_cache_save(cache_filename, &save_err))
    {
      char *normalized;
      
      normalized = st_dialog_normalize(save_err->message);
      
      st_error_dialog(_("Unable to save cache."), "%s", normalized);
      
      g_free(normalized);
      g_clear_error(&save_err);
    }
  
  return TRUE;
}
