/*
 * $Id: st-programs-tab.c,v 1.14 2004/01/05 19:44:10 jylefort Exp $
 *
 * Copyright (c) 2002, 2003 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <gtk/gtk.h>
#include "gettext.h"
#include "art/programs_preferences.h"
#include "sg-util.h"
#include "st-programs-tab.h"
#include "st-preferences-tab.h"
#include "st-action.h"

/*** type definitions ********************************************************/

enum {
  /* invisible columns */
  COLUMN_ID,
  COLUMN_EDITABLE,

  /* visible columns */
  COLUMN_LABEL,
  COLUMN_COMMAND,

  N_COLUMNS
};

/*** function declarations ***************************************************/

static void st_programs_tab_init		(STProgramsTab	      *tab);
static void st_programs_tab_cell_edited_h	(GtkCellRendererText  *cell,
						 const char    *path_string,
						 const char	  *new_text,
						 gpointer	       data);
static void st_programs_tab_fill		(STProgramsTab	      *tab);

/*** implementation **********************************************************/

GType
st_programs_tab_get_type (void)
{
  static GType programs_tab_type = 0;
  
  if (! programs_tab_type)
    {
      static const GTypeInfo programs_tab_info = {
	sizeof(STProgramsTabClass),
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	sizeof(STProgramsTab),
	0,
	(GInstanceInitFunc) st_programs_tab_init,
      };

      programs_tab_type = g_type_register_static(ST_TYPE_PREFERENCES_TAB,
						 "STProgramsTab",
						 &programs_tab_info,
						 0);
    }

  return programs_tab_type;
}

static void
st_programs_tab_init (STProgramsTab *tab)
{
  GdkPixbuf *pixbuf;
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;

  pixbuf = gdk_pixbuf_new_from_inline(sizeof(art_programs_preferences),
				      art_programs_preferences,
				      FALSE,
				      NULL);

  st_preferences_tab_construct_label(ST_PREFERENCES_TAB(tab),
				     pixbuf,
				     _("Programs"));
  g_object_unref(pixbuf);

  /* construct containers */
  
  tab->scrolled = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(tab->scrolled),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);

  /* construct the view */

  tab->store = gtk_list_store_new(N_COLUMNS,
				  G_TYPE_STRING,
				  G_TYPE_BOOLEAN,
				  G_TYPE_STRING,
				  G_TYPE_STRING);
  tab->view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(tab->store));

  /* construct the label column */

  column = gtk_tree_view_column_new_with_attributes(_("Action"),
						    gtk_cell_renderer_text_new(),
						    "text", COLUMN_LABEL,
						    NULL);
  gtk_tree_view_column_set_sort_column_id(column, COLUMN_LABEL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(tab->view), column);

  /* construct the command column */

  renderer = gtk_cell_renderer_text_new();
  g_signal_connect(G_OBJECT(renderer), "edited",
		   G_CALLBACK(st_programs_tab_cell_edited_h), tab);

  column = gtk_tree_view_column_new_with_attributes(_("Command"),
						    renderer,
						    "text", COLUMN_COMMAND,
						    "editable", COLUMN_EDITABLE,
						    NULL);
  gtk_tree_view_column_set_sort_column_id(column, COLUMN_COMMAND);
  gtk_tree_view_append_column(GTK_TREE_VIEW(tab->view), column);

  /* widget setup and packing */

  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(tab->store),
				       COLUMN_LABEL,
				       GTK_SORT_ASCENDING);
  
  gtk_container_add(GTK_CONTAINER(tab->scrolled), tab->view);
  gtk_container_add(GTK_CONTAINER(tab), tab->scrolled);
  gtk_widget_show_all(tab->scrolled);

  /* fill it */

  st_programs_tab_fill(tab);
}

static void
st_programs_tab_cell_edited_h (GtkCellRendererText *cell,
			       const char *path_string,
			       const char *new_text,
			       gpointer data)
{
  STProgramsTab *tab = data;
  GtkTreePath *path;
  GtkTreeIter iter;
  const char *id;
  
  g_return_if_fail(ST_IS_PROGRAMS_TAB(tab));

  path = gtk_tree_path_new_from_string(path_string);
  gtk_tree_model_get_iter(GTK_TREE_MODEL(tab->store), &iter, path);
  gtk_tree_path_free(path);

  gtk_tree_model_get(GTK_TREE_MODEL(tab->store), &iter,
		     COLUMN_ID, &id,
		     -1);

  gtk_list_store_set(tab->store, &iter,
		     COLUMN_COMMAND, new_text,
		     -1);
  st_action_associate(id, new_text);
}

static void
st_programs_tab_fill (STProgramsTab *tab)
{
  GSList *actions;
  GSList *l;

  g_return_if_fail(ST_IS_PROGRAMS_TAB(tab));

  actions = st_action_list();

  SG_LIST_FOREACH(l, actions)
    {
      const char *id = l->data;
      STAction *action;
      GtkTreeIter iter;

      action = st_action_get(id);
      g_return_if_fail(action != NULL);

      gtk_list_store_append(tab->store, &iter);
      gtk_list_store_set(tab->store, &iter,
			 COLUMN_ID, id,
			 COLUMN_EDITABLE, TRUE,
			 COLUMN_LABEL, action->label,
			 COLUMN_COMMAND, action->command,
			 -1);
    }
}

GtkWidget *
st_programs_tab_new (void)
{
  return g_object_new(ST_TYPE_PROGRAMS_TAB, NULL);
}
