/*
 * $Id: st-network-tab.c,v 1.25 2004/01/05 19:44:10 jylefort Exp $
 *
 * Copyright (c) 2002, 2003 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <gtk/gtk.h>
#include "gettext.h"
#include "art/network_preferences.h"
#include "sgtk-handlers.h"
#include "sgtk-hig.h"
#include "st-network-tab.h"
#include "st-preferences.h"
#include "st-settings.h"
#include "st-preferences-tab.h"
#include "st-shell.h"

/*** function declarations ***************************************************/

static void st_network_tab_init			(STNetworkTab	  *tab);
static void st_network_tab_check_toggled_h	(GtkToggleButton  *button,
						 gpointer	  data);
static void st_network_tab_update_sensitivity	(STNetworkTab	  *tab);

/*** implementation **********************************************************/

GType
st_network_tab_get_type (void)
{
  static GType network_tab_type = 0;
  
  if (! network_tab_type)
    {
      static const GTypeInfo network_tab_info = {
	sizeof(STNetworkTabClass),
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	sizeof(STNetworkTab),
	0,
	(GInstanceInitFunc) st_network_tab_init,
      };

      network_tab_type = g_type_register_static(ST_TYPE_PREFERENCES_TAB,
						"STNetworkTab",
						&network_tab_info,
						0);
    }

  return network_tab_type;
}

static void
st_network_tab_init (STNetworkTab *tab)
{
  GdkPixbuf *pixbuf;

  pixbuf = gdk_pixbuf_new_from_inline(sizeof(art_network_preferences),
				      art_network_preferences,
				      FALSE,
				      NULL);

  st_preferences_tab_construct_label(ST_PREFERENCES_TAB(tab),
				     pixbuf,
				     _("Network"));
  g_object_unref(pixbuf);

  gtk_container_set_border_width(GTK_CONTAINER(tab), SGTK_HIG_CONTROLS_BORDER_WIDTH);

  tab->proxy_check = gtk_check_button_new_with_mnemonic(_("_Use a HTTP proxy"));
  gtk_tooltips_set_tip(st_shell_tooltips, tab->proxy_check,
		       _("If enabled and the URL entry is not empty, streamtuner will use a HTTP proxy server for data transfers.\n"
		       "\n"
		       "See streamtuner(1) for more details."),
		       NULL);
  
  tab->url_label = gtk_label_new_with_mnemonic(_("UR_L:"));
  tab->url_entry = gtk_entry_new();
  gtk_label_set_mnemonic_widget(GTK_LABEL(tab->url_label), tab->url_entry);

  gtk_tooltips_set_tip(st_shell_tooltips, tab->url_entry,
		       _("The URL of the HTTP proxy server to use.\n"
		       "\n"
		       "Must be in the form [PROTOCOL://]HOSTNAME[:PORT] (PROTOCOL will be ignored).\n"
		       "\n"
		       "See streamtuner(1) and curl_easy_setopt(3) for more details."),
		       NULL);

  tab->auth_check = gtk_check_button_new_with_mnemonic(_("Use proxy _authentication"));
  gtk_tooltips_set_tip(st_shell_tooltips, tab->auth_check,
		       _("Enable this option if your proxy server requires user authentication."),
		       NULL);

  tab->name_label = gtk_label_new_with_mnemonic(_("_Name:"));
  tab->name_entry = gtk_entry_new();
  gtk_label_set_mnemonic_widget(GTK_LABEL(tab->name_label), tab->name_entry);

  gtk_tooltips_set_tip(st_shell_tooltips, tab->name_entry,
		       _("The login name used to authenticate yourself"),
		       NULL);

  tab->password_label = gtk_label_new_with_mnemonic(_("_Password:"));
  tab->password_entry = gtk_entry_new();
  gtk_label_set_mnemonic_widget(GTK_LABEL(tab->password_label), tab->password_entry);

  gtk_entry_set_visibility(GTK_ENTRY(tab->password_entry), FALSE);
  gtk_tooltips_set_tip(st_shell_tooltips, tab->password_entry,
		       _("The password used to authenticate yourself.\n"
		       "\n"
		       "If left blank, you'll be prompted for a password upon each transfer.\n"
		       "\n"
		       "See streamtuner(1) for more details."),
		       NULL);

  gtk_misc_set_alignment(GTK_MISC(tab->url_label), 1, 0.5);
  gtk_misc_set_alignment(GTK_MISC(tab->name_label), 1, 0.5);
  gtk_misc_set_alignment(GTK_MISC(tab->password_label), 1, 0.5);

  st_preferences_bind_boolean(GTK_TOGGLE_BUTTON(tab->proxy_check),
			      &st_settings.proxy_enabled);
  st_preferences_bind_string(GTK_ENTRY(tab->url_entry),
			     &st_settings.proxy_url);
  st_preferences_bind_boolean(GTK_TOGGLE_BUTTON(tab->auth_check),
			      &st_settings.proxy_auth_enabled);
  st_preferences_bind_string(GTK_ENTRY(tab->name_entry),
			     &st_settings.proxy_auth_name);
  st_preferences_bind_string(GTK_ENTRY(tab->password_entry),
			     &st_settings.proxy_auth_password);

  g_signal_connect(G_OBJECT(tab->proxy_check), "toggled",
		   G_CALLBACK(st_network_tab_check_toggled_h), tab);
  g_signal_connect(G_OBJECT(tab->auth_check), "toggled",
		   G_CALLBACK(st_network_tab_check_toggled_h), tab);

  g_signal_connect(G_OBJECT(tab->url_entry), "activate",
		   G_CALLBACK(sgtk_focus_entry_h), tab->name_entry);
  g_signal_connect(G_OBJECT(tab->name_entry), "activate",
		   G_CALLBACK(sgtk_focus_entry_h), tab->password_entry);
  g_signal_connect(G_OBJECT(tab->password_entry), "activate",
		   G_CALLBACK(sgtk_focus_entry_h), tab->url_entry);

  tab->table = gtk_table_new(5, 2, FALSE);

  gtk_table_set_row_spacings(GTK_TABLE(tab->table), SGTK_HIG_CONTROL_SPACING);
  gtk_table_set_col_spacings(GTK_TABLE(tab->table), SGTK_HIG_CONTROL_LABEL_SPACING);

  gtk_table_attach(GTK_TABLE(tab->table), tab->proxy_check, 0, 2, 0, 1,
		   GTK_EXPAND | GTK_FILL,
		   (GtkAttachOptions) 0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(tab->table), tab->url_label, 0, 1, 1, 2,
		   GTK_FILL,
		   (GtkAttachOptions) 0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(tab->table), tab->url_entry, 1, 2, 1, 2,
		   GTK_EXPAND | GTK_FILL,
		   (GtkAttachOptions) 0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(tab->table), tab->auth_check, 0, 2, 2, 3,
		   GTK_EXPAND | GTK_FILL,
		   (GtkAttachOptions) 0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(tab->table), tab->name_label, 0, 1, 4, 5,
		   GTK_FILL,
		   (GtkAttachOptions) 0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(tab->table), tab->name_entry, 1, 2, 4, 5,
		   GTK_EXPAND | GTK_FILL,
		   (GtkAttachOptions) 0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(tab->table), tab->password_label, 0, 1, 5, 6,
		   GTK_FILL,
		   (GtkAttachOptions) 0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(tab->table), tab->password_entry, 1, 2, 5, 6,
		   GTK_EXPAND | GTK_FILL,
		   (GtkAttachOptions) 0,
		   0,
		   0);

  gtk_widget_show_all(tab->table);
  gtk_container_add(GTK_CONTAINER(tab), tab->table);

  st_network_tab_update_sensitivity(tab);
}

static void
st_network_tab_check_toggled_h (GtkToggleButton *button, gpointer data)
{
  STNetworkTab *tab = data;

  g_return_if_fail(ST_IS_NETWORK_TAB(tab));

  st_network_tab_update_sensitivity(tab);
}

static void
st_network_tab_update_sensitivity (STNetworkTab *tab)
{
  gboolean proxy, auth;

  g_return_if_fail(ST_IS_NETWORK_TAB(tab));

  proxy = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tab->proxy_check));
  auth = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tab->auth_check));

  gtk_widget_set_sensitive(tab->url_label, proxy); 
  gtk_widget_set_sensitive(tab->url_entry, proxy);
  gtk_widget_set_sensitive(tab->auth_check, proxy);
  gtk_widget_set_sensitive(tab->name_label, proxy && auth);
  gtk_widget_set_sensitive(tab->name_entry, proxy && auth);
  gtk_widget_set_sensitive(tab->password_label, proxy && auth);
  gtk_widget_set_sensitive(tab->password_entry, proxy && auth);
}

GtkWidget *
st_network_tab_new (void)
{
  return g_object_new(ST_TYPE_NETWORK_TAB, NULL);
}
