/*
 * $Id: st-handler.c,v 1.25 2004/01/30 16:14:43 jylefort Exp $
 *
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <gtk/gtk.h>
#include <string.h>
#include "gettext.h"
#include "st-category.h"
#include "st-handler.h"
#include "st-handler-event.h"
#include "st-handler-field-api.h"
#include "st-stream-api.h"
#include "sg-util.h"

typedef struct
{
  const char	*name;
  STCategory	**category;
} GetStockInfo;

/*** function declarations ***************************************************/

static STCategory  *st_handler_category_new_default_cb	(gpointer    data);
static void	   st_handler_category_free_default_cb	(STCategory  *category,
							 gpointer    data);
static STStream    *st_handler_stream_new_default_cb	(gpointer    data);
static void	   st_handler_stream_free_default_cb	(STStream    *stream,
							 gpointer    data);

static void        st_handler_categories_free		(STHandler   *handler,
							 GNode       *categories);
static gboolean    st_handler_categories_free_cb	(GNode       *node,
							 STHandler   *handler);

static void	   st_handler_streams_free		(STHandler   *handler,
							 GList       *streams);

static gboolean    st_handler_get_category_cb		(GNode       *node,
							 gpointer    data);

static void	   st_handler_selection_restore_category    (STHandler *handler);
static gboolean    st_handler_selection_restore_category_cb (GNode     *node,
							     STHandler *handler);

static void	   st_handler_selection_restore_streams     (STHandler *handler);
static void	   st_handler_selection_restore_streams_cb  (gpointer  key,
							     gpointer  value,
							     gpointer  data);

/*** API implementation ******************************************************/

STHandler *
st_handler_new (const char *name)
{
  STHandler *handler;

  handler = g_new0(STHandler, 1);

  handler->name = g_strdup(name);
  handler->streams = g_hash_table_new_full(g_str_hash,
					   g_str_equal,
					   (GDestroyNotify) g_free,
					   NULL); /*
						   * streams are freed
						   * manually using
						   * handler_streams_free()
						   */

  /* init */

  handler->paned_position = 150;
  handler->mutex = g_mutex_new();

  /* provide handy defaults for some mandatory callbacks */

  st_handler_bind(handler, ST_HANDLER_EVENT_CATEGORY_NEW, st_handler_category_new_default_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_CATEGORY_FREE, st_handler_category_free_default_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_NEW, st_handler_stream_new_default_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FREE, st_handler_stream_free_default_cb, NULL);

  return handler;
}

static STCategory *
st_handler_category_new_default_cb (gpointer data)
{
  return g_new0(STCategory, 1);
}

static void
st_handler_category_free_default_cb (STCategory *category,
						 gpointer data)
{
  st_category_free(category);
}

static STStream *
st_handler_stream_new_default_cb (gpointer data)
{
  return g_new0(STStream, 1);
}

static void
st_handler_stream_free_default_cb (STStream *stream,
					       gpointer data)
{
  st_stream_free(stream);
}

void
st_handler_set_label (STHandler *handler, const char *label)
{
  g_return_if_fail(handler != NULL);
  g_return_if_fail(label != NULL);

  handler->label = g_strdup(label);
}

void
st_handler_set_description (STHandler *handler, const char *description)
{
  g_return_if_fail(handler != NULL);
  g_return_if_fail(description != NULL);

  handler->description = g_strdup(description);
}

void
st_handler_set_home (STHandler *handler, const char *home)
{
  g_return_if_fail(handler != NULL);
  g_return_if_fail(home != NULL);

  handler->home = g_strdup(home);
}

void
st_handler_set_copyright (STHandler *handler, const char *copyright)
{
  g_return_if_fail(handler != NULL);
  g_return_if_fail(copyright != NULL);

  handler->copyright = g_strdup(copyright);
}

/*
 * Deprecated
 */
void
st_handler_set_info (STHandler *handler,
		     const char *label,
		     const char *copyright)
{
  st_handler_set_label(handler, label);
  st_handler_set_copyright(handler, copyright);
}

void
st_handler_set_icon_from_inline (STHandler *handler,
				 int size,
				 const guint8 *data)
{
  g_return_if_fail(handler != NULL);
  g_return_if_fail(size > 0);
  g_return_if_fail(data != NULL);

  handler->pixbuf = gdk_pixbuf_new_from_inline(size, data, FALSE, NULL);
  g_return_if_fail(handler->pixbuf != NULL);
}

gboolean
st_handler_set_icon_from_file (STHandler *handler,
			       const char *filename,
			       GError **err)
{
  g_return_val_if_fail(handler != NULL, FALSE);
  g_return_val_if_fail(filename != NULL, FALSE);

  handler->pixbuf = gdk_pixbuf_new_from_file(filename, err);

  return handler->pixbuf != NULL;
}

/*
 * Deprecated.
 */
void
st_handler_set_icon (STHandler *handler, int size, const guint8 *data)
{
  st_handler_set_icon_from_inline(handler, size, data);
}

void
st_handler_set_stock_categories (STHandler *handler, GNode *categories)
{
  STCategory *category;

  g_return_if_fail(handler != NULL);
  g_return_if_fail(categories != NULL);

  /* add bookmarks category */

  category = st_category_new();
  category->name = "__bookmarks";
  category->label = _("Bookmarks");

  g_node_append_data(categories, category);

  /* set categories */

  handler->stock_categories = categories;
}

void
st_handler_add_field (STHandler *handler, STHandlerField *field)
{
  g_return_if_fail(handler != NULL);
  g_return_if_fail(field != NULL);

  handler->fields = g_slist_append(handler->fields, field);
}

void
st_handler_bind (STHandler *handler,
		 STHandlerEvent event,
		 gpointer cb,
		 gpointer data)
{
  g_return_if_fail(handler != NULL);

  handler->callbacks[event].cb = cb;
  handler->callbacks[event].data = data;
}

/*** private implementation **************************************************/

gboolean
st_handler_validate (STHandler *handler, GError **err)
{
  if (! handler)
    {
      g_set_error(err, 0, 0, _("passed handler is NULL"));
      return FALSE;
    }
  if (! handler->name)
    {
      g_set_error(err, 0, 0, _("handler->name is NULL"));
      return FALSE;
    }
  if (! handler->label)
    {
      g_set_error(err, 0, 0, _("handler->label is NULL"));
      return FALSE;
    }
  if (! handler->stock_categories)
    {
      g_set_error(err, 0, 0, _("handler->stock_categories is NULL"));
      return FALSE;
    }
  if (! st_handler_event_is_bound(handler, ST_HANDLER_EVENT_REFRESH))
    {
      g_set_error(err, 0, 0, _("ST_HANDLER_EVENT_REFRESH is not bound"));
      return FALSE;
    }
  if (! st_handler_event_is_bound(handler, ST_HANDLER_EVENT_STREAM_NEW))
    {
      g_set_error(err, 0, 0, _("ST_HANDLER_EVENT_STREAM_NEW is not bound"));
      return FALSE;
    }
  if (! st_handler_event_is_bound(handler, ST_HANDLER_EVENT_STREAM_FIELD_GET))
    {
      g_set_error(err, 0, 0, _("ST_HANDLER_EVENT_STREAM_FIELD_GET is not bound"));
      return FALSE;
    }
  if (! st_handler_event_is_bound(handler, ST_HANDLER_EVENT_STREAM_FIELD_SET))
    {
      g_set_error(err, 0, 0, _("ST_HANDLER_EVENT_STREAM_FIELD_SET is not bound"));
      return FALSE;
    }
  if (! st_handler_event_is_bound(handler, ST_HANDLER_EVENT_STREAM_FREE))
    {
      g_set_error(err, 0, 0, _("ST_HANDLER_EVENT_STREAM_FREE is not bound"));
      return FALSE;
    }
  if (! st_handler_event_is_bound(handler, ST_HANDLER_EVENT_CATEGORY_NEW))
    {
      g_set_error(err, 0, 0, _("ST_HANDLER_EVENT_CATEGORY_NEW is not bound"));
      return FALSE;
    }
  if (! st_handler_event_is_bound(handler, ST_HANDLER_EVENT_CATEGORY_FREE))
    {
      g_set_error(err, 0, 0, _("ST_HANDLER_EVENT_CATEGORY_FREE is not bound"));
      return FALSE;
    }
  
  return TRUE;
}

gboolean
st_handler_refresh (STHandler *handler,
		    STCategory *category,
		    GError **err)
{
  GNode *categories = NULL;
  GList *streams = NULL;

  GList *current_streams;

  g_return_val_if_fail(handler != NULL, FALSE);

  if (! st_handler_event_refresh(handler, category, &categories, &streams, err))
    {
      if (categories)
	st_handler_categories_free(handler, categories);
      st_handler_streams_free(handler, streams);

      return FALSE;
    }
  
  g_mutex_lock(handler->mutex);

  /* save the selection */
  st_handler_selection_save(handler);
  
  /* free the current data, if any */
  if (handler->categories)
    st_handler_categories_free(handler, handler->categories);
  if (st_handler_streams_get(handler, category->name, &current_streams))
    st_handler_streams_free(handler, current_streams);
  
  /* set the new data */
  st_handler_categories_set(handler, categories);
  st_handler_streams_set(handler, category->name, streams);
  
  /* restore the selection */
  st_handler_selection_restore(handler);

  g_mutex_unlock(handler->mutex);
  
  return TRUE;
}

void
st_handler_stream_field_get (STHandler *handler,
			     STStream *stream,
			     STHandlerField *field,
			     GValue *value)
{
  g_return_if_fail(handler != NULL);
  g_return_if_fail(stream != NULL);
  g_return_if_fail(field != NULL);
  
  g_value_init(value, field->type);
  st_handler_event_stream_field_get(handler, stream, field, value);
}

void
st_handler_categories_set (STHandler *handler, GNode *categories)
{
  handler->categories = categories;
}

gboolean
st_handler_streams_get (STHandler *handler,
			const char *category,
			GList **streams)
{
  return g_hash_table_lookup_extended(handler->streams,
				      category,
				      NULL,
				      (gpointer *) streams);
}

void
st_handler_streams_set (STHandler *handler,
			const char *category,
			GList *streams)
{
  g_hash_table_insert(handler->streams, g_strdup(category), streams);
}

static void
st_handler_categories_free (STHandler *handler, GNode *categories)
{
  g_return_if_fail(handler != NULL);
  g_return_if_fail(categories != NULL);
  
  g_node_traverse(categories,
		  G_POST_ORDER,
		  G_TRAVERSE_ALL,
		  -1,
		  (GNodeTraverseFunc) st_handler_categories_free_cb,
		  handler);
}

static gboolean
st_handler_categories_free_cb (GNode *node, STHandler *handler)
{
  STCategory *category = node->data;

  if (category)			/* no category == the root node */
    st_handler_event_category_free(handler, category);
  
  g_node_unlink(node);
  g_node_destroy(node);
  
  return FALSE;
}

static void
st_handler_streams_free (STHandler *handler, GList *streams)
{
  GList *l;
  
  SG_LIST_FOREACH(l, streams)
    st_handler_event_stream_free(handler, l->data);

  g_list_free(streams);
}

/*
 * Saves the name of the selected category, and streams if any, in the
 * HANDLER->selected_category_name and HANDLER->selected_stream_names
 * members.
 *
 * HANDLER->selected_streams becomes unavailable until a call to
 * st_handler_selection_restore().
 */
void
st_handler_selection_save (STHandler *handler)
{
  GSList *l;

  g_return_if_fail(handler != NULL);
  g_return_if_fail(handler->selected_category != NULL);

  handler->selected_category_name = g_strdup(handler->selected_category->name);
  handler->selected_category = NULL;

  handler->selected_stream_names = NULL;
  SG_LIST_FOREACH(l, handler->selected_streams)
    {
      STStream *stream = l->data;

      handler->selected_stream_names =
	g_slist_append(handler->selected_stream_names, g_strdup(stream->name));
    }

  g_slist_free(handler->selected_streams);
  handler->selected_streams = NULL;
}

/*
 * Restores the HANDLER->selected_category and
 * HANDLER->selected_streams members from the names stored in
 * HANDLER->selected_category, and HANDLER->selected_streams if any,
 * and the contents of the cache.
 */
void
st_handler_selection_restore (STHandler *handler)
{
  g_return_if_fail(handler != NULL);

  if (handler->selected_category_name)
    st_handler_selection_restore_category(handler);

  if (! handler->selected_category)
    {
      /* selected category not found, try again with __main */
      handler->selected_category_name = g_strdup("__main");
      st_handler_selection_restore_category(handler);

      /* having a selected category is _mandatory_ */
      g_assert(handler->selected_category != NULL);
    }
  
  if (handler->selected_stream_names)
    st_handler_selection_restore_streams(handler);
}

static void
st_handler_selection_restore_category (STHandler *handler)
{
  g_return_if_fail(handler != NULL);
  g_return_if_fail(handler->selected_category_name != NULL);

  if (handler->stock_categories)
    g_node_traverse(handler->stock_categories,
		    G_IN_ORDER,
		    G_TRAVERSE_ALL,
		    -1,
		    (GNodeTraverseFunc) st_handler_selection_restore_category_cb,
		    handler);

  if (handler->categories)
    g_node_traverse(handler->categories,
		    G_IN_ORDER,
		    G_TRAVERSE_ALL,
		    -1,
		    (GNodeTraverseFunc) st_handler_selection_restore_category_cb,
		    handler);

  g_free(handler->selected_category_name);
  handler->selected_category_name = NULL;
}

static gboolean
st_handler_selection_restore_category_cb (GNode *node, STHandler *handler)
{
  STCategory *category = node->data;
  
  if (category)
    {
      g_return_val_if_fail(category->name != NULL, FALSE);
      
      if (! strcmp(handler->selected_category_name, category->name))
	{
	  handler->selected_category = category;
	  return TRUE;
	}
    }
  
  return FALSE;
}

static void
st_handler_selection_restore_streams (STHandler *handler)
{
  GSList *l;

  g_return_if_fail(handler != NULL);
  g_return_if_fail(handler->selected_stream_names != NULL);

  g_hash_table_foreach(handler->streams,
		       st_handler_selection_restore_streams_cb,
		       handler);
  
  /* free handler->selected_stream_names */
  SG_LIST_FOREACH(l, handler->selected_stream_names)
    g_free(l->data);
  g_slist_free(handler->selected_stream_names);
  handler->selected_stream_names = NULL;
}

static void
st_handler_selection_restore_streams_cb (gpointer key,
					 gpointer value,
					 gpointer data)
{
  STHandler *handler = data;
  GList *streams = value;
  GList *l;

  SG_LIST_FOREACH(l, streams)
    {
      STStream *stream = l->data;
      GSList *m;
      
      g_return_if_fail(stream->name != NULL);
      
      SG_LIST_FOREACH(m, handler->selected_stream_names)
	if (! strcmp(stream->name, m->data))
	  handler->selected_streams = g_slist_append(handler->selected_streams, stream);
    }
}

/*
 * Deep-copy a category
 */
STCategory *
st_handler_copy_category (STHandler *handler, STCategory *category)
{
  STCategory *copy;

  copy = st_handler_event_category_new(handler);
  g_return_val_if_fail(copy != NULL, NULL);

  copy->name = category->name ? g_strdup(category->name) : NULL;
  copy->label = category->label ? g_strdup(category->label) : NULL;
  copy->url_postfix = category->url_postfix ? g_strdup(category->url_postfix) : NULL;
  copy->url_cb = category->url_cb;

  return copy;
}

/*
 * Deep-copy a stream.
 */
STStream *
st_handler_copy_stream (STHandler *handler, STStream *stream)
{
  STStream *copy;
  GSList *l;

  copy = st_handler_event_stream_new(handler);
  g_return_val_if_fail(copy != NULL, NULL);

  copy->name = g_strdup(stream->name);

  SG_LIST_FOREACH(l, handler->fields)
    { 
      STHandlerField *field = l->data;
      GValue value = { 0, };

      st_handler_stream_field_get(handler, stream, field, &value);
      st_handler_event_stream_field_set(handler, copy, field, &value);

      g_value_unset(&value);
    }
  
  return copy;
}

STCategory *
st_handler_get_category (STHandler *handler,
			 GNode *categories,
			 const char *name)
{
  STCategory *category = NULL;
  GetStockInfo info;

  g_return_val_if_fail(categories != NULL, NULL);
  g_return_val_if_fail(name != NULL, NULL);

  info.name = name;
  info.category = &category;

  g_node_traverse(categories,
		  G_IN_ORDER,
		  G_TRAVERSE_ALL,
		  -1,
		  st_handler_get_category_cb,
		  &info);

  return category;
}

static gboolean
st_handler_get_category_cb (GNode *node, gpointer data)
{
  STCategory *category = node->data;
  GetStockInfo *info = data;

  g_return_val_if_fail(info != NULL, TRUE);

  if (category && ! strcmp(category->name, info->name))
    {
      *info->category = category;
      return TRUE;
    }

  return FALSE;
}

void
st_handler_add_bookmark (STHandler *handler, STStream *stream)
{
  STStream *copy;
  GList *bookmarks = NULL;

  g_return_if_fail(handler != NULL);
  g_return_if_fail(stream != NULL);

  st_handler_streams_get(handler, "__bookmarks", &bookmarks);

  copy = st_handler_copy_stream(handler, stream);
  bookmarks = g_list_append(bookmarks, copy);

  st_handler_streams_set(handler, "__bookmarks", bookmarks);
}

void
st_handler_remove_bookmark (STHandler *handler, STStream *stream)
{
  GList *bookmarks = NULL;
  GList *elem;

  g_return_if_fail(handler != NULL);
  g_return_if_fail(stream != NULL);

  st_handler_streams_get(handler, "__bookmarks", &bookmarks);

  elem = g_list_find(bookmarks, stream);
  if (elem)
    {
      bookmarks = g_list_remove_link(bookmarks, elem);
      st_handler_event_stream_free(handler, elem->data);
      g_list_free(elem);

      st_handler_streams_set(handler, "__bookmarks", bookmarks);
    }
}
