/*
 * $Id: st-cache-save.c,v 1.17 2004/01/26 16:42:16 jylefort Exp $
 *
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <glib.h>
#include "gettext.h"
#include "sg-dumper.h"
#include "st-category.h"
#include "st-handler.h"
#include "st-handlers.h"
#include "st-settings.h"
#include "st-stream-api.h"
#include "sg-util.h"

/*** type definitions ********************************************************/

typedef struct
{
  SGDumper	*dumper;
  GError	**err;

  STHandler	*handler;
} STCacheSaveInfo;

/*** function declarations ***************************************************/

static void	st_cache_save_handler_cb	(STHandler	  *handler,
						 STCacheSaveInfo  *info);
static gboolean	st_cache_save_category_cb	(GNode		  *node,
						 STCacheSaveInfo  *info);
static void	st_cache_save_value		(GValue           *value,
						 STCacheSaveInfo  *info);
static void	st_cache_save_stream_cb		(STStream	  *stream,
						 STCacheSaveInfo  *info);

/*** implementation **********************************************************/

gboolean
st_cache_save (const char *filename, GError **err)
{
  SGDumper *dumper;
  STCacheSaveInfo info;

  dumper = sg_dumper_new(filename, err);
  if (! dumper)
    return FALSE;

  if (! sg_dumper_write(dumper, err,
			"#", _("streamtuner cache file, automatically generated"),
			"",
			NULL))
    goto error;

  info.dumper = dumper;
  info.err = err;
  info.handler = NULL;

  g_slist_foreach(st_handlers_list, (GFunc) st_cache_save_handler_cb, &info);
  if (*err)
    goto error;

  if (! sg_dumper_free(dumper, err))
    return FALSE;

  return TRUE;

 error:
  sg_dumper_free(dumper, NULL);	/* ignore errors since ERR is already set */
  return FALSE;
}

static void
st_cache_save_handler_cb (STHandler *handler, STCacheSaveInfo *info)
{
  info->handler = handler;
  
  if (*info->err)
    return;

  if (! sg_dumper_write(info->dumper, info->err,
			"ks{", "handler", handler->name,
			NULL))
    return;

  if (handler->stock_categories)
    g_node_traverse(handler->stock_categories,
		    G_PRE_ORDER,
		    G_TRAVERSE_ALL,
		    -1,
		    (GNodeTraverseFunc) st_cache_save_category_cb,
		    info);

  if (*info->err)
    return;

  if (handler->categories)
    g_node_traverse(handler->categories,
		    G_PRE_ORDER,
		    G_TRAVERSE_ALL,
		    -1,
		    (GNodeTraverseFunc) st_cache_save_category_cb,
		    info);

  if (! sg_dumper_write(info->dumper, info->err, "}", NULL))
    return;
}

static gboolean
st_cache_save_category_cb (GNode *node, STCacheSaveInfo *info)
{
  STCategory *category = node->data;
  GList *streams;

  if (! category)
    return FALSE;		/* it's the root node */
  
  if (! sg_dumper_write(info->dumper, info->err,
			"ks{", "category", category->name,
			NULL))
    return TRUE;

  if (node->parent && node->parent->data)
    {
      STCategory *parent = node->parent->data;
      if (! sg_dumper_write(info->dumper, info->err,
			    "ks", "parent", parent->name,
			    NULL))
	return TRUE;
    }

  if (! ST_CATEGORY_IS_STOCK(category))
    {
      if (! sg_dumper_write(info->dumper, info->err,
			    "ks", "label", category->label,
			    "ks", "url_postfix", category->url_postfix,
			    NULL))
	return TRUE;
    }
  
  if (st_handler_streams_get(info->handler, category->name, &streams))
    {
      if (! sg_dumper_write(info->dumper, info->err, "k{", "streams", NULL))
	return TRUE;

      g_list_foreach(streams, (GFunc) st_cache_save_stream_cb, info);
      if (*info->err)
	return TRUE;

      if (! sg_dumper_write(info->dumper, info->err, "}", NULL))
	return TRUE;
    }

  if (*info->err)
    return TRUE;

  if (! sg_dumper_write(info->dumper, info->err, "}", NULL))
    return TRUE;

  return FALSE;
}

static void
st_cache_save_value (GValue *value, STCacheSaveInfo *info)
{
  if (G_VALUE_HOLDS_BOOLEAN(value))
    {
      if (! sg_dumper_write(info->dumper, info->err,
			    "b", g_value_get_boolean(value),
			    NULL))
	return;
    }
  else if (G_VALUE_HOLDS_INT(value))
    {
      if (! sg_dumper_write(info->dumper, info->err,
			    "i", g_value_get_int(value),
			    NULL))
	return;
    }
  else if (G_VALUE_HOLDS_STRING(value))
    {
      if (! sg_dumper_write(info->dumper, info->err,
			    "s", g_value_get_string(value),
			    NULL))
	return;
    }
  else
    g_assert_not_reached();
}

static void
st_cache_save_stream_cb (STStream *stream, STCacheSaveInfo *info)
{
  GSList *l;
  
  if (*info->err)
    return;

  if (! sg_dumper_write(info->dumper, info->err,
			"ks{", "stream", stream->name,
			"k{", "fields",
			NULL))
    return;

  SG_LIST_FOREACH(l, info->handler->fields)
    {
      STHandlerField *field = l->data;
      GValue value = { 0, };
      
      st_handler_stream_field_get(info->handler, stream, field, &value);
      st_cache_save_value(&value, info);
      g_value_unset(&value);
    }
  
  sg_dumper_write(info->dumper, info->err, "}", "}", NULL);
}
