/*
 * $Id: st-cache-load.c,v 1.18 2004/01/27 14:16:42 jylefort Exp $
 *
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <glib.h>
#include <glib-object.h>
#include "gettext.h"
#include "sg-parser.h"
#include "st-category.h"
#include "st-handler.h"
#include "st-handler-event.h"
#include "st-handlers.h"
#include "st-parser.h"
#include "st-settings.h"
#include "st-stream-api.h"

/*** type definitions ********************************************************/

enum {
  STATEMENT_HANDLER = 1,

  STATEMENT_HANDLER_CATEGORY,
  STATEMENT_HANDLER_CATEGORY_PARENT,
  STATEMENT_HANDLER_CATEGORY_LABEL,
  STATEMENT_HANDLER_CATEGORY_URL_POSTFIX,
  
  STATEMENT_HANDLER_CATEGORY_STREAM,			/* deprecated */
  STATEMENT_HANDLER_CATEGORY_STREAM_FIELDS,		/* deprecated */

  STATEMENT_HANDLER_CATEGORY_STREAMS,
  STATEMENT_HANDLER_CATEGORY_STREAMS_STREAM,
  STATEMENT_HANDLER_CATEGORY_STREAMS_STREAM_FIELDS,
};

typedef struct
{
  SGParser		*parser;		/* the SGParser object */
  SGParserStatement	*statement;		/* the current statement */

  /* currently parsed handler */

  STHandler		*handler;
  GNode			*handler_categories;
  GHashTable		*handler_parents;	/* temporary hash used to store category parent nodes */
  
  /* currently parsed category */

  STCategory		*category;
  char			*category_name;
  char			*category_parent_name;
  gboolean		category_has_streams;
  GList			*category_streams;

  /* currently parsed stream */

  STStream		*stream;
  GSList		*stream_field_iter;
} STCacheLoadInfo;
  
/*** constant definitions ****************************************************/

static SGParserDefinition cache_definitions[] = {
  {
    0,
    STATEMENT_HANDLER,
    "handler",		TRUE,		G_TYPE_STRING
  },
  {
    STATEMENT_HANDLER,
    STATEMENT_HANDLER_CATEGORY,
    "category",		TRUE,		G_TYPE_STRING
  },
  {
    STATEMENT_HANDLER_CATEGORY,
    STATEMENT_HANDLER_CATEGORY_PARENT,
    "parent",		FALSE,		G_TYPE_STRING
  },
  {
    STATEMENT_HANDLER_CATEGORY,
    STATEMENT_HANDLER_CATEGORY_LABEL,
    "label",		FALSE,		G_TYPE_STRING
  },
  {
    STATEMENT_HANDLER_CATEGORY,
    STATEMENT_HANDLER_CATEGORY_URL_POSTFIX,
    "url_postfix",	FALSE,		G_TYPE_STRING
  },
  {				/* deprecated */
    STATEMENT_HANDLER_CATEGORY,
    STATEMENT_HANDLER_CATEGORY_STREAM,
    "stream",		TRUE,		G_TYPE_STRING
  },
  {				/* deprecated */
    STATEMENT_HANDLER_CATEGORY_STREAM,
    STATEMENT_HANDLER_CATEGORY_STREAM_FIELDS,
    "fields",		TRUE,		G_TYPE_NONE
  },
  {
    STATEMENT_HANDLER_CATEGORY,
    STATEMENT_HANDLER_CATEGORY_STREAMS,
    "streams",		TRUE,		G_TYPE_NONE
  },
  {
    STATEMENT_HANDLER_CATEGORY_STREAMS,
    STATEMENT_HANDLER_CATEGORY_STREAMS_STREAM,
    "stream",		TRUE,		G_TYPE_STRING
  },
  {
    STATEMENT_HANDLER_CATEGORY_STREAMS_STREAM,
    STATEMENT_HANDLER_CATEGORY_STREAMS_STREAM_FIELDS,
    "fields",		TRUE,		G_TYPE_NONE
  },
  { 0, 0, NULL, 0, 0 }
};

/*** function declarations ***************************************************/

static void st_cache_load_handler_begin		(STCacheLoadInfo *info);
static void st_cache_load_handler_end		(STCacheLoadInfo *info);

static void st_cache_load_category_begin	(STCacheLoadInfo *info);
static void st_cache_load_category_end		(STCacheLoadInfo *info);

static void st_cache_load_stream_begin		(STCacheLoadInfo *info);
static void st_cache_load_stream_end		(STCacheLoadInfo *info);

static void st_cache_load_stream_field		(STCacheLoadInfo *info);

/*** implementation **********************************************************/

gboolean
st_cache_load (const char *filename, GError **err)
{
  STCacheLoadInfo info = { 0, };
  
  info.parser = sg_parser_new(filename, err);
  if (! info.parser)
    return FALSE;

  sg_parser_set_message_cb(info.parser, st_parser_message_cb);
  sg_parser_definev(info.parser, cache_definitions);

  while ((info.statement = sg_parser_get_statement(info.parser)))
    {
      if (SG_PARSER_STATEMENT_IS_END(info.statement))
	{
	  switch (SG_PARSER_SCOPE(info.parser))
	    {
	    case STATEMENT_HANDLER:
	      st_cache_load_handler_end(&info);
	      break;
	      
	    case STATEMENT_HANDLER_CATEGORY:
	      st_cache_load_category_end(&info);
	      break;

	    case STATEMENT_HANDLER_CATEGORY_STREAM:
	    case STATEMENT_HANDLER_CATEGORY_STREAMS_STREAM:
	      st_cache_load_stream_end(&info);
	      break;
	    }
	}
      
      if (info.statement->definition)
	{
	  switch (info.statement->definition->id)
	    {
	    case STATEMENT_HANDLER:
	      st_cache_load_handler_begin(&info);
	      break;
	      
	    case STATEMENT_HANDLER_CATEGORY:
	      st_cache_load_category_begin(&info);
	      break;
	      
	    case STATEMENT_HANDLER_CATEGORY_PARENT:
	      if (info.handler && info.category)
		info.category_parent_name = g_value_dup_string(&info.statement->value);
	      break;
	      
	    case STATEMENT_HANDLER_CATEGORY_LABEL:
	      if (info.handler && info.category)
		info.category->label = g_value_dup_string(&info.statement->value);
	      break;
	      
	    case STATEMENT_HANDLER_CATEGORY_URL_POSTFIX:
	      if (info.handler)
		{
		  g_assert(info.category != NULL);
		  info.category->url_postfix = g_value_dup_string(&info.statement->value);
		}
	      break;
	      
	    case STATEMENT_HANDLER_CATEGORY_STREAMS:
	      info.category_has_streams = TRUE;
	      break;

	    case STATEMENT_HANDLER_CATEGORY_STREAM:		/* deprecated */
	    case STATEMENT_HANDLER_CATEGORY_STREAMS_STREAM:
	      st_cache_load_stream_begin(&info);
	      break;
	      
	    case STATEMENT_HANDLER_CATEGORY_STREAM_FIELDS:	/* deprecated */
	    case STATEMENT_HANDLER_CATEGORY_STREAMS_STREAM_FIELDS:
	      if (info.handler)
		{
		  g_assert(info.stream != NULL);
		  info.stream_field_iter = info.handler->fields;
		}
	      break;
	    }
	}
      else if (G_IS_VALUE(&info.statement->value))
	{
	  switch (SG_PARSER_SCOPE(info.parser))
	    {
	    case STATEMENT_HANDLER_CATEGORY_STREAM_FIELDS:
	    case STATEMENT_HANDLER_CATEGORY_STREAMS_STREAM_FIELDS:
	      st_cache_load_stream_field(&info);
	      break;

	    default:
	      sg_parser_warn(info.parser, _("unexpected value"));
	    }
	}

      sg_parser_statement_evaluate(info.statement);
      sg_parser_statement_free(info.statement);
    }

  sg_parser_free(info.parser);
  
  return TRUE;
}

static void
st_cache_load_handler_begin (STCacheLoadInfo *info)
{
  const char *name;

  name = g_value_get_string(&info->statement->value);
  info->handler = st_handlers_find_by_name(name);

  if (info->handler)
    {
      info->handler_parents = g_hash_table_new(g_str_hash, g_str_equal);
      info->handler_categories = g_node_new(NULL);
    }
  else
    sg_parser_warn(info->parser, _("%s: no such handler"), name);
}

static void
st_cache_load_handler_end (STCacheLoadInfo *info)
{
  if (! info->handler)
    return;

  g_hash_table_destroy(info->handler_parents);
  st_handler_categories_set(info->handler, info->handler_categories);
  
  info->handler = NULL;
  info->handler_categories = NULL;
  info->handler_parents = NULL;
}

static void
st_cache_load_category_begin (STCacheLoadInfo *info)
{
  if (! info->handler)
    return;

  info->category_name = g_value_dup_string(&info->statement->value);

  if (ST_CATEGORY_NAME_IS_STOCK(info->category_name))
    {
      info->category = st_handler_get_category(info->handler,
					       info->handler->stock_categories,
					       info->category_name);

      if (! info->category)
	sg_parser_warn(info->parser, _("%s: no such stock category"), info->category_name);
    }
  else
    {
      info->category = st_handler_event_category_new(info->handler);
      info->category->name = g_value_dup_string(&info->statement->value);
    }
}

static void
st_cache_load_category_end (STCacheLoadInfo *info)
{
  if (! info->handler)
    return;

  g_assert(info->category != NULL);
  g_assert(info->category_name != NULL);

  if (! ST_CATEGORY_IS_STOCK(info->category))
    {
      GNode *parent = NULL;

      if (info->category_parent_name)
	{
	  parent = g_hash_table_lookup(info->handler_parents,
				       info->category_parent_name);
	  if (! parent)
	    sg_parser_warn(info->parser,
			_("parent category \"%s\" not found"),
			info->category_parent_name);
	}
      else
	parent = info->handler_categories;

      if (parent)
	{
	  GNode *node;

	  node = g_node_append_data(parent, info->category);
	  g_hash_table_insert(info->handler_parents,
			      info->category->name,
			      node);
	}
      else
	{
	  sg_parser_warn(info->parser, _("unable to add category because it has no parent"));
	  st_handler_event_category_free(info->handler, info->category);
	}
    }
  
  if (info->category_has_streams)
    st_handler_streams_set(info->handler,
			   info->category_name,
			   info->category_streams);

  g_free(info->category_name);
  g_free(info->category_parent_name);

  info->category = NULL;
  info->category_name = NULL;
  info->category_parent_name = NULL;
  info->category_has_streams = FALSE;
  info->category_streams = NULL;
}

static void
st_cache_load_stream_begin (STCacheLoadInfo *info)
{
  if (! info->handler)
    return;

  info->category_has_streams = TRUE;
  info->stream = st_handler_event_stream_new(info->handler);
  info->stream->name = g_value_dup_string(&info->statement->value);
}

static void
st_cache_load_stream_end (STCacheLoadInfo *info)
{
  if (! info->handler)
    return;

  g_assert(info->stream != NULL);

  info->category_streams = g_list_append(info->category_streams, info->stream);

  info->stream = NULL;
  info->stream_field_iter = NULL;
}

static void
st_cache_load_stream_field (STCacheLoadInfo *info)
{
  if (! info->handler)
    return;

  g_assert(info->stream != NULL);

  if (info->stream_field_iter)
    {
      STHandlerField *field = info->stream_field_iter->data;
      
      if (G_VALUE_HOLDS(&info->statement->value, field->type))
	st_handler_event_stream_field_set(info->handler,
					  info->stream,
					  field,
					  &info->statement->value);
      else
	sg_parser_warn(info->parser, _("wrong field type"));
		      
      info->stream_field_iter = info->stream_field_iter->next;
    }
  else
    sg_parser_warn(info->parser, _("too many fields"));
}
