/*
 * $Id: st-browser-tab.c,v 1.67.2.2 2004/02/07 18:45:09 jylefort Exp $
 *
 * Copyright (c) 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <gtk/gtk.h>
#include "gettext.h"
#include "sgtk-tree-view.h"
#include "sgtk-util.h"
#include "sg-util.h"
#include "sg-printable.h"
#include "st-browser-tab.h"
#include "st-browser-tab-label.h"
#include "st-browser-window.h"
#include "st-category.h"
#include "st-category-view.h"
#include "st-dialog.h"
#include "st-handler-event.h"
#include "st-statusbar.h"
#include "st-stream-view.h"
#include "st-handler.h"
#include "st-thread.h"
#include "st-shell.h"
#include "st-settings.h"

/*** type definitions ********************************************************/

enum {
  PROP_0,
  PROP_WINDOW,
  PROP_HANDLER
};
    
typedef struct
{
  STBrowserTab		*tab;
  STCategory		*category;
} RefreshThreadInfo;

/*** variable declarations ***************************************************/

static GObjectClass *parent_class = NULL;

/*** function declarations ***************************************************/

static void st_browser_tab_class_init	(STBrowserTabClass	*class);
static void st_browser_tab_init		(STBrowserTab		*tab);

static GObject *st_browser_tab_constructor (GType type,
					    guint n_construct_properties,
					    GObjectConstructParam *construct_params);

static void st_browser_tab_set_property	(GObject	*object,
					 guint		prop_id,
					 const GValue	*value,
					 GParamSpec	*pspec);
static void st_browser_tab_finalize	(GObject	*object);
static void st_browser_tab_unrealize	(GtkWidget	*widget);

static void st_browser_tab_construct_view	(STBrowserTab	*tab,
						 GType		type,
						 GtkWidget	**scrolled_ptr,
						 GtkWidget	**view_ptr);
static void st_browser_tab_refresh_thread	(gpointer	data);
static void st_browser_tab_update_blinking	(STBrowserTab	*tab);
static void st_browser_tab_set_contents		(STBrowserTab	*tab,
						 STCategory	*category,
						 GNode		*stock_categories,
						 GNode		*categories,
						 GList		*streams,
						 gboolean	set_categories,
						 gboolean	set_streams);
static void st_browser_tab_update_counters	(STBrowserTab	*tab);
static void st_browser_tab_set_refreshing	(STBrowserTab	*tab,
						 gboolean	refreshing);

/*** implementation **********************************************************/

GType
st_browser_tab_get_type (void)
{
  static GType browser_tab_type = 0;
  
  if (! browser_tab_type)
    {
      static const GTypeInfo browser_tab_info = {
	sizeof(STBrowserTabClass),
	NULL,
	NULL,
	(GClassInitFunc) st_browser_tab_class_init,
	NULL,
	NULL,
	sizeof(STBrowserTab),
	0,
	(GInstanceInitFunc) st_browser_tab_init,
      };
      
      browser_tab_type = g_type_register_static(GTK_TYPE_HPANED,
						"STBrowserTab",
						&browser_tab_info,
						0);
    }

  return browser_tab_type;
}

static void
st_browser_tab_class_init (STBrowserTabClass *class)
{
  GObjectClass *object_class = G_OBJECT_CLASS(class);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(class);

  parent_class = g_type_class_peek_parent(class);

  object_class->constructor = st_browser_tab_constructor;
  object_class->set_property = st_browser_tab_set_property;
  object_class->finalize = st_browser_tab_finalize;

  widget_class->unrealize = st_browser_tab_unrealize;

  g_object_class_install_property(object_class,
				  PROP_WINDOW,
				  g_param_spec_object("window",
						      "Window",
						      _("The tab's parent window"),
						      ST_TYPE_BROWSER_WINDOW,
						      G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
  g_object_class_install_property(object_class,
				  PROP_HANDLER,
				  g_param_spec_pointer("handler",
						       "Handler",
						       _("The tab's STHandler object"),
						       G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}

static void
st_browser_tab_init (STBrowserTab *tab)
{
  tab->window = NULL;

  tab->labels = NULL;

  tab->handler = NULL;
  tab->statusbar = NULL;
  tab->statusbar_depth = -1;

  tab->refreshing = FALSE;

  tab->thread = NULL;
  
  tab->category_scrolled = NULL;
  tab->category_view = NULL;
  
  tab->stream_scrolled = NULL;
  tab->stream_view = NULL;
}

static GObject *
st_browser_tab_constructor (GType type,
			    guint n_construct_properties,
			    GObjectConstructParam *construct_params)
{
  GObject *object;
  STBrowserTab *tab;

  object = G_OBJECT_CLASS(parent_class)->constructor(type,
						     n_construct_properties,
						     construct_params);
  tab = ST_BROWSER_TAB(object);

  g_return_val_if_fail(tab->window != NULL, NULL);
  g_return_val_if_fail(tab->handler != NULL, NULL);

  tab->statusbar = st_statusbar_new();
  gtk_widget_show(tab->statusbar);
  g_object_ref(tab->statusbar);

  st_browser_tab_construct_view(tab, ST_TYPE_CATEGORY_VIEW,
				&tab->category_scrolled,
				&tab->category_view);
  st_browser_tab_construct_view(tab, ST_TYPE_STREAM_VIEW,
				&tab->stream_scrolled,
				&tab->stream_view);
  
  gtk_paned_add1(GTK_PANED(tab), tab->category_scrolled);
  gtk_paned_add2(GTK_PANED(tab), tab->stream_scrolled);

  gtk_paned_set_position(GTK_PANED(tab), tab->handler->paned_position);

  return object;
}

static void
st_browser_tab_set_property (GObject *object,
			     guint prop_id,
			     const GValue *value,
			     GParamSpec *pspec)
{
  STBrowserTab *tab = ST_BROWSER_TAB(object);

  switch (prop_id)
    {
    case PROP_WINDOW:
      tab->window = g_value_get_object(value);
      break;

    case PROP_HANDLER:
      tab->handler = g_value_get_pointer(value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
    }
}

static void
st_browser_tab_finalize (GObject *object)
{
  STBrowserTab *tab = ST_BROWSER_TAB(object);

  g_object_unref(tab->statusbar);

  G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
st_browser_tab_unrealize (GtkWidget *widget)
{
  STBrowserTab *tab = ST_BROWSER_TAB(widget);

  tab->handler->paned_position = gtk_paned_get_position(GTK_PANED(tab));

  GTK_WIDGET_CLASS(parent_class)->unrealize(widget);
}

static void
st_browser_tab_construct_view (STBrowserTab *tab,
			       GType type,
			       GtkWidget **scrolled_ptr,
			       GtkWidget **view_ptr)
{
  GtkWidget *scrolled;
  GtkWidget *view;

  g_return_if_fail(ST_IS_BROWSER_TAB(tab));
  g_return_if_fail(type == ST_TYPE_CATEGORY_VIEW || type == ST_TYPE_STREAM_VIEW);

  scrolled = gtk_scrolled_window_new(FALSE, FALSE);
  view = g_object_new(type,
		      "tab", tab,
		      NULL);

  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  gtk_widget_show(scrolled);
  gtk_widget_show(view);

  gtk_container_add(GTK_CONTAINER(scrolled), view);

  if (scrolled_ptr)
    *scrolled_ptr = scrolled;
  if (view_ptr)
    *view_ptr = view;
}

GtkWidget *
st_browser_tab_get_label (STBrowserTab *tab)
{
  GtkWidget *label;

  g_return_val_if_fail(ST_IS_BROWSER_TAB(tab), NULL);

  /*
   * A widget can't be shared by multiple containers, so we create a
   * new label for every call of this function.
   */

  label = st_browser_tab_label_new(tab->handler);
  tab->labels = g_slist_append(tab->labels, label);

  return label;
}

void
st_browser_tab_set_label_icon_visible (STBrowserTab *tab, gboolean visible)
{
  GSList *l;

  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  SG_LIST_FOREACH(l, tab->labels)
    st_browser_tab_label_set_icon_visible(ST_BROWSER_TAB_LABEL(l->data), visible);
}

GtkWidget *
st_browser_tab_new (GtkWidget *window, STHandler *handler)
{
  return g_object_new(ST_TYPE_BROWSER_TAB,
		      "window", window,
		      "handler", handler,
		      NULL);
}

void
st_browser_tab_update_sensitivity (STBrowserTab *tab)
{
  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  st_stream_view_update_sensitivity(ST_STREAM_VIEW(tab->stream_view));
}

/*
 * Returns true if the tab contains no categories nor streams.
 */
gboolean
st_browser_tab_is_empty (STBrowserTab *tab)
{
  g_return_val_if_fail(ST_IS_BROWSER_TAB(tab), FALSE);

  return sgtk_tree_view_is_empty(GTK_TREE_VIEW(tab->category_view))
    && sgtk_tree_view_is_empty(GTK_TREE_VIEW(tab->stream_view));
}

static void
st_browser_tab_update_blinking (STBrowserTab *tab)
{
  GSList *l;

  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  SG_LIST_FOREACH(l, tab->labels)
    st_browser_tab_label_set_blinking(ST_BROWSER_TAB_LABEL(l->data), tab->refreshing);
}

void
st_browser_tab_refresh (STBrowserTab *tab)
{
  STCategory *category = NULL;
  RefreshThreadInfo *info;

  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  /*
   * If the refresh is successful, the selected category will be freed
   * in st_browser_tab_refresh_thread() by st_handler_refresh(), so we
   * use our own copy.
   */
  g_mutex_lock(tab->handler->mutex);
  if (st_category_apply_url_cb(tab->handler->selected_category))
    category = st_handler_copy_category(tab->handler, tab->handler->selected_category);
  g_mutex_unlock(tab->handler->mutex);

  if (! category)		/* st_category_apply_url_cb() failed */
    return;

  /*
   * The selected_category label might have been changed by url_cb, so
   * we update the window title.
   */
  st_browser_window_update_title(ST_BROWSER_WINDOW(tab->window));

  if (tab->refreshing)
    {
      st_browser_tab_abort(tab);
      gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(ST_STATUSBAR(tab->statusbar)->progress_bar), 0);
    }
  else
    st_browser_tab_set_refreshing(tab, TRUE);
  
  info = g_new(RefreshThreadInfo, 1);
  info->tab = tab;
  info->category = category;
  
  tab->thread = st_thread_new(ST_THREAD_TYPE_REFRESH,
			      tab->handler,
			      st_browser_tab_refresh_thread,
			      info);

  tab->thread->tab = GTK_WIDGET(tab);
  tab->thread->printable = SG_PRINTABLE(tab->statusbar);
  tab->thread->progress_bar = ST_STATUSBAR(tab->statusbar)->progress_bar;

  st_thread_run(tab->thread);
}

static void
st_browser_tab_refresh_thread (gpointer data)
{
  RefreshThreadInfo *info = data;
  gboolean status;
  GError *err = NULL;

  g_return_if_fail(info != NULL);
  g_return_if_fail(ST_IS_BROWSER_TAB(info->tab));

  status = st_handler_refresh(info->tab->handler, info->category, &err);

  if (status)
    {
      GList *streams;

      status = st_handler_streams_get(info->tab->handler,
				      info->category->name,
				      &streams);
      g_return_if_fail(status == TRUE);

      GDK_THREADS_ENTER();
      st_browser_tab_set_refreshing(info->tab, FALSE);
      st_browser_tab_set_contents(info->tab,
				  info->category,
				  info->tab->handler->stock_categories,
				  info->tab->handler->categories,
				  streams,
				  TRUE,
				  TRUE);
      st_browser_tab_streams_selected(info->tab);
      GDK_THREADS_LEAVE();
    }
  else
    {
      if (err)			/* err unset means "aborted legally" */
	{
	  char *normalized;

	  GDK_THREADS_ENTER();
	  st_browser_tab_set_refreshing(info->tab, FALSE);
	  st_browser_tab_update_counters(info->tab);
	  GDK_THREADS_LEAVE();
	  
	  normalized = st_dialog_normalize(err->message);

	  st_error_dialog(_("Unable to refresh."), "%s", normalized);

	  g_free(normalized);
	  g_error_free(err);
	}
    }
  
  st_handler_event_category_free(info->tab->handler, info->category);
  g_free(info);
}

gboolean
st_browser_tab_can_stop (STBrowserTab *tab)
{
  g_return_val_if_fail(ST_IS_BROWSER_TAB(tab), FALSE);

  return tab->refreshing;
}

void
st_browser_tab_abort (STBrowserTab *tab)
{
  g_return_if_fail(ST_IS_BROWSER_TAB(tab));
  g_return_if_fail(st_browser_tab_can_stop(tab));

  st_thread_abort(tab->thread);
}

void
st_browser_tab_stop (STBrowserTab *tab)
{
  g_return_if_fail(ST_IS_BROWSER_TAB(tab));
  g_return_if_fail(st_browser_tab_can_stop(tab));

  st_browser_tab_abort(tab);
  st_browser_tab_set_refreshing(tab, FALSE);
  st_browser_tab_update_counters(tab);
}

static void
st_browser_tab_set_refreshing (STBrowserTab *tab, gboolean refreshing)
{
  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  tab->refreshing = refreshing;

  st_statusbar_set_active(ST_STATUSBAR(tab->statusbar), refreshing);
  st_browser_tab_update_blinking(tab);
  st_browser_window_update_sensitivity(ST_BROWSER_WINDOW(tab->window));
}

void
st_browser_tab_update (STBrowserTab *tab)
{
  gboolean is_bookmarks;
  gboolean found = FALSE;
  GList *streams = NULL;

  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  g_mutex_lock(tab->handler->mutex);
  is_bookmarks = ST_CATEGORY_IS_BOOKMARKS(tab->handler->selected_category);
  found = st_handler_streams_get(tab->handler, tab->handler->selected_category->name, &streams);
  g_mutex_unlock(tab->handler->mutex);

  if (is_bookmarks || found)
    {
      if (tab->refreshing)
	{
	  st_browser_tab_abort(tab);
	  st_browser_tab_set_refreshing(tab, FALSE);
	}
	  
      g_mutex_lock(tab->handler->mutex);
      st_browser_tab_set_contents(tab,
				  tab->handler->selected_category,
				  tab->handler->stock_categories,
				  tab->handler->categories,
				  streams,
				  sgtk_tree_view_is_empty(GTK_TREE_VIEW(tab->category_view)),
				  TRUE);
      g_mutex_unlock(tab->handler->mutex);
    }

  if ((! is_bookmarks) && ((! found) || st_settings.always_refresh))
    st_browser_tab_refresh(tab);
}

static void
st_browser_tab_set_contents (STBrowserTab *tab,
			     STCategory *category,
			     GNode *stock_categories,
			     GNode *categories,
			     GList *streams,
			     gboolean set_categories,
			     gboolean set_streams)
{
  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  /* FIXME: this has no effect from a thread */
  sg_printable_print(SG_PRINTABLE(tab->statusbar), _("Updating..."));
  sgtk_flush();

  if (set_categories)
    {
      STCategoryView *view = ST_CATEGORY_VIEW(tab->category_view);

      st_category_view_clear(view);
      st_category_view_append_categories(view, stock_categories);
      st_category_view_append_categories(view, categories);
    }

  if (set_streams)
    {
      STStreamView *view = ST_STREAM_VIEW(tab->stream_view);

      st_stream_view_clear(view);
      st_stream_view_append_streams(view, streams);
    }

  st_browser_tab_update_counters(tab);
}

static void
st_browser_tab_update_counters (STBrowserTab *tab)
{
  char *categories;
  char *streams;
  int n_categories;
  int n_streams;

  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  sg_printable_print(SG_PRINTABLE(tab->statusbar), _("Counting..."));
  sgtk_flush();

  n_categories = sgtk_tree_view_get_count(GTK_TREE_VIEW(tab->category_view));
  n_streams = sgtk_tree_view_get_count(GTK_TREE_VIEW(tab->stream_view));

  categories = g_strdup_printf(ngettext("%i category",
					"%i categories",
					n_categories),
			       n_categories);

  streams = g_strdup_printf(ngettext("%i stream",
				     "%i streams",
				     n_streams),
			    n_streams);

  sg_printable_printf(SG_PRINTABLE(tab->statusbar),
		      _("%s, %s"),
		      categories,
		      streams);

  g_free(categories);
  g_free(streams);
}

void
st_browser_tab_category_selected (STBrowserTab *tab)
{
  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  st_stream_view_clear(ST_STREAM_VIEW(tab->stream_view));
  st_browser_window_update_title(ST_BROWSER_WINDOW(tab->window));
  st_browser_window_update_sensitivity(ST_BROWSER_WINDOW(tab->window));

  st_browser_tab_update(tab);
}

void
st_browser_tab_streams_selected (STBrowserTab *tab)
{
  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  st_browser_window_update_sensitivity(ST_BROWSER_WINDOW(tab->window));
  st_shell_streams_selected();
}
