/*
 * $Id: sgtk-auth-dialog.c,v 1.23 2004/01/05 19:44:09 jylefort Exp $
 *
 * Copyright (c) 2003 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <stdarg.h>
#include "art/auth.h"
#include "gettext.h"
#include "sgtk-auth-dialog.h"
#include "sgtk-hig.h"

/*** function declarations ***************************************************/

static void sgtk_auth_dialog_class_init	(sGtkAuthDialogClass	*class);
static void sgtk_auth_dialog_init	(sGtkAuthDialog		*dialog);

static gboolean	sgtk_auth_dialog_entry_key_press_h  (GtkWidget    *widget,
						     GdkEventKey  *event,
						     gpointer     data);
     
/*** implementation **********************************************************/

GType
sgtk_auth_dialog_get_type (void)
{
  static GType auth_dialog_type = 0;
  
  if (! auth_dialog_type)
    {
      static const GTypeInfo auth_dialog_info = {
	sizeof(sGtkAuthDialogClass),
	NULL,
	NULL,
	(GClassInitFunc) sgtk_auth_dialog_class_init,
	NULL,
	NULL,
	sizeof(sGtkAuthDialog),
	0,
	(GInstanceInitFunc) sgtk_auth_dialog_init,
      };
      
      auth_dialog_type = g_type_register_static(SGTK_TYPE_ALERT_DIALOG,
						"sGtkAuthDialog",
						&auth_dialog_info,
						0);
    }

  return auth_dialog_type;
}

static void
sgtk_auth_dialog_class_init (sGtkAuthDialogClass *class)
{
  GtkIconFactory *factory;
  GtkIconSet *icon_set;
  GdkPixbuf *pixbuf;
  
  factory = gtk_icon_factory_new();
  gtk_icon_factory_add_default(factory);

  pixbuf = gdk_pixbuf_new_from_inline(sizeof(art_auth),
				      art_auth,
				      FALSE,
				      NULL);
  g_assert(pixbuf != NULL);

  icon_set = gtk_icon_set_new_from_pixbuf(pixbuf);
  gtk_icon_factory_add(factory, SGTK_STOCK_AUTH, icon_set);

  gtk_icon_set_unref(icon_set);
  g_object_unref(pixbuf);
  g_object_unref(factory);
}

static void
sgtk_auth_dialog_init (sGtkAuthDialog *dialog)
{
  GtkWidget *ok;

  gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
  ok = gtk_dialog_add_button(GTK_DIALOG(dialog), _("_Authenticate"), GTK_RESPONSE_OK);

  gtk_window_set_default(GTK_WINDOW(dialog), ok);

  gtk_image_set_from_stock(GTK_IMAGE(SGTK_ALERT_DIALOG(dialog)->image),
			   SGTK_STOCK_AUTH,
			   GTK_ICON_SIZE_DIALOG);

  dialog->label = gtk_label_new(NULL);

  gtk_misc_set_alignment(GTK_MISC(dialog->label),
			 SGTK_HIG_ALERT_LABEL_X_ALIGN,
			 SGTK_HIG_ALERT_LABEL_Y_ALIGN);
  gtk_label_set_line_wrap(GTK_LABEL(dialog->label), TRUE);
  gtk_label_set_selectable(GTK_LABEL(dialog->label), TRUE);

  dialog->name_label = gtk_label_new_with_mnemonic(_("_Name:"));
  dialog->name_entry = gtk_entry_new();

  dialog->password_label = gtk_label_new_with_mnemonic(_("_Password:"));
  dialog->password_entry = gtk_entry_new();

  gtk_label_set_mnemonic_widget(GTK_LABEL(dialog->name_label), dialog->name_entry);
  gtk_label_set_mnemonic_widget(GTK_LABEL(dialog->password_label), dialog->password_entry);

  gtk_misc_set_alignment(GTK_MISC(dialog->name_label), 0, 0.5);
  gtk_misc_set_alignment(GTK_MISC(dialog->password_label), 0, 0.5);
  gtk_entry_set_visibility(GTK_ENTRY(dialog->password_entry), FALSE);

  g_signal_connect(G_OBJECT(dialog->name_entry), "key-press-event",
		   G_CALLBACK(sgtk_auth_dialog_entry_key_press_h), dialog);
  g_signal_connect(G_OBJECT(dialog->password_entry), "key-press-event",
		   G_CALLBACK(sgtk_auth_dialog_entry_key_press_h), dialog);

  dialog->table = gtk_table_new(3, 2, FALSE);

  gtk_table_set_row_spacings(GTK_TABLE(dialog->table), SGTK_HIG_CONTROL_SPACING);
  gtk_table_set_col_spacings(GTK_TABLE(dialog->table), SGTK_HIG_CONTROL_LABEL_SPACING);
  
  gtk_table_attach(GTK_TABLE(dialog->table), dialog->label, 0, 2, 0, 1,
		   GTK_FILL,
		   (GtkAttachOptions) 0,
		   0,
		   0);
  
  gtk_table_attach(GTK_TABLE(dialog->table), dialog->name_label, 0, 1, 1, 2,
		   GTK_FILL,
		   (GtkAttachOptions) 0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(dialog->table), dialog->name_entry, 1, 2, 1, 2,
		   GTK_EXPAND | GTK_FILL,
		   (GtkAttachOptions) 0,
		   0,
		   0);

  gtk_table_attach(GTK_TABLE(dialog->table), dialog->password_label, 0, 1, 2, 3,
		   GTK_FILL,
		   (GtkAttachOptions) 0,
		   0,
		   0);
  gtk_table_attach(GTK_TABLE(dialog->table), dialog->password_entry, 1, 2, 2, 3,
		   GTK_EXPAND | GTK_FILL,
		   (GtkAttachOptions) 0,
		   0,
		   0);
 
  gtk_box_pack_start(GTK_BOX(SGTK_ALERT_DIALOG(dialog)->hbox), dialog->table, FALSE, FALSE, 0);

  gtk_widget_show_all(dialog->table);
}

static gboolean
sgtk_auth_dialog_entry_key_press_h (GtkWidget *widget,
				    GdkEventKey *event,
				    gpointer data)
{
  if (event->keyval == GDK_Return)
    {
      sGtkAuthDialog *dialog = data;

      g_return_val_if_fail(SGTK_IS_AUTH_DIALOG(dialog), FALSE);

      if (widget == dialog->name_entry)
	gtk_window_set_focus(GTK_WINDOW(dialog), dialog->password_entry);
      else if (widget == dialog->password_entry)
	gtk_window_activate_default(GTK_WINDOW(dialog));
      else
	g_assert_not_reached();
    }

  return FALSE;
}

GtkWidget *
sgtk_auth_dialog_new (GtkWindow *parent, const char *format, ...)
{
  sGtkAuthDialog *dialog;
  va_list args;
  char *message;

  g_return_val_if_fail(format != NULL, NULL);

  dialog = g_object_new(SGTK_TYPE_AUTH_DIALOG, NULL);

  if (parent)
    gtk_window_set_transient_for(GTK_WINDOW(dialog), parent);
  
  va_start(args, format);
  message = g_strdup_vprintf(format, args);
  va_end(args);

  gtk_label_set_text(GTK_LABEL(dialog->label), message);
  g_free(message);

  return GTK_WIDGET(dialog);
}

void
sgtk_auth_dialog_set_name_sensitive (sGtkAuthDialog *dialog,
				     gboolean sensitive)
{
  g_return_if_fail(SGTK_IS_AUTH_DIALOG(dialog));

  gtk_widget_set_sensitive(dialog->name_entry, sensitive);
}

void
sgtk_auth_dialog_set_name (sGtkAuthDialog *dialog, const char *name)
{
  g_return_if_fail(SGTK_IS_AUTH_DIALOG(dialog));

  gtk_entry_set_text(GTK_ENTRY(dialog->name_entry), name);
}

void
sgtk_auth_dialog_set_password (sGtkAuthDialog *dialog, const char *password)
{
  g_return_if_fail(SGTK_IS_AUTH_DIALOG(dialog));

  gtk_entry_set_text(GTK_ENTRY(dialog->password_entry), password);
}

const char *
sgtk_auth_dialog_get_name (sGtkAuthDialog *dialog)
{
  g_return_val_if_fail(SGTK_IS_AUTH_DIALOG(dialog), NULL);

  return gtk_entry_get_text(GTK_ENTRY(dialog->name_entry));
}

const char *
sgtk_auth_dialog_get_password (sGtkAuthDialog *dialog)
{
  g_return_val_if_fail(SGTK_IS_AUTH_DIALOG(dialog), NULL);

  return gtk_entry_get_text(GTK_ENTRY(dialog->password_entry));
}
