/*
 * $Id: st-transfer.c,v 1.46 2004/01/28 17:16:41 jylefort Exp $
 *
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <curl/curl.h>
#include <gtk/gtk.h>
#include <string.h>
#include "gettext.h"
#include "sg-util.h"
#include "sg-printable.h"
#include "sgtk-auth-dialog.h"
#include "st-dialog.h"
#include "st-settings.h"
#include "st-transfer.h"
#include "st-thread.h"
#include "st-shell.h"

#define AGENT_STRING			"streamtuner"

/* maintain compatibility with old libcurl versions */

#ifndef CURLOPT_WRITEDATA
#define CURLOPT_WRITEDATA		CURLOPT_FILE
#endif

#ifndef CURLOPT_HEADERDATA
#define CURLOPT_HEADERDATA		CURLOPT_WRITEHEADER
#endif

/*** type definitions ********************************************************/

/*
 * libcurl already provides this prototype as curl_write_callback, but
 * we are not gonna rely on it as it's undocumented
 */
typedef size_t (CurlCallback) (char *buffer,
			       size_t size,
			       size_t nitems,
			       void *data);

typedef struct
{
  const char			*url;
  int				flags;

  STTransferSession		*session;
  char				error[CURL_ERROR_SIZE];

  CurlCallback			*header_cb;
  gpointer			header_data;

  CurlCallback			*body_cb;
  gpointer			body_data;

  char				*proxy_url;
  char				*proxy_auth_name;
  char				*proxy_userpwd;
} STTransfer;

struct _STTransferSession
{
  CURL				*handle;
};

typedef struct
{
  STTransfer			*transfer;

  GString			*line;
  STTransferLineCallback	*line_cb;
  gpointer			line_cb_data;
} STTransferLineData;

/*** function declarations ***************************************************/

static gboolean st_transfer_perform (STTransfer *transfer, GError **err);

static void st_transfer_set_proxy_settings (STTransfer *transfer);
static void st_transfer_free_proxy_settings (STTransfer *transfer);

static gboolean st_transfer_progress_cb (void *data,
					 double dltotal,
					 double dlnow,
					 double ultotal,
					 double ulnow);
static gboolean st_transfer_password_prompt_cb (void *data,
						char *prompt,
						char *buffer,
						int buflen);

static gboolean st_transfer_session_get_internal (STTransferSession *session,
						  const char *url,
						  int flags,
						  char **headers,
						  char **body,
						  GError **err);
static gboolean st_transfer_session_get_by_line_internal (STTransferSession *session,
							  const char *url,
							  int flags,
							  STTransferLineCallback *header_cb,
							  gpointer header_data,
							  STTransferLineCallback *body_cb,
							  gpointer body_data,
							  GError **err);

static size_t st_transfer_session_get_cb (char *buffer,
					  size_t size,
					  size_t nitems,
					  gpointer data);
static size_t st_transfer_session_get_by_line_cb (char *buffer,
						  size_t size,
						  size_t nitems,
						  gpointer data);

static void st_transfer_zero_and_free (char *buffer);

/*** API implementation ******************************************************/

STTransferSession *
st_transfer_session_new (void)
{
  STTransferSession *session;

  session = g_new(STTransferSession, 1);
  session->handle = curl_easy_init();

  return session;
}

void
st_transfer_session_free (STTransferSession *session)
{
  g_return_if_fail(session != NULL);

  curl_easy_cleanup(session->handle);
  g_free(session);
}

/*
 * Deprecated.
 */
char *
st_transfer_get_full (const char *url, GError **err)
{
  STTransferSession *session;
  gboolean status;
  char *body;

  g_return_val_if_fail(url != NULL, FALSE);

  session = st_transfer_session_new();
  status = st_transfer_session_get_internal(session, url, 0, NULL, &body, err);
  st_transfer_session_free(session);

  return status ? body : NULL;
}

/*
 * Fetches URL.
 *
 * If HEADERS is not null, write a newly-allocated string containing
 * the HTTP headers at its address.
 *
 * A newly-allocated string containing the body will be written at the
 * address pointed to by BODY.
 *
 * Returns TRUE in case of success, FALSE in case of abort or error
 * (if there was an error, ERR will be set).
 */
gboolean
st_transfer_session_get (STTransferSession *session,
			 const char *url,
			 int flags,
			 char **headers,
			 char **body,
			 GError **err)
{
  gboolean status;

  g_return_val_if_fail(url != NULL, FALSE);
  g_return_val_if_fail(body != NULL, FALSE);

  status = st_transfer_session_get_internal(session, url, flags, headers, body, err);

  return status;
}

/*
 * Deprecated
 */
char *
st_transfer_get_full_with_session (STTransferSession *session,
				   const char *url,
				   GError **err)
{
  gboolean status;
  char *body;

  g_return_val_if_fail(session != NULL, FALSE);
  g_return_val_if_fail(url != NULL, FALSE);

  status = st_transfer_session_get_internal(session, url, 0, NULL, &body, err);
  
  return status ? body : NULL;
}

static gboolean
st_transfer_session_get_internal (STTransferSession *session,
				  const char *url,
				  int flags,
				  char **headers,
				  char **body,
				  GError **err)
{
  STTransfer transfer;
  GString *header_string = NULL;
  GString *body_string;
  gboolean status;

  g_return_val_if_fail(session != NULL, FALSE);
  g_return_val_if_fail(url != NULL, FALSE);
  g_return_val_if_fail(body != NULL, FALSE);
  
  transfer.url = url;
  transfer.flags = flags;
  transfer.session = session;

  if (headers)
    {
      header_string = g_string_new(NULL);
      transfer.header_cb = st_transfer_session_get_cb;
      transfer.header_data = header_string;
    }
  else
    {
      transfer.header_cb = NULL;
      transfer.header_data = NULL;
    }

  body_string = g_string_new(NULL);
  transfer.body_cb = st_transfer_session_get_cb;
  transfer.body_data = body_string;
  
  status = st_transfer_perform(&transfer, err);
  if (status)
    {
      if (headers)
	*headers = header_string->str;
      *body = body_string->str;
    }

  if (header_string)
    g_string_free(header_string, ! status);
  g_string_free(body_string, ! status);

  return status;
}

/*
 * Deprecated.
 */
gboolean
st_transfer_get_lines (const char *url,
		       STTransferLineCallback *cb,
		       gpointer data,
		       GError **err)
{
  STTransferSession *session;
  gboolean status;

  g_return_val_if_fail(url != NULL, FALSE);
  g_return_val_if_fail(cb != NULL, FALSE);

  session = st_transfer_session_new();
  status = st_transfer_session_get_by_line_internal(session, url, 0, NULL, NULL, cb, data, err);
  st_transfer_session_free(session);

  return status;
}

gboolean
st_transfer_session_get_by_line (STTransferSession *session,
				 const char *url,
				 int flags,
				 STTransferLineCallback *header_cb,
				 gpointer header_data,
				 STTransferLineCallback *body_cb,
				 gpointer body_data,
				 GError **err)
{
  gboolean status;

  g_return_val_if_fail(url != NULL, FALSE);
  g_return_val_if_fail(body_cb != NULL, FALSE);

  status = st_transfer_session_get_by_line_internal(session, url, flags, header_cb, header_data, body_cb, body_data, err);

  return status;
}

/*
 * Deprecated.
 */
gboolean
st_transfer_get_lines_with_session (STTransferSession *session,
				    const char *url,
				    STTransferLineCallback *cb,
				    gpointer data,
				    GError **err)
{
  g_return_val_if_fail(session != NULL, FALSE);
  g_return_val_if_fail(url != NULL, FALSE);
  g_return_val_if_fail(cb != NULL, FALSE);

  return st_transfer_session_get_by_line_internal(session, url, 0, NULL, NULL, cb, data, err);
}

static gboolean
st_transfer_session_get_by_line_internal (STTransferSession *session,
					  const char *url,
					  int flags,
					  STTransferLineCallback *header_cb,
					  gpointer header_data,
					  STTransferLineCallback *body_cb,
					  gpointer body_data,
					  GError **err)
{
  STTransfer transfer;
  STTransferLineData header_line_data;
  STTransferLineData body_line_data;
  gboolean status;

  g_return_val_if_fail(session != NULL, FALSE);
  g_return_val_if_fail(url != NULL, FALSE);
  g_return_val_if_fail(body_cb != NULL, FALSE);
  
  transfer.url = url;
  transfer.flags = flags;
  transfer.session = session;

  if (header_cb)
    {
      header_line_data.transfer = &transfer;
      header_line_data.line = g_string_new(NULL);
      header_line_data.line_cb = header_cb;
      header_line_data.line_cb_data = header_data;

      transfer.header_cb = st_transfer_session_get_by_line_cb;
      transfer.header_data = &header_line_data;
    }
  else
    {
      transfer.header_cb = NULL;
      transfer.header_data = NULL;
    }
   
  body_line_data.transfer = &transfer;
  body_line_data.line = g_string_new(NULL);
  body_line_data.line_cb = body_cb;
  body_line_data.line_cb_data = body_data;

  transfer.body_cb = st_transfer_session_get_by_line_cb;
  transfer.body_data = &body_line_data;

  status = st_transfer_perform(&transfer, err);
  
  if (header_cb)
    g_string_free(header_line_data.line, TRUE);
  g_string_free(body_line_data.line, TRUE);
  
  return status;
}

static gboolean
st_transfer_perform (STTransfer *transfer, GError **err)
{
  STThread *thread;
  int status;

  g_return_val_if_fail(transfer != NULL, FALSE);
  g_return_val_if_fail(transfer->url != NULL, FALSE);
  g_return_val_if_fail(transfer->session != NULL, FALSE);
  /*
   * libcurl assumes that eitheir CURLOPT_WRITEFUNCTION or
   * CURLOPT_WRITEDATA will be set, so we require a body callback.
   */
  g_return_val_if_fail(transfer->body_cb != NULL, FALSE);

  thread = st_thread_get();

  if (thread)
    {
      GDK_THREADS_ENTER();

      if (thread->printable)
	sg_printable_printf(thread->printable, _("Connecting to %s..."), transfer->url);

      if (thread->progress_bar)
	gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(thread->progress_bar), 0);

      GDK_THREADS_LEAVE();
    }
  
  st_transfer_set_proxy_settings(transfer);
  
  curl_easy_setopt(transfer->session->handle, CURLOPT_USERAGENT, AGENT_STRING);
  curl_easy_setopt(transfer->session->handle, CURLOPT_URL, transfer->url);
  curl_easy_setopt(transfer->session->handle, CURLOPT_HEADERFUNCTION, transfer->header_cb);
  curl_easy_setopt(transfer->session->handle, CURLOPT_HEADERDATA, transfer->header_data);
  curl_easy_setopt(transfer->session->handle, CURLOPT_WRITEFUNCTION, transfer->body_cb);
  curl_easy_setopt(transfer->session->handle, CURLOPT_WRITEDATA, transfer->body_data);
  curl_easy_setopt(transfer->session->handle, CURLOPT_NOPROGRESS, FALSE);
  curl_easy_setopt(transfer->session->handle, CURLOPT_PROGRESSFUNCTION, st_transfer_progress_cb);
  curl_easy_setopt(transfer->session->handle, CURLOPT_PROGRESSDATA, thread);
  curl_easy_setopt(transfer->session->handle, CURLOPT_ERRORBUFFER, transfer->error);
  curl_easy_setopt(transfer->session->handle, CURLOPT_PROXY, transfer->proxy_url);
  curl_easy_setopt(transfer->session->handle, CURLOPT_PROXYUSERPWD, transfer->proxy_userpwd);
  curl_easy_setopt(transfer->session->handle, CURLOPT_PASSWDFUNCTION, st_transfer_password_prompt_cb);
  curl_easy_setopt(transfer->session->handle, CURLOPT_PASSWDDATA, transfer);
  curl_easy_setopt(transfer->session->handle, CURLOPT_COOKIEFILE, "");

  status = curl_easy_perform(transfer->session->handle);
  
  st_transfer_free_proxy_settings(transfer);

  if (status != CURLE_OK
      && status != CURLE_ABORTED_BY_CALLBACK
      && status != CURLE_BAD_PASSWORD_ENTERED)
    g_set_error(err, 0, 0, "%s", transfer->error);

  return status == CURLE_OK;
}

static void
st_transfer_set_proxy_settings (STTransfer *transfer)
{
  g_return_if_fail(transfer != NULL);

  G_LOCK(st_settings);

  if (st_settings.proxy_enabled && st_settings.proxy_url)
    transfer->proxy_url = g_strdup(st_settings.proxy_url);
  else
    transfer->proxy_url = NULL;

  if (st_settings.proxy_auth_enabled && st_settings.proxy_auth_name)
    {
      transfer->proxy_auth_name = g_strdup(st_settings.proxy_auth_name);
      transfer->proxy_userpwd = g_strconcat(st_settings.proxy_auth_name,
					    ":",
					    st_settings.proxy_auth_password ? st_settings.proxy_auth_password : "",
					    NULL);
    }
  else
    {
      transfer->proxy_auth_name = NULL;
      transfer->proxy_userpwd = NULL;
    }

  G_UNLOCK(st_settings);
}

static void
st_transfer_free_proxy_settings (STTransfer *transfer)
{
  g_return_if_fail(transfer != NULL);
  
  g_free(transfer->proxy_url);
  st_transfer_zero_and_free(transfer->proxy_auth_name);
  st_transfer_zero_and_free(transfer->proxy_userpwd);
}

char *
st_transfer_escape (const char *url)
{
  return curl_escape(url, 0);
}

static gboolean
st_transfer_progress_cb (void *data,
			 double dltotal,
			 double dlnow,
			 double ultotal,
			 double ulnow)
{
  STThread *thread = data;
  gboolean aborted = FALSE;

  if (thread)
    {
      aborted = st_thread_is_aborted(thread);

      if (! aborted)
	{
	  GDK_THREADS_ENTER();

	  if (thread->printable)
	    {
	      if (dltotal == 0)
		sg_printable_printf(thread->printable,
				    ngettext("Receiving (%u byte so far)...",
					     "Receiving (%u bytes so far)...",
					     (unsigned int) dlnow),
				    (unsigned int) dlnow);
	      else
		sg_printable_printf(thread->printable,
				    ngettext("Receiving (%u byte out of %u)...",
					     "Receiving (%u bytes out of %u)...",
					     (unsigned int) dlnow),
				    (unsigned int) dlnow,
				    (unsigned int) dltotal);
	    }
	  
	  if (thread->progress_bar && dlnow != thread->downloaded)
	    {
	      if (dltotal == 0)
		gtk_progress_bar_pulse((GtkProgressBar *) thread->progress_bar);
	      else
		gtk_progress_bar_set_fraction((GtkProgressBar *) thread->progress_bar, dlnow / dltotal);
	    }

	  GDK_THREADS_LEAVE();

	  thread->downloaded = dlnow;
	}
    }

  return aborted;
}

static gboolean
st_transfer_password_prompt_cb (void *data,
				char *prompt,
				char *buffer,
				int buflen)
{
  STTransfer *transfer = data;
  GtkWidget *dialog;
  char *password = NULL;
  gboolean status = TRUE;	/* TRUE == failure */

  g_return_val_if_fail(transfer != NULL, TRUE);

  GDK_THREADS_ENTER();

  dialog = sgtk_auth_dialog_new(st_shell_get_transient(),
				_("Please enter your proxy password."));

  sgtk_auth_dialog_set_name(SGTK_AUTH_DIALOG(dialog), transfer->proxy_auth_name);
  sgtk_auth_dialog_set_name_sensitive(SGTK_AUTH_DIALOG(dialog), FALSE);

  if (sgtk_dialog_run_from_thread(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK)
    password = g_strdup(sgtk_auth_dialog_get_password(SGTK_AUTH_DIALOG(dialog)));

  gtk_widget_destroy(dialog);

  GDK_THREADS_LEAVE();

  if (password)
    {
      if (strlen(password) < buflen)
	{
	  strcpy(buffer, password);
	  status = FALSE;
	}

      st_transfer_zero_and_free(password);
    }
  else
    {
      STThread *thread;

      thread = st_thread_get();
      if (thread)
	{
	  GDK_THREADS_ENTER();
	  
	  switch (thread->type)
	    {
	    case ST_THREAD_TYPE_STREAM:
	      st_shell_stop_task(thread);
	      break;
	      
	    case ST_THREAD_TYPE_REFRESH:
	      st_browser_tab_stop(ST_BROWSER_TAB(thread->tab));
	      break;

	    default:
	      g_assert_not_reached();
	    }

	  GDK_THREADS_LEAVE();
	}
    }
  
  return status;
}

static size_t
st_transfer_session_get_cb (char *buffer,
			    size_t size,
			    size_t nitems,
			    void *data)
{
  GString *string = data;
  size_t len;

  len = size * nitems;
  g_string_append_len(string, buffer, len);
  
  return len;
}

static size_t
st_transfer_session_get_by_line_cb (char *buffer,
				    size_t size,
				    size_t nitems,
				    void *data)
{
  STTransferLineData *line_data = data;
  size_t len;
  int start;
  int i;

  len = size * nitems;

  /* handle lf (UNIX), crlf (DOS) and cr (Mac) */
  for (start = i = 0; i < len; i++)
    if (buffer[i] == '\n' || buffer[i] == '\r')
      {
	int newline_len;

	newline_len = buffer[i] == '\r' && i < len && buffer[i + 1] == '\n'
	  ? 2			/* crlf */
	  : 1;			/* cr or lf */

	g_string_append_len(line_data->line, &buffer[start], i - start);

	if (line_data->transfer->flags & ST_TRANSFER_PASS_NEWLINE)
	  g_string_append_len(line_data->line, &buffer[i], newline_len);

	line_data->line_cb(line_data->line->str, line_data->line_cb_data);
	g_string_truncate(line_data->line, 0);

	start = i + newline_len;
      }
  
  if (start < len)
    g_string_append_len(line_data->line, &buffer[start], len - start);
  
  return len;
}

static void
st_transfer_zero_and_free (char *buffer)
{
  if (buffer)
    {
      int i;

      for (i = 0; buffer[i]; i++)
	buffer[i] = 0;
      
      g_free(buffer);
    }
}
