/*
 * $Id: st-shell.c,v 1.39 2004/01/30 16:26:40 jylefort Exp $
 *
 * Copyright (c) 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "gettext.h"
#include "sgtk-message-dialog.h"
#include "sg-util.h"
#include "st-browser-window.h"
#include "st-about-dialog.h"
#include "st-category.h"
#include "st-handler.h"
#include "st-handler-event.h"
#include "st-handlers.h"
#include "st-dialog.h"
#include "st-shell.h"
#include "st-settings.h"
#include "st-preferences-dialog.h"
#include "st-stream-properties-dialog.h"
#include "st-action.h"
#include "st-session.h"
#include "st-plugins.h"
#include "st-stock.h"
#include "st-thread.h"

/*** type definitions ********************************************************/

typedef struct
{
  STHandler		*handler;
  gboolean		(* cb) (STHandler *, STStream *, GError **);
  gboolean		(* multiple_cb) (STHandler *, GSList *, GError **);
  const char		*errmsg;
  STThread		*thread;
} StreamThreadInfo;

/*** variable declarations ***************************************************/

G_LOCK_DEFINE_STATIC(st_shell_tasks);
static GSList		*st_shell_tasks = NULL;
static STThread		*st_shell_tune_in_thread = NULL;

GtkTooltips		*st_shell_tooltips;

static STBrowserWindow	*st_shell_browser = NULL;
static GtkWidget	*st_shell_stream_properties_dialog = NULL;
static GtkWidget	*st_shell_preferences_dialog = NULL;
static GtkWidget	*st_shell_about_dialog = NULL;

/*** function declarations ***************************************************/

static void	st_shell_select_handler		(void);

static void	st_shell_stream_properties_update_handler	(void);
static void	st_shell_stream_properties_update_stream	(void);

static void	st_shell_dialog_response_h	(GtkDialog	*dialog,
						 int		response,
						 gpointer	data);

static void	st_shell_stream_thread		(gpointer	data);

static void	st_shell_add_task		(STThread	*thread);
static void	st_shell_remove_task		(STThread	*thread);
static void	st_shell_visit_homepage		(const char	*url);

/*** implementation **********************************************************/

void
st_shell_main (void)
{
  st_shell_tooltips = gtk_tooltips_new();
  st_shell_browser = (STBrowserWindow *) st_browser_window_new();

  st_shell_select_handler();
  st_browser_window_update_sensitivity(st_shell_browser);

  gtk_widget_show(GTK_WIDGET(st_shell_browser));

  GDK_THREADS_ENTER();
  gtk_main();
  GDK_THREADS_LEAVE();
}

GtkWindow *st_shell_get_transient (void)
{
  return st_shell_browser ? GTK_WINDOW(st_shell_browser) : NULL;
}

static void
st_shell_select_handler (void)
{
  STHandler *handler = NULL;

  if (st_settings.selected_handler_name)
    handler = st_handlers_find_by_name(st_settings.selected_handler_name);

  if (! handler && st_handlers_list)
    /* fallback to the first handler of st_handlers_list */
    handler = st_handlers_list->data;

  if (handler)
    st_browser_window_set_tab_from_handler(st_shell_browser, handler);
}

gboolean
st_shell_can_refresh (void)
{
  gboolean can_refresh = FALSE;

  if (st_shell_browser->tab)
    {
      g_mutex_lock(st_shell_browser->tab->handler->mutex);
      can_refresh = ! ST_CATEGORY_IS_BOOKMARKS(st_shell_browser->tab->handler->selected_category);
      g_mutex_unlock(st_shell_browser->tab->handler->mutex);
    }

  return can_refresh;
}

void
st_shell_refresh (void)
{
  g_return_if_fail(st_shell_can_refresh());
  
  st_browser_tab_refresh(st_shell_browser->tab);
}

void
st_shell_update_tab (void)
{
  st_browser_tab_update(st_shell_browser->tab);
}

gboolean
st_shell_can_tune_in (void)
{
  gboolean can_tune_in = FALSE;

  if (st_shell_browser->tab)
    {
      g_mutex_lock(st_shell_browser->tab->handler->mutex);
      can_tune_in = st_shell_browser->tab->handler->selected_streams
	&& (st_handler_event_is_bound(st_shell_browser->tab->handler, ST_HANDLER_EVENT_STREAM_TUNE_IN)
	    || st_handler_event_is_bound(st_shell_browser->tab->handler, ST_HANDLER_EVENT_STREAM_TUNE_IN_MULTIPLE));
      g_mutex_unlock(st_shell_browser->tab->handler->mutex);
    }
  
  return can_tune_in;
}

void
st_shell_tune_in (void)
{
  StreamThreadInfo *info;
  
  g_return_if_fail(st_shell_can_tune_in());

  info = g_new(StreamThreadInfo, 1);
  
  info->handler = st_shell_browser->tab->handler;
  info->cb = st_handler_event_is_bound(info->handler, ST_HANDLER_EVENT_STREAM_TUNE_IN) ? st_handler_event_stream_tune_in : NULL;
  info->multiple_cb = st_handler_event_is_bound(info->handler, ST_HANDLER_EVENT_STREAM_TUNE_IN_MULTIPLE) ? st_handler_event_stream_tune_in_multiple : NULL;
  info->errmsg = _("Unable to tune in.");
  info->thread = st_thread_new(ST_THREAD_TYPE_STREAM,
			       info->handler,
			       st_shell_stream_thread,
			       info);

  /* abort previous tune in, if any */
  G_LOCK(st_shell_tasks);
  if (st_shell_tune_in_thread)
    {
      st_thread_abort(st_shell_tune_in_thread);
      st_shell_tasks = g_slist_remove(st_shell_tasks, st_shell_tune_in_thread);
    }
  st_shell_tune_in_thread = info->thread;
  G_UNLOCK(st_shell_tasks);
  
  st_shell_add_task(info->thread);
  st_thread_run(info->thread);
}

gboolean
st_shell_can_record (void)
{
  gboolean can_record = FALSE;

  if (st_shell_browser->tab)
    {
      g_mutex_lock(st_shell_browser->tab->handler->mutex);
      can_record = st_shell_browser->tab->handler->selected_streams
	&& st_handler_event_is_bound(st_shell_browser->tab->handler, ST_HANDLER_EVENT_STREAM_RECORD);
      g_mutex_unlock(st_shell_browser->tab->handler->mutex);
    }
  
  return can_record;
}

void
st_shell_record (void)
{
  StreamThreadInfo *info;

  g_return_if_fail(st_shell_can_record());

  info = g_new(StreamThreadInfo, 1);
 
  info->handler = st_shell_browser->tab->handler;
  info->cb = st_handler_event_stream_record;
  info->multiple_cb = NULL;
  info->errmsg = _("Unable to record.");
  info->thread = st_thread_new(ST_THREAD_TYPE_STREAM,
			       info->handler,
			       st_shell_stream_thread,
			       info);

  st_shell_add_task(info->thread);
  st_thread_run(info->thread);
}

gboolean
st_shell_can_browse (void)
{
  gboolean can_browse = FALSE;

  if (st_shell_browser->tab)
    {
      g_mutex_lock(st_shell_browser->tab->handler->mutex);
      can_browse = st_shell_browser->tab->handler->selected_streams
	&& st_handler_event_is_bound(st_shell_browser->tab->handler, ST_HANDLER_EVENT_STREAM_BROWSE);
      g_mutex_unlock(st_shell_browser->tab->handler->mutex);
    }
  
  return can_browse;
}

void
st_shell_browse (void)
{
  StreamThreadInfo *info;

  g_return_if_fail(st_shell_can_browse());

  info = g_new(StreamThreadInfo, 1);
 
  info->handler = st_shell_browser->tab->handler;
  info->cb = st_handler_event_stream_browse;
  info->multiple_cb = NULL;
  info->errmsg = _("Unable to browse.");
  info->thread = st_thread_new(ST_THREAD_TYPE_STREAM,
			       info->handler,
			       st_shell_stream_thread,
			       info);

  st_shell_add_task(info->thread);
  st_thread_run(info->thread);
}

void
st_shell_present_about_dialog (void)
{
  if (st_shell_about_dialog)
    gtk_window_present(GTK_WINDOW(st_shell_about_dialog));
  else
    {
      st_shell_about_dialog = st_about_dialog_new(GTK_WINDOW(st_shell_browser));

      g_signal_connect(G_OBJECT(st_shell_about_dialog),
		       "response",
		       G_CALLBACK(st_shell_dialog_response_h),
		       &st_shell_about_dialog);

      gtk_widget_show(st_shell_about_dialog);
    }
}

void
st_shell_present_preferences (void)
{
  if (st_shell_preferences_dialog)
    gtk_window_present(GTK_WINDOW(st_shell_preferences_dialog));
  else
    {
      st_shell_preferences_dialog = st_preferences_dialog_new(GTK_WINDOW(st_shell_browser));

      g_signal_connect(G_OBJECT(st_shell_preferences_dialog),
		       "response",
		       G_CALLBACK(st_shell_dialog_response_h),
		       &st_shell_preferences_dialog);

      gtk_widget_show(st_shell_preferences_dialog);
    }
}

gboolean
st_shell_can_present_stream_properties (void)
{
  gboolean can_present_stream_properties = FALSE;

  if (st_shell_browser->tab)
    {
      g_mutex_lock(st_shell_browser->tab->handler->mutex);
      can_present_stream_properties = st_shell_browser->tab->handler->selected_streams != NULL;
      g_mutex_unlock(st_shell_browser->tab->handler->mutex);
    }

  return can_present_stream_properties;
}

void
st_shell_present_stream_properties (void)
{
  g_return_if_fail(st_shell_can_present_stream_properties());

  if (st_shell_stream_properties_dialog)
    gtk_window_present(GTK_WINDOW(st_shell_stream_properties_dialog));
  else
    {
      st_shell_stream_properties_dialog = st_stream_properties_dialog_new(GTK_WINDOW(st_shell_browser));

      g_signal_connect(G_OBJECT(st_shell_stream_properties_dialog),
		       "response",
		       G_CALLBACK(st_shell_dialog_response_h),
		       &st_shell_stream_properties_dialog);
      
      st_shell_stream_properties_update_handler();
      st_shell_stream_properties_update_stream();
      
      gtk_widget_show(st_shell_stream_properties_dialog);
    }
}

static void
st_shell_stream_properties_update_handler (void)
{
  g_return_if_fail(st_shell_stream_properties_dialog != NULL);
  g_return_if_fail(st_shell_browser->tab != NULL);

  st_stream_properties_dialog_set_handler(ST_STREAM_PROPERTIES_DIALOG(st_shell_stream_properties_dialog),
					  st_shell_browser->tab->handler);
}

static void
st_shell_stream_properties_update_stream (void)
{
  STStream *stream;

  g_return_if_fail(st_shell_stream_properties_dialog != NULL);
  g_return_if_fail(st_shell_browser->tab != NULL);

  g_mutex_lock(st_shell_browser->tab->handler->mutex);
  stream = st_shell_browser->tab->handler->selected_streams
    ? st_shell_browser->tab->handler->selected_streams->data
    : NULL;
  st_stream_properties_dialog_set_stream(ST_STREAM_PROPERTIES_DIALOG(st_shell_stream_properties_dialog), stream);
  g_mutex_unlock(st_shell_browser->tab->handler->mutex);
}

static void
st_shell_dialog_response_h (GtkDialog *dialog, int response, gpointer data)
{
  GtkWidget **ptr = data;

  g_return_if_fail(GTK_IS_WIDGET(dialog));
  g_return_if_fail(ptr != NULL);
  
  gtk_widget_destroy(GTK_WIDGET(dialog));
  *ptr = NULL;
}

static void
st_shell_stream_thread (gpointer data)
{
  StreamThreadInfo *info = data;
  GSList *streams = NULL;
  GSList *l;
  gboolean status = TRUE;	/* we succeed if there is no stream */
  GError *err = NULL;

  g_return_if_fail(info != NULL);

  /*
   * We use our own copy of the streams, because they might be
   * destroyed by st_handler_refresh() while we still need them.
   */
  g_mutex_lock(info->handler->mutex);
  SG_LIST_FOREACH(l, info->handler->selected_streams)
    {
      STStream *stream = l->data;
      STStream *copy;
      
      copy = st_handler_copy_stream(info->handler, stream);
      streams = g_slist_append(streams, copy);
    }
  g_mutex_unlock(info->handler->mutex);

  /*
   * The following test is not superfluous: there can be no selected
   * stream, because something (refresh or selection change) might
   * have emptied info->tab->handler->selected_streams while we were
   * waiting for the handler lock.
   */
  if (streams)
    {
      if (info->multiple_cb)
	status = info->multiple_cb(info->handler, streams, &err);
      else
	{
	  g_return_if_fail(info->cb != NULL);
	  status = info->cb(info->handler, streams->data, &err);
	}

      /* free the copied streams */
      SG_LIST_FOREACH(l, streams)
	{
	  STStream *stream = l->data;
	  st_handler_event_stream_free(info->handler, stream);
	}
      g_slist_free(streams);
    }

  /*
   * If aborted, the following has already been done, so we don't need
   * to do it again.
   */
  if (status || err)		/* not aborted */
    {
      GDK_THREADS_ENTER();
      st_shell_remove_task(info->thread);
      GDK_THREADS_LEAVE();
    }

  if (! status)
    {
      if (err)			/* err unset means "aborted legally" */
	{
	  char *normalized;

	  normalized = st_dialog_normalize(err->message);

	  st_error_dialog(info->errmsg, "%s", normalized);

	  g_free(normalized);
	  g_error_free(err);
	}
    }

  g_free(info);
}

void
st_shell_quit (void)
{
  gboolean save = st_settings.save_automatic;
  
  if (! save)
    {
      GtkWidget *dialog;
      GtkWidget *yes;
      int response;
      
      dialog = sgtk_message_dialog_new(GTK_WINDOW(st_shell_browser),
				       GTK_STOCK_DIALOG_WARNING,
				       _("Save session before quitting?"),
				       _("If you quit without saving, changes to your session will be discarded."));
				       
      gtk_dialog_add_button(GTK_DIALOG(dialog), ST_STOCK_QUIT_WITHOUT_SAVING, GTK_RESPONSE_NO);
      gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
      yes = gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_SAVE, GTK_RESPONSE_YES);

      gtk_window_set_default(GTK_WINDOW(dialog), yes);

      response = gtk_dialog_run(GTK_DIALOG(dialog));
      gtk_widget_destroy(dialog);

      switch (response)
	{
	case GTK_RESPONSE_YES:
	  save = TRUE;
	  break;
	  
	case GTK_RESPONSE_NO:
	  break;
	  
	default:
	  return;
	}
    }
  
  st_browser_window_abort_all(st_shell_browser);

  gtk_widget_destroy(GTK_WIDGET(st_shell_browser));
  st_shell_browser = NULL;

  if (save)
    st_shell_save_session();
  
  gtk_main_quit();
}

void
st_shell_help (void)
{
  GError *err = NULL;
  
  if (! st_action_run("view-man", "streamtuner", &err))
    {
      char *normalized;

      normalized = st_dialog_normalize(err->message);

      st_error_dialog(_("Unable to view the manual."), "%s", normalized);

      g_free(normalized);
      g_error_free(err);
    }
}

static void
st_shell_visit_homepage (const char *url)
{
  GError *err = NULL;

  g_return_if_fail(url != NULL);

  if (! st_action_run("view-web", url, &err))
    {
      char *normalized;

      normalized = st_dialog_normalize(err->message);

      st_error_dialog(_("Unable to visit the homepage."), "%s", normalized);

      g_free(normalized);
      g_error_free(err);
    }
}

void
st_shell_homepage (void)
{
  st_shell_visit_homepage(PACKAGE_HOME);
}

gboolean
st_shell_can_visit_directory_homepage (void)
{
  return st_shell_browser->tab && st_shell_browser->tab->handler->home;
}

void
st_shell_visit_directory_homepage (void)
{
  g_return_if_fail(st_shell_can_visit_directory_homepage());
  
  st_shell_visit_homepage(st_shell_browser->tab->handler->home);
}

void
st_shell_load_session (void)
{
  GError *err = NULL;

  if (! st_session_load(&err))
    {
      char *normalized;
      
      normalized = st_dialog_normalize(err->message);

      st_error_dialog(_("Unable to load session."), "%s", normalized);

      g_free(normalized);
      g_error_free(err);
    }
}

void
st_shell_save_session (void)
{
  GError *err = NULL;
  
  if (! st_session_save(&err))
    {
      char *normalized;
      
      normalized = st_dialog_normalize(err->message);

      st_error_dialog(_("Unable to save session."), "%s", normalized);

      g_free(normalized);
      g_error_free(err);
    }
}

void
st_shell_load_plugins (void)
{
  GError *err = NULL;

  if (! st_plugins_load(&err))
    {
      char *normalized;

      normalized = st_dialog_normalize(err->message);

      st_error_dialog(_("Unable to load plugins."), "%s", normalized);
      
      g_free(normalized);
      g_error_free(err);
    }
}

void
st_shell_tab_selected (void)
{
  if (st_shell_stream_properties_dialog)
    {
      st_shell_stream_properties_update_handler();
      st_shell_stream_properties_update_stream();
    }
}

void
st_shell_streams_selected (void)
{
  if (st_shell_stream_properties_dialog)
    st_shell_stream_properties_update_stream();
}

void
st_shell_visibility_changed (void)
{
  st_browser_window_update_visibility(st_shell_browser);
}

void
st_shell_toolbar_style_changed (void)
{
  st_browser_window_update_toolbar_style(st_shell_browser);
}

void
st_shell_toolbar_size_changed (void)
{
  st_browser_window_update_toolbar_size(st_shell_browser);
}

void
st_shell_previous_tab (void)
{
  st_browser_window_previous_tab(st_shell_browser);
}

void
st_shell_next_tab (void)
{
  st_browser_window_next_tab(st_shell_browser);
}

void
st_shell_set_handler (STHandler *handler)
{
  st_browser_window_set_tab_from_handler(st_shell_browser, handler);
}

gboolean
st_shell_can_add_bookmark (void)
{
  gboolean can_add_bookmark = FALSE;

  if (st_shell_browser->tab)
    {
      g_mutex_lock(st_shell_browser->tab->handler->mutex);
      can_add_bookmark =
	st_shell_browser->tab->handler->selected_streams != NULL
	&& ! ST_CATEGORY_IS_BOOKMARKS(st_shell_browser->tab->handler->selected_category);
      g_mutex_unlock(st_shell_browser->tab->handler->mutex);
    }

  return can_add_bookmark;
}

void
st_shell_add_bookmark (void)
{
  GSList *l;

  g_return_if_fail(st_shell_can_add_bookmark());

  g_mutex_lock(st_shell_browser->tab->handler->mutex);
  SG_LIST_FOREACH(l, st_shell_browser->tab->handler->selected_streams)
    st_handler_add_bookmark(st_shell_browser->tab->handler, l->data);
  g_mutex_unlock(st_shell_browser->tab->handler->mutex);
}

gboolean
st_shell_can_remove_bookmark (void)
{
  gboolean can_remove_bookmark = FALSE;

  if (st_shell_browser->tab)
    {
      g_mutex_lock(st_shell_browser->tab->handler->mutex);
      can_remove_bookmark =
	st_shell_browser->tab->handler->selected_streams != NULL
	&& ST_CATEGORY_IS_BOOKMARKS(st_shell_browser->tab->handler->selected_category);
      g_mutex_unlock(st_shell_browser->tab->handler->mutex);
    }

  return can_remove_bookmark;
}

void
st_shell_remove_bookmark (void)
{
  GSList *l;

  g_return_if_fail(st_shell_can_remove_bookmark());

  g_mutex_lock(st_shell_browser->tab->handler->mutex);
  SG_LIST_FOREACH(l, st_shell_browser->tab->handler->selected_streams)
    st_handler_remove_bookmark(st_shell_browser->tab->handler, l->data);
  g_mutex_unlock(st_shell_browser->tab->handler->mutex);

  st_shell_update_tab();		/* update the bookmarks view */
}

static void
st_shell_add_task (STThread *thread)
{
  g_return_if_fail(thread != NULL);

  G_LOCK(st_shell_tasks);
  st_shell_tasks = g_slist_append(st_shell_tasks, thread);
  G_UNLOCK(st_shell_tasks);

  st_browser_window_update_sensitivity(st_shell_browser);
}

static void
st_shell_remove_task (STThread *thread)
{
  g_return_if_fail(thread != NULL);

  G_LOCK(st_shell_tasks);
  st_shell_tasks = g_slist_remove(st_shell_tasks, thread);
  if (thread == st_shell_tune_in_thread)
    st_shell_tune_in_thread = NULL;
  G_UNLOCK(st_shell_tasks);

  st_browser_window_update_sensitivity(st_shell_browser);
}

void
st_shell_stop_task (STThread *thread)
{
  st_thread_abort(thread);
  st_shell_remove_task(thread);
}

gboolean
st_shell_can_stop (void)
{
  gboolean can_stop;

  can_stop = st_shell_browser->tab && st_browser_tab_can_stop(st_shell_browser->tab);

  if (! can_stop)
    {
      G_LOCK(st_shell_tasks);
      can_stop = st_shell_tasks != NULL;
      G_UNLOCK(st_shell_tasks);
    }

  return can_stop;
}

void
st_shell_stop (void)
{
  GSList *l;

  g_return_if_fail(st_shell_can_stop());

  if (st_shell_browser->tab && st_browser_tab_can_stop(st_shell_browser->tab))
    st_browser_tab_stop(st_shell_browser->tab);

  G_LOCK(st_shell_tasks);
  SG_LIST_FOREACH(l, st_shell_tasks)
    {
      st_thread_abort(l->data);
      if (l->data == st_shell_tune_in_thread)
	st_shell_tune_in_thread = NULL;
    }
  g_slist_free(st_shell_tasks);
  st_shell_tasks = NULL;
  G_UNLOCK(st_shell_tasks);

  st_browser_window_update_sensitivity(st_shell_browser);
}
