/*
 * $Id: st-category-view.c,v 1.38 2004/01/30 16:26:40 jylefort Exp $
 *
 * Copyright (c) 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <gtk/gtk.h>
#include <string.h>
#include "gettext.h"
#include "sg-util.h"
#include "sgtk-handlers.h"
#include "st-browser-window.h"
#include "st-category-view.h"
#include "st-stock.h"

/*** type definitions ********************************************************/

enum {
  PROP_0,
  PROP_TAB
};

enum {
  COLUMN_PTR,
  COLUMN_STOCK_ID,
  COLUMN_LABEL,
  N_COLUMNS
};

typedef struct
{
  STCategoryView *view;
  GHashTable *iterators;
} AppendInfo;

/*** variable declarations ***************************************************/

static GObjectClass *parent_class = NULL;

/*** function declarations ***************************************************/

static void st_category_view_init	(STCategoryView		*view);
static void st_category_view_class_init	(STCategoryViewClass	*class);

static GObject *st_category_view_constructor (GType type,
					      guint n_construct_properties,
					      GObjectConstructParam *construct_params);

static void st_category_view_set_property (GObject	*object,
					   guint	prop_id,
					   const GValue	*value,
					   GParamSpec	*pspec);

static void st_category_view_construct_store	(STCategoryView	*view);
static void st_category_view_construct_columns	(STCategoryView	*view);
static void st_category_view_realize		(GtkWidget	*widget);
static void st_category_view_bind		(STCategoryView *view);

static void st_category_view_row_expanded_h	(GtkTreeView	*gtk_view,
						 GtkTreeIter	*iter,
						 GtkTreePath	*path,
						 gpointer	data);
static void st_category_view_row_collapsed_h	(GtkTreeView	*gtk_view,
						 GtkTreeIter	*iter,
						 GtkTreePath	*path,
						 gpointer	data);

static void st_category_view_selection_changed_h (GtkTreeSelection *selection,
						  gpointer	   data);

static gboolean st_category_view_append_categories_cb (GNode	*node,
						       gpointer	data);

static void	st_category_view_restore_state    (STCategoryView	*view);
static gboolean st_category_view_restore_state_cb (GtkTreeModel		*model,
						   GtkTreePath		*path,
						   GtkTreeIter		*iter,
						   gpointer		data);

/*** implementation **********************************************************/

GType
st_category_view_get_type (void)
{
  static GType category_view_type = 0;
  
  if (! category_view_type)
    {
      static const GTypeInfo category_view_info = {
	sizeof(STCategoryViewClass),
	NULL,
	NULL,
	(GClassInitFunc) st_category_view_class_init,
	NULL,
	NULL,
	sizeof(STCategoryView),
	0,
	(GInstanceInitFunc) st_category_view_init,
      };
      
      category_view_type = g_type_register_static(GTK_TYPE_TREE_VIEW,
						  "STCategoryView",
						  &category_view_info,
						  0);
    }

  return category_view_type;
}

static void
st_category_view_class_init (STCategoryViewClass *class)
{
  GObjectClass *object_class;
  GtkWidgetClass *widget_class;

  parent_class = g_type_class_peek_parent(class);

  object_class = G_OBJECT_CLASS(class);
  object_class->constructor = st_category_view_constructor;
  object_class->set_property = st_category_view_set_property;

  widget_class = GTK_WIDGET_CLASS(class);
  widget_class->realize = st_category_view_realize;

  g_object_class_install_property(object_class,
				  PROP_TAB,
				  g_param_spec_object("tab",
						      "Tab",
						      _("The view's parent STBrowserTab"),
						      ST_TYPE_BROWSER_TAB,
						      G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}

static void
st_category_view_init (STCategoryView *view)
{
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(view), FALSE);
  
  view->tab = NULL;
  view->store = NULL;

  view->selection_changed_hid = -1;

  view->pixbuf_renderer = NULL;
}

static GObject *
st_category_view_constructor (GType type,
			      guint n_construct_properties,
			      GObjectConstructParam *construct_params)
{
  GObject *object; 
  STCategoryView *view;

  object = G_OBJECT_CLASS(parent_class)->constructor(type,
						     n_construct_properties,
						     construct_params);
  view = ST_CATEGORY_VIEW(object);
  
  g_return_val_if_fail(view->tab != NULL, NULL);

  st_category_view_construct_store(view);
  st_category_view_construct_columns(view);
  st_category_view_bind(view);

  return object;
}

static void
st_category_view_set_property (GObject *object,
			       guint prop_id,
			       const GValue *value,
			       GParamSpec *pspec)
{
  STCategoryView *view;

  view = ST_CATEGORY_VIEW(object);

  switch (prop_id)
    {
    case PROP_TAB:
      view->tab = g_value_get_object(value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
    }
}

static void
st_category_view_construct_store (STCategoryView *view)
{
  g_return_if_fail(ST_IS_CATEGORY_VIEW(view));

  view->store = gtk_tree_store_new(N_COLUMNS,
				   G_TYPE_POINTER,
				   G_TYPE_STRING,
				   G_TYPE_STRING);

  gtk_tree_view_set_model(GTK_TREE_VIEW(view), GTK_TREE_MODEL(view->store));
}

static void
st_category_view_construct_columns (STCategoryView *view)
{
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;

  g_return_if_fail(ST_IS_CATEGORY_VIEW(view));

  column = gtk_tree_view_column_new();

  view->pixbuf_renderer = gtk_cell_renderer_pixbuf_new();

  gtk_tree_view_column_pack_start(column, view->pixbuf_renderer, FALSE);
  gtk_tree_view_column_set_attributes(column, view->pixbuf_renderer,
				      "stock-id", COLUMN_STOCK_ID,
				      NULL);

  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_column_pack_start(column, renderer, TRUE);
  gtk_tree_view_column_set_attributes(column, renderer,
				      "text", COLUMN_LABEL,
				      NULL);
  
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);
}

static void
st_category_view_realize (GtkWidget *widget)
{
  STCategoryView *view;
  GdkPixbuf *pixbuf_expander_closed;
  GdkPixbuf *pixbuf_expander_open;

  view = ST_CATEGORY_VIEW(widget);

  pixbuf_expander_closed = gtk_widget_render_icon(widget,
						  ST_STOCK_CATEGORY,
						  GTK_ICON_SIZE_MENU,
						  NULL);
  pixbuf_expander_open = gtk_widget_render_icon(widget,
						ST_STOCK_CATEGORY_OPEN,
						GTK_ICON_SIZE_MENU,
						NULL);
  
  /* those are used only for cells where is-expander is true */
  g_object_set(view->pixbuf_renderer,
	       "pixbuf-expander-closed", pixbuf_expander_closed,
	       "pixbuf-expander-open", pixbuf_expander_open,
	       NULL);

  g_object_unref(pixbuf_expander_closed);
  g_object_unref(pixbuf_expander_open);

  GTK_WIDGET_CLASS(parent_class)->realize(widget);
}

static void
st_category_view_bind (STCategoryView *view)
{
  GtkTreeSelection *selection;

  g_return_if_fail(ST_IS_CATEGORY_VIEW(view));

  g_signal_connect(G_OBJECT(view), "row-expanded",
		   G_CALLBACK(st_category_view_row_expanded_h), NULL);
  g_signal_connect(G_OBJECT(view), "row-collapsed",
		   G_CALLBACK(st_category_view_row_collapsed_h), NULL);

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(view));
  view->selection_changed_hid = g_signal_connect(G_OBJECT(selection),
						 "changed",
						 G_CALLBACK(st_category_view_selection_changed_h),
						 view);
}

static void
st_category_view_selection_changed_h (GtkTreeSelection *selection,
				      gpointer data)
{
  STCategoryView *view = data;
  GtkTreeIter iter;
  GtkTreeModel *model;

  g_return_if_fail(ST_IS_CATEGORY_VIEW(view));
  
  if (gtk_tree_selection_get_selected(selection, &model, &iter))
    {
      STCategory *category;

      gtk_tree_model_get(model, &iter, COLUMN_PTR, &category, -1);

      g_mutex_lock(view->tab->handler->mutex);
      view->tab->handler->selected_category = category;
      g_mutex_unlock(view->tab->handler->mutex);

      st_browser_tab_category_selected(view->tab);
    }
}

static void
st_category_view_row_expanded_h (GtkTreeView *gtk_view,
				 GtkTreeIter *iter,
				 GtkTreePath *path,
				 gpointer data)
{
  STCategoryView *view = (STCategoryView *) gtk_view;
  STCategory *category;

  g_return_if_fail(ST_IS_CATEGORY_VIEW(view));

  gtk_tree_model_get(gtk_tree_view_get_model(gtk_view), iter,
		     COLUMN_PTR, (gpointer *) &category,
		     -1);

  if (! g_slist_find_custom(view->tab->handler->expanded_categories,
			    category->name,
			    (GCompareFunc) strcmp))
    view->tab->handler->expanded_categories
      = g_slist_append(view->tab->handler->expanded_categories,
		       g_strdup(category->name));
}

static void
st_category_view_row_collapsed_h (GtkTreeView *gtk_view,
				  GtkTreeIter *iter,
				  GtkTreePath *path,
				  gpointer data)
{
  STCategoryView *view = (STCategoryView *) gtk_view;
  STCategory *category;
  GSList *node;

  g_return_if_fail(ST_IS_CATEGORY_VIEW(view));

  gtk_tree_model_get(gtk_tree_view_get_model(gtk_view), iter,
		     COLUMN_PTR, (gpointer *) &category,
		     -1);

  if ((node = g_slist_find_custom(view->tab->handler->expanded_categories,
				  category->name,
				  (GCompareFunc) strcmp)))
    {
      g_free(node->data);
      view->tab->handler->expanded_categories
	= g_slist_delete_link(view->tab->handler->expanded_categories, node);
    }
}

GtkWidget *
st_category_view_new (STBrowserTab *tab)
{
  return g_object_new(ST_TYPE_CATEGORY_VIEW,
		      "tab", tab,
		      NULL);
}

void
st_category_view_append_categories (STCategoryView *view, GNode *categories)
{
  AppendInfo info;

  g_return_if_fail(ST_IS_CATEGORY_VIEW(view));
  g_return_if_fail(categories != NULL);

  info.view = view;
  info.iterators = g_hash_table_new_full(g_str_hash,
					 g_str_equal,
					 NULL,
					 (GDestroyNotify) g_free);

  g_node_traverse(categories,
		  G_IN_ORDER,
		  G_TRAVERSE_ALL,
		  -1,
		  (GNodeTraverseFunc) st_category_view_append_categories_cb,
		  &info);

  g_hash_table_destroy(info.iterators);

  st_category_view_restore_state(view);
}

static gboolean
st_category_view_append_categories_cb (GNode *node, gpointer data)
{
  AppendInfo *info = data;
  STCategory *category = node->data;
  GtkTreeIter *root = NULL;
  GtkTreeIter *iter = NULL;
  const char *stock_id;

  if (! category)
    return FALSE;		/* it's the root node */

  if ((iter = g_hash_table_lookup(info->iterators, category->name)))
    return FALSE;		/* already inserted */

  if (node->parent && node->parent->data)
    {
      STCategory *parent = node->parent->data;
      
      st_category_view_append_categories_cb(node->parent, info);
      root = g_hash_table_lookup(info->iterators, parent->name);
      
      g_return_val_if_fail(root != NULL, FALSE);
    }
  
  iter = g_new(GtkTreeIter, 1);
  g_hash_table_insert(info->iterators, category->name, iter);

  /* stock-id is only used when is-expander is false */
  if (! strcmp(category->name, "__main"))
    stock_id = GTK_STOCK_INDEX;
  else if (! strcmp(category->name, "__search"))
    stock_id = GTK_STOCK_FIND;
  else if (! strcmp(category->name, "__bookmarks"))
    stock_id = ST_STOCK_BOOKMARKS;
  else
    stock_id = ST_STOCK_CATEGORY;

  gtk_tree_store_append(info->view->store, iter, root);
  gtk_tree_store_set(info->view->store, iter,
		     COLUMN_PTR, category,
		     COLUMN_STOCK_ID, stock_id,
		     COLUMN_LABEL, category->label,
		     -1);

  return FALSE;
}

/*
 * Restores the state (expanded categories and selected category).
 */
static void
st_category_view_restore_state (STCategoryView *view)
{
  g_return_if_fail(ST_IS_CATEGORY_VIEW(view));
  g_return_if_fail(view->tab->handler->selected_category != NULL);
  
  gtk_tree_model_foreach(GTK_TREE_MODEL(view->store),
			 st_category_view_restore_state_cb,
			 view);
}

static gboolean
st_category_view_restore_state_cb (GtkTreeModel *model,
				   GtkTreePath *path,
				   GtkTreeIter *iter,
				   gpointer data)
{
  GtkTreeView *gtk_view = data;
  STCategoryView *view = data;
  STCategory *category;

  gtk_tree_model_get(model, iter, COLUMN_PTR, &category, -1);

  g_return_val_if_fail(category != NULL, FALSE);
  g_return_val_if_fail(category->name != NULL, FALSE);
  
  /* expand the category, if needed */
  if (g_slist_find_custom(view->tab->handler->expanded_categories,
			  category->name,
			  (GCompareFunc) strcmp))
    gtk_tree_view_expand_row(gtk_view, path, FALSE);

  /* select the category, if needed */
  if (category == view->tab->handler->selected_category)
    {
      GtkTreeSelection *selection;

      selection = gtk_tree_view_get_selection(gtk_view);

      g_signal_handler_block(selection, view->selection_changed_hid);
      gtk_tree_selection_select_iter(selection, iter);
      g_signal_handler_unblock(selection, view->selection_changed_hid);
    }

  return FALSE;
}

void
st_category_view_clear (STCategoryView *view)
{
  g_return_if_fail(ST_IS_CATEGORY_VIEW(view));

  gtk_tree_store_clear(view->store);
}
