/*
 * $Id: shoutcast.c,v 1.126 2004/01/26 16:42:16 jylefort Exp $
 *
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <streamtuner.h>
#include <string.h>
#include <stdlib.h>
#include "gettext.h"
#include "art/shoutcast.h"

#define STREAMS_PER_PAGE		100

#define SHOUTCAST_ROOT			"http://www.shoutcast.com/"

/*** type definitions ********************************************************/

typedef struct
{
  STStream	stream;

  char		*genre;
  char		*description;
  char		*now_playing;
  int		listeners;
  int		max;
  int		bitrate;

  char		*url_postfix;
} SHOUTcastStream;

enum {
  FIELD_GENRE,
  FIELD_DESCRIPTION,
  FIELD_NOW_PLAYING,
  FIELD_LISTENERS,
  FIELD_MAX,
  FIELD_BITRATE,
  FIELD_URL_POSTFIX
};

typedef struct
{
  STCategory		*category;
  GNode			**categories;
  GList			**streams;

  int			page;
  int			npages;

  char			*charset;
  GNode			*parent_node;
  SHOUTcastStream	*stream;

  STTransferSession	*session;
} RefreshInfo;

/*** variable declarations ***************************************************/

static regex_t re_header_charset;
static regex_t re_body_charset;
static regex_t re_genre;
static regex_t re_playing;
static regex_t re_listeners;
static regex_t re_bitrate;
static regex_t re_stream;
static regex_t re_category;
static regex_t re_page;

/*** function declarations ***************************************************/

static gboolean	refresh_cb	(STCategory	*category,
				 GNode		**categories,
				 GList		**streams,
				 gpointer	data,
				 GError		**err);

static gboolean	refresh_page		(const char	*url,
					 RefreshInfo	*info,
					 GError		**err);
static void	refresh_header_cb	(const char	*line,
					 gpointer	data);
static void	refresh_body_cb		(const char	*line,
					 gpointer	data);

static SHOUTcastStream *stream_new_cb (gpointer data);
static void stream_field_get_cb (SHOUTcastStream	*stream,
				 STHandlerField		*field,
				 GValue			*value,
				 gpointer		data);
static void stream_field_set_cb (SHOUTcastStream	*stream,
				 STHandlerField		*field,
				 const GValue		*value,
				 gpointer		data);
static void stream_free_cb	(SHOUTcastStream	*stream,
				 gpointer		data);

static gboolean url_list_get		(SHOUTcastStream	*stream,
					 GSList			**url_list,
					 GError			**err);
static void     url_list_free		(GSList			*url_list);

static gboolean	stream_tune_in_cb	(SHOUTcastStream	*stream,
					 gpointer		data,
					 GError			**err);
static gboolean	stream_record_cb	(SHOUTcastStream	*stream,
					 gpointer		data,
					 GError			**err);
static gboolean	stream_browse_cb	(SHOUTcastStream	*stream,
					 gpointer		data,
					 GError			**err);

static int	search_url_cb		(STCategory		*category);

static gboolean	init_re		(void);
static void	init_handler	(void);

/*** implementation **********************************************************/

static SHOUTcastStream *
stream_new_cb (gpointer data)
{
  return g_new0(SHOUTcastStream, 1);
}

static void
stream_field_get_cb (SHOUTcastStream *stream,
		     STHandlerField *field,
		     GValue *value,
		     gpointer data)
{
  switch (field->id)
    {
    case FIELD_GENRE:
      g_value_set_string(value, stream->genre);
      break;

    case FIELD_DESCRIPTION:
      g_value_set_string(value, stream->description);
      break;

    case FIELD_NOW_PLAYING:
      g_value_set_string(value, stream->now_playing);
      break;

    case FIELD_LISTENERS:
      g_value_set_int(value, stream->listeners);
      break;

    case FIELD_MAX:
      g_value_set_int(value, stream->max);
      break;

    case FIELD_BITRATE:
      g_value_set_int(value, stream->bitrate);
      break;

    case FIELD_URL_POSTFIX:
      g_value_set_string(value, stream->url_postfix);
      break;

    default:
      g_assert_not_reached();
    }
}

static void
stream_field_set_cb (SHOUTcastStream *stream,
		     STHandlerField *field,
		     const GValue *value,
		     gpointer data)
{
  switch (field->id)
    {
    case FIELD_GENRE:
      stream->genre = g_value_dup_string(value);
      break;

    case FIELD_DESCRIPTION:
      stream->description = g_value_dup_string(value);
      break;

    case FIELD_NOW_PLAYING:
      stream->now_playing = g_value_dup_string(value);
      break;

    case FIELD_LISTENERS:
      stream->listeners = g_value_get_int(value);
      break;

    case FIELD_MAX:
      stream->max = g_value_get_int(value);
      break;

    case FIELD_BITRATE:
      stream->bitrate = g_value_get_int(value);
      break;

    case FIELD_URL_POSTFIX:
      stream->url_postfix = g_value_dup_string(value);
      break;

    default:
      g_assert_not_reached();
    }
}

static void
stream_free_cb (SHOUTcastStream *stream, gpointer data)
{
  g_free(stream->genre);
  g_free(stream->description);
  g_free(stream->now_playing);
  g_free(stream->url_postfix);

  st_stream_free((STStream *) stream);
}

static gboolean
url_list_get (SHOUTcastStream *stream, GSList **url_list, GError **err)
{
  STTransferSession *session;
  char *url;
  char *playlist;
  gboolean status;

  g_return_val_if_fail(stream != NULL, FALSE);
  g_return_val_if_fail(url_list != NULL, FALSE);

  url = g_strconcat(SHOUTCAST_ROOT, stream->url_postfix, NULL);
  session = st_transfer_session_new();
  status = st_transfer_session_get(session, url, 0, NULL, &playlist, err);
  st_transfer_session_free(session);
  g_free(url);

  if (status)
    {
      *url_list = st_pls_parse(playlist);
      g_free(playlist);

      if (! *url_list)
	{
	  g_set_error(err, 0, 0, _("stream is empty"));
	  return FALSE;
	}
    }
  
  return status;
}

static void
url_list_free (GSList *url_list)
{
  GSList *l;

  for (l = url_list; l; l = l->next)
    g_free(l->data);

  g_slist_free(url_list);
}

static gboolean
stream_tune_in_cb (SHOUTcastStream *stream,
		   gpointer data,
		   GError **err)
{
  GSList *url_list;
  char *m3uname;
  gboolean status;
  
  /* fetch the URL list */

  if (! url_list_get(stream, &url_list, err))
    return FALSE;

  if (st_is_aborted())
    {
      url_list_free(url_list);
      return FALSE;
    }

  /* write the .m3u file */

  m3uname = st_m3u_mktemp("streamtuner.shoutcast.XXXXXX", url_list, err);
  url_list_free(url_list);

  if (! m3uname)
    return FALSE;

  /* pass the .m3u file to the player */

  status = st_action_run("play-m3u", m3uname, err);
  g_free(m3uname);

  return status;
}

static gboolean
stream_record_cb (SHOUTcastStream *stream,
		  gpointer data,
		  GError **err)
{
  GSList *url_list;
  gboolean status;

  /* fetch the URL list */

  if (! url_list_get(stream, &url_list, err))
    return FALSE;

  if (st_is_aborted())
    {
      url_list_free(url_list);
      return FALSE;
    }
  
  /* pass the first URL to the recorder */

  status = st_action_run("record-stream", url_list->data, err);
  url_list_free(url_list);

  return status;
}

static gboolean
stream_browse_cb (SHOUTcastStream *stream,
		  gpointer data,
		  GError **err)
{
  GSList *url_list;
  gboolean status;

  /* fetch the URL list */

  if (! url_list_get(stream, &url_list, err))
    return FALSE;

  if (st_is_aborted())
    {
      url_list_free(url_list);
      return FALSE;
    }

  /* pass the first URL to the browser */

  status = st_action_run("view-web", url_list->data, err);
  url_list_free(url_list);

  return status;
}

static gboolean
refresh_cb (STCategory *category,
	    GNode **categories,
	    GList **streams,
	    gpointer data,
	    GError **err)
{
  RefreshInfo info;
  char *url;
  gboolean status;

  g_return_val_if_fail(category != NULL, FALSE);
  g_return_val_if_fail(category->url_postfix != NULL, FALSE);

  *categories = g_node_new(NULL);
  *streams = NULL;

  info.category = category;
  info.categories = categories;
  info.streams = streams;

  info.session = st_transfer_session_new();

  url = g_strdup_printf(SHOUTCAST_ROOT "directory/?numresult=%i%s",
			STREAMS_PER_PAGE, category->url_postfix);
  status = refresh_page(url, &info, err);
  g_free(url);

  if (! status)
    goto end;

  /*
   * The main category contains _all_ SHOUTcast streams (a lot), so we
   * just load the first page.
   */
  if (strcmp(category->name, "__main") != 0)
    while (info.page > 0 && info.page < info.npages)
      {
	if (st_is_aborted())
	  {
	    status = FALSE;
	    goto end;
	  }

	url = g_strdup_printf(SHOUTCAST_ROOT "directory/index.phtml?startat=%i",
			      info.page * STREAMS_PER_PAGE);
	status = refresh_page(url, &info, err);
	g_free(url);
	
	if (! status)
	  goto end;
      }
  
 end:
  st_transfer_session_free(info.session);
  
  return status;
}

static gboolean
refresh_page (const char *url, RefreshInfo *info, GError **err)
{
  gboolean status;
  GError *tmp_err = NULL;

  info->page = 0;
  info->npages = 0;

  info->charset = NULL;
  info->parent_node = NULL;
  info->stream = NULL;

  status = st_transfer_session_get_by_line(info->session,
					   url,
					   0,
					   refresh_header_cb,
					   info,
					   refresh_body_cb,
					   info,
					   &tmp_err);

  g_free(info->charset);

  if (info->stream)
    {
      stream_free_cb(info->stream, NULL);
      if (status) /* only display warning if the transfer was otherwise correct */
	st_notice(_("SHOUTcast:EOF: found unterminated stream"));
    }

  if (! status)
    {
      if (tmp_err)
	{
	  g_set_error(err, 0, 0, _("unable to transfer: %s"), tmp_err->message);
	  g_error_free(tmp_err);
	}
      
      return FALSE;
    }
  
  return TRUE;
}

static void
refresh_header_cb (const char *line, gpointer data)
{
  RefreshInfo *info = data;

  if (! info->charset)
    st_re_parse(&re_header_charset, line, &info->charset);
}

static void
refresh_body_cb (const char *line, gpointer data)
{
  RefreshInfo *info = data;
  char *sub1, *sub2;
  char *converted = NULL;

  if (! info->charset && ! strncasecmp(line, "</head>", 6))
    info->charset = g_strdup("ISO8859-1"); /* we got no charset info, use a fallback */
  else
    {
      char *charset;

      if (st_re_parse(&re_body_charset, line, &charset))
	{			/* body charset overrides header charset */
	  g_free(info->charset);
	  info->charset = charset;
	}
    }
  
  if (info->charset)
    {
      GError *convert_err = NULL;
      
      if ((converted = g_convert(line,
				 strlen(line),
				 "UTF-8",
				 info->charset,
				 NULL,
				 NULL,
				 &convert_err)))
	line = converted;
      else
	{
	  st_notice(_("SHOUTcast: unable to convert line to UTF-8: %s"),
		    convert_err->message);
	  
	  g_error_free(convert_err);
	  return;
	}
    }
  
  if (st_re_parse(&re_stream, line, &sub1))
    {
      if (info->stream)	/* a malformed stream remains, free it */
	{
	  st_notice(_("SHOUTcast: found unterminated stream"));
	  stream_free_cb(info->stream, NULL);
	}
	  
      info->stream = stream_new_cb(NULL);
      info->stream->url_postfix = sub1;
    }
  else if (st_re_parse(&re_bitrate, line, &sub1))
    {
      if (info->stream)
	{
	  info->stream->bitrate = atoi(sub1);
	  
	  if (info->stream->genre && info->stream->description)
	    {
	      ((STStream *) info->stream)->name =
		g_strdup_printf("%s%s%i",
				info->stream->genre,
				info->stream->description,
				info->stream->bitrate);
	      
	      *(info->streams) = g_list_append(*(info->streams), info->stream);
	    }
	  else
	    {
	      st_notice(_("SHOUTcast: found incomplete stream"));
	      stream_free_cb(info->stream, NULL);
	    }

	  info->stream = NULL;
	}
      else
	{
	  st_notice(_("SHOUTcast: found misplaced bitrate"));
	  g_free(sub1);
	}
    }
  else if (info->page < 2
	   && st_re_parse(&re_category, line, &sub1, &sub2)
	   && strcmp(sub1, "TopTen") != 0)
    {
      STCategory *category;
      GNode *node;
      char *escaped;
      
      category = st_category_new();
      
      category->name = sub1;
      category->label = sub2;
      
      escaped = st_transfer_escape(sub1);
      category->url_postfix = g_strconcat("&sgenre=", escaped, NULL);
      g_free(escaped);
      
      node = g_node_new(category);
      
      if (! strncmp(category->label, " - ", 3))
	{
	  if (info->parent_node)
	    {
	      category->label = g_strdup(&sub2[3]);
	      g_free(sub2);
	      
	      g_node_append(info->parent_node, node);
	    }
	  else
	    {
	      st_notice(_("SHOUTcast: parent category not found"));
	      st_category_free(category);
	      g_node_destroy(node);
	    }
	}
      else
	{
	  g_node_append(*(info->categories), node);
	  info->parent_node = node;
	}
    }
  else if (st_re_parse(&re_genre, line, &sub1, &sub2))
    {
      if (info->stream)
	{
	  info->stream->genre = sub1;
	  info->stream->description = sub2;
	}
      else
	{
	  st_notice(_("SHOUTcast: found misplaced genre and description"));
	  g_free(sub1);
	  g_free(sub2);
	}
    }
  else if (st_re_parse(&re_playing, line, &sub1))
    {
      if (info->stream)
	info->stream->now_playing = sub1;
      else
	{
	  st_notice(_("SHOUTcast: found misplaced now_playing"));
	  g_free(sub1);
	}
    }
  else if (st_re_parse(&re_listeners, line, &sub1, &sub2))
    {
      if (info->stream)
	{
	  info->stream->listeners = atoi(sub1);
	  info->stream->max = atoi(sub2);
	}
      else
	{
	  st_notice(_("SHOUTcast: found misplaced listeners and max"));
	  g_free(sub1);
	  g_free(sub2);
	}
    }
  else if (st_re_parse(&re_page, line, &sub1, &sub2))
    {
      info->page = atoi(sub1);
      info->npages = atoi(sub2);

      g_free(sub1);
      g_free(sub2);
    }
  
  if (converted)
    g_free(converted);
}

static gboolean
search_url_cb (STCategory *category)
{
  char *str;

  str = st_search_dialog();
  if (str)
    {
      char *escaped;

      g_free(category->label);
      category->label = g_strdup_printf(_("Search results for \"%s\""), str);

      escaped = st_transfer_escape(str);
      g_free(str);

      g_free(category->url_postfix);
      category->url_postfix = g_strconcat("&s=", escaped, NULL);
      g_free(escaped);

      return TRUE;
    }
  else
    return FALSE;
}

static gboolean
init_re (void)
{
  int status;

  status = regcomp(&re_header_charset, "^Content-Type: .*charset=(.*)", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_body_charset, "<meta http-equiv=.* content=.*charset=(.*)\"", REG_EXTENDED | REG_ICASE);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_genre, "^      <td .*<b>\\[(.*)\\]</font>.*<a .*>(.*)</a>", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_playing, "^<font .*>Now Playing:</font>(.*)</font></font></td>", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_listeners, "      <td .*>([0-9]+)/([0-9]+)</font>", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_bitrate, "^      <td .*>([0-9]+)</font>", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_stream, "<a href=\"/(sbin/shoutcast-playlist.pls.*filename.pls)\"", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_category, "^\t\t?<OPTION VALUE=\"(.*)\">(.*)$", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);
  status = regcomp(&re_page, "^Page ([0-9]+) of ([0-9]+)&nbsp;", REG_EXTENDED);
  g_return_val_if_fail(status == 0, FALSE);

  return TRUE;
}

static void
init_handler (void)
{
  STHandler *handler;
  
  GNode *stock_categories;
  STCategory *category;

  stock_categories = g_node_new(NULL);

  category = st_category_new();
  category->name = "__main";
  category->label = _("Top streams");
  category->url_postfix = "&sgenre=TopTen";
  
  g_node_append_data(stock_categories, category);
  
  category = st_category_new();
  category->name = "__search";
  category->label = g_strdup(_("Search"));
  category->url_cb = search_url_cb;

  g_node_append_data(stock_categories, category);

  handler = st_handler_new("shoutcast");

  st_handler_set_label(handler, "SHOUTcast");
  st_handler_set_copyright(handler, COPYRIGHT);
  st_handler_set_description(handler, _("SHOUTcast Yellow Pages"));
  st_handler_set_home(handler, "http://www.shoutcast.com/");
  st_handler_set_icon_from_inline(handler, sizeof(art_shoutcast), art_shoutcast);
  st_handler_set_stock_categories(handler, stock_categories);
  
  st_handler_bind(handler, ST_HANDLER_EVENT_REFRESH, refresh_cb, NULL);

  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_NEW, stream_new_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FIELD_GET, stream_field_get_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FIELD_SET, stream_field_set_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FREE, stream_free_cb, NULL);

  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_TUNE_IN, stream_tune_in_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_RECORD, stream_record_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_BROWSE, stream_browse_cb, NULL);

  st_handler_add_field(handler, st_handler_field_new(FIELD_GENRE,
						     _("Genre"),
						     G_TYPE_STRING,
						     TRUE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_DESCRIPTION,
						     _("Description"),
						     G_TYPE_STRING,
						     TRUE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_NOW_PLAYING,
						     _("Now playing"),
						     G_TYPE_STRING,
						     TRUE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_LISTENERS,
						     _("Listeners"),
						     G_TYPE_INT,
						     TRUE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_MAX,
						     _("Max"),
						     G_TYPE_INT,
						     TRUE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_BITRATE,
						     _("Bitrate"),
						     G_TYPE_INT,
						     TRUE));
  st_handler_add_field(handler, st_handler_field_new(FIELD_URL_POSTFIX,
						     _("URL postfix"),
						     G_TYPE_STRING,
						     FALSE));

  st_handlers_add(handler);
}

#ifdef WITH_STATIC_SHOUTCAST
gboolean
shoutcast_init (GError **err)
#else
gboolean
plugin_init (GError **err)
#endif
{
  gboolean status;
  
  if (! st_check_api_version(5, 4))
    {
      g_set_error(err, 0, 0, _("API version mismatch"));
      return FALSE;
    }
  
  status = init_re();
  g_return_val_if_fail(status == TRUE, FALSE);
  
  init_handler();
  
  st_action_register("play-m3u", _("Listen to a .m3u file"), "xmms %q");
  st_action_register("record-stream", _("Record a stream"), "xterm -hold -e streamripper %q");
  st_action_register("view-web", _("Browse a website"), "epiphany %q");

  return TRUE;
}
