import straw
import error
import locale
from StringIO import StringIO

PSEUDO_ALL_KEY = 'ALL'
PSEUDO_UNCATEGORIZED_KEY = 'UNCATEGORIZED'

class FeedCategoryList(object, straw.SignalEmitter):
    def __init__(self):
        straw.SignalEmitter.__init__(self)
        self.initialize_slots(straw.FeedCategoryChangedSignal,
                              straw.FeedCategoryListChangedSignal,
                              straw.FeedCategoryAddedSignal,
                              straw.FeedCategoryRemovedSignal)
        # have to define this here so the titles can be translated
        PSEUDO_TITLES = {PSEUDO_ALL_KEY: _('All'),
                         PSEUDO_UNCATEGORIZED_KEY: _('Uncategorized')}
        self._all_category = PseudoCategory(PSEUDO_TITLES[PSEUDO_ALL_KEY],
                                            PSEUDO_ALL_KEY)
        self._un_category = PseudoCategory(
            PSEUDO_TITLES[PSEUDO_UNCATEGORIZED_KEY], PSEUDO_UNCATEGORIZED_KEY)
        self._user_categories = []
        self._pseudo_categories = (self._un_category, self._all_category)
        self._loading = False
        self._feedlist = straw.FeedList.get_instance()
        self._feedlist.signal_connect(straw.FeedDeletedSignal,
                                self.feed_deleted)
        self._feedlist.signal_connect(straw.FeedCreatedSignal,
                                self.feed_created)

    def load_data(self):
        """Loads categories from config.
        Data format: [[{'title': '', 'subscription': {}, 'pseudo': pseudo key},
        {'id', feed_id1, 'from_subscription': bool} ...], ...]
        """
        cats = straw.Config.get_instance().categories or []
        categorized = {}
        for c in cats:
            head = c[0]
            tail = c[1:]
            pseudo_key = head.get('pseudo', None)
            if pseudo_key == PSEUDO_ALL_KEY:
                fc = self.all_category
            elif pseudo_key == PSEUDO_UNCATEGORIZED_KEY:
                fc = self.un_category
            else:
                fc = FeedCategory(head['title'])
                sub = head.get('subscription', None)
                if sub is not None:
                    fc.subscription = undump_subscription(sub)
            for f in tail:
                # backwards compatibility for stuff saved with versions <= 0.23
                if type(f) is int:
                    fid = f
                    from_sub = False
                else:
                    fid = f['id']
                    from_sub = f['from_subscription']
                feed = self._feedlist.get_feed_with_id(fid)
                
                if feed is not None:
                    if feed in fc.feeds:
                        error.log("%s (%d) was already in %s, skipping" % (str(feed), fid, str(fc)))
                        continue

                    fc.append_feed(feed, from_sub)
                    categorized[feed] = True
            # User categories: connect pseudos later
            if not pseudo_key:
                fc.signal_connect(straw.FeedCategoryChangedSignal,
                                  self.category_changed)
                self._user_categories.append(fc)
        # just in case we've missed any feeds, go through the list
        # and add to the pseudocategories. cache the feed list of all_category
        # so we don't get a function call (and a list comprehension loop
        # inside it) on each feed. it should be ok here, there are no
        # duplicates in feedlist. right?
        pseudos_changed = False
        all_feeds = self.all_category.feeds
        for f in self._feedlist:
            if f not in all_feeds:
                self.all_category.append_feed(f, False)
                pseudos_changed = True
            uf =  categorized.get(f, None)
            if uf is None:
                self.un_category.append_feed(f, False)
                pseudos_changed = True
        if pseudos_changed:
           self.save_data()
        for cat in self.pseudo_categories:
            cat.signal_connect(
                straw.FeedCategoryChangedSignal, self.pseudo_category_changed)

    def save_data(self):
        straw.Config.get_instance().categories = [
            cat.dump() for cat in self]

    def pseudo_category_changed(self, signal):
        self.save_data()
        self.emit_signal(signal)

    def category_changed(self, signal):
        if signal.feed is not None:
            uncategorized = True
            for cat in self.user_categories:
                if signal.feed in cat.feeds:
                    uncategorized = False
                    break
            if uncategorized:
                self.un_category.append_feed(signal.feed, False)
            else:
                try:
                    self.un_category.remove_feed(signal.feed)
                except ValueError:
                    pass
        self.save_data()
        self.emit_signal(signal)

    def feed_deleted(self, signal):
        for c in self:
            try:
                c.remove_feed(signal.feed)
            except ValueError:
                pass
        #self.emit_signal(straw.FeedCategoriesChangedSignal(self))

    def feed_created(self, signal):
        value = signal.feed
        category = signal.category
        index = signal.index

        #cm = CategoryMember(value, False)
        if (category is not None) and (category not in self.pseudo_categories):
            if index is not None:
                category.insert_feed(index, value, False)
            else:
                category.append_feed(value, False)
        else:
            self.un_category.append_feed(value, False)

        self.all_category.append_feed(value, False)

    def get_user_categories(self):
        return self._user_categories

    user_categories = property(get_user_categories)

    def get_pseudo_categories(self):
        return self._pseudo_categories

    pseudo_categories = property(get_pseudo_categories)

    def get_all_categories(self):
        return self.pseudo_categories + tuple(self.user_categories)

    all_categories = property(get_all_categories)

    def get_all_category(self):
        return self._all_category

    all_category = property(get_all_category)

    def get_un_category(self):
        return self._un_category

    un_category = property(get_un_category)

    class CategoryIterator:
        def __init__(self, fclist):
            self._fclist = fclist
            self._index = -1

        def __iter__(self):
            return self

        def _next(self):
            self._index += 1
            i = self._index
            uclen = len(self._fclist.user_categories)
            if i < uclen:
                return self._fclist.user_categories[i]
            elif i < uclen + len(self._fclist.pseudo_categories):
                return self._fclist.pseudo_categories[i - uclen]
            else:
                raise StopIteration

        def next(self):
            v = self._next()
            return v

    def __iter__(self):
        return self.CategoryIterator(self)

    def add_category(self, category):
        category.signal_connect(straw.FeedCategoryChangedSignal,
                                self.category_changed)
        self._user_categories.append(category)
        self._user_categories.sort(lambda a, b: locale.strcoll(a.title.lower(), b.title.lower()))
        self.emit_signal(straw.FeedCategoryAddedSignal(self, category))
        self.save_data()

    def remove_category(self, category):
        category.signal_disconnect(straw.FeedCategoryChangedSignal,
                                   self.category_changed)
        self._user_categories.remove(category)
        self.emit_signal(straw.FeedCategoryRemovedSignal(self, category))
        self.save_data()

# It might be good to have a superclass FeedCategorySubscription or something
# so we could support different formats. However, I don't know of any other
# relevant format used for this purpose, so that can be done later if needed.
# Of course, they could also just implement the same interface.
class OPMLCategorySubscription(object, straw.SignalEmitter):
    REFRESH_DEFAULT = -1
    
    def __init__(self, location=None):
        straw.SignalEmitter.__init__(self)
        self.initialize_slots(straw.FeedCategoryChangedSignal,
                              straw.SubscriptionContentsUpdatedSignal)
        self._location = location
        self._username = None
        self._password = None
        self._previous_etag = None
        self._contents = None
        self._frequency = OPMLCategorySubscription.REFRESH_DEFAULT
        self._last_poll = 0
        self._error = None

    def get_location(self): return self._location

    def set_location(self, location):
        self._location = location
        self.emit_signal(straw.FeedCategoryChangedSignal(self))

    location = property(get_location, set_location)

    def get_username(self): return self._username

    def set_username(self, username):
        self._username = username
        self.emit_signal(straw.FeedCategoryChangedSignal(self))

    username = property(get_username, set_username)

    def get_password(self): return self._password

    def set_password(self, password):
        self._password = password
        self.emit_signal(straw.FeedCategoryChangedSignal(self))

    password = property(get_password, set_password)

    def get_previous_etag(self): return self._previous_etag

    def set_previous_etag(self, etag):
        self._previous_etag = etag
        self.emit_signal(straw.FeedCategoryChangedSignal(self))

    previous_etag = property(get_previous_etag, set_previous_etag)

    def get_frequency(self): return self._frequency

    def set_frequency(self, frequency):
        self._frequency = frequency
        self.emit_signal(straw.FeedCategoryChangedSignal(self))

    frequency = property(get_frequency, set_frequency)

    def get_last_poll(self): return self._last_poll
    def set_last_poll(self, last_poll):
        self._last_poll = last_poll
        self.emit_signal(straw.FeedCategoryChangedSignal(self))
    last_poll = property(get_last_poll, set_last_poll)

    def get_error(self): return self._error
    def set_error(self, error):
        self._error = error
        self.emit_signal(straw.FeedCategoryChangedSignal(self))
    error = property(get_error, set_error)

    def parse(self, data):
        datastream = StringIO(data)
        entries = straw.OPMLImport.read(datastream)
        contents = [(e.url, e.text) for e in entries]
        updated = contents == self._contents
        self._contents = contents
        if updated:
            self.emit_signal(straw.SubscriptionContentsUpdatedSignal(self))

    def get_contents(self):
        return self._contents

    contents = property(get_contents)

    def undump(klass, dictionary):
        sub = klass()
        sub.location = dictionary.get('location')
        sub.username = dictionary.get('username')
        sub.password = dictionary.get('password')
        sub.frequency = dictionary.get(
            'frequency', OPMLCategorySubscription.REFRESH_DEFAULT)
        sub.last_poll = dictionary.get('last_poll', 0)
        sub.error = dictionary.get('error')
        return sub
    undump = classmethod(undump)

    def dump(self):
        return {'type': 'opml',
                'location': self.location,
                'username': self.username,
                'password': self.password,
                'frequency': self.frequency,
                'last_poll': self.last_poll,
                'error': self.error}

def undump_subscription(dictionary):
    try:
        if dictionary.get('type') == 'opml':
            return OPMLCategorySubscription.undump(dictionary)
    except Exception, e:
        error.log("exception while undumping subscription: " + str(e))
        raise

class CategoryMember(object):
    def __init__(self, feed=None, from_sub=False):
        self._feed = feed
        self._from_subscription = from_sub

    def get_feed(self): return self._feed
    def set_feed(self, feed): self._feed = feed
    feed = property(get_feed, set_feed)

    def get_from_subscription(self): return self._from_subscription
    def set_from_subscription(self, p): self._from_subscription = p
    from_subscription = property(get_from_subscription, set_from_subscription)

class FeedCategory(list, straw.SignalEmitter):
    def __init__(self, title=""):
        straw.SignalEmitter.__init__(self)
        self.initialize_slots(straw.FeedCategoryChangedSignal,
                              straw.SaveFeedsSignal)
        self._title = title
        self._subscription = None

    def get_title(self):
        return self._title

    def set_title(self, title):
        self._title = title
        self.emit_signal(straw.FeedCategoryChangedSignal(self))

    title = property(get_title, set_title)

    def read_contents_from_subscription(self):
        if self.subscription is None:
            return
        subfeeds = self.subscription.contents
        sfdict = dict(subfeeds)
        feedlist = straw.FeedList.get_instance()
        current = dict([(feed.location, feed) for feed in self.feeds])
        allfeeds = dict([(feed.location, feed) for feed in feedlist])
        common, toadd, toremove = straw.utils.listdiff(sfdict.keys(),
                                                       current.keys())
        existing, nonexisting, ignore = straw.utils.listdiff(
            toadd, allfeeds.keys())
        for f in nonexisting:
            feed = straw.create_new_feed(sfdict[f], f)
            feedlist.append(None, feed)
            self.append_feed(feed, True)
        for f in existing:
            self.append_feed(allfeeds[f], True)
        for f in toremove:
            index = self.index_feed(allfeeds[f])
            member = self[index]
            if member.from_subscription:
                self.remove_feed(allfeeds[f])
        return

    def _subscription_changed(self, signal):
        self.emit_signal(straw.FeedCategoryChangedSignal(self))

    def _subscription_contents_updated(self, signal):
        self.read_contents_from_subscription()

    def get_subscription(self):
        return self._subscription

    def set_subscription(self, sub):
        if self._subscription is not None:
            self._subscription.signal_disconnect(
                straw.FeedCategoryChangedSignal, self._subscription_changed)
            self._subscription.signal_disconnect(
                straw.SubscriptionContentsUpdatedSignal,
                self._subscription_contents_updated)
        if sub is not None:
            sub.signal_connect(straw.FeedCategoryChangedSignal,
                               self._subscription_changed)
            sub.signal_connect(straw.SubscriptionContentsUpdatedSignal,
                               self._subscription_contents_updated)
        self._subscription = sub
        self.emit_signal(straw.FeedCategoryChangedSignal(self))

    subscription = property(get_subscription, set_subscription)

    def __str__(self):
        return "FeedCategory %s" % self.title

    def __hash__(self):
        return hash(id(self))

    def append(self, value):
        error.log("warning, probably should be using append_feed?")
        list.append(self, value)
        self.emit_signal(straw.FeedCategoryChangedSignal(self, feed=value.feed))

    def append_feed(self, value, from_sub):
        list.append(self, CategoryMember(value, from_sub))
        self.emit_signal(straw.FeedCategoryChangedSignal(self, feed=value))

    def insert(self, index, value):
        error.log("warning, probably should be using insert_feed?")
        list.insert(self, index, value)
        self.emit_signal(straw.FeedCategoryChangedSignal(self, feed=value.feed))

    def insert_feed(self, index, value, from_sub):
        list.insert(self, index, CategoryMember(value, from_sub))
        self.emit_signal(straw.FeedCategoryChangedSignal(self, feed=value))

    def remove(self, value):
        list.remove(self, value)
        self.emit_signal(straw.FeedCategoryChangedSignal(self, feed=value.feed))

    def remove_feed(self, value):
        for index in range(len(self)):
            if self[index].feed is value:
                del self[index]
                break
        else:
            raise ValueError(value)
        self.emit_signal(straw.FeedCategoryChangedSignal(self, feed=value))

    def reverse(self):
        list.reverse(self)
        self.emit_signal(straw.FeedCategoryChangedSignal(self))

    def index_feed(self, value):
        for index in range(len(self)):
            if self[index].feed is value:
                return index
        raise ValueError(value)

    def sort(self, indices=None):
        if (not indices) or (len(indices) == 1):
            list.sort(self, lambda a, b: locale.strcoll(
                a.feed.title.lower(), b.feed.title.lower()))
        else:
            items = []
            for i in indices:
                items.append(self[i])
            items.sort(lambda a, b: locale.strcoll(
                a.feed.title.lower(), b.feed.title.lower()))
            for i in range(len(items)):
                list.__setitem__(self, indices[i], items[i])
        self.emit_signal(straw.FeedCategoryChangedSignal(self))

    def move_feed(self, source, target):
        if target > source:
            target -= 1
        if target == source:
            return
        t = self[source]
        del self[source]
        list.insert(self, target, t)
        self.emit_signal(straw.FeedCategoryChangedSignal(self))

    def dump(self):
        head = {'title': self.title}
        if self.subscription is not None:
            head['subscription'] = self.subscription.dump()
        return [head] + [
            {'id': f.feed.id, 'from_subscription': f.from_subscription}
            for f in self]

    def get_feeds(self):
        return [f.feed for f in self]

    feeds = property(get_feeds)

    def __eq__(self, ob):
        if isinstance(ob, FeedCategory):
            return self.title == ob.title and list.__eq__(self, ob)
        else:
            raise NotImplementedError

    def __contains__(self, item):
        error.log("warning, should probably be querying the feeds property instead?")
        return list.__contains__(self, item)

class PseudoCategory(FeedCategory):
    def __init__(self, title="", key=None):
        if key not in (PSEUDO_ALL_KEY, PSEUDO_UNCATEGORIZED_KEY):
            raise ValueError, "Invalid key"
        FeedCategory.__init__(self, title)
        self._pseudo_key = key

    def __str__(self):
        return "PseudoCategory %s" % self.title

    def dump(self):
        return [{'pseudo': self._pseudo_key, 'title': ''}] + [
            {'id': f.feed.id, 'from_subscription': False} for f in self]

    def append_feed(self, feed, from_sub):
        assert not from_sub
        FeedCategory.append_feed(self, feed, False)

    def insert_feed(self, index, feed, from_sub):
        assert not from_sub
        FeedCategory.insert_feed(self, index, feed, False)

fclist = None

def get_instance():
    global fclist
    if fclist is None:
        fclist = FeedCategoryList()
    return fclist
