import straw
from error import log
import locale

PSEUDO_ALL_KEY = 'ALL'
PSEUDO_UNCATEGORIZED_KEY = 'UNCATEGORIZED'

class FeedCategoryList(object, straw.SignalEmitter):
    def __init__(self):
        straw.SignalEmitter.__init__(self)
        self.initialize_slots(straw.FeedCategoriesChangedSignal)
        # have to define this here so the titles can be translated
        PSEUDO_TITLES = {PSEUDO_ALL_KEY: _('All'),
                         PSEUDO_UNCATEGORIZED_KEY: _('Uncategorized')}
        self._all_category = PseudoCategory(PSEUDO_TITLES[PSEUDO_ALL_KEY],
                                            PSEUDO_ALL_KEY)
        self._un_category = PseudoCategory(
            PSEUDO_TITLES[PSEUDO_UNCATEGORIZED_KEY], PSEUDO_UNCATEGORIZED_KEY)
        self._user_categories = []
        self._pseudo_categories = (self._un_category, self._all_category)
        self._loading = False
        self._feedlist = straw.FeedList.get_instance()
        self._feedlist.signal_connect(straw.FeedDeletedSignal,
                                self.feed_deleted)
        self._feedlist.signal_connect(straw.FeedCreatedSignal,
                                self.feed_created)

    def load_data(self):
        cats = straw.Config.get_instance().categories or []
        categorized = {}
        for c in cats:
            head = c[0]
            tail = c[1:]
            pseudo_key = head.get('pseudo', None)
            if pseudo_key == PSEUDO_ALL_KEY:
                fc = self.all_category
            elif pseudo_key == PSEUDO_UNCATEGORIZED_KEY:
                fc = self.un_category
            else:
                fc = FeedCategory(head['title'])
            for fid in tail:
                feed = self._feedlist.get_feed_with_id(fid)
                if feed is not None:
                    if feed in fc:
                        log("%s was already in %s, skipping" % (str(feed), str(fc)))
                        continue
                    fc.append(feed)
                    categorized[feed] = True
            # User categories: connect pseudos later
            if not pseudo_key:
                fc.signal_connect(straw.FeedCategoriesChangedSignal,
                                  self.category_changed)
                self._user_categories.append(fc)
        # just in case we've missed any feeds, go through the list
        # and add to the pseudocategories
        pseudos_changed = False
        for f in self._feedlist:
            if f not in self.all_category:
                self.all_category.append(f)
                pseudos_changed = True
            uf =  categorized.get(f, None)
            if uf is None:
                self.un_category.append(f)
                pseudos_changed = True
        if pseudos_changed:
           self.save_data()
        for cat in self.pseudo_categories:
            cat.signal_connect(
                straw.FeedCategoriesChangedSignal, self.pseudo_category_changed)

    def save_data(self):
        straw.Config.get_instance().categories = [
            cat.dump() for cat in self]

    def pseudo_category_changed(self, signal):
        self.save_data()
        self.emit_signal(signal)

    def category_changed(self, signal):
        if signal.feed is not None:
            uncategorized = True
            for cat in self.user_categories:
                if signal.feed in cat:
                    uncategorized = False
                    break
            if uncategorized:
                self.un_category.append(signal.feed)
            else:
                try:
                    self.un_category.remove(signal.feed)
                except ValueError:
                    pass
        self.save_data()
        self.emit_signal(signal)

    def feed_deleted(self, signal):
        for c in self:
            try:
                c.remove(signal.feed)
            except ValueError:
                pass
        self.emit_signal(straw.FeedCategoriesChangedSignal(self))

    def feed_created(self, signal):
        value = signal.feed
        category = signal.category
        index = signal.index

        if (category is not None) and (category not in self.pseudo_categories):
            if index is not None:
                category.insert(index,value)
            category.append(value)
        else:
            self.un_category.append(value)

        self.all_category.append(value)

    def get_user_categories(self):
        return self._user_categories

    user_categories = property(get_user_categories)

    def get_pseudo_categories(self):
        return self._pseudo_categories

    pseudo_categories = property(get_pseudo_categories)

    def get_all_category(self):
        return self._all_category

    all_category = property(get_all_category)

    def get_un_category(self):
        return self._un_category

    un_category = property(get_un_category)

    class CategoryIterator:
        def __init__(self, fclist):
            self._fclist = fclist
            self._index = -1

        def __iter__(self):
            return self

        def _next(self):
            self._index += 1
            i = self._index
            uclen = len(self._fclist.user_categories)
            if i < uclen:
                return self._fclist.user_categories[i]
            elif i < uclen + len(self._fclist.pseudo_categories):
                return self._fclist.pseudo_categories[i - uclen]
            else:
                raise StopIteration

        def next(self):
            v = self._next()
            return v

    def __iter__(self):
        return self.CategoryIterator(self)

    def add_category(self, category):
        category.signal_connect(straw.FeedCategoriesChangedSignal,
                                self.category_changed)
        self._user_categories.append(category)
        self._user_categories.sort(lambda a, b: locale.strcoll(a.title.lower(), b.title.lower()))
        self.emit_signal(straw.FeedCategoriesChangedSignal(self))
        self.save_data()

    def remove_category(self, category):
        category.signal_disconnect(straw.FeedCategoriesChangedSignal,
                                   self.category_changed)
        self._user_categories.remove(category)
        self.emit_signal(straw.FeedCategoriesChangedSignal(self))
        self.save_data()

class FeedCategory(list, straw.SignalEmitter):
    def __init__(self, title=""):
        straw.SignalEmitter.__init__(self)
        self.initialize_slots(straw.FeedCategoriesChangedSignal,
                              straw.SaveFeedsSignal)
        self._title = title

    def get_title(self):
        return self._title

    def set_title(self, title):
        self._title = title
        self.emit_signal(straw.FeedCategoriesChangedSignal(self))

    title = property(get_title, set_title)

    def __str__(self):
        return "FeedCategory %s" % self.title

    def __hash__(self):
        h = hash(self.title)
        for item in self:
            h ^= hash(item)
        return h

    def append(self, value):
        list.append(self, value)
        self.emit_signal(straw.FeedCategoriesChangedSignal(self, feed=value))

    def insert(self, index, value):
        list.append(self, index, value)
        self.emit_signal(straw.FeedCategoriesChangedSignal(self, feed=value))

    def remove(self, value):
        list.remove(self, value)
        self.emit_signal(straw.FeedCategoriesChangedSignal(self, feed=value))

    def reverse(self):
        list.reverse(self)
        self.emit_signal(straw.FeedCategoriesChangedSignal(self))

    def sort(self, indices=None):
        if (not indices) or (len(indices) == 1):
            list.sort(self, lambda a, b: locale.strcoll(a.title.lower(), b.title.lower()))
        else:
            items = []
            for i in indices:
                items.append(self[i])
            items.sort(lambda a, b: locale.strcoll(a.title.lower(), b.title.lower()))
            for i in range(len(items)):
                list.__setitem__(self, indices[i], items[i])
        self.emit_signal(straw.FeedCategoriesChangedSignal(self))

    def move_feed(self, source, target):
        if target > source:
            target -= 1
        if target == source:
            return
        t = self[source]
        del self[source]
        list.insert(self, target, t)
        self.emit_signal(straw.FeedCategoriesChangedSignal(self))

    def dump(self):
        return [{'title': self.title}] + [
            f.id for f in self]

    def __eq__(self, ob):
        if isinstance(ob, FeedCategory):
            return self.title == ob.title and list.__eq__(self, ob)
        else:
            raise NotImplemented

class PseudoCategory(FeedCategory):
    def __init__(self, title="", key=None):
        if key not in (PSEUDO_ALL_KEY, PSEUDO_UNCATEGORIZED_KEY):
            raise ValueError, "Invalid key"
        FeedCategory.__init__(self, title)
        self._pseudo_key = key
    
    def __str__(self):
        return "PseudoCategory %s" % self.title

    def dump(self):
        return [{'pseudo': self._pseudo_key, 'title': ''}] + [
            f.id for f in self]

fclist = None

def get_instance():
    global fclist
    if fclist is None:
        fclist = FeedCategoryList()
    return fclist
