cvs_id = "$Id: PollManager.py,v 1.4 2004/04/24 12:01:23 jpakaste Exp $"

import straw
import time
import gtk
import socket
import error

class PollManager:
    def __init__(self):
        self._config = straw.Config.get_instance()
        self._mmgr = straw.MessageManager.get_instance()
        self._feedlist = straw.FeedList.get_instance()

        self._config.signal_connect(straw.PollFrequencyChangedSignal, self.poll_changed)
        self._config.signal_connect(straw.OfflineModeChangedSignal, self.offline_changed)
        self._feedlist.signal_connect(straw.FeedsChangedSignal, self.feeds_changed)

        self._is_offline = self._config.offline
        self._poll_frequency = self._config.poll_frequency
        self._last_poll = self._config.last_poll
        self._feeds = self._feedlist.flatten_list()

    def poll_changed(self, signal):
        """ called when global poll frequency has changed"""
        self._poll_frequency = signal.value

    def offline_changed(self, signal):
        self._is_offline = self._config.offline
        if not self._is_offline:
            gtk.timeout_add(straw.NetworkConstants.POLL_INTERVAL, self.maybe_poll)

    def feeds_changed(self, *args):
        self._feedlist = straw.FeedList.get_instance()
        self._feeds = self._feedlist.flatten_list()

    def start_polling_loop(self):
        if not self._is_offline:
            gtk.timeout_add(straw.NetworkConstants.POLL_INTERVAL, self.maybe_poll)

    def maybe_poll(self):

        self.poll([feed for feed, timediff in [(feed, int(time.time()) - feed.last_poll) for feed in self._feeds]
                     if ((timediff > feed.poll_frequency > 0) or
                         (feed.poll_frequency == straw.Feed.DEFAULT and timediff > self._config.poll_frequency > 0))])

        gtk.timeout_add(straw.NetworkConstants.POLL_INTERVAL, self.maybe_poll)

    def poll(self, obj):
        """ obj must be a list of Feed objects"""

        for f in obj:
            Poller(f, self).poll()

        self.poll_network()

    def poll_network(self):

        try:
            straw.URLFetch.connection_manager.poll(straw.NetworkConstants.POLL_TIMEOUT)
        except socket.error, ex:
            error.log("%s" % ex)


class Poller:
    def __init__(self, feed, pm):
        self._feed = feed
        self._pm = pm
        self._mmgr = straw.MessageManager.get_instance()

    def poll(self):
        url, user, pw = self._feed.get_access_info()
        parsed = None
        headers = {}
        if self._feed.previous_etag is not None:
            headers['If-None-Match'] = self._feed.previous_etag
        try:
            pc = straw.URLFetch.connection_manager.request(
                url, self, headers, user, pw,
                priority=straw.NetworkConstants.PRIORITY_RSS)
        except Exception, e:
            self.http_failed(e)
        else:
            self._feed.last_poll = int (time.time())
            self._feed.router.start_polling()

    def http_results(self, status, header, data):
        self._feed.poll_done()
        error = ""
        if status is None:
            error = _("No data (%s)" % status[1])
        elif status[1] == 304:
            self._feed.router.route_no_data()
        elif status[1] == 410 or status[1] == 404:
            error = _("Unable to find the feed (%s)" % status[1])
        elif status[1] > 299:
            error = _("Updating feed resulted in abnormal status '%s' (code %d)") % (status[2].strip(), status[1].strip())

        if error:
            self._feed.router.set_error(error)
            return

        try:
            parsed = straw.SummaryParser.parse(data, self._feed)
        except Exception, e:
            self._feed.router.set_error(_("An error occurred while processing feed: %s") % str(e))
        else:
            self._feed.router.route_all(header, parsed)

        self._mmgr.post_message(_("Updating %s done.") % self._feed.title)

    def http_failed(self, exception):
        if isinstance(exception, socket.error):
            err = exception.args[1]
        else:
            err = str(exception)
        self._feed.router.set_error(err)
        self._mmgr.post_message(_("Updating %s failed") % self._feed.title)
        self._feed.poll_done()

    def http_permanent_redirect(self, location):
        (oldloc, u, p) = self._feed.access_info
        self._feed.access_info = (location, u, p)

pollmanager_instance = None

def get_instance():
    global pollmanager_instance
    if pollmanager_instance is None:
        pollmanager_instance = PollManager()
    return pollmanager_instance
