cvs_id = "$Id: PreferencesDialog.py,v 1.20 2003/11/30 00:49:34 juri Exp $"

import gobject
import gtk
import pango
import straw
import locale

class PreferencesDialog:
    SEC_PER_MINUTE = 60
    FEED_COLUMN_TITLE = 0
    FEED_COLUMN_OBJ = 1
    CATEGORY_COLUMN_TITLE = 0
    CATEGORY_COLUMN_OBJ = 1
    CATEGORY_COLUMN_BOLD = 2
    CATEGORY_COLUMN_EDITABLE = 3

    def __init__(self, xml):
        config = straw.Config.get_instance()
        self._window = xml.get_widget('preferences_dialog')
        self._window.set_transient_for(straw.main_window.get_window())

        self._selcategory = None # currently selected category
        self._selfeed = None     # currently selected feed

        self._delete_button = xml.get_widget('category_delete_button')
        self._delete_button.set_sensitive(False)

        # setup categories treeview
        self._category_list = straw.FeedCategoryList.get_instance()
        model = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_PYOBJECT,
                              gobject.TYPE_INT, gobject.TYPE_BOOLEAN)

        for category in self._category_list.pseudo_categories:
            iter = model.append()
            model.set(iter, self.CATEGORY_COLUMN_TITLE, category.title,
                      self.CATEGORY_COLUMN_OBJ, category,
                      self.CATEGORY_COLUMN_BOLD, pango.WEIGHT_BOLD,
                      self.CATEGORY_COLUMN_EDITABLE, gtk.FALSE)

        for category in self._category_list.user_categories:
            iter = model.append()
            model.set(iter, self.CATEGORY_COLUMN_TITLE, category.title,
                      self.CATEGORY_COLUMN_OBJ, category,
                      self.CATEGORY_COLUMN_BOLD, pango.WEIGHT_NORMAL,
                      self.CATEGORY_COLUMN_EDITABLE, gtk.TRUE)

        self._category_treeview = xml.get_widget('category_treeview')
        self._category_treeview.set_model(model)
        self._category_treeview.set_rules_hint(False)
        renderer = gtk.CellRendererText()
        renderer.connect('edited', self.category_cell_edited, (model,
                                                      self.CATEGORY_COLUMN_TITLE,
                                                      self.CATEGORY_COLUMN_OBJ))
        column = gtk.TreeViewColumn(_('_Categories'), renderer,
                                    text=self.CATEGORY_COLUMN_TITLE,
                                    weight=self.CATEGORY_COLUMN_BOLD,
                                    editable=self.CATEGORY_COLUMN_EDITABLE)
        self._category_treeview.append_column(column)
        self._category_treeview.get_selection().connect('changed',
                                         self.category_changed,
                                         self.CATEGORY_COLUMN_OBJ)

        # setup feeds treeview.  Put the model in a gtk.TreeModelSort
        store = gtk.ListStore( gobject.TYPE_STRING, gobject.TYPE_PYOBJECT)
        model = gtk.TreeModelSort(store)
        model.set_sort_func(self.FEED_COLUMN_TITLE,self.sort_feeds,self.FEED_COLUMN_OBJ)
        model.connect('sort-column-changed',self.column_changed,self.FEED_COLUMN_OBJ)

        self._feed_properties_button = xml.get_widget('feed_properties_button')

        self._feeds_treeview = xml.get_widget('feeds_treeview')
        self._feeds_treeview.set_model(model)
        self._feeds_treeview.set_rules_hint(False)
        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn(_('_Feeds:'), renderer,
                                    text=self.FEED_COLUMN_TITLE)
        column.set_sort_column_id(self.FEED_COLUMN_TITLE)
        self._feeds_treeview.append_column(column)
        self._feeds_treeview.connect('cursor-changed', self.feeds_cursor_changed, self.FEED_COLUMN_OBJ)
        # default to ascending
        model.set_sort_column_id(self.FEED_COLUMN_TITLE, gtk.SORT_ASCENDING)

        # General
        config = straw.Config.get_instance()
        poll_frequency = int(config.poll_frequency/self.SEC_PER_MINUTE)
        items_stored = int(config.number_of_items_stored)

        xml.get_widget('poll_frequency_spin').set_value(poll_frequency)
        xml.get_widget('number_of_items_spin').set_value(items_stored)
        xml.get_widget(['item_order_oldest',
                        'item_order_newest'][config.item_order]).set_active(1)

        nameFuncMap = {}
        for key in dir(self.__class__):
            if key[:3] == 'on_':
                nameFuncMap[key] = getattr(self, key)
        xml.signal_autoconnect(nameFuncMap)


    def show(self, *args):
        selection = self._category_treeview.get_selection()
        # always select first row
        selection.select_path((0,))

        self._window.present()

    def hide(self, *args):
        self._window.hide()
        self._reset()

    def on_preferences_dialog_delete_event(self, *args):
        self.hide()
        return gtk.TRUE

    def on_preferences_close_button_clicked(self, button):
        self.hide()
        return

    def on_poll_frequency_spin_value_changed(self, spin):
        straw.Config.get_instance().poll_frequency = int(spin.get_value() * self.SEC_PER_MINUTE)

    def on_number_of_items_spin_value_changed(self, spin):
        straw.Config.get_instance().number_of_items_stored = int(spin.get_value())

    def on_item_order_newest_toggled(self, radio):
        config = straw.Config.get_instance()
        config.item_order = not config.item_order

    def on_item_order_oldest_toggled(self, radio):
        pass

    def _reset(self):
        selection = self._category_treeview.get_selection()
        if not selection.path_is_selected((0,)):
            # if 1st row is selected, don't clear it as
            # 'changed' signal is not going to be emitted
            # therefore we'll end up with an empty feeds model
            self._feeds_treeview.get_model().get_model().clear()
        selection.unselect_all()
        selection = self._feeds_treeview.get_selection()
        selection.unselect_all()
        self._delete_button.set_sensitive(False)

    def _unsorted_state(self):
        self._feeds_treeview.get_model().reset_default_sort_func()

    def feeds_cursor_changed(self, tv, column):
        selection = self._feeds_treeview.get_selection()
        model, iter = selection.get_selected()
        self._selfeed = model.get_value(iter, column)
        self._feed_properties_button.set_sensitive(True)

    def column_changed(self, model, obj):
        model = self._feeds_treeview.get_model()
        if model.get_iter_first() is None:
            # don't sort if there's nothing to sort
            return

        col, order = model.get_sort_column_id()

        if order == gtk.SORT_ASCENDING:
            self._selcategory.sort()
        elif order == gtk.SORT_DESCENDING:
            self._selcategory.reverse()
        else:
            self._unsorted_state()

        return

    def on_feed_properties_button_clicked(self, button):
        straw.main.show_feed_properties(self._selfeed)

    def category_cell_edited(self, cell, path_string, text, (model, column_title, column_obj)):
        iter = model.get_iter_from_string(path_string)
        if not iter: return
        category = model.get_value(iter, column_obj)
        category.title = text
        model.set(iter, column_title, category.title,
                  column_obj, category,
                  self.CATEGORY_COLUMN_BOLD, pango.WEIGHT_NORMAL,
                  self.CATEGORY_COLUMN_EDITABLE, gtk.TRUE)

    def on_category_add_button_clicked(self, *args):
        model = self._category_treeview.get_model()
        iter = model.append()
        path = model.get_path(iter)
        cat = straw.FeedCategoryList.FeedCategory(_("Category name"))
        model.set(iter, self.CATEGORY_COLUMN_TITLE, cat.title,
                  self.CATEGORY_COLUMN_OBJ, cat,
                  self.CATEGORY_COLUMN_BOLD, pango.WEIGHT_NORMAL,
                  self.CATEGORY_COLUMN_EDITABLE, gtk.TRUE)
        straw.FeedCategoryList.get_instance().add_category(cat)

        # set_cursor emits 'edited', calls cell_edited
        column = self._category_treeview.get_column(self.CATEGORY_COLUMN_TITLE)
        self._category_treeview.scroll_to_cell(path)
        self._category_treeview.set_cursor(path, column, gtk.TRUE)

    def on_category_delete_button_clicked(self, *args):
        selection = self._category_treeview.get_selection()
        model, iter = selection.get_selected()
        if iter:
            path = model.get_path(iter)
            category = model.get_value(iter, self.CATEGORY_COLUMN_OBJ)
            x=0

            # remove the feeds first
            while x < (len(category)):
                category.remove(category[x])

            model.remove(iter)
            selection.select_path(path)

            if not selection.path_is_selected(path):
                path = path[0]-1
                if path >= 0:
                    selection.select_path((path,))
                    self._category_treeview.set_cursor(path)

            straw.FeedCategoryList.get_instance().remove_category(category)

        return

    def category_changed(self, selection, obj):
        model, iter = selection.get_selected()
        if iter is None:
            # if nodes are unselect_all'd
            return

        self._selcategory = model.get_value(iter, obj)

        if self._selcategory in self._category_list.pseudo_categories:
            self._delete_button.set_sensitive(False)
        else:
            self._delete_button.set_sensitive(True)

        self._unsorted_state()

        self._feed_properties_button.set_sensitive(False)

        column = self._feeds_treeview.get_column(self.FEED_COLUMN_TITLE)
        column.set_sort_indicator(False)

        self.create_feed_model(self._selcategory)
        return

    def create_feed_model(self, category=None):
        if category is None:
            if self._selcategory is not None:
                category = self._selcategory
            else:
                return

        model = self._feeds_treeview.get_model().get_model()
        model.clear()

        for feed in category:
            iter = model.append()
            model.set(iter, self.FEED_COLUMN_TITLE, feed.title,
                      self.FEED_COLUMN_OBJ, feed)
        return

    def sort_feeds(self, model, a, b, column):
        A = model.get_value(a, column)
        B = model.get_value(b, column)

        if A is not None or B is not None:
            try:
                return locale.strcoll(A.title.lower(), B.title.lower())
            except AttributeError:
                return 1
        else:
            return 0
