cvs_id = "$Id: MainWindow.py,v 1.105 2003/11/25 12:51:35 jmalonzo Exp $"

import StringIO

import gobject
import gtk
import gnome
import gtkhtml2
import gnome.ui
import pango
from gtk import glade
from xml.sax import saxutils

import straw
from error import log, logparam

class ExceptionView:
    def __init__(self, frame, label):
        self._frame = frame
        self._label = label

    def show_exception(self, exception):
        self._label.set_text(
            _("<b>An error occured while reading this feed:</b>\n%s" %
              str(exception)))
        self._label.set_property("use_markup", True)
        self._frame.show()
        return

    def hide(self):
        self._frame.hide()
        return

class DummyDocStream(StringIO.StringIO):
    def write_stream(self, s):
        self.write(s)

class ItemView:
    def __init__(self, item_view_container):
        widget_tree = glade.get_widget_tree(item_view_container)

        self._source_item_hbox = widget_tree.get_widget('item_source_hbox')
        self._source_subs_hbox = widget_tree.get_widget('item_subscribe_hbox')
        self._source_subscribe_button = widget_tree.get_widget('source_subscribe_button')
        self._source_url = gnome.ui.HRef(url='', text='')
        self._source_item_hbox.pack_start(self._source_url, expand=gtk.FALSE, fill=gtk.TRUE)
        widget_tree.signal_connect('on_source_subscribe_button_clicked',
                                   self.on_source_subscribe_button_clicked)

        scrolled_window = widget_tree.get_widget('html_scrolled_window')
        htmlview = gtkhtml2.View()
        self._document = gtkhtml2.Document()
        self._document.connect("link-clicked", self.link_clicked)
        htmlview.connect("on_url", self.on_url)
        self._document.connect("request-url", self.request_url)
        htmlview.get_vadjustment().set_value(0)
        scrolled_window.set_hadjustment(htmlview.get_hadjustment())
        scrolled_window.set_vadjustment(htmlview.get_vadjustment())
        self._document.clear()
        htmlview.set_document(self._document)
        scrolled_window.add(htmlview)
        scrolled_window.show_all()
        self._scrolled_window = scrolled_window
        self._item = None
        self._htmlview = htmlview

    def on_url(self, view, url):
        if url is not None and (len(url) > 0):
            feed = straw.main.get_visible_feed()
            url = straw.utils.complete_url(url, feed.location)
        else:
            url = ''
        straw.MessageManager.get_instance().post_message(url)
        return

    def link_clicked(self, document, link):
        link = link.strip()
        feed = straw.main.get_visible_feed()
        link = straw.utils.complete_url(link, feed.location)
        try:
            gnome.url_show(link)
        except Exception, ex:
            straw.hig_alert.reportError(_("Error Loading Browser"),
                                        _("Straw cannot find a browser to view this item. Please check your browser settings and try again."))
        return

    def request_url(self, document, url, stream):
        try:
            feed = straw.main.get_visible_feed()
            url = straw.utils.complete_url(url, feed.location)
            image = straw.ImageCache.cache[url]
            stream.write(image.get_data())
        except Exception, ex:
            log(ex)

        stream.close()
        return

    def on_source_subscribe_button_clicked(self, button, *data):
        f = straw.create_new_feed(self._item.source['text'],
                                  location = self._item.source['url'])
        straw.FeedList.get_instance().append(f)
        straw.PollManager.get_instance().poll(f)
        return

    def display_item(self, item):
        self._document.clear()
        self._document.open_stream("text/html")
        enc = None
        html = (
            """<html><head>
            <meta http-equiv="Content-Type" content="text/html; charset=utf-8">
            <style type="text/css">
            q { font-style: italic;}
            blockquote { display: block; font-style: italic; }
            .stitle {background-color:transparent;font-size:small;font-weight:bold;}
            .sdate {font-size:small}
            .content {padding-left:10px;margin-top:10px}
            </style>
            <title>title</title></head><body>""" +
                self.htmlify_item(item) +
                "</body></html>")
        html = html.encode('utf-8')
	self._document.write_stream(html)
        self._document.close_stream()
        va = self._scrolled_window.get_vadjustment()
        va.set_value(va.lower)
        if item.source is not None:
            self._source_url.set_text(saxutils.escape(item.source['url']))
            self._source_url.set_url(item.source['url'])
            self._source_url.show()
            self._source_item_hbox.show()
            self._source_subs_hbox.show()
        else:
            self._source_item_hbox.hide()
            self._source_subs_hbox.hide()
        self._item = item
        return

    def display_empty_feed(self, feed):
        self._document.clear()
        self._document.open_stream("text/html")
        self._document.write_stream(self.htmlify_empty_feed(feed))
        self._document.close_stream()
        self._item = None
        self._source_item_hbox.hide()
        self._source_subs_hbox.hide()
        return

    def htmlify_empty_feed(self, feed):
        return ("<html><head><title>title</title></head><body><p>" +
                   _("No data yet, need to poll first.") +
                "</p></body></html>")

    def htmlify_item(self, item):
        ret = []
        if item.title is not None:
            ret.append('<div class="stitle">%s</div>' % item.title)
        if item.pub_date is not None:
            timestr = item.pub_date.strftime('%a, %d %b %H:%M:%S')
            timeustr = unicode(timestr, straw.utils.get_encoding())
            ret.append('<div class="sdate">%s</div>' % timeustr)
        ret.append('<div class="content">')
        if item.description is not None:
            ret.append('%s ' % item.description)
        ret.append('</div>')
        if item.creator is not None:
            ret.append('<p><b>%s:</b> %s</p>' % (_("Creator"), item.creator))
        if item.fm_license != '' and item.fm_license is not None:
            ret.append('<p><b>%s:</b> %s</p>' % (_("Software license"), item.fm_license))
        if item.fm_changes != '' and item.fm_changes is not None:
            ret.append('<p><b>%s:</b> %s</p>' % (_("Changes"), item.fm_changes))
        if item.license_urls is not None:
            ret.append("<p>")
            for l in item.license_urls:
                ret.append('<a href="%s">%s</a> ' % (l, _("License")))
            ret.append('</p>')
        if item.publication_name is not None:
            ret.append('<p>')
            ret.append('<b>%s:</b> %s ' % (_("Publication"),
                                           item.publication_name))
            if item.publication_volume is not None:
                ret.append('%s' % item.publication_volume)
                if item.publication_number is not None:
                    ret.append('(%s)' % item.publication_number)
            if item.publication_section is not None:
                ret.append(' %s' % item.publication_section)
            if item.publication_starting_page is not None:
                ret.append(', %s %s' % (_("Page"),
                                        item.publication_starting_page))
            ret.append('</p>')
        if item.guid is not None and straw.utils.is_url(item.guid):
            ret.append('<p><a href="%s">%s &gt;&gt;</a></p>' %
                       (item.guid, _("Open in web browser (permanent link)")))
        if item.link is not None:
            ret.append('<p><a href="%s">%s &gt;&gt;</a></p>' %
                       (item.link,_("Open in web browser (article link)")))
        return "".join(ret)

    def scroll_down(self):
        va = self._scrolled_window.get_vadjustment()
        old_value = va.get_value()
        new_value = old_value + va.page_increment
        limit = va.upper - va.page_size
        if new_value > limit:
            new_value = limit
        va.set_value(new_value)
        return new_value > old_value

class ItemList:
    COLUMN_TITLE = 0
    COLUMN_STICKY = 1
    COLUMN_ITEM = 2
    COLUMN_BOLD = 3
    COLUMN_STICKY_FLAG = 4

    def __init__(self, widget):
        self._widget = widget

        model = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_OBJECT,
                              gobject.TYPE_PYOBJECT, gobject.TYPE_INT,
                              gobject.TYPE_BOOLEAN)

        widget.set_model(model)
        widget.set_rules_hint(gtk.TRUE)

        self.create_item_selection_columns(self._widget)

        widget.get_selection().connect("changed", self.item_selection_changed)

        self._feed = None
        return

    def create_item_selection_columns(self, item_list_view):
        renderer = gtk.CellRendererToggle()
        column = gtk.TreeViewColumn(_('Keep'), renderer,
                                    active=self.COLUMN_STICKY_FLAG)
        column.set_resizable(gtk.TRUE)
        column.set_reorderable(gtk.TRUE)
        item_list_view.append_column(column)
        renderer.connect('toggled', self.sticky_toggled)

        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn(_('Title'), renderer,
                                    text=self.COLUMN_TITLE,
                                    weight=self.COLUMN_BOLD)
        column.set_resizable(gtk.TRUE)
        column.set_reorderable(gtk.TRUE)
        item_list_view.append_column(column)

        return

    def sticky_toggled(self, cell, path):
        model = self._widget.get_model()
        iter = model.get_iter((int(path),))
        item = model.get_value(iter, self.COLUMN_ITEM)
        item.sticky = not item.sticky
        model.set(iter, self.COLUMN_STICKY_FLAG, item.sticky)

    def render_feed(self, feed):
        model = self._widget.get_model()
        selection = self._widget.get_selection()
        selection.unselect_all()
        model.clear()
        count = 0
        for item in feed.items:
            title = ""
            if item.title is not None and item.title != "":
                title = item.title.strip()
                title = straw.utils.convert_entities(straw.utils.read_text(title, len(title)))
            elif item.description is not None and item.description != "":
                # read only 50 characters
                desc = straw.utils.read_text(item.description, 50)
                title = straw.utils.convert_entities(desc)
                title += " ..."
            else:
                # empty title, empty description, let's not show it
                continue

            count += 1

            if item.seen:
                bold = pango.WEIGHT_NORMAL
            else:
                bold = pango.WEIGHT_BOLD

            iter = model.append()
            model.set(iter, self.COLUMN_TITLE, title, self.COLUMN_ITEM, item,
                      self.COLUMN_BOLD, bold, self.COLUMN_STICKY_FLAG, item.sticky)

        return count

    def display_feed(self, feed, select_first = 1):
        if self._feed is not None:
            self._feed.signal_disconnect(straw.RefreshFeedDisplaySignal,
                                         self.feed_order_changed)
        self._feed = feed
        count = self.render_feed(feed)
        if count and select_first:
            self._widget.get_selection().select_path((0,))
        feed.signal_connect(straw.RefreshFeedDisplaySignal,
                            self.feed_order_changed)
        return

    def feed_order_changed(self, event):
        if event.sender is self._feed:
            selection = self._widget.get_selection()
            liststore, iter = selection.get_selected()
            path = liststore.get_path(iter)
            item = self._widget.get_model()[path[0]][self.COLUMN_ITEM]
            self.render_feed(event.sender)
            selection.select_path(event.sender.get_item_index(item))

    def display_empty_feed(self, feed):
        self._widget.get_model().clear()

    def item_selection_changed(self, selection):
        s = selection.get_selected()
        if s:
            model, iter = s
            if iter is None:
                return
            path = model.get_path(iter)
            index = path[0]
            item = model[index][self.COLUMN_ITEM]
            if not straw.main.is_visible_item(item):
                self.display_item(item)

    def mark_item(self, index):
        model = self._widget.get_model()
        path = (index,)
        try:
            iter = model.get_iter(path)
        except ValueError:
            return
        model.set_value(iter, self.COLUMN_BOLD, pango.WEIGHT_NORMAL)
        return

    def select_item(self, item):
        index = item.feed.get_item_index(item)
        path = (index,)
        self._widget.get_selection().select_path(path)
        self._widget.set_cursor(path, None, 0)
        self._widget.scroll_to_cell(path, None, 0, 0.5, 0)

    def display_item(self, item):
        self.mark_item(item.feed.get_item_index(item))
        straw.main.display_item(item)

class OfflineToggle:
    def __init__(self, widget):
        self._widget = widget
        widget_tree = glade.get_widget_tree(widget)
        for key in dir(self.__class__):
            if key[:3] == 'on_':
                widget_tree.signal_connect(key, getattr(self, key))
        self._widget.remove(self._widget.get_child())
        self._widget.add(gtk.Image())
        self._tooltips = gtk.Tooltips()
        straw.Config.get_instance().signal_connect(
            straw.OfflineModeChangedSignal, self.mode_changed)

    def show(self):
        if straw.Config.get_instance().offline:
            imagename = "offline"
            tooltip = _("Straw is currently offline. Click to work online.")
            self._widget.set_active(1)
        else:
            imagename = "online"
            tooltip = _("Straw is currently online. Click to work offline.")
            self._widget.set_active(0)
        straw.main.show_image(self._widget.get_child(), imagename)
        self._tooltips.set_tip(self._widget, tooltip, tooltip)
        self._tooltips.enable()
        self._widget.show_all()

    def on_offline_toggle_toggled(self, *args):
        active = self._widget.get_active()
        config = straw.Config.get_instance()
        if active != config.offline:
            config.offline = not config.offline

    def mode_changed(self, signal):
        self.show()

class Toolbar:
    def __init__(self,widget):
        # set toolbar tooltips
        widget_tree = glade.get_widget_tree(widget)
        self._widget = widget
        self._forward = widget_tree.get_widget('toolbar_forward_button')
        self._poll = widget_tree.get_widget('toolbar_poll_button')
        self._new = widget_tree.get_widget('toolbar_new_feed_button')
        self._find = widget_tree.get_widget('toolbar_find_button')
        self._tooltips = gtk.Tooltips()

    def show_tips(self):
        poll = _("Get news items")
        next = _("Go to next unread feed or item")
        new =  _("Subscribe to a new feed")
        find = _("Find a word or a phrase on feeds")

        self._tooltips.set_tip(self._forward, next, next)
        self._tooltips.set_tip(self._poll, poll, poll)
        self._tooltips.set_tip(self._new, new, new)
        self._tooltips.set_tip(self._find, find, find)
        self._tooltips.enable()

class FeedInfoDisplay:
    def __init__(self, widget):
        widget_tree = glade.get_widget_tree(widget)
        self._widget = widget
        self._title_hbox = widget_tree.get_widget('feed_info_title_hbox')
        self._link_display = gnome.ui.HRef(url='', text='')
        self._description_display = widget_tree.get_widget('feed_info_description')
        self._description_display.set_redraw_on_allocate(True)
        self._copyright_display = widget_tree.get_widget('feed_info_copyright')

        self._tooltips = gtk.Tooltips()
        self._title_hbox.pack_start(self._link_display,expand=gtk.FALSE,fill=gtk.FALSE, padding=6)

    def show(self):
        self._widget.show()
        self.set_visibility(True)

    def hide(self):
        self._widget.hide()
        self.set_visibility(False)

    def set_visibility(self, mode):
        straw.Config.get_instance().feed_info_visibility = mode

    def display_feed(self, feed):
        title = straw.utils.convert_entities(feed.channel_title)
        link = feed.channel_link
        if len(title) == 0:
            title = feed.title
        if len(title) == 0:
            title = feed.channel_url
        title = title.strip()
        if len(title) > 0:
            self._link_display.set_text(saxutils.escape(title))
            self._link_display.set_url(link)
            self._link_display.show()
        else:
            self._link_display.hide()

        description = feed.channel_description.strip()
        size = len(description)
        if size and description != title:
            if straw.utils.is_html(description):
                description = straw.utils.read_text(description, size)
            self._description_display.set_text(straw.utils.convert_entities(description))
            self._description_display.show()
        else:
            self._description_display.hide()

        copyright = feed.channel_copyright
        self._copyright_display.set_text(copyright)
        if len(copyright):
            self._copyright_display.show()
        else:
            self._copyright_display.hide()

    def set_size(self, size):
        w = max(size - 10, 0)
        window_height = straw.Config.get_instance().main_window_size[1]
        max_height = window_height / 4
        for widget in (self._description_display,
                       self._copyright_display):
            widget.set_size_request(w, -1)
            if widget.size_request()[1] > max_height:
                widget.set_size_request(w, max_height)

class FindResultList:
    COLUMN_TITLE = 0
    COLUMN_FEED = 1
    COLUMN_ITEM = 2

    def __init__(self, widget):
        widget_tree = glade.get_widget_tree(widget)
        self._treeview = widget_tree.get_widget('find_results_treeview')
        self._countlabel = widget_tree.get_widget('find_results_count_display')
        model = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_STRING,
                              gobject.TYPE_PYOBJECT)
        self._treeview.set_model(model)
        self._treeview.set_rules_hint(gtk.TRUE)
        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn('Title', renderer, text=self.COLUMN_TITLE)
        self._treeview.append_column(column)
        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn('Feed', renderer, text=self.COLUMN_FEED)
        self._treeview.append_column(column)
        self._treeview.get_selection().connect("changed", self.item_selection_changed)
        self._find_interrupted = 0

    def render_results(self, items):
        model = self._treeview.get_model()
        model.clear()
        self._countlabel.set_text(_('%d items found') % len(items))
        i = 0
        for item in items:
            if self._find_interrupted:
                self._find_interrupted = 0
                break
            i += 1
            iter = model.append()
            title = ""
            if item.title is not None:
                title = item.title[:50]
            if title == "":
                title = straw.utils.read_text(item.description, 50) + "..."
            model.set(iter, self.COLUMN_TITLE, title,
                      self.COLUMN_FEED, item.feed.title,
                      self.COLUMN_ITEM, item)
            if i % 10 == 0:
                # gtk bug avoidance?
                # without threads_*, after returning from the timeout
                # callback everything will lock up
                gtk.threads_enter()
                while gtk.events_pending():
                    gtk.mainiteration(gtk.FALSE)
                gtk.threads_leave()

    def item_selection_changed(self, selection):
        s = selection.get_selected()
        if s:
            model, iter = s
            if iter is None:
                return
            path = model.get_path(iter)
            index = path[0]
            item = model[index][self.COLUMN_ITEM]
            straw.main.display_item_unselected(item)

    def interrupt_find(self, signal):
        self._find_interrupted = 42

    def register_find_dialog(self, fd):
        fd.signal_connect(straw.FindInterruptSignal, self.interrupt_find)

class FeedListView:
    COLUMN_NAME = 0
    COLUMN_UNREAD = 1
    COLUMN_STATUS_DISPLAY = 2
    COLUMN_BOLD = 3
    COLUMN_ID = 4
    COLUMN_STATUS_FLAG = 5
    COLUMN_STATUS_PIXBUF = 6

    def __init__(self, widget):
        widget_tree = glade.get_widget_tree(widget)
        self._widget = widget
        model = gtk.TreeStore(
            gobject.TYPE_STRING, gobject.TYPE_STRING, gobject.TYPE_OBJECT,
            gobject.TYPE_INT, gobject.TYPE_INT, gobject.TYPE_BOOLEAN, gobject.TYPE_STRING)

        widget.set_model(model)
        widget.set_rules_hint(True)
        widget.set_search_column(0)
        self.create_feed_selection_columns(widget)

        popup_menu_items = (
            (_("/_Poll"), None, self.on_menu_poll_selected_activate, 0, "<Item>"),
            (_("/_Mark as Read"), None, self.on_menu_mark_all_as_read_activate, 0, "<Item>"),
            (_("/sep"), None, None, 0, "<Separator>"),
            (_("/_Delete"), None, self.delete_selected_feed, 0, "<Item>"),
            ("/sep", None, None, 0, "<Separator>"),
            (_("/P_roperties"), None, self.display_properties_feed, 0, "<Item>"))

        item_factory = gtk.ItemFactory(gtk.Menu, '<feed_list_popup>')
        item_factory.create_items(popup_menu_items)
        self._popup = item_factory.get_widget('<feed_list_popup>')
        self._widget.connect("popup-menu", self.on_popup_menu)
        self._widget.connect("button_press_event", self.on_button_press_event)
        self._widget.get_selection().connect(
            "changed", self.feed_selection_changed)
        # have to store this or bad things happen
        self._item_factory = item_factory

    def create_feed_selection_columns(self, feed_list_view):
        column = gtk.TreeViewColumn()
        column.set_title("_Subscriptions")

        status_renderer = gtk.CellRendererPixbuf()
        column.pack_start(status_renderer, gtk.FALSE)
        column.add_attribute(status_renderer, "stock_id", self.COLUMN_STATUS_PIXBUF)
        column.add_attribute(status_renderer, "visible", self.COLUMN_STATUS_FLAG)

        title_renderer = gtk.CellRendererText()
        column.pack_start(title_renderer, gtk.TRUE)
        column.add_attribute(title_renderer, "text", self.COLUMN_NAME)
        column.add_attribute(title_renderer, "weight", self.COLUMN_BOLD)

        unread_renderer = gtk.CellRendererText()
        column.pack_end(unread_renderer, gtk.FALSE)
        column.add_attribute(unread_renderer, "text", self.COLUMN_UNREAD)
        column.add_attribute(unread_renderer, "weight", self.COLUMN_BOLD)

        feed_list_view.append_column(column)
        
    def on_popup_menu(self, *args):
        self._popup.popup( None, None, None, 0, 0)
        return 1

    def on_button_press_event(self, treeview, event):
        if event.button == 3:
            x = int(event.x)
            y = int(event.y)
            time = gtk.get_current_event_time()
            path, col, cellx, celly = treeview.get_path_at_pos(x, y)
            treeview.grab_focus()
            treeview.set_cursor( path, col, 0)
            self._popup.popup(None, None, None, event.button, time)
            return 1

    def on_menu_poll_selected_activate(self, *args):
        straw.PollManager.get_instance().poll(straw.main.get_visible_feed())

    def on_menu_mark_all_as_read_activate(self, *args):
        straw.main.mark_as_read()

    def display_properties_feed(self, *args):
       straw.main.show_feed_properties_dialog()

    def delete_selected_feed(self, *args):
        response = straw.hig_alert.confirmDelete(_("Delete subscription?"),
                    _("Deleting a subscription will remove it from your subscription list."))
        if (response == gtk.RESPONSE_YES):
           selection = self._widget.get_selection() 
           model, iter = selection.get_selected()
           if iter:
               path = model.get_path(iter)
               del straw.FeedList.get_instance()[path[0]]
        return 1

    def feed_selection_changed(self, selection):
        s = selection.get_selected()
        if s:
            model, iter = s
            if iter is None:
                return
            path = model.get_path(iter)
            feed = straw.FeedList.get_instance()[path[0]]
            if feed:
                straw.main.display_feed(feed)
        return

    def display_feed_unselected(self, feed, index = None):
        if index is None:
            index = straw.FeedList.get_instance().index(feed)
        self._widget.set_cursor(
            (index,), self._widget.get_column(0), 0)
        self._widget.get_selection().select_path((index,))

    def number_of_new(self, feed):
        nu = feed.number_of_unread
        if nu != 0:
            return ("(%s)" % nu, nu)
        else:
            return (" ", nu)

    def _select_feed_status_icon(self, feed):
        if feed.process_status is not straw.Feed.STATUS_IDLE:
            return (1, gtk.STOCK_EXECUTE)
        elif feed.error:
            return (1, gtk.STOCK_DIALOG_ERROR)
        return (0, None)

    def display_feeds(self, feeds, visible_feed):
        model = self._widget.get_model()
        selection = self._widget.get_selection()
        fc = 0
        mlines = len(model)
        # first iterate through either the currently visible feeds,
        # OR the feeds to be displayed, whichever is the smaller set
        while fc < mlines and fc < len(feeds):
            feed = feeds[fc]
            if model[fc][self.COLUMN_ID] != feed.id:
                try:
                    feed.signal_disconnect(
                        straw.ItemReadSignal, straw.main_window.item_read)
                    feed.signal_disconnect(
                        straw.ItemsAddedSignal, straw.main_window.feed_updated)
                    feed.signal_disconnect(
                        straw.AllItemsReadSignal,
                        straw.main_window.all_items_read)
                    feed.signal_disconnect(
                        straw.FeedPolledSignal, straw.main_window.feed_updated)
                    feed.signal_disconnect(
                        straw.FeedStatusChangedSignal,
                        straw.main_window.feed_updated)
                except KeyError:
                    log("Key error in display_feeds in MainWindow")
                feed.signal_connect(straw.ItemReadSignal,
                                    straw.main_window.item_read, fc)
                feed.signal_connect(straw.AllItemsReadSignal,
                                    straw.main_window.all_items_read, fc)
                feed.signal_connect(
                    straw.ItemsAddedSignal, straw.main_window.feed_updated, fc)
                feed.signal_connect(
                    straw.FeedPolledSignal, straw.main_window.feed_updated, fc)
                feed.signal_connect(
                    straw.FeedStatusChangedSignal,
                    straw.main_window.feed_updated, fc)
                new_string, new = self.number_of_new(feed)
                weight = (pango.WEIGHT_NORMAL, pango.WEIGHT_BOLD)[new > 0]
                model[fc][self.COLUMN_NAME] = feed.title
                model[fc][self.COLUMN_UNREAD] = new_string
                model[fc][self.COLUMN_BOLD] = weight
                model[fc][self.COLUMN_ID] = feed.id
                status, pixbuf = self._select_feed_status_icon(feed)
                model[fc][self.COLUMN_STATUS_FLAG] = status
                if pixbuf:
                    model[fc][self.COLUMN_STATUS_PIXBUF] = pixbuf
                if feed is visible_feed:
                    selection.select_path((fc,))
            elif model[fc][self.COLUMN_NAME] != feed.title:
                model[fc][self.COLUMN_NAME] = feed.title
            fc += 1
        # if there were fewer feeds given as argument than currently displayed,
        # remove the extra rows
        if fc < mlines:
            treeiter = model.get_iter((fc,))
            while treeiter:
                next_iter = model.iter_next(treeiter)
                model.remove(treeiter)
                treeiter = next_iter
        # otherwise, add the extra rows and display the rest of the feeds
        # on them
        else:
            while fc < len(feeds):
                feed = feeds[fc]
                iter = model.append(None)
                new_string, new = self.number_of_new(feed)
                feed.signal_connect(straw.ItemReadSignal,
                                    straw.main_window.item_read, fc)
                feed.signal_connect(straw.AllItemsReadSignal,
                                    straw.main_window.all_items_read, fc)
                feed.signal_connect(straw.ItemsAddedSignal,
                                    straw.main_window.feed_updated, fc)
                feed.signal_connect(straw.FeedPolledSignal,
                                    straw.main_window.feed_updated, fc)
                feed.signal_connect(straw.FeedStatusChangedSignal,
                                    straw.main_window.feed_updated, fc)
                weight = (pango.WEIGHT_NORMAL, pango.WEIGHT_BOLD)[new > 0]
                status, pixbuf = self._select_feed_status_icon(feed)
                model.set(iter, self.COLUMN_NAME, feed.title,
                          self.COLUMN_UNREAD, new_string,
                          self.COLUMN_BOLD, weight,
                          self.COLUMN_ID, feed.id,
                          self.COLUMN_STATUS_FLAG, status)
                if pixbuf:
                    model.set(iter, self.COLUMN_STATUS_PIXBUF, pixbuf)
                if feed is visible_feed:
                    selection.select_iter(iter)
                fc += 1
        self._widget.columns_autosize()

    def all_items_read(self, signal, feed_index):
        model = self._widget.get_model()
        new = self.number_of_new(signal.sender)
        weight = (pango.WEIGHT_NORMAL, pango.WEIGHT_BOLD)[new[1] > 0]
        model[feed_index][self.COLUMN_UNREAD] = new[0]
        model[feed_index][self.COLUMN_BOLD] = weight
        self._widget.queue_draw()

    def item_read(self, signal, feed_index):
        model = self._widget.get_model()
        new = self.number_of_new(signal.sender.feed)
        weight = (pango.WEIGHT_NORMAL, pango.WEIGHT_BOLD)[new[1] > 0]
        model[feed_index][self.COLUMN_UNREAD] = new[0]
        model[feed_index][self.COLUMN_BOLD] = weight
        self._widget.queue_draw()

    def feed_updated(self, signal, feed_index):
        model = self._widget.get_model()
        new = self.number_of_new(signal.sender)
        weight = (pango.WEIGHT_NORMAL, pango.WEIGHT_BOLD)[new[1] > 0]
        model[feed_index][self.COLUMN_UNREAD] = new[0]
        model[feed_index][self.COLUMN_BOLD] = weight
        status, pixbuf = self._select_feed_status_icon(signal.sender)
        model[feed_index][self.COLUMN_STATUS_FLAG] = status
        if pixbuf:
            model[feed_index][self.COLUMN_STATUS_PIXBUF] = pixbuf
        self._widget.queue_draw()

    def disable_feed_selection(self):
        self._widget.set_sensitive(0)

    def enable_feed_selection(self):
        self._widget.set_sensitive(1)

class MainWindow:
    def __init__(self, xml):
        self._item_view = ItemView(xml.get_widget('item_view_container'))
        self._exception_view = ExceptionView(
            xml.get_widget('feed_exception_frame'),
            xml.get_widget('feed_exception_label'))
        self._feed_info_view = FeedInfoDisplay(
            xml.get_widget('feed_info_container'))
        self._find_results_view = FindResultList(
            xml.get_widget('find_results_treeview'))
        self._offline_toggle = OfflineToggle(
            xml.get_widget('offline_toggle'))
        self._toolbar = Toolbar(
            xml.get_widget('toolbar_default'))
        self._feed_list_view = FeedListView(
            xml.get_widget('feed_selection_treeview'))
        self._offline_toggle.show()
        self._toolbar.show_tips()
        self.create_ui(xml)

        # check if feed info is visible
        self._menu_feed_info = glade.get_widget_tree(xml.get_widget('menubar_default')).get_widget('menu_feed_information')

        if straw.Config.get_instance().feed_info_visibility:
            self._menu_feed_info.set_active(gtk.TRUE)
            self._feed_info_view.show()
        else:
            self._menu_feed_info.set_active(gtk.FALSE)
            self._feed_info_view.hide()

        gtk.timeout_add(300, self._display_status_messages)

    def get_find_results_view(self):
        return self._find_results_view

    def on_straw_main_destroy_event(self, *args):
        return straw.main.quit()

    def on_straw_main_delete_event(self, *args):
        return straw.main.quit()

    def on_straw_main_configure_event(self, widget, event, *args):
        def check_size((width, height, widget)):
            if width == widget.allocation.width and height == widget.allocation.height:
                straw.Config.get_instance().main_window_size = (width, height)
        if event.width != widget.allocation.width or event.height != widget.allocation.height:
            gtk.timeout_add(1000, check_size, (
                (event.width, event.height, widget)))
        return gtk.FALSE

    def on_main_main_pane_size_allocate(self, widget, *args):
        config = straw.Config.get_instance()
        def check_position((position, widget)):
            if position == widget.get_position():
                self._feed_info_view.set_size(position)
                config.main_pane_position = position
        pos = widget.get_position()
        if pos != config.main_pane_position:
            gtk.timeout_add(1000, check_position, (pos, widget))

    def on_main_sub_pane_size_allocate(self, widget, *args):
        config = straw.Config.get_instance()
        def check_position((position, widget)):
            if position == widget.get_position():
                config.sub_pane_position = position
        pos = widget.get_position()
        if pos != config.sub_pane_position:
            gtk.timeout_add(1000, check_position, (pos, widget))

    def on_toolbar_poll_button_clicked(self, button, *args):
        straw.PollManager.get_instance().poll()
        return

    def on_toolbar_find_button_clicked(self, button, *args):
        self.show_find_mode()

    def on_toolbar_new_feed_button_clicked(self, button, *args):
        straw.main.show_subscribe_dialog()

    def on_toolbar_forward_button_clicked(self, button, *args):
        if not self._item_view.scroll_down():
            straw.main.display_next_unread_item()

    def on_menu_about_activate(self, menuitem, *args):
        description = _("Straw is a desktop news aggregator for GNOME")
        gnome.ui.About(
            straw.APPNAME, straw.VERSION, "Copyright (c) 2002-2003 Juri Pakaste", description,
            ["Juri Pakaste <juri@iki.fi>",
             "Jan Alonzo <jmalonzo@unpluggable.com>",
             "Scott Douglas-Watson <sdouglaswatson@yahoo.co.uk>",
             "feedparser.py and rssfinder.py by Mark Pilgrim",
             "Icons by Jakub 'jimmac' Steiner and Juri Pakaste"],
            [],
            "Juri Pakaste <juri@iki.fi>\n"
            "Martin Steldinger <tribble@hanfplantage.de>\n"
            "David Rousseau <boiteaflood@wanadoo.fr>\n"
            "Sergei Vavinov <svv@cmc.msu.ru>\n"
            u"Terje R\xf8sten <terjeros@phys.ntnu.no>\n"
            "Francisco J. Fernandez <franciscojavier.fernandez.serrador@hispalinux.es>\n"
            "Elros Cyriatan (Dutch)",
            gtk.gdk.pixbuf_new_from_file(straw.main._libdir +
                                         "/straw.png")).show()

    def on_menu_quit_activate(self, *args):
        straw.main.quit()

    def on_menu_preferences_activate(self, *args):
        straw.main.show_preferences_dialog()

    def on_menu_feed_properties_activate(self, *args):
        self._feed_list_view.display_properties_feed()

    def on_menu_poll_all_activate(self, *args):
        straw.PollManager.get_instance().poll()

    def on_menu_next_activate(self, *args):
        straw.main.display_next_item()
        return

    def on_menu_previous_activate(self, *args):
        straw.main.display_previous_item()

    def on_menu_scroll_next_activate(self, *args):
        if not self._item_view.scroll_down():
            straw.main.display_next_unread_item()

    def on_menu_next_feed_activate(self, *args):
        straw.main.display_next_feed()
        return

    def on_menu_previous_feed_activate(self, *args):
        straw.main.display_previous_feed()
        return

    def on_menu_next_feed_unread_activate(self, *args):
        straw.main.display_next_unread_feed()
        return

    def on_menu_feed_information_activate(self, *args):
        if self._menu_feed_info.get_active():
            self._feed_info_view.show()
        else:
            self._feed_info_view.hide()

    def on_menu_find_activate(self, *args):
        self.show_find_mode()

    def on_menu_import_subscriptions_activate(self, *args):
        straw.main.show_import_subscriptions_dialog()

    def on_menu_export_subscriptions_activate(self, *args):
        straw.main.show_export_subscriptions_dialog()

    def on_menu_poll_selected_activate(self, *args):
        straw.PollManager.get_instance().poll(straw.main.get_visible_feed())

    def on_menu_mark_all_as_read_activate(self, *args):
        straw.main.mark_as_read()

    def on_straw_main_focus_in_event(self, *args):
        straw.DashboardFrontend.get_instance().focus_in()

    def on_straw_main_focus_out_event(self, *args):
        straw.DashboardFrontend.get_instance().focus_out()

    def show_find_mode(self):
        self._mode_view_notebook.set_current_page(1)
        self._feed_selection_vbox.hide()
        straw.main.show_find_dialog()

    def hide_find_mode(self):
        self._mode_view_notebook.set_current_page(0)
        self._feed_selection_vbox.show()

    def show_find_results(self, items):
        self._find_results_view.render_results(items)

    def display_feed(self, feed, select_first = 1):
        if feed.error is None:
            self._exception_view.hide()
        else:
            self._exception_view.show_exception(feed.error)
        self._item_list_view.display_feed(feed, select_first)
        self._feed_info_view.display_feed(feed)

    def display_feed_unselected(self, feed, index = None):
        self._feed_list_view.display_feed_unselected(feed, index)

    def display_empty_feed(self, feed):
        if feed.error is None:
            self._exception_view.hide()
        else:
            self._exception_view.show_exception(feed.error)
        self._item_list_view.display_empty_feed(feed)
        self._item_view.display_empty_feed(feed)

    def display_item_unselected(self, item):
        self._item_list_view.select_item(item)

    def display_item(self, item):
        self._item_view.display_item(item)

    def _display_status_messages(self):
        mmgr = straw.MessageManager.get_instance()
        ql = mmgr.number_of_messages()
        if ql:
            cid = self._status_view.get_context_id("asdf")
            self._status_view.pop(cid)
            m = mmgr.read_message()
            if len(m):
                self._status_view.push(cid, m)
        gtk.timeout_add(300, self._display_status_messages)

    def update_total_unread_indicator(self):
        uritems = urfeeds = 0
        for ur in [f.number_of_unread for f in straw.FeedList.get_instance()]:
            if ur:
                uritems += ur
                urfeeds += 1
        map = {'uritems': uritems, 'urfeeds': urfeeds}
        self._window.set_title(('%s - ' % straw.APPNAME) +
                               _('%(uritems)d unread in %(urfeeds)d feeds') %
                               map )

    def all_items_read(self, signal, feed_index):
        self._feed_list_view.all_items_read(signal, feed_index)
        for index, item in signal.changed:
            self._item_list_view.mark_item(index)
        self.update_total_unread_indicator()

    def item_read(self, signal, feed_index):
        self._feed_list_view.item_read(signal, feed_index)
        self._item_list_view.mark_item(
            signal.sender.feed.get_item_index(signal.sender))
        self.update_total_unread_indicator()

    def feed_updated(self, signal, feed_index):
        self._feed_list_view.feed_updated(signal, feed_index)
        if signal.sender is straw.main.get_visible_feed():
            self.display_feed(signal.sender)
        self.update_total_unread_indicator()

    def create_ui(self, xml):
        config = straw.Config.get_instance()
        self._window = xml.get_widget('straw_main')
        self._window.set_default_size(*config.main_window_size)
        xml.get_widget('main_main_pane').set_position(
            config.main_pane_position)
        xml.get_widget('main_sub_pane').set_position(
            config.sub_pane_position)
        self._feed_info_view.set_size(config.main_pane_position)

        self._window.show()

        while gtk.events_pending():
            gtk.mainiteration(gtk.FALSE)

        self._mode_view_notebook = xml.get_widget('mode_view_notebook')
        self._feed_selection_vbox = xml.get_widget('feed_selection_vbox')
        self._status_view = xml.get_widget("main_statusbar")

        for key in dir(self.__class__):
            if key[:3] == 'on_':
                xml.signal_connect(key, getattr(self, key))

        self._item_list_view = ItemList(
            xml.get_widget('item_selection_treeview'))

        self._window.drag_dest_set(
            gtk.DEST_DEFAULT_ALL,
            [('_NETSCAPE_URL', 0, 0), ('text/uri-list ', 0, 1),
             ('x-url/http', 0, 2)],
            gtk.gdk.ACTION_COPY | gtk.gdk.ACTION_MOVE)

        return

    def display_feeds(self, feeds, visible_feed):
        self._feed_list_view.display_feeds(feeds, visible_feed)
        self.update_total_unread_indicator()

    def get_window(self):
        return self._window

    def on_straw_main_drag_data_received(self, w, context, x, y, data, info, time):
        if data and data.format == 8:
            straw.main.show_subscribe_dialog(data.data.split("\n")[0])
            context.finish(gtk.TRUE, gtk.FALSE, time)
        else:
            context.finish(gtk.FALSE, gtk.FALSE, time)

    def disable_feed_selection(self):
        self._feed_list_view.disable_feed_selection()

    def enable_feed_selection(self):
        self._feed_list_view.enable_feed_selection()

