cvs_id = "$Id: ItemStore.py,v 1.26 2003/11/08 14:46:07 juri Exp $"

import cPickle as pickle
import os
import time
from error import *
import gtk

try:
    from bsddb.db import *
except ImportError:
    from bsddb3.db import *

import straw

DATABASE_FILE_NAME = "itemstore.db"

class MyDB:
    def __init__(self, filename, dbhome, create=0, truncate=0, mode=0600,
                 recover=0, dbflags=0):
        myflags = DB_THREAD
        if create:
            myflags = myflags | DB_CREATE
        flagsforenv = (DB_INIT_MPOOL | DB_INIT_LOCK | DB_INIT_LOG |
                       DB_INIT_TXN | DB_RECOVER)
        try:
            dbflags |= DB_AUTO_COMMIT
        except NameError:
            pass
        self._env = DBEnv()
        # enable auto deadlock avoidance
        self._env.set_lk_detect(DB_LOCK_DEFAULT)
        self._env.set_lg_max(2**20)
        self._env.open(dbhome, myflags | flagsforenv)
        if truncate:
            myflags = myflags | DB_TRUNCATE
        self._db = DB(self._env)
        self._db.open(filename, DB_BTREE, dbflags | myflags, mode)

    def __del__(self):
        self.close()

    def close(self):
        if self._db is not None:
            self._db.close()
            self._db = None
        if self._env is not None:
            self._env.close()
            self._env = None

    def checkpoint(self):
        self._env.txn_checkpoint()
        deletees = self._env.log_archive(DB_ARCH_ABS)
        for d in deletees:
            os.remove(d)

    def begin_transaction(self):
        return self._env.txn_begin()

    def get_feed_item_ids(self, feed, txn=None):
        key = "fids:%d" % feed.id
        dids = self._db.get(key, txn=txn)
        ids = []
        if dids:
            ids = pickle.loads(dids)
        return ids

    def save_feed_item_ids(self, feed, ids, txn=None):
        rowid = "fids:%d" % feed.id
        commit = 0
        if txn is None:
            txn = self.begin_transaction()
            commit = 1
        try:
            try:
                self._db.delete(rowid, txn=txn)
            except DBNotFoundError:
                pass
            self._db.put(rowid, pickle.dumps(ids), txn=txn)
            if commit:
                txn.commit()
        except:
            if commit:
                txn.abort()
            raise

    def get_item(self, feed, item_id, txn=None):
        item = unstringify_item(
            self._db.get("%d:%d" % (feed.id, item_id), txn=txn))
        if item is None:
            return None
        return item

    def add_item(self, item):
        txn = self.begin_transaction()
        try:
            feed_item_ids = self.get_feed_item_ids(item.feed, txn=txn)
            feed_item_ids.append(item.id)
            self.save_feed_item_ids(feed, feed_ids, txn)
            self._db.put("%d:%d" % (item.feed.id, item.id),
                         stringify_item(item), txn=txn)
            txn.commit()
        except:
            txn.abort()
            raise

    def add_items(self, feed, items):
        txn = self.begin_transaction()
        try:
            feed_item_ids = self.get_feed_item_ids(feed, txn=txn)
            for item in items:
                feed_item_ids.append(item.id)
                self._db.put("%d:%d" % (item.feed.id, item.id),
                             stringify_item(item), txn=txn)
            self.save_feed_item_ids(feed, feed_item_ids, txn)
            txn.commit()
        except:
            txn.abort()
            raise

    def del_item(self, feed, item):
        txn = self.begin_transaction()
        try:
            feed_item_ids = self.get_feed_item_ids(feed, txn=txn)
            if item.id in feed_item_ids:
                feed_item_ids.remove(item.id)
                self.save_feed_item_ids(feed, feed_item_ids, txn)
                self._db.delete("%d:%d" % (feed.id, item.id), txn=txn)
            txn.commit()
        except:
            txn.abort()
            raise

    def modify_items(self, items):
        txn = self.begin_transaction()
        try:
            for item in items:
                self._db.put("%d:%d" % (item.feed.id, item.id),
                             stringify_item(item), txn=txn)
            txn.commit()
        except:
            txn.abort()
            raise

    def get_feed_items(self, feed):
        txn = self.begin_transaction()
        items = []
        try:
            ids = self.get_feed_item_ids(feed, txn=txn)
            for id in ids:
                item = self.get_item(feed, id, txn=txn)
                if item is not None:
                    items.append(item)
            txn.commit()
        except:
            txn.abort()
            raise
        return items

    def get_image_urls(self, txn=None):
        dkeys = self._db.get("images", txn=txn)
        keys = []
        if dkeys is not None:
            keys = pickle.loads(dkeys)
        return keys

    def save_image_urls(self, urls, txn=None):
        self._db.put("images", pickle.dumps(urls), txn=txn)

    def update_image(self, url, image):
        key = "image:%s" % str(url)
        txn = self.begin_transaction()
        try:
            image_urls = self.get_image_urls(txn)
            if image is not None:
                self._db.put(key.encode('utf-8'), image, txn=txn)
                if url not in image_urls:
                    image_urls.append(url)
                    self.save_image_urls(image_urls, txn)
            else:
                if url in image_urls:
                    self._db.delete(key, txn=txn)
                    image_urls.remove(url)
                    self.save_image_urls(image_urls, txn=txn)
            txn.commit()
        except:
            txn.abort()
            raise

    def get_image_data(self, url, txn=None):
        return self._db.get(
            "image:%s" % url.encode('utf-8'), default = None, txn=txn)

    def _image_print(self, key, data):
        if key[:6] == "image:":
            print key

    def _data_print(self, key, data):
        data = pickle.loads(data)
        print `{key: data}`

    def _db_print(self, helper):
        """Print the database to stdout for debugging"""
        print "******** Printing raw database for debugging ********"
        cur = self._db.cursor()
        try:
            key, data = cur.first()
            while 1 :
                helper(key, data)
                next = cur.next()
                if next:
                    key, data = next
                else:
                    cur.close()
                    return
        except DBNotFoundError:
            cur.close()

class ModifyItemAction:
    def __init__(self, item):
        self._item = item

    def doit(self, db):
        db.modify_items([self._item])

class ModifyItemsAction:
    def __init__(self, items):
        self._items = items

    def doit(self, db):
        db.modify_items(self._items)

class ItemsAddedAction:
    def __init__(self, feed, items):
        self._feed = feed
        self._items = items

    def doit(self, db):
        db.add_items(self._feed, self._items)

class DeleteItemAction:
    def __init__(self, feed, item):
        self._feed = feed
        self._item = item

    def doit(self, db):
        db.del_item(self._feed, self._item)

class ImageUpdateAction:
    def __init__(self, url, image):
        self._url = url
        self._image = image

    def doit(self, db):
        db.update_image(self._url, self._image)

class ItemStore:
    def __init__(self):
        config = straw.Config.get_instance()
        feedlist = straw.FeedList.get_instance()
        self._db = MyDB(DATABASE_FILE_NAME,
                        config._straw_dir, create = 1)
        self.connect_signals(feedlist)
        feedlist.signal_connect(
            straw.FeedsChangedSignal, self.feeds_changed)
        straw.ImageCache.cache.signal_connect(straw.ImageUpdatedSignal,
                                              self.image_updated)
        self._stop = 0
        self._action_queue = []

    def initialize_db(self):
        self._db.CreateTable("items", ["feed_id", "item_id", "item_data"])

    def feeds_changed(self, signal):
        self.connect_signals(signal.sender)

    def connect_signals(self, feeds):
        for f in feeds:
            f.signal_connect(straw.NewItemsSignal, self.items_added)
            f.signal_connect(straw.ItemReadSignal, self.item_modified)
            f.signal_connect(straw.ItemStickySignal, self.item_modified)
            f.signal_connect(straw.AllItemsReadSignal, self.all_items_read)
            f.signal_connect(straw.ItemDeletedSignal, self.item_deleted)

    def modify_item(self, item):
        self._action_queue.append(ModifyItemAction(item))
        return

    def image_updated(self, signal):
        self._action_queue.append(
            ImageUpdateAction(signal.url, signal.data))

    def read_image(self, url):
        return self._db.get_image_data(url)

    def item_deleted(self, signal):
        self._action_queue.append(DeleteItemAction(signal.sender, signal.item))

    def item_modified(self, signal):
        self.modify_item(signal.sender)

    def all_items_read(self, signal):
        self._action_queue.append(ModifyItemsAction(
            [item for index, item in signal.changed]))

    def items_added(self, signal):
        self._action_queue.append(
            ItemsAddedAction(signal.sender, signal.items))

    def read_feed_items(self, feed):
        return self._db.get_feed_items(feed)

    def start(self):
        gtk.timeout_add(5000, self.run)

    def stop(self):
        self._stop = 1

    def run(self):
        self._db.checkpoint()
        freq = 5
        timer = freq
        cpfreq = 60
        cptimer = cpfreq
        prevtime = time.time()
        if not self._stop:
            tmptime = time.time()
            timer += tmptime - prevtime
            cptimer += tmptime - prevtime
            prevtime = tmptime
            if timer > freq:
                try:
                    while len(self._action_queue):
                        action = self._action_queue.pop(0)
                        if action is None:
                            break
                        action.doit(self._db)
                except IndexError, e:
                    pass
                timer = 0
            if cptimer > cpfreq:
                self._db.checkpoint()
                cptimer = 0
            gtk.timeout_add(freq * 1000, self.run)
        else:
            self._db.checkpoint()
            self._db.close()

def stringify_item(item):
    itemdict = {
        'title': item.title,
        'link': item.link,
        'description': item.description,
        'guid': item.guid,
        'pub_date': item.pub_date,
        'source': item.source,
        'images': item.image_keys(),
        'seen': item.seen,
        'id': item.id,
        'fm_license': item.fm_license,
        'fm_changes': item.fm_changes,
        'creator': item.creator,
        'license_urls': item.license_urls,
        'publication_name': item.publication_name,
        'publication_volume': item.publication_volume,
        'publication_number': item.publication_number,
        'publication_section': item.publication_section,
        'publication_starting_page': item.publication_starting_page,
        'sticky': item._sticky}
    return pickle.dumps(itemdict)

def unstringify_item(itemstring):
    if itemstring is None:
        return None
    try:
        dict = pickle.loads(itemstring)
    except ValueError, e:
        log("ItemStore.unstringify_item: pickle.loads raised ValueError, argument was %s" % repr(itemstring))
        return None
    item = straw.SummaryItem()
    item.title = dict['title']
    item.link = dict['link']
    item.description = dict['description']
    item.guid = dict['guid']
    item.pub_date = dict['pub_date']
    item.source = dict['source']
    for i in dict['images']:
        item.restore_image(i)
    item.seen = dict['seen']
    item.id = dict['id']
    item.fm_license = dict.get('fm_license', None)
    item.fm_changes = dict.get('fm_changes', None)
    item.creator = dict.get('creator', None)
    item.license_urls = dict.get('license_urls', None)
    item._sticky = dict.get('sticky', 0)
    item.publication_name = dict.get('publication_name', None)
    item.publication_volume = dict.get('publication_volume', None)
    item.publication_number = dict.get('publication_number', None)
    item.publication_section = dict.get('publication_section', None)
    item.publication_starting_page = dict.get('publication_starting_page',
                                              None)
    return item

if __name__ == '__main__':
    db = MyDB("itemstore.db", "%s/.straw" % os.getenv('HOME'), create = 1)
    db._db_print(db._image_print)
