# Copyright (c) 2002 Juri Pakaste
# You may use and distribute this software under the terms of the
# GNU General Public License, version 2 or later
#

cvs_id = "$Id: FeedList.py,v 1.10 2003/11/09 09:57:46 jmalonzo Exp $"

import straw

class FeedList(list, straw.SignalEmitter):
    def __init__(self, init_seq = []):
        straw.SignalEmitter.__init__(self)
        self.initialize_slots(straw.FeedsChangedSignal)

        for item in init_seq:
            try:
                self.append(item)
            except ValueError:
                raise ValueError, "%s is not of type Feed" % item

    def load_data(self):
        feeds = straw.Config.get_instance().feeds
        if feeds is not None:
            for df in feeds:
                f = straw.create_empty_feed()
                f.undump(df)
                self.append(f)

    def __setitem__(self, key, value):
        if type(value) is not straw.Feed:
            raise ValueError, "%s is not of type Feed" % value
        list.__setitem__(self, key, value)
        value.signal_connect(straw.FeedsChangedSignal, self.feeds_changed)
        self.save_feeds_and_notify()

    def append(self, value):
        list.append(self, value)
        value.signal_connect(straw.FeedsChangedSignal, self.feeds_changed)
        self.save_feeds_and_notify()

    def switch(self, key1, key2):
        tmp = self[key1]
        list.__setitem__(self, key1, self[key2])
        list.__setitem__(self, key2, tmp)
        self.save_feeds_and_notify()

    def reorder(self, move, delta):
        k = self[:]
        move = list(move)
        move.sort()
        if delta > 0:
            move.reverse()
        if move[0] == 0 and delta < 0 or move[-1] == (len(self) - 1) and delta > 0:
            return
        for m in move:
            k[m + delta], k[m] = k[m], k[m + delta]
        for i in range(len(k)):
            list.__setitem__(self, i, k[i])
        self.emit_signal(straw.FeedsChangedSignal(self))

    def __delitem__(self, value):
        list.__delitem__(self, value)
        self.save_feeds_and_notify()

    def save_feeds_and_notify(self):
        straw.Config.get_instance().feeds = [f.dump() for f in self] # XXX
        self.emit_signal(straw.FeedsChangedSignal(self))

    def feeds_changed(self, signal):
        self.save_feeds_and_notify()

    def sort(self, indices = None):
        if not indices or len(indices) == 1:
            list.sort(self, lambda a, b: cmp(a.title.lower(), b.title.lower()))
        else:
            items = []
            for i in indices:
                items.append(self[i])
            items.sort(lambda a, b: cmp(a.title.lower(), b.title.lower()))
            for i in range(len(items)):
                list.__setitem__(self, indices[i], items[i])
        self.save_feeds_and_notify()
        
    def __hash__(self):
        h = 0
        for item in self:
            h ^= hash(item)
        return h

    def get_feed_with_id(self, id):
        for f in self:
            if f.id == id:
                return f
        return None

feedlist_instance = None

def get_instance():
    global feedlist_instance
    if feedlist_instance is None:
        feedlist_instance = FeedList()
    return feedlist_instance

