# Copyright (C) 2012, 2013, 2014 Julian Marchant <onpon4@riseup.net>
# 
# This file is part of the Pygame SGE.
# 
# The Pygame SGE is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# The Pygame SGE is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public License
# along with the Pygame SGE.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import division
from __future__ import absolute_import
from __future__ import print_function
from __future__ import unicode_literals

import pygame
import six

import sge
from sge import r
from sge.r import (_get_dot_sprite, _get_line_sprite, _get_rectangle_sprite,
                   _get_ellipse_sprite, _get_circle_sprite,
                   _get_polygon_sprite, _get_text_sprite,
                   o_update_collision_lists, r_reset, r_update_fade,
                   r_update_dissolve, r_update_pixelate, r_update_wipe_left,
                   r_update_wipe_right, r_update_wipe_up, r_update_wipe_down,
                   r_update_wipe_upleft, r_update_wipe_upright,
                   r_update_wipe_downleft, r_update_wipe_downright,
                   r_update_wipe_matrix, r_update_iris_in, r_update_iris_out,
                   s_get_image)


__all__ = ['Room']


class Room(object):

    """
    This class stores the settings and objects found in a room.  Rooms
    are used to create separate parts of the game, such as levels and
    menu screens.

    .. attribute:: width

       The width of the room in pixels.  If set to :const:`None`,
       :attr:`sge.game.width` is used.

    .. attribute:: height

       The height of the room in pixels.  If set to :const:`None`,
       :attr:`sge.game.height` is used.

    .. attribute:: views

       A list containing all :class:`sge.View` objects in the room.

    .. attribute:: background

       The :class:`sge.Background` object used.

    .. attribute:: background_x

       The horizontal position of the background in the room.

    .. attribute:: background_y

       The vertical position of the background in the room.

    .. attribute:: alarms

       A dictionary containing the alarms of the room.  Each value
       decreases by 1 each frame (adjusted for delta timing if it is
       enabled).  When a value is at or below 0,
       :meth:`sge.Room.event_alarm` is executed with ``alarm_id`` set to
       the respective key, and the item is deleted from this dictionary.

    .. attribute:: objects

       A list containing all :class:`sge.Object` objects in the
       room.  (Read-only)

    .. attribute:: rd

       Reserved dictionary for internal use by the SGE.  (Read-only)
    """

    def __init__(self, objects=(), width=None, height=None, views=None,
                 background=None, background_x=0, background_y=0):
        """
        Arguments:

        - ``views`` -- A list containing all :class:`sge.View` objects
          in the room.  If set to :const:`None`, a new view will be
          created with ``x=0``, ``y=0``, and all other arguments
          unspecified, which will become the first view of the room.
        - ``background`` -- The :class:`sge.Background` object used.  If
          set to :const:`None`, a new background will be created with no
          layers and the color set to black.

        All other arguments set the respective initial attributes of the
        room.  See the documentation for :class:`sge.Room` for more
        information.
        """
        self.rd = {}
        self.width = width if width is not None else sge.game.width
        self.height = height if height is not None else sge.game.height
        self.rd["swidth"] = self.width
        self.rd["sheight"] = self.height
        self.background_x = background_x
        self.background_y = background_y
        self.alarms = {}
        self.__new_objects = []
        self.rd["projections"] = []

        if views is not None:
            self.views = list(views)
        else:
            self.views = [sge.View(0, 0)]
        self.rd["sviews"] = []

        self.rd["view_sx"] = {}
        self.rd["view_sy"] = {}
        self.rd["view_sxport"] = {}
        self.rd["view_syport"] = {}
        self.rd["view_swidth"] = {}
        self.rd["view_sheight"] = {}

        if background is not None:
            self.background = background
        else:
            self.background = sge.Background((), sge.Color("black"))
        self.rd["sbackground"] = self.background
        self.rd["sbackground_x"] = self.background_x
        self.rd["sbackground_y"] = self.background_y

        self.rd["started"] = False
        self.__has_started = False

        self.objects = []

        self.add(sge.game.mouse)
        for obj in objects:
            self.add(obj)
        self.rd["sobjects"] = []

        self.rd["object_sx"] = {}
        self.rd["object_sy"] = {}
        self.rd["object_sz"] = {}
        self.rd["object_ssprite"] = {}
        self.rd["object_svisible"] = {}
        self.rd["object_schecks_collisions"] = {}
        self.rd["object_stangible"] = {}
        self.rd["object_sbbox_x"] = {}
        self.rd["object_sbbox_y"] = {}
        self.rd["object_sbbox_width"] = {}
        self.rd["object_sbbox_height"] = {}
        self.rd["object_scollision_ellipse"] = {}
        self.rd["object_scollision_precise"] = {}

        if self.views:
            size = max(int(self.views[0].width / 10),
                       int(self.views[0].height / 10))
        else:
            size = sge.COLLISION_AREA_SIZE_DEFAULT

        self.rd["collision_area_size"] = size
        self.rd["collision_areas"] = []
        for i in six.moves.range(0, self.width, size):
            column = [[] for j in six.moves.range(0, self.height, size)]
            self.rd["collision_areas"].append(column)

        # The "Void" is the area outside the room.  This area is
        # infinite, so everything in the Void is checked against
        # everything else in the Void (otherwise an infinite grid, which
        # is impossible, would be needed).
        self.rd["collision_area_void"] = []

    def add(self, obj):
        """
        Add an object to the room.

        Arguments:

        - ``obj`` -- The :class:`sge.Object` object to add.

        """
        obj.alive = True
        if obj not in self.objects:
            objects = self.objects[:]
            objects.append(obj)
            self.objects = objects

            if self is sge.game.current_room and self.rd["started"]:
                obj.event_create()
            else:
                self.__new_objects.append(obj)

            o_update_collision_lists(obj)

    def remove(self, obj):
        """
        Remove an object from the room.

        Arguments:

        - ``obj`` -- The :class:`sge.Object` object to remove.
        """
        objects = self.objects[:]
        while obj in objects:
            objects.remove(obj)
        self.objects = objects

        o_update_collision_lists(obj)
        if self is sge.game.current_room:
            obj.event_destroy()

    def start(self, transition=None, transition_time=1500):
        """
        Start the room, resetting to its original state if it has been
        started previously.

        Arguments:

        - ``transition`` -- The type of transition to use.  Should be
          one of the following:

          - :const:`None` (no transition)
          - ``"fade"`` (fade to black)
          - ``"dissolve"``
          - ``"pixelate"``
          - ``"wipe_left"`` (wipe right to left)
          - ``"wipe_right"`` (wipe left to right)
          - ``"wipe_up"`` (wipe bottom to top)
          - ``"wipe_down"`` (wipe top to bottom)
          - ``"wipe_upleft"`` (wipe bottom-right to top-left)
          - ``"wipe_upright"`` (wipe bottom-left to top-right)
          - ``"wipe_downleft"`` (wipe top-right to bottom-left)
          - ``"wipe_downright"`` (wipe top-left to bottom-right)
          - ``"wipe_matrix"``
          - ``"iris_in"``
          - ``"iris_out"``

          If an unsupported value is given, default to :const:`None`.

        - ``transition_time`` -- The time the transition should take in
          milliseconds.  Has no effect if ``transition`` is
          :const:`None`.
        """
        if self.__has_started:
            r_reset(self)

        self.resume(transition, transition_time)

    def resume(self, transition=None, transition_time=1500):
        """
        Start the room without resetting to its original state if it has
        been started previously.  See the documentation for
        :meth:`sge.Room.start` for more information.
        """
        transitions = {
            "fade": r_update_fade, "dissolve": r_update_dissolve,
            "pixelate": r_update_pixelate, "wipe_left": r_update_wipe_left,
            "wipe_right": r_update_wipe_right, "wipe_up": r_update_wipe_up,
            "wipe_down": r_update_wipe_down,
            "wipe_upleft": r_update_wipe_upleft,
            "wipe_upright": r_update_wipe_upright,
            "wipe_downleft": r_update_wipe_downleft,
            "wipe_downright": r_update_wipe_downright,
            "wipe_matrix": r_update_wipe_matrix,
            "iris_in": r_update_iris_in, "iris_out": r_update_iris_out}

        if transition in transitions and transition_time > 0:
            self.rd["t_update"] = transitions[transition]
            self.rd["t_sprite"] = sge.Sprite.from_screenshot()
            self.rd["t_duration"] = transition_time
            self.rd["t_time_passed"] = 0
            self.rd["t_complete_last"] = 0
            self.rd["t_matrix_remaining"] = None
        else:
            self.rd["t_update"] = None

        sge.game.unpause()
        sge.game.current_room = self

        if not self.__has_started:
            self.rd["sobjects"] = self.objects[:]
            for obj in self.objects:
                self.rd["object_sx"][id(obj)] = obj.x
                self.rd["object_sy"][id(obj)] = obj.y
                self.rd["object_sz"][id(obj)] = obj.z
                self.rd["object_ssprite"][id(obj)] = obj.sprite
                self.rd["object_svisible"][id(obj)] = obj.visible
                self.rd["object_schecks_collisions"][id(obj)] = obj.checks_collisions
                self.rd["object_stangible"][id(obj)] = obj.tangible
                self.rd["object_sbbox_x"][id(obj)] = obj.bbox_x
                self.rd["object_sbbox_y"][id(obj)] = obj.bbox_y
                self.rd["object_sbbox_width"][id(obj)] = obj.bbox_width
                self.rd["object_sbbox_height"][id(obj)] = obj.bbox_height
                self.rd["object_scollision_ellipse"][id(obj)] = obj.collision_ellipse
                self.rd["object_scollision_precise"][id(obj)] = obj.collision_precise

            self.rd["sviews"] = self.views[:]
            for view in self.views:
                self.rd["view_sx"][id(view)] = view.x
                self.rd["view_sy"][id(view)] = view.y
                self.rd["view_sxport"][id(view)] = view.xport
                self.rd["view_syport"][id(view)] = view.yport
                self.rd["view_swidth"][id(view)] = view.width
                self.rd["view_sheight"][id(view)] = view.height

        for obj in self.objects:
            obj.rd["collision_areas"] = []
            o_update_collision_lists(obj)

        started = self.rd["started"]
        self.rd["started"] = True
        self.__has_started = True
        if not started:
            self.event_room_start()
            for obj in self.objects:
                obj.event_create()
        else:
            while self.__new_objects:
                self.__new_objects[0].event_create()
                del self.__new_objects[0]
            self.event_room_resume()

    def project_dot(self, x, y, z, color):
        """
        Project a single-pixel dot onto the room.

        Arguments:

        - ``x`` -- The horizontal location relative to the room to
          project the dot.
        - ``y`` -- The vertical location relative to the room to project
          the dot.
        - ``z`` -- The Z-axis position of the projection in the room.

        See the documentation for :meth:`sge.Sprite.draw_dot` for more
        information.
        """
        if not isinstance(color, sge.Color):
            e = "`{}` is not a sge.Color object.".format(repr(color))
            raise TypeError(e)

        sprite = _get_dot_sprite(color)
        self.project_sprite(sprite, 0, x, y, z)

    def project_line(self, x1, y1, x2, y2, z, color, thickness=1,
                     anti_alias=False):
        """
        Project a line segment onto the room.

        Arguments:

        - ``x1`` -- The horizontal location relative to the room of the
          first endpoint of the projected line segment.
        - ``y1`` -- The vertical location relative to the room of the
          first endpoint of the projected line segment.
        - ``x2`` -- The horizontal location relative to the room of the
          second endpoint of the projected line segment.
        - ``y2`` -- The vertical location relative to the room of the
          second endpoint of the projected line segment.
        - ``z`` -- The Z-axis position of the projection in the room.

        See the documentation for :meth:`sge.Sprite.draw_line` for more
        information.
        """
        if not isinstance(color, sge.Color):
            e = "`{}` is not a sge.Color object.".format(repr(color))
            raise TypeError(e)

        thickness = abs(thickness)
        x = min(x1, x2) - thickness // 2
        y = min(y1, y2) - thickness // 2
        x1 -= x
        y1 -= y
        x2 -= x
        y2 -= y

        sprite = _get_line_sprite(x1, y1, x2, y2, color, thickness, anti_alias)
        self.project_sprite(sprite, 0, x, y, z)

    def project_rectangle(self, x, y, z, width, height, fill=None,
                          outline=None, outline_thickness=1):
        """
        Project a rectangle onto the room.

        Arguments:

        - ``x`` -- The horizontal location relative to the room to
          project the rectangle.
        - ``y`` -- The vertical location relative to the room to project
          the rectangle.
        - ``z`` -- The Z-axis position of the projection in the room.

        See the documentation for :meth:`sge.Sprite.draw_rectangle` for
        more information.
        """
        if fill is not None and not isinstance(fill, sge.Color):
            e = "`{}` is not a sge.Color object.".format(repr(fill))
            raise TypeError(e)
        if outline is not None and not isinstance(outline, sge.Color):
            e = "`{}` is not a sge.Color object.".format(repr(outline))
            raise TypeError(e)

        outline_thickness = abs(outline_thickness)
        draw_x = outline_thickness // 2
        draw_y = outline_thickness // 2
        x -= draw_x
        y -= draw_y
        sprite = _get_rectangle_sprite(width, height, fill, outline,
                                       outline_thickness)
        self.project_sprite(sprite, 0, x, y, z)

    def project_ellipse(self, x, y, z, width, height, fill=None,
                        outline=None, outline_thickness=1, anti_alias=False):
        """
        Project an ellipse onto the room.

        Arguments:

        - ``x`` -- The horizontal location relative to the room to
          position the imaginary rectangle containing the ellipse.
        - ``y`` -- The vertical location relative to the room to
          position the imaginary rectangle containing the ellipse.
        - ``z`` -- The Z-axis position of the projection in the room.
        - ``width`` -- The width of the ellipse.
        - ``height`` -- The height of the ellipse.
        - ``outline_thickness`` -- The thickness of the outline of the
          ellipse.
        - ``anti_alias`` -- Whether or not anti-aliasing should be used.

        See the documentation for :meth:`sge.Sprite.draw_ellipse` for
        more information.
        """
        if fill is not None and not isinstance(fill, sge.Color):
            e = "`{}` is not a sge.Color object.".format(repr(fill))
            raise TypeError(e)
        if outline is not None and not isinstance(outline, sge.Color):
            e = "`{}` is not a sge.Color object.".format(repr(outline))
            raise TypeError(e)

        outline_thickness = abs(outline_thickness)
        draw_x = outline_thickness // 2
        draw_y = outline_thickness // 2
        x -= draw_x
        y -= draw_y
        sprite = _get_ellipse_sprite(width, height, fill, outline,
                                     outline_thickness, anti_alias)
        self.project_sprite(sprite, 0, x, y, z)

    def project_circle(self, x, y, z, radius, fill=None, outline=None,
                       outline_thickness=1, anti_alias=False):
        """
        Project a circle onto the room.

        Arguments:

        - ``x`` -- The horizontal location relative to the room to
          position the center of the circle.
        - ``y`` -- The vertical location relative to the room to
          position the center of the circle.
        - ``z`` -- The Z-axis position of the projection in the room.

        See the documentation for :meth:`sge.Sprite.draw_circle` for
        more information.
        """
        if fill is not None and not isinstance(fill, sge.Color):
            e = "`{}` is not a sge.Color object.".format(repr(fill))
            raise TypeError(e)
        if outline is not None and not isinstance(outline, sge.Color):
            e = "`{}` is not a sge.Color object.".format(repr(outline))
            raise TypeError(e)

        sprite = _get_circle_sprite(radius, fill, outline, outline_thickness,
                                    anti_alias)
        self.project_sprite(sprite, 0, x - radius, y - radius, z)

    def project_polygon(self, points, z, fill=None, outline=None,
                        outline_thickness=1, anti_alias=False):
        """
        Draw a polygon on the sprite.

        Arguments:

        - ``points`` -- A list of points relative to the room to
          position each of the polygon's angles.  Each point should be a
          tuple in the form ``(x, y)``, where x is the horizontal
          location and y is the vertical location.
        - ``z`` -- The Z-axis position of the projection in the room.

        See the documentation for :meth:`sge.Sprite.draw_polygon` for
        more information.
        """
        if fill is not None and not isinstance(fill, sge.Color):
            e = "`{}` is not a sge.Color object.".format(repr(fill))
            raise TypeError(e)
        if outline is not None and not isinstance(outline, sge.Color):
            e = "`{}` is not a sge.Color object.".format(repr(outline))
            raise TypeError(e)

        xlist = []
        ylist = []
        for point in points:
            xlist.append(point[0])
            ylist.append(point[1])
        x = min(xlist)
        y = min(ylist)

        sprite = _get_polygon_sprite(points, fill, outline, outline_thickness,
                                     anti_alias)
        self.project_sprite(sprite, 0, x, y, z)

    def project_sprite(self, sprite, image, x, y, z, blend_mode=None):
        """
        Project a sprite onto the room.

        Arguments:

        - ``x`` -- The horizontal location relative to the room to
          project ``sprite``.
        - ``y`` -- The vertical location relative to the room to project
          ``sprite``.
        - ``z`` -- The Z-axis position of the projection in the room.

        See the documentation for :meth:`sge.Sprite.draw_sprite` for
        more information.
        """
        img = s_get_image(sprite, image)
        x -= sprite.origin_x
        y -= sprite.origin_y
        self.rd["projections"].append((img, x, y, z, blend_mode))

    def project_text(self, font, text, x, y, z, width=None, height=None,
                    color=sge.Color("black"), halign=sge.ALIGN_LEFT,
                    valign=sge.ALIGN_TOP, anti_alias=True):
        """
        Project text onto the room.

        Arguments:

        - ``x`` -- The horizontal location relative to the room to
          project the text.
        - ``y`` -- The vertical location relative to the room to project
          the text.
        - ``z`` -- The Z-axis position of the projection in the room.

        See the documentation for :meth:`sge.Sprite.draw_text` for more
        information.
        """
        if not isinstance(color, sge.Color):
            e = "`{}` is not a sge.Color object.".format(repr(color))
            raise TypeError(e)

        sprite = _get_text_sprite(font, text, width, height, color, halign,
                                  valign, anti_alias)
        self.project_sprite(sprite, 0, x, y, z)

    def event_room_start(self):
        """
        Called when the room starts for the first time, and when it is
        restarted with :meth:`sge.Room.start`.  It is always called
        after any :meth:`sge.Game.event_game_start` and before any
        :class:`sge.Object.event_create` occurring at the same time.
        """
        pass

    def event_room_resume(self):
        """
        Called when the room is resumed with :meth:`sge.Room.resume`
        after it has already previously been started.  It is always
        called before any :meth:`sge.Object.event_create` occurring at
        the same time.
        """
        pass

    def event_room_end(self):
        """
        Called when another room is started or the game ends while this
        room is the current room.  It is always called before any
        :meth:`sge.Game.event_game_end` occurring at the same time.
        """
        pass

    def event_step(self, time_passed, delta_mult):
        """
        See the documentation for :meth:`sge.Game.event_step` for more
        information.
        """
        pass

    def event_alarm(self, alarm_id):
        """
        See the documentation for :meth:`sge.Room.alarms` for more
        information.
        """
        pass

    def event_key_press(self, key, char):
        """
        See the documentation for :class:`sge.input.KeyPress` for more
        information.
        """
        pass

    def event_key_release(self, key):
        """
        See the documentation for :class:`sge.input.KeyRelease` for more
        information.
        """
        pass

    def event_mouse_move(self, x, y):
        """
        See the documentation for :class:`sge.input.MouseMove` for more
        information.
        """
        pass

    def event_mouse_button_press(self, button):
        """Mouse button press event.

        See the documentation for :class:`sge.input.MouseButtonPress`
        for more information.

        """
        pass

    def event_mouse_button_release(self, button):
        """
        See the documentation for :class:`sge.input.MouseButtonRelease`
        for more information.
        """
        pass

    def event_joystick_axis_move(self, js_name, js_id, axis, value):
        """
        See the documentation for :class:`sge.input.JoystickAxisMove`
        for more information.
        """
        pass

    def event_joystick_hat_move(self, js_name, js_id, hat, x, y):
        """
        See the documentation for :class:`sge.input.JoystickHatMove` for
        more information.
        """
        pass

    def event_joystick_trackball_move(self, js_name, js_id, ball, x, y):
        """
        See the documentation for
        :class:`sge.input.JoystickTrackballMove` for more information.
        """
        pass

    def event_joystick_button_press(self, js_name, js_id, button):
        """
        See the documentation for :class:`sge.input.JoystickButtonPress`
        for more information.
        """
        pass

    def event_joystick_button_release(self, js_name, js_id, button):
        """
        See the documentation for
        :class:`sge.input.JoystickButtonRelease` for more information.
        """
        pass

    def event_gain_keyboard_focus(self):
        """
        See the documentation for :class:`sge.input.KeyboardFocusGain`
        for more information.
        """
        pass

    def event_lose_keyboard_focus(self):
        """
        See the documentation for :class:`sge.input.KeyboardFocusLose`
        for more information.
        """
        pass

    def event_gain_mouse_focus(self):
        """
        See the documentation for :class:`sge.input.MouseFocusGain` for
        more information.
        """
        pass

    def event_lose_mouse_focus(self):
        """
        See the documentation for :class:`sge.input.MouseFocusLose` for
        more information.
        """
        pass

    def event_close(self):
        """
        This is always called before any :meth:`sge.Game.event_close`
        occurring at the same time.

        See the documentation for :class:`sge.input.QuitRequest` for
        more information.
        """
        pass

    def event_paused_key_press(self, key, char):
        """
        See the documentation for :class:`sge.input.KeyPress` for more
        information.
        """
        pass

    def event_paused_key_release(self, key):
        """
        See the documentation for :class:`sge.input.KeyRelease` for more
        information.
        """
        pass

    def event_paused_mouse_move(self, x, y):
        """
        See the documentation for :class:`sge.input.MouseMove` for more
        information.
        """
        pass

    def event_paused_mouse_button_press(self, button):
        """
        See the documentation for :class:`sge.input.MouseButtonPress`
        for more information.
        """
        pass

    def event_paused_mouse_button_release(self, button):
        """
        See the documentation for :class:`sge.input.MouseButtonRelease`
        for more information.
        """
        pass

    def event_paused_joystick_axis_move(self, js_name, js_id, axis, value):
        """
        See the documentation for :class:`sge.input.JoystickAxisMove`
        for more information.
        """
        pass

    def event_paused_joystick_hat_move(self, js_name, js_id, hat, x, y):
        """
        See the documentation for :class:`sge.input.JoystickHatMove` for
        more information.
        """
        pass

    def event_paused_joystick_trackball_move(self, js_name, js_id, ball, x, y):
        """
        See the documentation for
        :class:`sge.input.JoystickTrackballMove` for more information.
        """
        pass

    def event_paused_joystick_button_press(self, js_name, js_id, button):
        """
        See the documentation for :class:`sge.input.JoystickButtonPress`
        for more information.
        """
        pass

    def event_paused_joystick_button_release(self, js_name, js_id, button):
        """
        See the documentation for
        :class:`sge.input.JoystickButtonRelease` for more information.
        """
        pass

    def event_paused_gain_keyboard_focus(self):
        """
        See the documentation for :class:`sge.input.KeyboardFocusGain`
        for more information.
        """
        pass

    def event_paused_lose_keyboard_focus(self):
        """
        See the documentation for :class:`sge.input.KeyboardFocusLose`
        for more information.
        """
        pass

    def event_paused_gain_mouse_focus(self):
        """
        See the documentation for :class:`sge.input.MouseFocusGain` for
        more information.
        """
        pass

    def event_paused_lose_mouse_focus(self):
        """
        See the documentation for :class:`sge.input.MouseFocusLose` for
        more information.
        """
        pass

    def event_paused_close(self):
        """
        See the documentation for :meth:`sge.Room.event_close` for more
        information.
        """
        pass
