# Copyright (C) 2012, 2013, 2014 Julian Marchant <onpon4@riseup.net>
# 
# This file is part of the Pygame SGE.
# 
# The Pygame SGE is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# The Pygame SGE is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public License
# along with the Pygame SGE.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import division
from __future__ import absolute_import
from __future__ import print_function
from __future__ import unicode_literals

import pygame

import sge


__all__ = ['Background']


class Background(object):

    """Background class.

    This class stores the layers that make up the background (which
    contain the information about what images to use and where) as well
    as the color to use where no image is shown.

    .. attribute:: color

       A :class:`sge.Color` object representing the color used in parts
       of the background where no layer is shown.

    .. attribute:: id

       The unique identifier for this background.  (Read-only)

    .. attribute:: layers

       A list containing all :class:`sge.BackgroundLayer` objects used
       in this background.  (Read-only)

    """

    @property
    def color(self):
        return self._color

    @color.setter
    def color(self, value):
        if isinstance(value, sge.Color):
            self._color = value
        else:
            e = "`{}` is not a sge.Color object.".format(repr(value))
            raise TypeError(e)

    def __init__(self, layers, color, ID=None):
        """Constructor method.

        Arguments:

        - ``ID`` -- The value to set :attr:`id` to.  If set to
          :const:`None`, the SGE chooses the value.

        All other arguments set the respective initial attributes of the
        background.  See the documentation for :class:`sge.Background`
        for more information.

        """
        self.color = color

        backgrounds = sge.game.backgrounds.copy()
        if ID is not None:
            self.id = ID
        else:
            ID = 0
            while ID in backgrounds:
                ID += 1
            self.id = ID

        backgrounds[self.id] = self
        sge.game.backgrounds = backgrounds

        unsorted_layers = []
        sorted_layers = []

        for layer in layers:
            if isinstance(layer, sge.BackgroundLayer):
                unsorted_layers.append(layer)
            else:
                if layer in sge.game.background_layers:
                    unsorted_layers.append(sge.game.background_layers[layer])

        for layer in unsorted_layers:
            i = 0
            while i < len(sorted_layers) and layer.z >= sorted_layers[i].z:
                i += 1

            sorted_layers.insert(i, layer)

        self.layers = sorted_layers

    def destroy(self):
        """Destroy the background."""
        backgrounds = sge.game.backgrounds.copy()
        if self.id in backgrounds:
            del backgrounds[self.id]
            sge.game.backgrounds = backgrounds
