/* Pattern.pm -- generates a plot of an antenna radiation pattern.
 *
 * Copyright (C) 2001-2003, John Kodis <john@kodis.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gd.h>
#include <gdfontl.h>
#include <gdfontmb.h>
#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

double rms = 1.0, erp = 1.0; // FIX THIS

static int
gen_ant_plot(char *plot_file, char *id1, char *id2, int patno, double *pat)
{
  int p, r = 160; // plot radius
  gdImage *im = gdImageCreate(3*r+1, 2*r+1);
  int white = gdImageColorAllocate(im, 255, 255, 255);
  int gray  = gdImageColorAllocate(im, 220, 220, 220);
  int black = gdImageColorAllocate(im,   0,   0,   0);
  int red   = gdImageColorAllocate(im, 255,   0,   0);
  int green = gdImageColorAllocate(im,   0, 255,   0);
  int blue  = gdImageColorAllocate(im,   0,   0, 255);

  im->transparent = white;
  gdImageInterlace(im, 1);
  gdImageRectangle(im, 0,0, 3*r,2*r, black);

  /* Loop through the radiation pattern elements to find the min and
     max pattern magnitudes. */
  double pat_min = 0, pat_max = 1;
  if (patno > 0)
    pat_min = pat_max = pat[1];

  for (p = 1; p < patno; p++)
    {
      double d = pat[p * 2 + 1];
      if (d < pat_min) pat_min = d;
      if (pat_max < d) pat_max = d;
    }
  printf("pat_min = %g, pat_max = %g\n", pat_min, pat_max);

  /* Outline the max radiation circle in red, and gray fill it. */
  gdImageArc(im, 2*r, r, 2*r, 2*r, 0, 360, red);
  gdImageFill(im, 2*r, r, gray);

  /* Draw cross-hairs at center of radiation pattern. */
  gdImageLine(im, 2*r - 10, r, 2*r + 10, r, black);
  gdImageLine(im, 2*r, r - 10, 2*r, r + 10, black);

  if (pat_min < pat_max)
    {
      /* Outline the min radiation circle in blue.  Don't white fill it. */
      double ratio = pat_min / pat_max;
      gdImageArc (im, 2*r, r, 2*r*ratio, 2*r*ratio, 0,360, blue);
      //gdImageFill(im, 2*r, r, white);

      /* For AM stations, show the 1 mV/m at 1 km circle in black. */
      if (/* pat_min < 1 && */ 1 < pat_max)
	gdImageArc(im, 2*r, r, 2*r / pat_max, 2*r / pat_max, 0,360, black);
    }

  /* Perform various fix-ups to scale or use defaults for radiation
     values.  Scale radiation by rms, but leave pat_min/max for plotting. */

  double rad_min = rms * pat_min;
  double rad_avg = rms;
  double rad_max = rms * pat_max;

  if (patno <= 0)
    {
      if (rms <= 0)
	{
	  /* Fix-ups for FM/TV omni -- use 1 for min/avg/max. */
	  rad_min = rad_avg = rad_max = 1;
	}
      else
	{
	  /* Fix-ups for AM antennas.  John Byrns explains: For AM
             omnidirectional antennas, the FCC database doesn't give
             the RMS value of the pattern for the actual transmitter
             power like with the directional arrays, but rather gives
             the signal strength at 1 km for a transmitter power of 1
             kW.  So, for omnidirectional antennas, you need to
             multiply the signal strength given in the database by the
             square root of the transmitter power in kW's, to get a
             value that is comparable to the RMS value for the
             directional arrays. */
	    rad_min = rad_avg = rad_max = rms * sqrt(erp);
	}
    }

  /* Print the max, avg, min, and rms pattern values in the upper left. */
  char buf[20];
  gdFontPtr mb_font = gdFontMediumBold;
  sprintf(buf, " Maximum:   %8.3f", rad_max);
  gdImageString(im, mb_font, 0, mb_font->h, buf, red);

  char *rad_fmt = rms ? " Radiation: %8.3f RMS" : " Radiation: --------";
  sprintf(buf, rad_fmt, rad_avg);
  gdImageString(im, mb_font, 0, 2 * mb_font->h, buf, green);

  sprintf(buf, " Minimum:   %8.3f", rad_min);
  gdImageString(im, mb_font, 0, 3 * mb_font->h, buf, blue);

  if (rms > 0)
    gdImageString(im, mb_font, 0, 4 * mb_font->h,
      " Units: mV/m at 1 km", black);

  /* Draw North-up arrow. */
  gdImageString(im, gdFontLarge, r/4, r, " N", black);
  gdImageLine(im, r/4, r - r/4, r/4, r + r/4, black);
  gdImageLine(im, r/4, r - r/4, r/4 + r/32, r - r/8, black);

  /* Print the station identification in the lower left. */
  gdFontPtr big_font = gdFontLarge;
  gdImageString(im, big_font, big_font->w, 2*r - big_font->h*5/2, id1, black);
  gdImageString(im, big_font, big_font->w, 2*r - big_font->h*3/2, id2, black);

  if (patno > 0)
    {
      gdPoint *poly = malloc(patno * sizeof(*poly));
      for (p = 0; p < patno; p++)
	{
	  double t = pat[p * 2 + 0];
	  double d = pat[p * 2 + 1] / pat_max;
	  poly[p].x = 2*r + r * d * sin(t);
	  poly[p].y = 1*r - r * d * cos(t);
	  gdImageArc(im, poly[p].x,poly[p].y, 9,9, 0,360, green);
	}
      gdImagePolygon(im, poly, patno, green);
    }

  /* Finally, write out the plot. */
  FILE *fd;
  if ((fd = fopen(plot_file, "w")) == NULL)
    {
      fprintf(stderr, "station-info-pattern: %s: %s\n",
	strerror(errno), plot_file);
      return EXIT_FAILURE;
    }
  gdImagePng(im, fd);
  fclose(fd);
  return EXIT_SUCCESS;
}

/* main -- for the station-info-pattern program.
   Usage: pattern fn id1 id2 angle1 rad1 ... anglen radn */
int
main(int argc, char **argv)
{
  if (argc < 4)
    {
      fprintf(stderr, "Too few args (%d); fn id1 id2 required\n", argc);
      return EXIT_FAILURE;
    }

  int i, n = (argc - 4) / 2;
  // printf("argc is %d; n is %d\n", argc, n);
  double *p = malloc(2 * n * sizeof(*p));
  for (i = 0; i < n*2; i += 2)
    {
      p[i + 0] = atof(argv[i + 4]);
      p[i + 1] = atof(argv[i + 5]);
      // printf("p[%d] = %g, %g\n", i/2, p[i + 0], p[i + 1]);
    }

  return gen_ant_plot(argv[1], argv[2], argv[3], n, p);
}
