#! /usr/bin/perl -w
#
# details.cgi -- generate a web page with detailed info on one station.
#
# Copyright (C) 2001-2003, John Kodis <john@kodis.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

use strict;
use CGI;
use Paths;
use Detail_vars;

my $pi = 3.14159265358979323846;

sub rad2deg {
    my ($radians) = @_;
    return 180. * ($radians / $pi);
}

sub arc2dist {
    my ($arc, $unit) = @_;
    if ($unit =~ /km/) { return 40000.000 * $pi * $arc }
    if ($unit =~ /mi/) { return 24854.848 * $pi * $arc }
    if ($unit =~ /nm/) { return 21598.272 * $pi * $arc }
    return 0;
}

sub rad2point {
    my ($rad) = @_;
    if ($rad < 0) {
	$rad = 2 * $pi + $rad;
    }
    my $point = ($rad + $pi / 16) / ($pi / 8);
    my @pstr = ( "North", "North North East", "North East", "East North East",
		 "East", "East South East", "South East", "South South East",
		 "South", "South South West", "South West", "West South West",
		 "West", "West North West", "North West", "North North West" );
  return $pstr[$point % 16];
}

sub referrer_extract_loc_unit {
    my ($referrer) = @_;
    $referrer =~ s/^.*[?]//;
    my %refs = new CGI($referrer)->Vars;
    $loc = $refs{'loc'};
    $unit = $refs{'unit'};

    # print "loc=$loc; unit=$unit; ref=$referrer\n";
}

sub extract_si_output {
    my $query = new CGI;
    my %param = $query->Vars;

    # while (my ($k,$v) = each %param) { print "$k = $v\n"; }

    my %data = ();
    if (! open(CHILD, "-|")) {
	open(STDERR, ">&STDOUT") || die "can't dup stderr onto stdout";
	exec("$prefix/bin/station-info-cli",
	     "--location", $loc, "--details", $param{'off'});
	exit 1;
    }
    my $first_line = <CHILD>;
    if ($first_line !~ /^success/) {
	print "<h1> Error </h1>\n",
        "<p> An error seems to have occurred. <pre>\n", $first_line;
	while (<CHILD>) { print $_; }
	exit 1;
    }
    while (<CHILD>) {
	chomp;
	my ($key, $val) = split(/\s/, $_, 2);
	$data{$key} = $val;
    }
    # while (my ($k,$v) = each %data) { print "$k = $v\n"; }

    $call    = $data{'fac.callsign'};
    $band    = $data{'fac.service'};
    $corf    = $data{'sloc.corf'} / 10.0;
    $khz     = $data{'khz'};
    $chan    = $data{'fac.channel'};
    $city    = $data{'fac.comm_city'};
    $state   = $data{'fac.comm_state'},
    $rms     = $data{'ant.pat.rms'};
    $pat_min = $data{'ant.pat.min'};
    $pat_max = $data{'ant.pat.max'};

    $lat   = 180.0 * $data{'sloc.lat'} / (2 ** 31);
    $lon   = 180.0 * $data{'sloc.lon'} / (2 ** 31);

    $band_corf = "";
    if ($band eq "AM") { $band_corf = sprintf "%d AM", $corf; }
    if ($band eq "FM") { $band_corf = sprintf "%.1f FM", $corf; }
    if ($band eq "TV") { $band_corf = sprintf "TV %d", $corf; }

    return %data;
}

sub row {
    my ($id, $val) = @_;
    print "<tr> <td> $id </td><td> $val </td></tr>\n";
}

sub emit_generic_table {
    my ($band, $unit, %data) = @_;

    print "<p> <table>\n";
    print "<tr> <th> Parameter </th><th> Value </th></tr>\n";
    printf "<tr><td> Located at </td><td>%+.3f, %+.3f</td></tr>\n", $lat,$lon;

    row("Heading", sprintf("%.1f %s %s at %d degrees",
        arc2dist($data{'arc'}, $unit), $unit,
        rad2point($data{'head'}), rad2deg($data{'head'})));
    row("From", sprintf("%s at %+.3f,%+.3f",
        $loc, rad2deg($data{'search.lat'}), rad2deg($data{'search.lon'})));

    row("Serving", "$city, $state");
    row("Facility", $data{'fac.fac_addr1'} . ", " . $data{'fac.fac_addr2'});
    row("",         $data{'fac.fac_country'} . ", " . $data{'fac.fac_zip'});

    my $freq_and_chan = "";
    if ($band eq "AM") {
	$freq_and_chan = sprintf("%d KHz", $khz);
    } else {
	$freq_and_chan = sprintf("%.1f MHz (Channel %d)", $khz/1000.0, $chan);
    }
    row("Frequency", $freq_and_chan);
    print "</td></tr> </table>\n";
}

sub emit_band_specific_table {
    my ($band, %data) = @_;

    print "<p> <table>\n<tr> <th> Parameter </th><th> Value </th></tr>\n";
    if ($band eq "AM") {
	row("Antenna Type", $data{'am.ant_type'});
	row("Antenna System ID", $data{'am.ant_sys_id'});
	row("Antenna Towers", $data{'am.towers'});
	row("Antenna Power", $data{'am.power'});
    }
    if ($band eq "FM") {
	row("Antenna ID", $data{'fm.ant_id'});
	row("Antenna Type", $data{'fm.ant_type'});
	row("Antenna Polarization", $data{'fm.ant_pol'});
	row("Antenna Rotation", $data{'fm.ant_rot'});
	row("Horizontal ERP", $data{'fm.horiz_erp'});
	row("Vertical ERP", $data{'fm.vert_erp'});
	row("Elevation AMSL", $data{'fm.eamsl'});
    }
    if ($band eq "TV") {
	row("Antenna ID", $data{'tv.ant_id'});
	row("Antenna Type", $data{'tv.ant_type'});
	row("Antenna Polarization", $data{'tv.ant_pol'});
	row("Antenna Rotation", $data{'tv.ant_rot'});
	row("ERP", $data{'tv.erp'});
	row("Elevation AMSL", $data{'tv.eamsl'});
    }
    print "</table>\n";
}

sub emit_tiger_map_img {
    my ($lat, $lon, $call, $band_corf) = @_;

    print "<p> <h2> Map of $call, $band_corf </h2>\n";
    printf "<img height=600 width=600 alt='TMS Map' ";
    printf "src='http://tiger.census.gov/cgi-bin/mapper/map.gif?";
    printf "lon=%.4f&lat=%.4f&", $lon, $lat;
    printf "wid=%.4f&iwd=%d&ht=%.4f&iht=%d&", 1.0, 600, 1.0, 600;
    printf "mlat=%.4f&mlon=%.4f&msym=%s&mlabel=%s'>\n",
        $lat, $lon, "cross", $call;
}

sub emit_details_page {

    print "Content-type: text/html\n\n";
    print "<html> <head>\n";
    print "<title> Station Information Details Page </title>\n";
    print "</head><body>\n";

    referrer_extract_loc_unit($ENV{'HTTP_REFERER'});
    my (%data) = extract_si_output();

    print "<h1> $call, $band_corf </h1>\n";

    emit_generic_table($band, $unit, %data);
    emit_band_specific_table($band, %data);

#    eval {
#	require Pattern;
#	Pattern::emit_pattern_img(%data);
#    };
#    if ($@) { print "<pre>\n$@</pre>\n"; }

    my $pat_prog = "$prefix/bin/station-info-pattern";
    my $plot_file = "$band-$data{'sloc.eng_off'}-$call.gd.png";
    my $id1 = "$call, $band_corf";
    my $id2 = "$city, $state";
    my @pattern = split(' ', $data{'ant.pat.az_mag'});
    my $spew = `$pat_prog $plot_dir/$plot_file "$id1" "$id2" @pattern 2>&1`;
    if ($?) {
	print "<pre>\n$pat_prog\n$plot_dir/$plot_file\n$id1\n$id2\n" .
	    "Spewage:\n$spew\n</pre>\n";
    } else {
	print "<p> <h2> Radiation Plot of $call, $band_corf </h2>\n";
	print "<img src='$plot_dir_url/$plot_file'>\n";
    }

    #emit_tiger_map_img($lat, $lon, $call, $band_corf);

    print "<p> That's all, folks! </body> </html>\n";
    return 1;
}

# print "Content-type: text/plain\n\n";
# printf "sloc.old, new, anc = %d, %d, %d\n",
#     $data{'sloc.old'}, $data{'sloc.new'}, $data{'sloc.anc'}, "\n";
# system "echo -n 'pwd is '; pwd; echo -n 'I am '; whoami";
# system "printenv";

exit ! emit_details_page();
