/* main -- for the station-info program.
 *
 * Copyright (C) 2001 John Kodis <kodis@jagunet.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <callbacks.h>
#include <guess-where.h>
#include <gnome.h>
#include <interface.h>
#include <main.h>
#include <support.h>
#include <utils.h>

Search search;
GtkCList *clist, *slist;
GtkWidget *station_info, *appbar;

static int geometry_flags;
static int default_w = 600, default_h = 450;
static int geometry_w = -1, geometry_h = -1;
static int geometry_x = -1, geometry_y = -1;

static void
popt_arg_extractor(
  poptContext state, enum poptCallbackReason reason,
  const struct poptOption *opt, const char *arg, void *data)
{
  if (opt->val == 'g')
    {
      geometry_flags = gnome_parse_geometry(arg,
	&geometry_x, &geometry_y, &geometry_w, &geometry_h);
      debug("geo: (%s): 0x%x, %d,%d; %d,%d\n", arg,
        geometry_flags, geometry_x, geometry_y, geometry_w, geometry_h);
    }
  else
    process_option(&search, opt->val, (char *)arg);
}

static struct
poptOption popt_options[] =
{
  { NULL,      '\0', POPT_ARG_CALLBACK, popt_arg_extractor },
  { "bands", 'b', POPT_ARG_STRING, NULL, 'b',
    N_("Bands: am,fm,tv"), N_("BANDS") },
  { "distance", 'd', POPT_ARG_STRING, NULL, 'd',
    N_("Distance: [min,] max[km,mi,nm]"), N_("DIST") },
  { "keep", 'k', POPT_ARG_STRING, NULL, 'k',
    N_("Max stations per channel: max"), N_("MAX") },
  { "location", 'l', POPT_ARG_STRING, NULL, 'l',
    N_("Location: lat,lon"), N_("LOC") },
  { "options", 'o', POPT_ARG_STRING, NULL, 's',
    N_("Options: new,old,anc"), N_("OPTS") },
  { "range", 'r', POPT_ARG_STRING, NULL, 'r',
    N_("Channel or Frequency range: [min,] max"), N_("CORF") },
  { "sort", 's', POPT_ARG_STRING, NULL, 's',
    N_("Sort keys: freq, dist, power"), N_("KEY") },
  { "geometry", 'g', POPT_ARG_STRING, NULL, 'g',
    N_("Geometry string: WxH+X+Y"), N_("GEO") },
  { "verbose", 'v', POPT_ARG_NONE, NULL, 'v',
    N_("Print some debugging information to stderr"), NULL },
  { "antennas", 'A', POPT_ARG_NONE, NULL, 'A',
    N_("Show only stations with interesting antennas"), NULL },
  { NULL,             '\0', 0, NULL, 0 }
};

#ifdef SHOW_SPLASH
static GtkWidget *splash;

static void
on_splash_destroy(GtkObject *object, void *user_data)
{
  show_splash = 0;
}

static void
progress(char *fn, int unused_recs, int entries, int unused_allocation)
{
  static int ingest, cl_row;
  char msg[100], rec_str[20], *stext[] = { fn, rec_str };

  if (fn == NULL)
    return;

  sprintf(msg, "%s:  %d records", fn, entries);
  gnome_appbar_set_status(GNOME_APPBAR(appbar), msg);
  gnome_appbar_set_progress(GNOME_APPBAR(appbar), (++ingest + 1) / 10.0);

  if (show_splash)
    {
      static GdkColor *red;
      sprintf(rec_str, "%d", entries);
      cl_row = gtk_clist_append(slist, stext);
      if (entries <= 0)
	{
	  if (red == NULL)
	    red = gdk_color_new("#ff0000", FALSE);
	  gtk_clist_set_foreground(slist, cl_row, red);
	}
      gtk_clist_columns_autosize(slist);
    }

  while (gtk_events_pending())
    gtk_main_iteration();
}
#endif

static int
station_info_idle(void *unused)
{
  while (gtk_events_pending())
    gtk_main_iteration();

  search_set_widget_state(&search, station_info);
  on_search_button_clicked(NULL, NULL);
  gnome_appbar_set_progress(GNOME_APPBAR(appbar), 1);

  return FALSE;
}

int
main (int argc, char *argv[])
{
#ifdef ENABLE_NLS
  bindtextdomain(PACKAGE, PACKAGE_LOCALE_DIR);
  textdomain(PACKAGE);
#endif

  fcc_init(&search, argc, argv);
  guess_where(&search.lat, &search.lon);
  search_set_location(&search, NULL);

  gnome_init_with_popt_table(_(PACKAGE), VERSION, argc, argv,
    popt_options, 0, NULL);

#ifdef SHOW_SPLASH
  if (show_splash)
    {
      splash = create_splash();
      slist = GTK_CLIST(lookup_widget(splash, "splash_clist"));
      gtk_clist_set_column_justification(slist, 1, GTK_JUSTIFY_RIGHT);
      gtk_signal_connect(GTK_OBJECT(splash),
	"destroy", GTK_SIGNAL_FUNC(on_splash_destroy), NULL);
      gtk_widget_show(splash);
    }
#endif

  station_info = create_station_info();

  clist = GTK_CLIST(lookup_widget(station_info, "clist"));
  gtk_clist_set_column_justification(clist, 0, GTK_JUSTIFY_RIGHT);
  gtk_clist_set_column_justification(clist, 3, GTK_JUSTIFY_RIGHT);
  gtk_clist_set_column_justification(clist, 4, GTK_JUSTIFY_RIGHT);
  gtk_clist_set_column_justification(clist, 5, GTK_JUSTIFY_RIGHT);

  gtk_clist_column_titles_passive(clist);
  gtk_clist_column_title_active(clist, 0);
  gtk_clist_column_title_active(clist, 3);
  gtk_clist_column_title_active(clist, 4);

  gtk_window_set_default_size(GTK_WINDOW(station_info),
    geometry_w < 0 ? default_w : geometry_w, 
    geometry_h < 0 ? default_h : geometry_h);
  gtk_widget_set_uposition(station_info, geometry_x, geometry_y);

  appbar = lookup_widget(station_info, "appbar");
  gnome_appbar_set_status(GNOME_APPBAR(appbar), "Loading databases...");
  gnome_appbar_set_progress(GNOME_APPBAR(appbar), 1 / 7.0);
  gtk_widget_show(station_info);

  gtk_idle_add(station_info_idle, NULL);
  gtk_main();
  return EXIT_SUCCESS;
}
