/* guess-where -- tries to guess the user's location.
 *
 * Copyright (C) 2001 John Kodis <kodis@jagunet.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include "guess-where.h"
#include "zone.h"

static int
lat_for_lon[] =
  {
    +52, // gmt+00, London
    +54, // gmt+01, Berlin
    +54, // gmt+02, Minsk
    +56, // gmt+03, Moscow
    +57, // gmt+04, Yekaterinburg
    +52, // gmt+05, Astana
    +24, // gmt+06, Dhaka
    +22, // gmt+07, Hanoi
    -32, // gmt+08, Perth
    -35, // gmt+09, Adelaide
    -34, // gmt+10, Sydney
    -42, // gmt+11, Wellington
    -22, // gmt+12, Tonga
    +23, // gmt-11, Necker Is
    +20, // gmt-10, Hilo
    +58, // gmt-09, Juneau
    +37, // gmt-08, San Francisco
    +39, // gmt-07, Denver
    +39, // gmt-06, Springfield
    +39, // gmt-05, Annapolis
    +45, // gmt-04, Sydney
    -25, // gmt-03, Sao Paulo
    -8 , // gmt-02, Recife
    +14, // gmt-01, Dakar
  };

static int
guess_from_gmt_offset(double *lat, double *lon)
{
  time_t now = time(NULL);
  struct tm local = *localtime(&now);
  int soff = -timezone/60/60 + (local.tm_isdst == 1);
  int uoff = soff < 0 ? 24 + soff : soff;
  *lon = 15 * soff;
  *lat = lat_for_lon[uoff];

#ifdef MAIN
  printf("gmt offset: (%s,%s)[%d(%d)] %lds, %ldm, %ldh; off=%d\n",
    tzname[0], tzname[1], local.tm_isdst, daylight,
    timezone, timezone/60, timezone/60/60, soff);
#endif

  return 1;
}

static int
guess_from_zone_table(double *lat, double *lon)
{
  char *tz_str = getenv("TZ");

#ifdef MAIN
  printf("zone table: getenv(\"TZ\") = %s\n", tz_str);
#endif

  if (tz_str)
    {
      struct zone_t *zp;
      for (zp = zone; zp->name && strcmp(zp->name, tz_str); zp++)
	;
      if (zp->name)
	{
	  *lat = zp->lat;
	  *lon = zp->lon;
	  return 1;
	}
    }
  return 0;
}

static int
guess_from_dot_emacs_file(double *lat, double *lon)
{
  FILE *emacs;
  char *home, path[FILENAME_MAX], line[1000];
  int got_lat = 0, got_lon = 0;

  home = getenv("HOME");
  if (!home || !*home)
    return 0;
  strcpy(path, getenv("HOME"));
  strcat(path, "/.emacs");
  if ((emacs = fopen(path, "r")) == NULL)
    return 0;

  while (fgets(line, sizeof(line), emacs) > 0)
    {
      char *tail, *head;
      if ((head = strstr(line, "calendar-latitude")) != NULL)
	{
	  head += sizeof("calendar-latitude");
	  *lat = strtod(head, &tail);
	  if (head != tail)
	    got_lat++;
	}
      if ((head = strstr(line, "calendar-longitude")) != NULL)
	{
	  head += sizeof("calendar-longitude");
	  *lon = strtod(head, &tail);
	  if (head != tail)
	    got_lon++;
	}
    }

  fclose(emacs);
  return got_lat && got_lon;
}

static double
deg2rad(double degrees)
{
  return M_PI * (degrees / 180.);
}

int
guess_where(double *lat, double *lon)
{
  if ( (guess_from_dot_emacs_file(lat, lon))
    || (guess_from_zone_table(lat, lon))
    || (guess_from_gmt_offset(lat, lon)) )
    {
      *lat = deg2rad(*lat);
      *lon = deg2rad(*lon);
      return 1;
    }
  return 0;
}

#ifdef MAIN

static double
rad2deg(double radians)
{
  return 180. * (radians / M_PI);
}

static int
gen_marker_file(void)
{
  int hr;

  for (hr = 0; hr < 24; hr++)
    {
      int soff = hr <= 12 ? hr : hr - 24;
      int uoff = soff < 0 ? 24 + soff : soff;
      int lon = 15 * soff;
      int lat = lat_for_lon[uoff];
      printf("%+4d %+4d %+3d(%2d)\n", lat, lon, soff, uoff);
    }

  return 1;
}

static int
report_on_guess(char *guess_name, int (*guess)(double *, double*))
{
  int got;
  double lat, lon;

  got = guess(&lat, &lon);
  if (!got)
    printf("%s: failed\n", guess_name);
  else
    printf("%s: %+8.3f, %+8.3f\n", guess_name, rad2deg(lat), rad2deg(lon));

  return got;
}

int
main(int argc, char **argv)
{
  if (0)
    gen_marker_file();
  else
    {
      report_on_guess("emacs file", guess_from_dot_emacs_file);
      report_on_guess("zone table", guess_from_zone_table);
      report_on_guess("gmt offset", guess_from_gmt_offset);
      report_on_guess("best guess", guess_where);
    }
  return EXIT_SUCCESS;
}

// Local Variables:
// mode: C
// compile-command: "gcc -Wall -g -DMAIN -o guess-where guess-where.c"
// End:

#endif
