# Pattern.pm -- generates a plot of an antenna radiation pattern.
#
# Copyright (C) 2001-2003, John Kodis <john@kodis.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

package Pattern;

require Exporter;
@ISA = qw(Exporter);
@EXPORT = qw( emit_pattern_img );

use strict;
use GD;
use Paths;
use Detail_vars;

#
# gen_ant_plot -- generates an antenna pattern .png using GD
#
sub gen_ant_plot {
    my ($plot_file, @pat) = @_;

    my $r = 160; # plot radius
    my $im = new GD::Image(3*$r+1, 2*$r+1);
    my $white = $im->colorAllocate(255,255,255);
    my $gray  = $im->colorAllocate(220,220,220);
    my $black = $im->colorAllocate(  0,  0,  0);
    my $red   = $im->colorAllocate(255,  0,  0);
    my $green = $im->colorAllocate(  0,255,  0);
    my $blue  = $im->colorAllocate(  0,  0,255);

    # $im->transparent($white);
    # $im->rectangle(0,0, 3*$r,2*$r, $black);
    $im->interlaced('true');

    $im->arc(2*$r,$r, 2*$r,2*$r, 0,360, ($#pat > 0) ? $red : $green);
    if ($pat_max > $pat_min) {
	my $ratio = $pat_min / $pat_max;
	$im->fill(2*$r,$r, $gray);
	$im->arc (2*$r,$r, 2*$ratio*$r,2*$ratio*$r, 0,360, $blue);
	$im->fill(2*$r,$r, $white);
	if ($pat_min < 1 && 1 < $pat_max) {
	    $im->arc(2*$r,$r, 2*$r/$pat_max,2*$r/$pat_max, 0,360, $black);
	}
    }

    # Perform various fix-ups to scale or use defaults for radiation values.
    my $lno = 1;
    my ($rad_min, $rad_avg, $rad_max) = ($pat_min, $pat_max, $pat_max);
    if (! $rms) {
	if ($#pat <= 0) {
	    # Fix-ups for FM/TV omni -- use 1 for min/avg/max.
	    $rad_min = $rad_avg = $rad_max = 1;
	}
    } else {
	# Fix-ups for AM antennas.
	if ($#pat <= 0) {
	    # John Byrns explains: For AM omnidirectional antennas,
	    # the FCC database doesn't give the RMS value of the
	    # pattern for the actual transmitter power like with the
	    # directional arrays, but rather gives the signal strength
	    # at 1 km for a transmitter power of 1 kW.  So, for
	    # omnidirectional antennas, you need to multiply the
	    # signal strength given in the database by the square root
	    # of the transmitter power in kW's, to get a value that is
	    # comparable to the RMS value for the directional arrays.
	    $rad_min = $rad_avg = $rad_max = $rms * sqrt($erp);
	} else {
	    # Scale radiation by rms, but leave $pat_min/max for plotting.
	    $rad_min = $pat_min * $rms;
	    $rad_avg = $rms;
	    $rad_max = $pat_max * $rms;
	}
    }
    if (1) {
	$im->string(gdMediumBoldFont, 0, $lno*gdMediumBoldFont->height(),
	    sprintf(" Maximum:   %8.3f", $rad_max), $red);
	$lno++;
    }
    my $rad_str	= " Radiation: --------";
    if ($rms) {
	$rad_str = sprintf " Radiation: %8.3f RMS", $rad_avg;
    }
    $im->string(gdMediumBoldFont, 0, $lno*gdMediumBoldFont->height(),
        $rad_str, $green);
    $lno++;
    if (1) {
        $im->string(gdMediumBoldFont, 0, $lno*gdMediumBoldFont->height(),
	    sprintf(" Minimum:   %8.3f", $rad_min), $blue);
	$lno++;
    }
    if ($rms) {
        $im->string(gdMediumBoldFont, 0, $lno*gdMediumBoldFont->height(),
	    " Units: mV/m at 1 km", $black);
	$lno++;
    }

    $im->string(gdLargeFont, 0, 2*$r-(gdLargeFont->height)*5/2,
        "$call, $band_corf", $black);
    $im->string(gdLargeFont, 0, 2*$r-(gdLargeFont->height)*3/2,
        "$city, $state", $black);

    if ($#pat > 0) {
	my $poly = new GD::Polygon();
	for (my $pt = 0; $pt < $#pat; $pt += 2) {
	    my $t = $pat[$pt];
	    my $d = $pat[$pt + 1] / $pat_max;
	    my $x = 2*$r + $r * $d * +sin($t);
	    my $y = 1*$r + $r * $d * -cos($t);
	    $im->arc($x,$y, 9,9, 0,360, $green);
	    $poly->addPt($x,$y);
	}
	$im->polygon($poly, $green);
    }
    # Draw cross-hairs at center of radiation pattern.
    $im->line(2*$r-10,$r, 2*$r+10,$r, $black);
    $im->line(2*$r,$r-10, 2*$r,$r+10, $black);

    # Draw North-up arrow.
    if (1) {
	$im->string(gdLargeFont, $r/4, $r, " N", $black);
	$im->line($r/4,$r-$r/4, $r/4,$r+$r/4, $black);
	$im->line($r/4,$r-$r/4, $r/4+$r/32,$r-$r/8, $black);
    }

    open (PNG, ">$plot_dir/$plot_file") ||
	die "can't write to $plot_dir/$plot_file";
    print PNG $im->png;
    close(PNG);
}

sub emit_pattern_img {
    my (%data) = @_;

    my $plot_file = "$band-$data{'sloc.eng_off'}-$call.png";
    my @pattern = split(' ', $data{'ant.pat.az_mag'});

    # $plot_dir = ".";
    if (gen_ant_plot($plot_file, @pattern)) {
	print "<p> <h2> Radiation Plot of $call, $band_corf </h2>\n";
	print "<img src='$plot_dir_url/$plot_file'>\n";
    }
}

1;
