/* details.c -- displays detailed station information.
 *
 * Copyright (C) 2001 John Kodis <kodis@jagunet.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <callbacks.h>
#include <details.h>
#include <fcc.h>
#include <interface.h>
#include <math.h>
#include <main.h>
#include <support.h>
#include <utils.h>

static GdkColor *blue;

static void
canvas_circle(GnomeCanvasGroup *group,
  double x, double y, double r, char *color)
{
  gnome_canvas_item_new(group, GNOME_TYPE_CANVAS_ELLIPSE,
    "x1", x-r, "y1", y-r, "x2", x+r, "y2", y+r,
    "outline_color", color, NULL);
}

static void
canvas_cross(GnomeCanvasGroup *group,
  double x, double y, double r, char *color)
{
  double *dp;
  GnomeCanvasPoints *cross = gnome_canvas_points_new(2);

  dp = cross->coords;
  *dp++ = x; *dp++ = y+r; *dp++ = x; *dp++ = y-r;
  gnome_canvas_item_new(group, GNOME_TYPE_CANVAS_LINE,
    "points", cross, "fill_color", color, NULL);

  dp = cross->coords;
  *dp++ = x+r; *dp++ = y; *dp++ = x-r; *dp++ = y;
  gnome_canvas_item_new(group, GNOME_TYPE_CANVAS_LINE,
    "points", cross, "fill_color", color, NULL);

  gnome_canvas_points_free(cross);
}

static void
canvas_rose(GnomeCanvasGroup *group, char *color)
{
  double *dp;
  GnomeCanvasPoints *rose = gnome_canvas_points_new(3);

  dp = rose->coords;
  *dp++ = +1.20; *dp++ = +0.50;
  *dp++ = +1.20; *dp++ = -0.50;
  *dp++ = +1.30; *dp++ = -0.25;
  gnome_canvas_item_new(group, GNOME_TYPE_CANVAS_LINE,
    "points", rose, "fill_color", color, NULL);
  gnome_canvas_points_free(rose);
  gnome_canvas_item_new(group, GNOME_TYPE_CANVAS_TEXT,
    "x", 1.20, "y", 0.00, "text", "N", "fill_color", color,
    "font", "-*-times-medium-r-normal-*-24-*-*-*-p-*-*", NULL);
}

static void
canvas_pattern(GnomeCanvasGroup *group,
  Pattern *pattern, double rotation, char *pattern_color, char *points_color)
{
  int p;
  double *dp;
  GnomeCanvasPoints *outline = gnome_canvas_points_new(pattern->n_points + 1);

  for (p = 0, dp = outline->coords; p < pattern->n_points; p++)
    {
      double r = pattern->magnitude[p] / pattern->max;
      double t = M_PI/2 - (pattern->azimuth[p] + rotation);
      double x = +r * cos(t);
      double y = -r * sin(t);

      *dp++ = x; *dp++ = y;
      canvas_circle(group, x, y, 0.02, points_color);
    }
  *dp++ = outline->coords[0]; *dp++ = outline->coords[1]; 
  gnome_canvas_item_new(group, GNOME_TYPE_CANVAS_LINE,
    "points", outline, "fill_color", pattern_color, NULL);
  gnome_canvas_points_free(outline);
}

static void
pattern_plot(Sinfo *sinfo, Pattern *pattern, GtkWidget *details)
{
  GnomeCanvas *canvas = GNOME_CANVAS(lookup_widget(details, "pattern_canvas"));
  GnomeCanvasGroup *group = gtk_object_get_data(GTK_OBJECT(canvas), "group");

  if (group)
    gtk_object_destroy(GTK_OBJECT(group));
  group = GNOME_CANVAS_GROUP(gnome_canvas_item_new(
    gnome_canvas_root(canvas), GNOME_TYPE_CANVAS_GROUP, NULL));
  gtk_object_set_data(GTK_OBJECT(canvas), "group", group);

  canvas_circle(group, 0, 0, 0.02, "black");
  canvas_cross(group, 0, 0, 0.10, "black");
  canvas_circle(group, 0, 0, 1.00, "white");

  if (pattern && pattern->n_points)
    canvas_pattern(group, pattern, sinfo_get_ant_rot(sinfo), "cyan", "blue");
  else
    canvas_circle(group, 0, 0, 1.00, "cyan");

  canvas_rose(group, "red");
  gnome_canvas_item_new(group, GNOME_TYPE_CANVAS_TEXT,
    "text", sinfo->fac->callsign, "fill_color", "black",
    "x", 2.00, "y", -1.00, "anchor", GTK_ANCHOR_NORTH_EAST,
    "font", "-*-times-medium-r-normal-*-24-*-*-*-p-*-*", NULL);
}

static void
am_antenna_info(GtkWidget *details, GtkCList *clist, Sinfo *sinfo)
{
  int cl_row, ant_sys_id, rel;
  char *kv[2], val[1000];
  static const char *band_to_str[] = { "am", "fm", "tv" };
  Am *am = sinfo->am;
  Ant *ant = sinfo_get_ant(sinfo);

  if (ant == NULL)
    fatal_need_for_data();

  ant_sys_id = am->ant_sys_id;

  kv[0] = " "; kv[1] = "";
  gtk_clist_append(clist, kv);
  
  kv[0] = "Antenna Info ..."; kv[1] = "";
  cl_row = gtk_clist_append(clist, kv);
  gtk_clist_set_foreground(clist, cl_row, blue);
  
  kv[0] = "App, Fac, Ant"; kv[1] = val;
  sprintf(val, "%d, %d, %d",
    -1, sinfo->fac->fac_id, ant->id);
  gtk_clist_append(clist, kv);

  kv[0] = "Towers(rel), Svc, Type"; kv[1] = val;
  rel = ant->tower_set ? ant->tower_set->rel : 0;
  sprintf(val, "%d(%d), %s, %c",
    am->towers, rel, band_to_str[sinfo->sloc.band], am->ant_type);
  gtk_clist_append(clist, kv);

  kv[0] = " "; kv[1] = "";
  gtk_clist_append(clist, kv);

  pattern_plot(sinfo, ant->pattern, details);
}

static void
fm_tv_antenna_info(GtkWidget *details, GtkCList *clist, Sinfo *sinfo)
{
  int cl_row;
  char val[1000], *kv[2];
  Ant *ant = sinfo_get_ant(sinfo);

  if (ant == NULL)
    fatal_need_for_data();

  kv[0] = " "; kv[1] = "";
  gtk_clist_append(clist, kv);
  
  kv[0] = "Antenna Info ..."; kv[1] = "";
  cl_row = gtk_clist_append(clist, kv);
  gtk_clist_set_foreground(clist, cl_row, blue);
  
  if (ant == NULL)
    {
      kv[0] = "Make / Model"; kv[1] = "Generic Omnidirectional";
      gtk_clist_append(clist, kv);
    }
  else
    {
      kv[0] = "Make / Model"; kv[1] = val;
      sprintf(val, "%s, %s", ant->make, ant->model);
      gtk_clist_append(clist, kv);

      kv[0] = "Pts, ID, Rot, Svc, Std, Date"; kv[1] = val;
      sprintf(val, "%d, %d, %.3f, %s, %s, %s",
	ant->pattern->n_points, ant->id, rad2deg(sinfo_get_ant_rot(sinfo)),
	ant->service, ant->standard, ant->updated);
      gtk_clist_append(clist, kv);

      kv[0] = " "; kv[1] = "";
      gtk_clist_append(clist, kv);
    }

  pattern_plot(sinfo, ant ? ant->pattern : NULL, details);
}

static GSList *details_windows;

static GtkWidget *
last_details(void)
{
  return details_windows ? (GtkWidget *)details_windows->data : NULL;
}

static void
free_details(GtkWidget *details)
{
  gtk_widget_destroy(details);
  details_windows = g_slist_remove(details_windows, details);
}

static GtkWidget *
new_details(void)
{
  GtkWidget *details = create_details_window();
  GtkWidget *close = lookup_widget(details, "details_close_button");

  details_windows = g_slist_prepend(details_windows, details);
  gtk_window_set_default_size(GTK_WINDOW(details), 400, 600);
  gtk_signal_connect_object(GTK_OBJECT(close),
    "clicked", GTK_SIGNAL_FUNC(free_details), GTK_OBJECT(details));
  gtk_signal_connect_object(GTK_OBJECT(details),
    "delete_event", GTK_SIGNAL_FUNC(free_details), GTK_OBJECT(details));

  return (GtkWidget *)details_windows->data;
}

static GdkColor *
gdk_color_new(const char *spec, int writable)
{
  GdkColor *color = g_malloc(sizeof(*color));

  if (!gdk_color_parse(spec, color))
    {
      g_free(color);
      return NULL;
    }
  gdk_colormap_alloc_color(gdk_colormap_get_system(), color, writable, TRUE);
  return color;
}

static void
show_details_window(GtkCList *main_clist, Sinfo *sinfo, int button)
{
  int cl_row;
  char val[1000], *kv[2];
  Fac *fac;
  GtkCList *clist;
  GtkWidget *label, *details;

  if (blue == NULL)
    blue = gdk_color_new("#0000ff", FALSE);

  details = last_details();
  if (!details || button == 3)
    details = new_details();

  fac = sinfo->fac;
  label = lookup_widget(details, "details_label");
  gtk_label_set_text(GTK_LABEL(label), fac->callsign);
  gtk_widget_show(details);

  clist = GTK_CLIST(lookup_widget(details, "details_clist"));
  gtk_clist_clear(clist);
  gtk_clist_freeze(clist);

  /* Facility record information... */

  kv[0] = "Facility Info ..."; kv[1] = "";
  cl_row = gtk_clist_append(clist, kv);
  gtk_clist_set_foreground(clist, cl_row, blue);

  kv[0] = "Callsign"; kv[1] = fac->callsign;
  gtk_clist_append(clist, kv);
  
  kv[0] = "Chan, Freq"; kv[1] = val;
  sprintf(val, "%d, %.1f %s", fac->channel, fac->freq, fac->service);
  gtk_clist_append(clist, kv);
  
  kv[0] = "Location"; kv[1] = val;
  sprintf(val, "%s, %s", fac->comm_city, fac->comm_state);
  gtk_clist_append(clist, kv);
  
  kv[0] = "Address"; kv[1] = fac->fac_addr1;
  gtk_clist_append(clist, kv);

  kv[0] = " "; kv[1] = fac->fac_addr2;
  if (*fac->fac_addr2)
    gtk_clist_append(clist, kv);

  kv[0] = " "; kv[1] = val;
  sprintf(val, "%s, %s %s, %s",
    fac->fac_city, fac->fac_state, fac->fac_zip, fac->fac_country);
  gtk_clist_append(clist, kv);

  /* Station ID info... */
  
  kv[0] = " "; kv[1] = "";
  gtk_clist_append(clist, kv);
  
  kv[0] = "Station Info ..."; kv[1] = "";
  cl_row = gtk_clist_append(clist, kv);
  gtk_clist_set_foreground(clist, cl_row, blue);
  
  kv[0] = "Location"; kv[1] = val;
  sprintf(val, "%c %.4f, %c %.4f",
    "NS"[sinfo->sloc.lat < 0], fabs(rad2deg(rot2rad(sinfo->sloc.lat))),
    "EW"[sinfo->sloc.lon < 0], fabs(rad2deg(rot2rad(sinfo->sloc.lon))));
  gtk_clist_append(clist, kv);
  
  kv[0] = "Dist / Dir"; kv[1] = val;
  sprintf(val, "%.1f %s, %.0f %s",
    sinfo->arc * search.rad2units, search.unit_name,
    rad2deg(sinfo->head), rad2point(sinfo->head));
  gtk_clist_append(clist, kv);
  
  kv[0] = "Freq, Corf"; kv[1] = val;
  sprintf(val, "%d, %.1f", sinfo->khz, sinfo->sloc.corf / 10.0);
  gtk_clist_append(clist, kv);

  /* Antenna Information... */
  if (sinfo->sloc.band == band_am)
    am_antenna_info(details, clist, sinfo);
  else
   fm_tv_antenna_info(details, clist, sinfo);

  gtk_clist_columns_autosize(clist);
  gtk_clist_thaw(clist);
}

int
on_clist_button_release_event(
  GtkWidget *widget, GdkEventButton *event, void *user_data)
{
  int x = event->x, y = event->y, button = event->button, r, c;

  if (button == 1 || button == 3)
    {
      Sinfo **sa;
      GtkCList *clist = GTK_CLIST(widget);
      gtk_clist_set_button_actions(clist, 1, GTK_BUTTON_IGNORED);
      gtk_clist_get_selection_info(clist, x, y, &r, &c);
      gtk_clist_select_row(clist, r, c);
      sa = gtk_object_get_data(GTK_OBJECT(clist), "sinfo-array");
      if (sa == NULL)
	;
      else
	{
	  Sinfo *sinfo = sa[r];
	  show_details_window(clist, sinfo, button); 
	}
    }

  return FALSE;
}
