/**
 * test suite for the *Vector classes
 * @version 25.09.2007
 * @author Michael J. Beer (mibeer@uni-osnabrueck.de)
 * Copyright (C) 2007 Michael Beer 
 */
/*      
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


#include <iostream>

// #include "DenseVector.h"
#include "utils.h"
#include "staf.h"
#include <cmath>


using namespace std;
using namespace STAF;
/**
 * length to be tested
 */
#define LENGTH 100

/**
 * the VectorType to be testet
 * always use double as storageType
 */
typedef DenseVector<double> TVec;
typedef DenseMathVector<double>::Type TMathVec;

/**
 * if this value is given to checkVec as content, it just checks 
 * for accessibility
 */
#define JUST_ACCESSIBLE -1   

bool checkVec(TVec &vec, double content = JUST_ACCESSIBLE)
  {
    for (size_t count = 0; count < vec.size(); count++)
      {
        if ( (vec.at(count) != content))
          {
            if (content != JUST_ACCESSIBLE)
              {
                return false;
              };
          };
      };
    return true;
  }
;

/**
 * testing the functions declared by BaseVector
 */
int baseVectorTest()
  {

    TVec *vec1, *vec2;

    bool correct = true;

    cout << "BaseVector tests"<< endl << endl;
    ////////////////////////////////////////////////////////
    // constructors
    cout << "Testing constructors"<< endl << endl;

    vec1 = new TVec(LENGTH);
    TEST("constructor(size_t)", checkVec(*vec1));

    vec2 = new TVec(LENGTH, 1);
    TEST("constructor(size_t, storageType)", checkVec(*vec2, 1));
    ////////////////////////////////////////////////////////
    // BASIC access functions
    cout << endl << "Testing basic input/output functionality"<< endl << endl;

    TEST("size()", vec1->size() == LENGTH);

    for (unsigned int count = 0; count < vec1->size(); count++)
      {
        vec1->assign(count, count);
        vec2->assign(count, count);
      };

    for (unsigned int count = 0; count < vec1->size(); count++)
      {
        if (vec1->at(count) != count)
          {
            correct = false;
            break;
          }
      };

    TEST("assign() / at() test 1", correct);

    for (unsigned int count = 0; count < vec1->size(); count++)
      {
        if (vec1->at(count) != vec2->at(count) )
          {
            correct = false;
            break;
          }
      };

    TEST("assign() / at() test 2", correct);

    ///////////////////////////////////////////////////////////////
    // operator tests
    cout << endl << "testing operators"<< endl << endl;

    TEST("operator==() test 1", *vec1 == *vec2);

    vec1->assign(LENGTH - 1, 0);

    TEST("operator==() test 2", !(*vec1 == *vec2));

    cout << endl << "BaseVector functionality tested"<< endl << endl << endl
        << endl;
    delete vec1;
    delete vec2;
    // delete vec3;
    // delete vec4;

    return 0;
  }
;

/**
 * testing the functions of MathVector
 */
int mathVectorTest()
  {
    TMathVec *vec1, *vec2, *vec3;

    bool passed = true;

    double diff = 0;
    cout << "MathVector tests"<< endl << endl;

    ///////////////////////////////////////////////////////////////////
    // constructor test
    cout << "Constructor tests..."<< endl << endl;

    vec1 = new TMathVec(LENGTH);
    TEST("MathVector(size_t)", vec1 != 0);

    vec2 = new TMathVec(LENGTH, 13);
    for (size_t count = 0; count < LENGTH; count++)
      {
        if (vec2->at(count) != 13)
          {
            passed = false;
            break;
          };
      };
    TEST("constructor(size_t, storageType)", passed);
    passed = true;

    vec3 = new TMathVec(*vec2);
    for (size_t count = 0; count < LENGTH; count++)
      {
        if (vec2->at(count) != vec3->at(count))
          {
            passed = false;
            break;
          };
      };

    TEST("constructor(MathVector &) (Copyconstructor)", passed);
    delete vec3;
    passed = true;

    ///////////////////////////////////////////////////////////////////
    // access tests
    cout << endl << "Access tests..."<< endl << endl;

    TEST("size()", vec1->size() == LENGTH);

    for (size_t count = 0; count < LENGTH; count++)
      {
        vec1->assign(count, count);
      };

    for (size_t count = 0; count < LENGTH; count++)
      {
        if (vec1->at(count) != count)
          {
            passed = false;
            break;
          }
      };

    TEST("assign()/at()", passed);

    delete vec1;

    ///////////////////////////////////////////////////////////////////
    // testing the scalar product
    cout << endl << "Product tests..."<< endl << endl;

    vec1 = new TMathVec(LENGTH, 10);
    TEST("scalarProduct()", (vec1->scalarProduct(*vec2) == 10*13*LENGTH));

    delete vec1;
    delete vec2;

    // vector product test
    vec1 = new TMathVec(3);
    vec2 = new TMathVec(3);

    vec1->assign(0, 5);
    vec1->assign(1, 10);
    vec1->assign(2, 2);
    vec2->assign(0, 7);
    vec2->assign(1, 3);
    vec2->assign(2, 4);
    vec3 = new TMathVec(vec1->vectorProduct(*vec2));
    // vec3->assign(0 , 4 * 10 - 2 * 3);
    // vec3->assign(1, 5 * 4 - 2 * 7);
    // vec3->assign(2, 5 * 3 - 7 * 10);
    TEST("vectorProduct()", (vec3->at(0) == 10 * 4- 2 * 3) &&(vec3->at(1) == 5
        * 4- 2 * 7)&&(vec3->at(2) == 5 * 3- 10 * 7));

    delete vec3;
    vec3 = new TMathVec(vec1->multiply(2.5));

    TEST("multiply(storageType )", (vec3->at(0) == vec1->at(0) * 2.5)
        &&(vec3->at(1) == vec1->at(1) * 2.5)
        &&(vec3->at(2) == vec1->at(2) * 2.5));

    /////////////////////////////////////////////////////////////////////
    // testing the one - forms
    cout << endl << "Norm tests..."<< endl << endl;

    TEST("p1Norm() test 1", absolute(vec1->p1Norm() - 17) < MIN_DIFFERENCE);
    delete vec3;
    vec3 = new TMathVec(LENGTH, 12);
    TEST("p1Norm() test 2", absolute(vec3->p1Norm() - LENGTH * 12) < MIN_DIFFERENCE);
    TEST("piNorm(1) test 1", absolute(vec1->p1Norm() - vec1->piNorm(1)) < MIN_DIFFERENCE);
    
    cout << "vec3->piNorm(1) : " << vec3->piNorm(1) << "   ref : " << vec3->p1Norm() << endl;
    TEST("piNorm(1) test 2", absolute(vec3->p1Norm() - vec3->piNorm(1)) < MIN_DIFFERENCE);
    
    diff = (vec1->piNorm(2) - sqrt(5.0 * 5 + 10.0 * 10 + 2.0 * 2));
    TEST("piNorm(2) test 1", absolute(diff) < MIN_DIFFERENCE);
    // cout << "vec1->piNorm(2) : " << vec1->piNorm(2) << "   res : " << sqrt(5.0 * 5 + 10 * 10 + 2 * 2) << endl;

    diff = (vec3->piNorm(2) - sqrt(LENGTH * 12.0 * 12));
    TEST("piNorm(2) test 2", absolute(diff) < MIN_DIFFERENCE);
    // cout << "vec3->piNorm(2) : " << vec3->piNorm(2) << "   res : " << sqrt(LENGTH * 12.0 * 12) << endl;

    diff= (vec1->piNorm(3)- pow(5.0 * 5 * 5 + 10.0 * 10 * 10 + 2.0 * 2 * 2, 1/ 3.0));
    TEST("piNorm(3) test 1", (absolute(diff) < MIN_DIFFERENCE));
    // cout << "vec1->piNorm(3) : " << vec1->piNorm(3) << "   res : " << pow(5.0 * 5 * 5 + 10 * 10 * 10 + 2 * 2 * 2, 1 / 3.0) << endl;

    diff = (vec3->piNorm(3) - pow(LENGTH * 12.0 * 12 * 12, 1 / 3.0));
    TEST("piNorm(3) test 2", (absolute(diff) < MIN_DIFFERENCE));
    // cout << "vec3->piNorm(3) : " << vec3->piNorm(3) << "   res : " << pow(LENGTH * 12.0 * 12 * 12, 1 / 3.0) << endl;

    diff = (vec1->piNorm(10) - pow( 5.0 * 5 * 5 * 5 * 5 * 5 * 5 * 5 * 5 * 5 + 10.0 * 10
        * 10 * 10 * 10 * 10 * 10 * 10 * 10 * 10 + 2.0 * 2 * 2 * 2 * 2 * 2 * 2 * 2 * 2 * 2, 1
        / 10.0));
    TEST("piNorm(10) test 1", (absolute(diff) < MIN_DIFFERENCE));
    /* cout << "vec1->piNorm(10) : " << vec1->piNorm(10) << "   res : " << pow( 5.0 *  5 *  5 *  5 *  5 *  5 *  5 *  5 *  5 *  5 +
     10.0 * 10 * 10 * 10 * 10 * 10 * 10 * 10 * 10 * 10 +
     2.0 *  2 *  2 *  2 *  2 *  2 *  2 *  2 *  2 *  2, 1 / 10.0) << endl; */
    diff
        = vec3->piNorm(10)
            - pow(LENGTH * 12.0 * 12 * 12 * 12 * 12 * 12 * 12 * 12 * 12 * 12, 1 / 10.0);
    TEST("piNorm(10) test 2", absolute(diff) < MIN_DIFFERENCE);
    // cout << "vec3->piNorm(10) : " << vec3->piNorm(10) << "   res : " << pow(LENGTH * 12.0 * 12 * 12 * 12 * 12 * 12 * 12 * 12 * 12 * 12, 1 / 10.0) << endl;       

    delete vec1;
    delete vec2;
    delete vec3;
    return 0;
  }
;
