/**
 * Representation of a matrix
 * @version 06.09.2007
 * @author Michael J. Beer (mibeer@uni-osnabrueck.de)
 * Copyright (C) 2007 Michael Beer 
 */
/*      
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#ifndef __MATRIX_H__
#define __MATRIX_H__

#include <iostream>
#include "utils.h"
#include "config.h"
#include <cmath>

#ifdef USE_VIRTUAL_FUNCTIONS
#include "BaseVector.h"
#endif


// not well looking, adjust it to the correct (abstract classes of the STL) 

namespace STAF
{

/**
 * Representation of a Matrix that is indepenent of the type of the
 * line vector types and the storage types 
 * minimum requirements of vectorType:
 *    - handling a sorted set of a predefined length
 *    - template class with variable storage types
 *    - constructor vectorType(int noEntries)
 *    - at(uint) and assign(uint, storageType value) for accessing elements
 *    - uint size(void) function
 *
 * storageType can be anything that has
 *    - Operators *, + 
 *    - Operator  =
 *    - needs no constructor
 * @see BaseVector
 * @see Matrix
 */
template<class vectorType, class storageType> class Matrix
  {

    // TODO: in case of USE_VIRTUAL_FUNCTIONS :
    //       enable runtime - sided support for vector type change
    //       ( using an extra argument in constructor) if possible

public:

    /**
     * Constructor, constructs a matrix with given size
     * @param m the number of lines
     * @param n the number of columns
     */
    Matrix<vectorType, storageType>(size_t m, size_t n)
      {

        this->createNew(m, n);

      }
    ;

    /**
     * Constructor, constructs a matrix with given size, inits it with desired value
     * @param m the number of lines
     * @param n the number of columns
     * @param filling the initial value of the entries
     */
    Matrix<vectorType, storageType>(size_t m, size_t n, storageType filling)
      {
        // create matrix
        this->createNew(m , n);
        // ini the columns containing vectors
        for (size_t line = 0; line < this->getNoLines(); line++)
          {
            for (size_t col = 0; col < this->getNoCols(); col++)
              {
                this->entries[line]->assign(col, filling);
              }
          }
      }
    ;

    /**
     * Copyconstructor
     * @param m the Matrix to be copied into this 
     */
    Matrix<vectorType, storageType>(const Matrix<vectorType, storageType>& m)
      {

        this->entries = 0;
        this->resize(m.getNoLines(), m.getNoCols());
        // copy rhs into this
        this->assignWithoutDestroy(m);

      }
    ;

    /**
     * initialize matrix using an array
     * @param lines no of lines 
     * @param cols number of cols
     * @param array the array containing the new values
     */
    Matrix<vectorType, storageType>(size_t lines, size_t cols,
        storageType *array)
      {
        this->createNew(lines, cols);
        // fill it
        this->assign(array);
      }
    ;

    /**
     * destructor - ?
     */
    ~Matrix<vectorType, storageType>()
      {
        this->destroy();
      }
    ;

    /** 
     * Get number of lines of the matrix
     * @return the Number of Lines
     */
    inline size_t getNoLines(void) const
      {
        return this->noLines;
      }
    ;

    /**
     * Get number of columns of the matrix
     * @return the number of columns
     */
    inline size_t getNoCols(void) const
      {
        return this->noCols;
      }
    ;

    /**
     * set value of an entry
     * @param m line of the entry
     * @param n column of the entry
     * @param value new value
     */
    void setValue(unsigned int m, unsigned int n, storageType value)
    // CHECKED
      {

        assert((m < this->getNoLines()) && (n < this->getNoCols()));

        (( this->entries) [m])->assign(n, value);
      }
    ;

    /**
     * get value of an entry
     * @param m line of the entry
     * @param n column of the entry
     * @return value of the entry
     */
    storageType getValue(unsigned int m, unsigned int n) const
    // CHECKED
      {

        assert((m < ( this->getNoLines() )) && (n < ( this->getNoCols() )));

        return ( this->entries)[m]->at(n);
      }
    ;

    /**
     * multiply with a vector
     * @param vec the vector to be multiplied with
     * @return the resulting vector
     */
    vectorType operator*(const vectorType& vec) const
    // CHECKED
      {

        assert(this->getNoLines() == vec.size() );

        int maxLine = this->getNoLines();

        vectorType sol(maxLine);

        for (int line = 0; line < maxLine; line++)
          {
            sol.assign(line, 0);

            for (unsigned int col = 0; col < this->getNoCols(); col++)
              {
                sol.assign(line, sol.at(line) + (this->getValue(line, col)
                    * vec.at(col)));
              } /* end for */
          } /* end for */

        return sol;

      }
    ; /* end operator*() */

    /**
     * multiply ( rightwise) by another matrix
     * @param matrix the matrix to be multiplied with
     * @return the new matrix
     */
    Matrix<vectorType, storageType> operator*(
        const Matrix<vectorType, storageType>& matrix) const
    // CHECKED
      {

        assert((this->getNoCols() == matrix.getNoLines()));
        // moving those querries out of the loop heads could save some time (??)
        int maxCol = matrix.getNoCols();
        int maxLine = this->getNoLines();
        int matrixLines = matrix.getNoLines();

        Matrix<vectorType, storageType> sol(maxLine, maxCol);

        for (int line = 0; line < maxLine; line++)
          {
            for (int col = 0; col < maxCol; col++)
              {
                sol.setValue(line, col, 0);
                // cout "   Betrachte Eintrag " << line
                for (int count = 0; count < matrixLines; count++)
                  {
                    sol.setValue(line, col, sol.getValue(line, col)
                        + this->getValue(line, count) * matrix.getValue(count,
                            col) );
                    // cout << this->getValue(line,count) 
                  }
              }
          }

        return sol;

      }
    ;

    /**
     * multiply by a scalar
     * @param s the scalar to be multiplied with
     * @return the new matrix
     */
    Matrix<vectorType, storageType> operator*(const storageType s) const
    // CHECKED
      {
        Matrix<vectorType, storageType> sol(this->getNoLines(),
            this->getNoCols());

        for (unsigned int line = 0; line < this->getNoLines(); line++)
          {
            for (unsigned int col = 0; col < this->getNoCols(); col++)
              {
                sol.setValue(line, col, this->getValue(line, col) * s);
              }
          }
        return sol;
      }
    ;

    /**
     * divide by a scalar
     * @param s the scalar that should be divided by
     * @return the new matrix
     */
    inline Matrix<vectorType, storageType> operator/(const storageType s) const
    // CHECKED
      {
        return this->operator*( 1 / s);
      }

    /**
     * add a matrix to this one
     * @param matrix the matrix to be added
     * @return the resulting matrix
     */
    Matrix<vectorType, storageType> operator+(
        const Matrix<vectorType, storageType>& matrix) const
    // CHECKED
      {

        assert((this->getNoCols() == matrix.getNoCols()) && (this->getNoLines() == matrix.getNoLines()));

        Matrix<vectorType, storageType> sol(this->getNoLines(),
            this->getNoCols());

        // perhaps faster if this query is placed in here rather than in the loops head?
        int maxLine = this->getNoLines();
        int maxCol = this->getNoCols();

        for (int line = 0; line < maxLine; line++)
          {
            for (int col = 0; col < maxCol; col++)
              {
                sol.setValue(line, col, this->getValue(line, col)
                    +matrix.getValue(line, col));
              }
          }
        return sol;

      }
    ;

    /**
     * subtract a matrix to this one
     * could be done by just multiplying with (-1) and then add but its 
     * faster this way (who care about those few more bytes :)
     * @param matrix the matrix to be subtracted
     * @return the resulting matrix
     */
    Matrix<vectorType, storageType> operator-(
        const Matrix<vectorType, storageType>& matrix) const
    // TODO: check
      {

        assert((this->getNoCols() == matrix.getNoCols()) && (this->getNoLines() == matrix.getNoLines()));

        Matrix<vectorType, storageType> sol(this->getNoLines(),
            this->getNoCols());

        // perhaps faster if this query is placed in here rather than in the loops head?
        int maxLine = this->getNoLines();
        int maxCol = this->getNoCols();

        for (int line = 0; line < maxLine; line++)
          {
            for (int col = 0; col < maxCol; col++)
              {
                sol.setValue(line, col, this->getValue(line, col)
                    - matrix.getValue(line, col));
              }
          }
        return sol;

      }
    ;

    /**
     * assignment to this-object
     * @param rhs right hand side of assignment
     * @return the right hand side
     */
    Matrix<vectorType, storageType>& assign(
        const Matrix<vectorType, storageType> &rhs)
    // CHECKED
      {
        // check wether theres need for resize
        if ( !equalSize(rhs))
          {
            this->destroy();
            this->resize(rhs.getNoLines(), rhs.getNoCols());
          }
        // copy rhs into this
        this->assignWithoutDestroy(rhs);
        return *this;
      }
    ;

    /**
     * Assigns the content of an array to the matrix without changing the size
     * @param array the array the content of to be assigned 
     */
    Matrix<vectorType, storageType>& assign(storageType *array)
    // CHECKED
      {
        size_t noLines = this->getNoLines();
        size_t noCols = this->getNoCols();

        for (size_t line = 0; line < noLines; line++)
          {
            for (size_t col = 0; col < noCols; col++)
              {
                this->setValue(line, col, ARRAY_2D(array, noCols, line, col));
              }
          }
        return *this;
      }
    ;

    /**
     * assignment to this-object
     * @param rhs right hand side of assignment
     * @return the right hand side
     */
    Matrix<vectorType, storageType>& operator=(
        Matrix<vectorType, storageType> &rhs)
    // CHECKED
      {
        return this->assign(rhs);
      }
    ;

    /**
     * compare a matrix to this one entry by entry
     * does not check for equal sizes!
     * @param m the other matrix
     * @return true if both are equal, else false
     */
    bool operator==(const Matrix<vectorType, storageType> &m) const
    // CHECKED
      {
        // if not equal in size, matrices cant be equal
        assert( (m.getNoLines() == this->getNoLines()) || (m.getNoCols()
                == this->getNoCols()) );

        for (size_t line = 0; line < this->getNoLines(); line++)
          {
            for (size_t col = 0; col < this->getNoCols(); col++)
              if (absolute(this->getValue(line, col) - m.getValue(line, col)) >= MIN_DIFFERENCE)
                {
                  return false;
                }
          }

        return true;
      }
    ;

    /**
     * Print out the Matrix
     * @param stream the stream to be printed out on
     */
    void print(::std::ostream &stream) const
    // CHECKED
      {
        for (unsigned int line = 0; line < this->getNoLines(); line++)
          {
            for (unsigned int col = 0; col < this->getNoCols(); col++)
              {
                ::std::cout << this->getValue(line, col) << " ";
              }
            stream << ::std::endl;
          }
      }
    ;

    /**
     * checks wether matrix m has equal size as this
     * @param m the matrix to be compared to this 
     * @return true if both are equally size, false elsewise
     */
    bool equalSize(const Matrix<vectorType, storageType> &m) const
    // CHECKED
      {
        if ( (m.getNoLines() != this->getNoLines()) || (m.getNoCols()
            != this->getNoCols()))
          {
            return false;
          }
        return true;
      }
    ;

private:

    /** 
     * initializes new data structure without destroying old one 
     * matrix must have correct size!
     * used in constructors and assign()
     * @param m the Matrix to be copied into this one
     */
    void assignWithoutDestroy(const Matrix<vectorType, storageType> &m)
      {

        assert((m.getNoLines() == this->getNoLines()) && (m.getNoCols() == this->getNoCols()) );

        // copy everyting into the new matrix
        for (unsigned int line = 0; line < this->getNoLines(); line++)
          {
            for (unsigned int col = 0; col < this->getNoCols(); col++)
              {
                this->setValue(line, col, m.getValue(line, col));
              }
          }
      }
    ;

    /**
     * destroys current data structure
     * used by destructor & assign()
     */
    void destroy(void)
      {
        for (unsigned int index = 0; index < this->getNoLines(); index++)
          {
            delete (this->entries)[index];
          }

        delete[] this->entries;
        this->entries = 0;
      }
    ;

    /**
     * changes size of matrix, data structures must have been destroyed before!
     * @param lines new number of lines
     * @param cols new number of cols
     */
    void resize(size_t lines, size_t cols)
      {
        assert(this->entries == 0);
        // data struct should be freed
#ifdef USE_VIRTUAL_FUNCTIONS
        this->entries = (STAF::BaseVector<storageType> **)new vectorType *[lines]; // create the cols containing array
#else
        this->entries = new vectorType *[lines];
#endif

        this->noLines = lines;
        this->noCols = cols;

        // ini the columns containing vectors
        for (size_t i = 0; i < this->getNoLines(); i++)
          {
            this->entries[i] = new vectorType(cols);
          }
      }
    ;

    /**
     * create data structure new
     * @param m number of lines 
     * @param n number of columns
     */

    void createNew(size_t m, size_t n)
      {
        this->entries = 0;
        this->resize(m, n);
      }
    ;

    //////////////////////////////////////////////////////////////////////
    // T H E    D A T A


#ifdef USE_VIRTUAL_FUNCTIONS
    STAF::BaseVector<storageType> **entries;
#else
    /**
     * structure to store entries
     */
    vectorType **entries;
#endif

    /**
     * measures of the matrix
     * */
    size_t noLines, noCols;
  };

}
;
#endif
