/**
 * @version 20.09.2007
 * @author Michael J. Beer (mibeer@uni-osnabrueck.de)
 * Copyright (C) 2007 Michael Beer 
 */
/*      
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA. */


#ifndef MATHVECTOR_H_
#define MAThVECTOR_H_

#include <cassert>
#include <cmath>
#include "BaseVector.h"


namespace STAF
{
/**
 * defines basic functions of a mathematical vector object
 * still lacks of actual implementation of the storage structures
 * actually just for using with types that are castable to double
 * @see BaseVector
 */
/*      
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

template<class vectorType, class storageType> class MathVector :
  public virtual BaseVector<storageType>
  {
public:

    /**
     * creates a vector wit hgiven length
     * @param len length of the new vector
     */
    MathVector<vectorType, storageType>(size_t len) :
      BaseVector<storageType>(len)
      // CHECKED
      {
        content = new vectorType(len);
      }
    ;

    /**
     * creates a vector with given length and fills it with given value
     * @param len length of new vector
     * @param filling filling of new vector
     */
    MathVector<vectorType, storageType>(size_t len, storageType filling) :
      BaseVector<storageType>(len, filling)
      // CHECKED
      {
        content = new vectorType(len, filling);
      }
    ;

    /**
     * destructor
     */
    ~MathVector<vectorType, storageType>()
      {
        delete content;
      }
    ;

    /**
     * Copyconstructor
     * @param vec the vector to be copied into this
     */
    MathVector<vectorType, storageType>(
        const MathVector<vectorType, storageType> &vec) :
      BaseVector<storageType>(vec)
      // CHECKED
      {
        delete content;

        // its better done implementing a copy constructor that allows BaseVectors to be copied 
        // for e.g. SparseVector could optimize the copying
        content = new vectorType((BaseVector<storageType> &) vec);
      }
    ;

    /**
     * returns the value at point pos
     * @pos the position which the value is wanted of
     * @return teh value at pos
     */
    inline storageType at(size_t pos) const
    // CHECKED
      {
        return content->at(pos);
      }
    ;

    /**
     * sets entry pos to value val
     * @pos the entry to be reset
     * @val the new value of teh entry
     */
    inline void assign(size_t pos, storageType val)
    // CHECKED
      {
        content->assign(pos, val);
      }
    ;

    /**
     * returns the length of the vector
     * @return the length of the vector
     */
    inline size_t size(void) const
    // CHECKED
      {
        return content->size();
      }
    ;

    /**
     * calculate scalar - product
     * @param vec another vector
     * @return the scalar product of this and vec
     */
    storageType scalarProduct(const MathVector<vectorType, storageType> &vec) const
    // CHECKED
      {
        assert(this->content->size() == vec.size());

        storageType result = 0;

        for (unsigned int count = 0; count < this->size(); count++)
          {
            result += this->at(count) * vec.at(count);
          }

        return result;
      }

    /**
     * calculates the vector product 
     * only for 3 dim vectors!
     * @param vec another vector
     * @result the vector product this x vec
     */
    MathVector<vectorType, storageType> vectorProduct(
        const MathVector<vectorType, storageType> &vec) const
    // CHECKED
      {
        assert(this->size() == 3&& vec.size() == 3);
        // impl. just for 3 dims

        MathVector<vectorType, storageType> result(3);

        result.assign(0, this->at(1) * vec.at(2) - this->at(2) * vec.at(1));
        result.assign(1, this->at(0) * vec.at(2) - this->at(2) * vec.at(0));
        result.assign(2, this->at(0) * vec.at(1) - this->at(1) * vec.at(0));

        return result;
      }
    ;

    /**
     * multiplies this by a scalar
     * @param s the scalar
     * @return the result of the scalar multiplication
     */
    MathVector<vectorType, storageType> multiply(const storageType s) const
    // CHECKED
      {
        MathVector<vectorType, storageType> result(this->size());
        for (unsigned int count = 0; count < this->size(); count++)
          {
            result.assign(count, this->at(count) * s);
          }
        return result;
      }
    ;

    /**
     * calculate the p1 - norm of the vector
     * could be done by PiNorm(1) as well, but would waste time 
     * @return the p1 - norm of the vector
     */
    storageType p1Norm(void) const
    // CHECKED
      {
        storageType result = 0;
        for (size_t count = 0; count < this->size(); count++)
          {
            result += absolute(this->at(count));
          }
        return result;
      }

    /**
     * calculates the P-i norm of the vector
     * @param i the number pf the p - norm
     * @return the result
     */
    storageType piNorm(unsigned int i) const
    // CHECKED for PINORM_AVOID_OVERFLOW false
    // TODO: DEBUG for PINORM_AVOID_OVERFLOW true
      {
        storageType result = 0;
        storageType bak = 0;
#ifdef PINORM_AVOID_OVERFLOW
        double normalize = this->p1Norm();       
#endif
        for (size_t count = 0; count < this->size(); count++)
          {
            bak = absolute(this->at(count));
#ifdef PINORM_AVOID_OVERFLOW
            bak /= normalize;
#endif
            result += pow((double ) bak, (double ) i);
          }

        result = pow(result, 1.0 / ((double) i));
#ifdef PINORM_AVOID_OVERFLOW
        result *= normalize;
#endif
        return result;
      }
    ;

    /**
     * calculate the P - inf norm ( the max - norm ) of
     * this vector
     * @return the P - inf norm
     */
    storageType pInfNorm(void) const
    // TODO: CHECK
      {
        storageType result = this->at(0), bak;

        for (size_t count = 0; count < this->size(); count++)
          {
            bak = abs(this->at(count));
            result = (result < bak) ? bak : result;
          }
        return result;
      }
    ;

    /**
     * adds one vector to this
     * @vec another vector
     * @return the result of this + vec
     */
    MathVector<vectorType, storageType> add(
        const MathVector<vectorType, storageType> &vec) const
        // TODO: CHECK
      {
        assert(vec.size() == this->size());

        size_t len = this->size();
        MathVector<vectorType, storageType> result(len);

        for (size_t count = 0; count < len; count++)
          {
            result.assign(count, this->at(count) + vec.at(count));
          }
        return result;
      }
    ;

    /**
     * compares  the vector with another one
     * @param vec the other vector
     * @return true if both are equal, false elsewise
     */
    bool operator==(const BaseVector<storageType> &vec) const
    // TODO: CHECK
      {
        assert(this->size() == vec.size());

        size_t len = this->size();

        for (unsigned int count = 0; count < len; count++)
          {
            if (absolute(this->at(count) - vec.at(count)) >= MIN_DIFFERENCE)
              {
                return false;
              }
          }

        return true;
      }

    /**
     * adds one vector to this
     * @vec another vector
     * @return the result of this + vec
     */
    MathVector<vectorType, storageType> operator+(
        const MathVector<vectorType, storageType> &vec) const
        // TODO: CHECK
      {
        return add(vec);
      }
    ;

    /**
     * multiplies this by a scalar
     * @param s the scalar
     * @return the result of the scalar multiplication
     */
    MathVector<vectorType, storageType> operator*(const storageType s) const
    // TODO: CHECK
      {
        return multiply(s);
      }
    ;

    /**
     * calculate scalar - product
     * @param vec another vector
     * @return the scalar product of this and vec
     */
    storageType operator*(const MathVector<vectorType, storageType> &vec) const
    // TODO: CHECK
      {
        return scalarProduct(vec);
      }

    /**
     * assign the entries read from an array of fitting size to this
     * @param array the new entries
     */
    void assign(const storageType *array)
    // TODO: CHECK
      {
        for (size_t count = 0; count < this->size(); count++)
          {
            this->assign(count, array[count]);
          }
      }
    ;

protected:

    void destroy(void)
      {
        ::std::cerr << "Called prohibited function MathVector::destroy()"<< ::std::endl;
        abort();
      }
    ;

    void resize(size_t n)
      {
        ::std::cerr << "Called prohibited function MathVector::resize()"<< ::std::endl;
        abort();
      }
    ;

    BaseVector<storageType> *content;

  };
  
};
#endif /*MATHVECTOR_H_*/
