/**
 * (Hopefully) Fast representation of a densely filled vector
 * @version 14.09.2007
 * @author Michael J. Beer (mibeer@uni-osnabrueck.de)
 * Copyright (C) 2007 Michael Beer 
 */
/*
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#ifndef __DENSEVECTOR_H__
#define __DENSEVECTOR_H__

#include <cassert>
#include "BaseVector.h"
#include "config.h"
#include <cmath>

namespace STAF
{
  
#ifdef USE_VIRTUAL_FUNCTIONS
template<class storageType> class DenseVector :
  public virtual BaseVector<storageType>
#else
/**
 * Defines a (hopefully) fast implementation for densely populated vectors 
 * minimalistic implementation for use with Matrix - objects according to 
 * the BaseVector - declarations 
 * @see BaseVector
 * @see Matrix
 */
template<class storageType> class DenseVector
#endif

  {

public:

    /**
     * creates a denseVector object with given length
     * @param len desired length
     */
    DenseVector<storageType>(size_t len)
#ifdef USE_VIRTUAL_FUNCTIONS
    :
      BaseVector<storageType>(len)
#endif
      {

        this->length = len;
        this->entries = new storageType[len];

      }
    ;

    /**
     * creates a denseVector object with given length and inits all entries
     * @param len desired length
     * @param filling the initial filling of the entries
     */
    DenseVector<storageType>(size_t len, storageType filling)
#ifdef      USE_VIRTUAL_FUNCTIONS
    :
      BaseVector<storageType>(len)
#endif
      {

        this->length = len;
        this->entries = new storageType[len];

        for (size_t i = 0; i < this->size(); i++)
          {
            this->entries[i] = filling;
          }
      }
    ;

    /**
     * Copyconstructor
     * @param vec the BaseVector object to be copied into this 
     */
    DenseVector<storageType>(const BaseVector<storageType>& vec)
    : BaseVector<storageType>(vec)
      {
        if (this->size() != vec.size())
          {
            this->destroy();
            this->resize(vec.size());
          }

        for (size_t count = 0; count < this->size(); count++)
          {
            this->assign(count, vec.at(count));
          }

      }
    ;

    /**
     * cleans up
     */
    ~DenseVector<storageType>()
      {
        this->destroy();
      }
    ;

    /**
     * returns length of the vector
     * @return Length
     */
    size_t size(void) const
      {
        return this->length;
      }
    ;

    /** 
     * returns desired entry
     * @param no number of the entry
     * @return the entrie s value
     */
    storageType at(const unsigned int no) const
      {
        assert(no < this->size());

        return this->entries[no];
      }
    ;

    /**
     * sets desired entry to value
     * @param no number of desierd entry
     * @value the new value
     */
    void assign(const unsigned int no, const storageType value)
      {
        assert( no < this->size());

        entries[no] = value;
      }
    ;

    /**
     * compares this with another vector entry by entry
     * @param vec the other vector
     * @return true if equal, false elsewise
     */
    bool operator==(const BaseVector<storageType> &vec) const
      {
        assert(this->size() == vec.size());

        for (size_t index = 0; index < this->size(); index++)
          {
            if (absolute((this->at(index)) - vec.at(index)) >= MIN_DIFFERENCE)
              {
                return false;
              }
          }
        return true;
      }
    ;

protected:

    /**
     * must be implemented acc. to BaseVector
     */
    void destroy(void)
      {
        delete[] this->entries;
        this->entries = 0;
      }
    ;

    /**
     * must be implemented acc. to BaseVector
     */
    void resize(size_t n)
      {
        assert(entries == 0);

        entries = new storageType[n];
        length = n;
      }
    ;

    storageType *entries;

    size_t length;
  };

}
;
#endif
