/*
# Copyright (C) 2009-2011
# Raffaele Granito <raffaele.granito@tiscali.it>
#
# This file is part of GnuLibXML :
# Other XML Library ;)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "libSpiderXML.h"

/*
.-------------------------------------------------------------.
|                                                             |                                                          
|  funzion : parser                                           |
|  remark  : This is XML parser.                              |
|  input   : fileXML, debug_level {0..5}                      |
|  output  : rPharse.return_code :                            |
|            [  0] Ok. The fileXML read is correct            |
|            [ -1] fileXML not found or you havent permission |  
|            [ -2] overflow buffering [DIM_BUFFER]            |
|            [ -3] found a remark into delimiter tag          |
|            [ -4] unknown token on left the remark           | 
|            [ -5] unknown token on left the XML delimiter    |
|            [ -6] initial delimiter xml not found            | 
|            [ -7] unknown token on left the XML delimiter    |
|            [ -8] unknown token on left the XML delimiter    |
|            [ -9] found more tag delimiter                   |
|            [-10] not found close delimiter for tag          |
|            [-11] a previous definition tag is open          |
|            [-12] a previous tag is not close                |
|            [-13] found another root node                    |
|            [-14] unknown token                              |
|            [-15] a previous definition tag xml is open      | 
|            [-16] found TAG XML after DATE TAG.              |
|            [-17] unknown token                              |
|            [-18] unknown token                              |
|            [-19] found tail tag but there arent heads.      |
|            [-20] a previous head tag [%s] is not close      |
|            [-21] a previous tail tag [%s] is not close      |
|            [-22] delimiter tail tag but missing father      |
|            [-23] not found open delimiter remark            |
|            [-24] i not fount tag XML first row but remark   |
|            [-25] found a remark into delimiter tag XML      |
|            [-26] unknown token                              |
|            [-27] ???                                        |
|            [-28] ???                                        |
|            [-29] ???                                        |
|            [-30] XML file is EMPTY                          |
|            [-31] It isnt XML file. Not found TAG.           |
|            [-32] XML File incomplete - TAG XML not Close    |
|            [-33] XML File incomplete - TAG not Close        |
|            [-34] Not found open delimiter tail TAG          |
|            [-35] found tag delimiter close but no tag       |
|            [-36] the three is complete.                     | 
|            [-37] not found ASSIGN but unknown token         |
|            [-38] not found ASSIGN but unknown token XML     | 
|            [-39] not found ASSIGN but unknown token         |
|            [-40] not found XML head.                        |
|            [-41] unknown token                              |
|            [-42] not found ASSIGN but unknown token         |
|            [-43] the TAG close have not a name.             |
|            [-44] the TAG close have variable not definition |
|            [-45] missing intestation XML                    |    
|            [-46] Last TAG XML isnt close                    |
|            [-47] delimiter close but there isnt tag         | 
|            [-48] delimiter close but there isnt tag open    |
|            [-49] missing delimiter close for tag [%s]       |
|            [-50] missing tag name                           |
|            [-51] missing assign and value for value.        |
|            [-52] missing value for value.                   |
|            [-53] unknown token on left the XML delimiter    |
|            [-54] found more tag delimiter                   |
|            [-55] not found close delimiter for tag          |
|            [-56] not found TAG definition XML before assign |
|            [-57] found token assign. Last TAG is TAIL close |
|            [-58] last tag found have not a name.            |
|            [-59] not found var for token assign             |
|            [-60] *Unknown Token* [%s] left token assign     |
|            [-61] replicate token assign for var [%s]        |
|            [-62] last tag found have not a name.            |
|            [-63] not found var for token assign             |
|            [-64] *Unknown Token* [%s] left token assign     |
|            [-65] duplicate token assign for var [%s]        |
|            [-66] found token assign. Last TAG is HEAD close |
|            [-67] found token assign. Last TAG is TAIL open  |
|            [-68] found token assign. Last TAG is TAIL close |
|            [-69] not found XML head.                        |
|            [-70] Last TAG XML isnt close [%s]               |
|            [-71] Found delimiter close but no TAG open.     |
|            [-72] delimiter close but TAG havent name.       |
|            [-73] delimiter close but missing assign var [%s]|
|            [-74] delimiter close but missing value var [%s] |
|            [-75] delimiter close but missing assign var [%s]|
|            [-76] delimiter close but missing value var [%s] |
|            [-77] Found delimiter close but no TAG open.     |
|            [-78] unknown token on left the XML delimiter    |
|            [-79] found more tag delimiter                   |
|            [-80] not found close delimiter for tag          |
|                                                             | 
`-------------------------------------------------------------' */ 
tPharse pharseXML ( char *fileXML, int debug_level ) 
{

    FILE *fp                 ;   
    char carattere[1]        ;   
    char buffer[1024]        ;
    int  i, j            = 0 ;
    int  countRowXMLFile = 0 ;

    /*
    .-------------.
    | Event Flags |
    '-------------' */
    int  foundToken   = 0;
    int  remark       = 0;
    int  ignore       = 0;
    

    /*
    .---------------.
    | Tabella TOKEN |
    '---------------' */
    typedef struct tToken {
            char *type  ;
            char *value ;
    } tToken;

    tToken token[100];


    /*
    .-----------------------.
    | Return Struct Funtion |
    '-----------------------' */
    tPharse rPharse; 

    /*
    .---------------------------.
    | Max Dimension for a Token | 
    '---------------------------' */
    #define __DIM_BUFFER__         1024
 
    /*
    .---------------.
    | Generic Alias |
    '---------------' */ 
    #define __ZERO__                  0
    #define __ONE__                   1
    #define __NULL__                  0
    #define __TAB__                   9
    #define __LF__                   10
    #define __BLANK__                32          
    
    #define __ON_TO_OFF__             2
    #define __ON__                    1
    #define __OFF__                   0

    #define __TRUE__                  1
    #define __FALSE__                 0

    /*   
    .------------------.
    | Delimitatori XML | 
    '------------------' */
    #define __TAG_INI_REMARK__    "<!--"
    #define __TAG_END_REMARK__     "-->"
    #define __TAG_INI_XML__         "<?"  
    #define __TAG_END_XML__         "?>"

    #define __TAG_INI_OPEN__         "<"
    #define __TAG_INI_OPEN2__        '<' 
    #define __TAG_END_OPEN__        "</"
    #define __TAG_INI_END__         "/>"
    #define __TAG_END__              ">"  

    #define __TAG_ASSIGN__           "="
    #define __TAG_SEPARATORE__       " "

    /*    
    .-----------------------.
    | Apre il file di input |
    '-----------------------' */
    fp = fopen(fileXML, "r");
    if (!fp) {
        rPharse.return_code = -1;
        return rPharse;
    };

    /*
    .---------------------------.
    | Analizza il file di input |
    '---------------------------' */
    buffer[0]  = __NULL__  ; 
    i          = __ZERO__  ;
    foundToken = __FALSE__ ;
    remark     = __OFF__   ;
    while (fread(carattere, 1, sizeof(carattere), fp)) 
    {      
          /*
          .--------------------------------------------------------.
          | Il parsen XML ignora: LF(10), TAB(9) e DOPPI BLANK(32) |
          '--------------------------------------------------------' */
          ignore = __OFF__ ;

          if ( carattere[0] == __LF__  ) {
               countRowXMLFile++;
               ignore = __ON__ ;
          };

          if ( carattere[0] == __TAB__ ) 
               ignore = __ON__ ;

          if ( i == 0 && carattere[0] == __BLANK__ ) 
               ignore = __ON__ ;

          if ( i > 0 ) {
               if ( carattere[0] == __BLANK__ && buffer[i-1] == __BLANK__ ) {
                    ignore = __ON__ ; }; };  



          /*
          .------------------------------------------.
          | Bufferizzazione per individuazione token |
          '------------------------------------------' */
          if ( foundToken == __FALSE__ && ignore == __OFF__ )
          {
              buffer[i]=carattere[0];
              buffer[i+1] = 0;
              i++; 

              /*
              .---------------------------------------------------------.
              | Gestione OVERFLOW Buffer                                |
              .---------------------------------------------------------.
              | Se il parsen ha bufferizzato 1k lo svuota salvando  gli |
              | ultimi 4 byte se si è in un commento. Tali byte salvati | 
              | potrebbero in teoria contenere l'inizio del tab di fine |
              | commento ("-->\0"|"?--\0"|"??-\0"). Se non si è  in  un |
              | commento il parsen si blocca, è sicuramente  un  errore |
              | se dopo 1k non si è terminato nessun token anche se  lo |
              | standard XML non prevede limiti di lunghezza ad esempio |
              | per i nomi dei tag.                                     | 
              '---------------------------------------------------------' */ 
              if ( i == __DIM_BUFFER__ ) 
              { 
                   /*
                   .------------------------------------------------------.
                   | Si è in un commento perchè il flag "remark" è ad ON. |
                   | In questo caso il token è classificato come Remark   |
                   | ed il buffer non viene svuotato completamente. Si    |
                   | salvano gli ultimi 4 byte per non perdersi un        |
                   | eventuale fine commento parziale.                    |
                   '------------------------------------------------------' */
                   if (remark == __ON__) 
                   {
                       token[foundToken].type  = malloc(strlen("Remark\0"));
                       token[foundToken].value = malloc(strlen(buffer));
                       strcpy(token[foundToken].type  , "Remark\0");
                       strcpy(token[foundToken].value , buffer) ;
                       foundToken++;
     
                       buffer[3] = buffer[i]   ; 
                       buffer[2] = buffer[i-1] ; 
                       buffer[1] = buffer[i-2] ; 
                       buffer[0] = buffer[i-3] ;
                       i = 3 ; 
                   };
                   
                   /*
                   .-------------------------------------------.
                   | Se non si è in un commento il superamento |
                   | del __DIM_BUFFER__ viene considerato un   |
                   | errore malgrado lo standard XML che non   |
                   | pone limiti.                              |
                   '-------------------------------------------' */
                   if (remark == __OFF__) { 
                       rPharse.return_code = -2; 
                       return rPharse;
                   }; 
              };


                
              /*
              .-------------------------------------------------------------------------------------.
              |                                I Commenti in  X M L                                 |
              .-------------------------------------------------------------------------------------.
              | XML prevede la possiblità di inserire commenti. Questi devono essere racchiusi dai  |  
              | seguenti delimitatori: "<!--" e "-->". Si pi può commentare il singolo nodo: radice |
              | ramo o foglia, purchè questo sia definito al di fuori di esso. Il commento legato   |
              | al nodo radice lo si può considerare come commento del documento XML. Il commento   |
              | Il commento può essere inserito anche per descrivere i TAG XML con l'eccezione che  |
              | un file XML non può iniziare con un commento. La prima occorrenza è l'intestazione  |
              | XML (<?xml ... ?>).                                                                 |
              '-------------------------------------------------------------------------------------' */

              /*
              .-------------------------------------------------------------------------------------.
              |                                                                                     |
              |                                                                                     | 
              |                                                                                     |
              |                                __TAG_INI_REMARK__                                   |
              |                                  inizio  Remark                                     | 
              |                                                                                     |
              |                                                                                     | 
              |                                                                                     |
              '-------------------------------------------------------------------------------------' */
              if ( search(buffer,__TAG_INI_REMARK__) == __TRUE__ ) 
              { 
                   /*
                   .-------------------------------------------------------------.
                   | Il commento in XML non può essere inserito prima del TAG    |
                   | di intestazione del documento. E' un errore quindi se viene |
                   | trovato un inizio commento a stack XML ancora vuoto.        |
                   '-------------------------------------------------------------' */
                   if (rPharse.iTabTagXML == __ZERO__) {
                       rPharse.row_error = countRowXMLFile + 1;
                       rPharse.token_unknown = malloc(strlen(__TAG_INI_REMARK__));
                       strcpy(rPharse.token_unknown,__TAG_INI_REMARK__);
                       rPharse.return_code = -24;
                       return rPharse; };

                   /*
                   .-------------------------------------------------------------.
                   | Il commento non può iniziare all'interno dei delimitatori   |
                   | di un tag di apertura o di un tag di chiusura XML.          |
                   | esempio di errore: <?xml... <!-- ...                        |
                   '-------------------------------------------------------------' */
                   if (rPharse.rTabTagXML[rPharse.iTabTagXML-1].stato == 0 || 
                       rPharse.rTabTagXML[rPharse.iTabTagXML-1].stato == 2) {
                       rPharse.row_error = countRowXMLFile + 1;
                       rPharse.token_unknown = malloc(strlen(rPharse.rTabTagXML[rPharse.iTabTagXML-1].tag));
                       strcpy(rPharse.token_unknown,rPharse.rTabTagXML[rPharse.iTabTagXML-1].tag);
                       rPharse.return_code = -25;
                       return rPharse; };

                   /*
                   .-------------------------------------------------------------.
                   | Il commento non può iniziare all'interno dei delimitatori   |
                   | di un tag di apertura o di un tag di chiusura.              |
                   | esempio di errori: [0] <tag <!-- ... , [2] </tag <!-- ...   |
                   '-------------------------------------------------------------' */ 
                   if (rPharse.iTabTag > __ZERO__) {
                       if (rPharse.rTabTag[rPharse.iTabTag-1].stato == 0 || rPharse.rTabTag[rPharse.iTabTag-1].stato == 2) {
                           rPharse.row_error = countRowXMLFile + 1;
                           rPharse.token_unknown = malloc(strlen(rPharse.rTabTag[rPharse.iTabTag-1].tag));
                           strcpy(rPharse.token_unknown,rPharse.rTabTag[rPharse.iTabTag-1].tag);
                           rPharse.return_code = -3; 
                           return rPharse; };};

                   /*
                   .-----------------------------------------------------.
                   | Estrapola la parte alla sinistra del TAG di inizio  |
                   | Remark. In teoria non ci dovrebbe essere nulla alla |
                   | sinistra se il file XML è corretto. Qualsiasi altro |
                   | delimitatore sarebbe stato intercettato già.        |
                   | E prima del delimitatore del commento non ci può    |
                   | essere che un altro delimitatore.                   |
                   '-----------------------------------------------------' */
                   i = i - strlen(__TAG_INI_REMARK__);
                   buffer[i] = __NULL__; 

                   /*
                   .------------------------------------------------.
                   | Questo è il caso in cui è stato trovato un     |
                   | qualcosa alla sinistra del delimitatore di     |
                   | Remark. Il parsen esce.                        |
                   '------------------------------------------------' */
                   if (i > __ZERO__) {
                       rPharse.row_error = countRowXMLFile + 1;
                       rPharse.token_unknown = malloc(strlen(buffer));
                       strcpy(rPharse.token_unknown,buffer);
                       rPharse.return_code = -4;
                       return rPharse; };

                   /* 
                   .-------------------------------------------.
                   | Attiva il flag "remark" ad __ON__ questo  |
                   | significa che da questo momento e fino a  |
                   | quando non si trova il token di chiusura  |
                   | ci troviamo in blocco commentato.         |
                   '-------------------------------------------' */
                   remark    = __ON__  ;

                   /*
                   .----------------------------------------------.
                   | Classfica e registra il token trovato        |
                   '----------------------------------------------' */
                   token[foundToken].type  = malloc(strlen("RemarkIni\0"));
                   token[foundToken].value = malloc(strlen(__TAG_INI_REMARK__));
                   strcpy(token[foundToken].type  , "RemarkIni");
                   strcpy(token[foundToken].value , __TAG_INI_REMARK__) ;
                   foundToken++;
              }; 

              /*
              .-------------------------------------------------------------------------------------.
              |                                                                                     |              
              |                                                                                     |
              |                                                                                     |
              |                                __TAG_END_REMARK__                                   |
              |                                   Fine  Remark                                      |
              |                                                                                     |
              |                                                                                     | 
              |                                                                                     |
              '-------------------------------------------------------------------------------------' */
              if ( search(buffer,__TAG_END_REMARK__) == __TRUE__ ) { 

                   /*
                   .---------------------------------------------------------.
                   | Il delimitatore di fine commento deve essere preveduto  |
                   | da un delimitatore di inizio commento che imposta il    |
                   | flag "remark" ad __ON__. Se questo non è vero il parsen |
                   | si blocca e segnala l'errore.                           |
                   '---------------------------------------------------------' */
                   if ( remark != __ON__ ) {
                        rPharse.row_error = countRowXMLFile + 1;
                        rPharse.token_unknown = malloc(strlen(__TAG_END_REMARK__));
                        strcpy(rPharse.token_unknown,__TAG_END_REMARK__);
                        rPharse.return_code = -23;
                        return rPharse; };
                   
                   /*
                   .---------------------------------------------------------.
                   | Isolamento della stringa alla sinistra del delimitatore |
                   '---------------------------------------------------------' */     
                   i = i - strlen(__TAG_END_REMARK__);
                   buffer[i] = __NULL__;       

                   /*
                   .------------------------------------------------------------.
                   | Valorizza il flag "remark" in uno stato intermedio, quindi |
                   | non lo abbassa ancora per effettuale l'ultimo ciclo        |
                   | completo.                                                  |
                   '------------------------------------------------------------' */ 
                   remark = __ON_TO_OFF__;

                   /*
                   .----------------------------------------------------. 
                   | La parte alla sinistra del token, se esiste, non   |
                   | può che essere una parte del commento e pertanto   |
                   | viene classificata in questo modo.                 |
                   '----------------------------------------------------' */
                   if (i > __ZERO__) {
                       token[foundToken].type  = malloc(strlen("Remark\0"));
                       token[foundToken].value = malloc(strlen(buffer));
                       strcpy(token[foundToken].type  , "Remark\0");
                       strcpy(token[foundToken].value , buffer);
                       foundToken++; 
                   };

                   /*
                   .----------------------------------------------------.
                   | Viene classificato anche il token di fine commento |
                   '----------------------------------------------------' */
                   token[foundToken].type  = malloc(strlen("RemarkEnd\0"));
                   token[foundToken].value = malloc(strlen(__TAG_END_REMARK__));
                   strcpy(token[foundToken].type  , "RemarkEnd"); 
                   strcpy(token[foundToken].value , __TAG_END_REMARK__) ;
                   foundToken++;
              }; 



              /*
              .-------------------------------------------------------------------------------------.
              |                         I n t e s t a z i o n i    X M L                            | 
              .-------------------------------------------------------------------------------------.
              | I file XML devono avere un intestazione ben specifica prima di qualsiasi altra cosa |
              | (commento, struttura dati). Questa prima riga ha il seguente formato:               |
              | <?xml [var-1=[value-1]] ... [var-N=[value-N]] ?> - "xml" è il nome di questo primo  |
              | TAG obbligatorio, ed è anche la parola chiave non utilizzabile in nessun modo nella |
              | nella struttura dati per nominare i tag utente. Oltre a questo primo tag che        |
              | rappresenta l'intestazione di un documento XML potrebbero seguire altri tag con il  |
              | prefisso xml sempre racchiuso tra i delimitatori <? ... ?>. Non sono obbligatori e  |
              | <? ... ?>. Non sono obbligatori e non verranno al momento verficati anche se        |
              | analizzati e memorizzati dal pharsen.                                               |
              '-------------------------------------------------------------------------------------' */

              /*
              .-------------------------------------------------------------------------------------.
              |                                                                                     |
              |                                                                                     |
              |                                                                                     |
              |                                   __TAG_INI_XML__                                   |
              |                                                                                     |
              |                                                                                     |
              |                                                                                     | 
              |                                                                                     |
              '-------------------------------------------------------------------------------------' */
              if ( search(buffer,__TAG_INI_XML__) == __TRUE__ && remark == __OFF__ && foundToken == __FALSE__ ) {

                   /*
                   .---------------------------------------------------------.
                   | Isolamento della stringa alla sinistra del delimitatore |
                   '---------------------------------------------------------' */
                   i = i - strlen(__TAG_INI_XML__);
                   buffer[i] = __NULL__; 
                   
                   /*
                   .----------------------------------------------------.
                   | Prima di un delimitatore iniziale XML (<?) non può |
                   | esserci nessun token sconosciuto. L'unico permesso |
                   | è il delimitatore finale XML (oppure forse e se    |
                   | si tratta di intestazione del documeto da un token |
                   | di fine commento (-->)  ma questo sarebbe stato    |
                   | intercettato dal pharsen prima. Questa casistica è |
                   | considerata quindi un errore.                      |
                   '----------------------------------------------------' */    
                   if ( i > __ZERO__ ) {
                        rPharse.row_error = countRowXMLFile + 1;
                        rPharse.token_unknown = malloc(strlen(buffer));
                        strcpy(rPharse.token_unknown,buffer);
                        rPharse.return_code = -5;
                        return rPharse; };

                   /*
                   .---------------------------------------------------.
                   | Non ci può essere un'altro TAG XML aperto.        |
                   | Per effettuare questa verifica: [1] ricerca nella |
                   | stack XML l'ultimo inserito e, [2] verifica che   |
                   | lo stato di quest'ultimo sia impostato 3 (chiuso) |
                   '---------------------------------------------------' */
                   if ( rPharse.iTabTagXML > 0 ) {
                        if ( rPharse.rTabTagXML[rPharse.iTabTagXML-1].stato != 3 ) {
                             rPharse.row_error = countRowXMLFile + 1;
                             rPharse.token_unknown = malloc(strlen(rPharse.rTabTagXML[rPharse.iTabTagXML-1].tag));
                             strcpy(rPharse.token_unknown,rPharse.rTabTagXML[rPharse.iTabTagXML-1].tag);
                             rPharse.return_code = -15;
                             return rPharse;
                        };
                   };

                   /*
                   .---------------------------------------------------.
                   | I tag XML devono trovarsi nel documento in testa  |
                   | prima della struttura dati. Se durante l'analisi  |
                   | di un nuovo TAG XML invece lo stack dei TAG è     |
                   | pieno (con almeno un elemento) significa che il   |
                   | tag XML è posizionato al centro del documento e   |
                   | quindi è un errore ed il pharser esce.            |
                   '---------------------------------------------------' */
                   if ( rPharse.iTabTag > 0 ) {
                        rPharse.row_error = countRowXMLFile + 1;
                        rPharse.token_unknown = malloc(strlen(rPharse.rTabTag[rPharse.iTabTag-1].tag));
                        strcpy(rPharse.token_unknown,rPharse.rTabTag[rPharse.iTabTag-1].tag);
                        rPharse.return_code = -16;
                        return rPharse;
                   };

                   /*
                   .-------------------------------------------------.
                   | Classifica il token                             |
                   '-------------------------------------------------' */
                   token[foundToken].type  = malloc(strlen("XMLIni\0"));
                   token[foundToken].value = malloc(strlen(__TAG_INI_XML__)); 
                   strcpy(token[foundToken].type  , "XMLIni\0"); 
                   strcpy(token[foundToken].value , __TAG_INI_XML__);
                   foundToken++;

                   /*
                   .-----------------------------------------------.
                   | Inserimento di un nuovo TAG XML nello stack   |
                   | Non avendoci ancora il nome del TAG inserisce |
                   | un nome temporaneo "Unknown".                 | 
                   '-----------------------------------------------' */
                   rPharse.rTabTagXML[rPharse.iTabTagXML].tag = malloc(strlen("*Unknown*"));
                   strcpy(rPharse.rTabTagXML[rPharse.iTabTagXML].tag, "*Unknown*");
                   rPharse.rTabTagXML[rPharse.iTabTagXML].stato = 0;
                   rPharse.rTabTagXML[rPharse.iTabTagXML].stato_last_variable = 0;
                   rPharse.rTabTagXML[rPharse.iTabTagXML].dimTabVar = 0;
                   rPharse.rTabTagXML[rPharse.iTabTagXML].level = 0;
                   rPharse.iTabTagXML++;


              };

              /*
              .-------------------------------------------------------------------------------------.
              |                                                                                     |
              |                                                                                     |
              |                                                                                     |
              |                                   __TAG_END_XML__                                   |
              |                                                                                     |
              |                                                                                     |
              |                                                                                     |
              |                                                                                     |
              '-------------------------------------------------------------------------------------' */
              if ( search(buffer,__TAG_END_XML__) == __TRUE__ && remark == __OFF__ && foundToken == __FALSE__ ) {

                   /*
                   .-----------------------------------------------------------------------------------.
                   | Un delimitatore di chiusura di un tag xml deve sempre essere preceduto            |
                   | da un delimitatore di apertura (<? ... ?>). Questo quando viene analizzato        |
                   | genera l'inserimento di una nuova occorrenza nello stack TAGXML anche se          |
                   | non è conosciuto in alcuni casi ancora il nome del TAG che viene momentaneamente  | 
                   | impostato a "Unknown" e sostituito solo successivamente con quello reale.         |
                   | Quindi se nello stack XML l'ultima TAG inserito non ha stato a __ZERO__ (aperto)  |
                   | oppure lo stack dei TAGXML è vuoto significa che in precedenza non è stato aperto |
                   | alcun TAGXML e quindi il parse esce con errore.                                   |
                   '-----------------------------------------------------------------------------------' */
                   if (rPharse.iTabTagXML == 0 || 
                      (rPharse.iTabTagXML  > 0 && rPharse.rTabTagXML[rPharse.iTabTagXML-1].stato != 0)) { 
                       rPharse.row_error = countRowXMLFile + 1;
                       rPharse.token_unknown = malloc(strlen("\0")); 
                       strcpy(rPharse.token_unknown,"\0");
                       rPharse.return_code = -6;
                       return rPharse; };
                   /*
                   .---------------------------------------------------------.
                   | Verifica se c'è qualcosa alla sinistra del delimitatore |
                   | di fine tag xml. I soliti token ammessi sono: il tag    |
                   | e la value di una varibile. In tutti gli altri casi si  |
                   | tratta di un errore.                                    |
                   '---------------------------------------------------------' */ 

                   /*
                   .---------------------------------------------------------.
                   | Isolamento della stringa alla sinistra del delimitatore |
                   '---------------------------------------------------------' */
                   i = i - strlen(__TAG_END_XML__);
                   buffer[i] = __NULL__;
                   if ( i > __ZERO__ ) 
                   {

                        /*
                        .--------------------------------------------------------------.
                        | CASO 1: Tag (es. <? [tagXML]?>)                              |
                        .--------------------------------------------------------------.
                        |                                                              |
                        |                                                              |
                        |                                                              |
                        |                                                              |
                        |                                                              |
                        '--------------------------------------------------------------'                              
                        .---------------------------------------------------------------------.
                        | Alla sinistra del delimitatore di chiusura (?>) del TAG XML come    |
                        | scritto in precedenza ci si aspetta: nome tag oppure valore di una  |
                        | variabile. Come riconoscere l'una dall'altra? Si tratta di nome TAG |
                        | se questo nello stack  è impostato a "*Unknown*" nome temporaneo    |
                        | inserito quando si è analizzato il delimitatore di apertura e non   |
                        | si conosceva il nome TAG quindi se il nome è ancora impostato così  |
                        | vuol dire che non lo si ha ancora incontrato e quindi quello che si | 
                        | sta in quetso momento è proprio quello. E' il "value" di una        |
                        | variabile se lo stato_last_variabile è ad 2.                        | 
                        '---------------------------------------------------------------------' */ 
                        if ( strcmp(rPharse.rTabTagXML[rPharse.iTabTagXML-1].tag,"*Unknown*") == 0 ) 
                        {
                                /* 
                                .----------------------------------------------.
                                |  Classificazione e valorizzazione del Token  |
                                '----------------------------------------------' */
                                token[foundToken].type  = malloc(strlen("XML\0"));
                                strcpy(token[foundToken].type  , "XML\0");
                                token[foundToken].value  = malloc(strlen(buffer));
                                strcpy(token[foundToken].value , buffer);

                                /*  
                                .----------------------------------------------.
                                |          Aggiornamento nome TAG XML          |
                                '----------------------------------------------' */
                                rPharse.rTabTagXML[rPharse.iTabTagXML-1].tag = malloc(strlen(buffer));
                                strcpy(rPharse.rTabTagXML[rPharse.iTabTagXML-1].tag, buffer);
                        }
                        else
                        {
                             switch (rPharse.rTabTagXML[rPharse.iTabTagXML-1].stato_last_variable)
                             {
                                case 0: 

                                    /*
                                    .--------------------------------------------------------------.
                                    | CASO 2: Token Unknown (es. <?tagXML token_unknown?>)         |
                                    .--------------------------------------------------------------.
                                    |                                                              |
                                    |                                                              |
                                    |                                                              |
                                    |                                                              |
                                    |                                                              |
                                    '--------------------------------------------------------------' */
                                    rPharse.row_error = countRowXMLFile + 1;
                                    rPharse.token_unknown = malloc(strlen(buffer));
                                    strcpy(rPharse.token_unknown,buffer);
                                    rPharse.return_code = -41;
                                    return rPharse;
                                    break;


                                case 1: 

                                    /*
                                    .--------------------------------------------------------------------.
                                    | CASO 3: Token Unknown - Missing ASSIGN (es. <?tagXML var [word]?>) |
                                    .--------------------------------------------------------------------.
                                    |                                                                    |
                                    |                                                                    |
                                    |                                                                    |
                                    |                                                                    |
                                    |                                                                    |
                                    '--------------------------------------------------------------------' */
                                    rPharse.row_error = countRowXMLFile + 1;
                                    rPharse.token_unknown = malloc(strlen(buffer));
                                    strcpy(rPharse.token_unknown,buffer);
                                    rPharse.return_code = -42;
                                    return rPharse;
                                    break;



                               case 2: 

                                    /*
                                    .--------------------------------------------------------------.
                                    | CASO 4: Value (es. <?tagXML var = [value]?>)                 |
                                    .--------------------------------------------------------------.
                                    |                                                              |
                                    |                                                              |
                                    |                                                              |
                                    |                                                              |
                                    |                                                              |
                                    '--------------------------------------------------------------' */

                                    /*
                                    .----------------------------------------------.
                                    |  Classificazione e valorizzazione del Token  |
                                    '----------------------------------------------' */
                                    token[foundToken].type = malloc(strlen("Value\0"));
                                    strcpy(token[foundToken].type,"Value\0");
                                    token[foundToken].value = malloc(strlen(buffer));
                                    strcpy(token[foundToken].value,buffer);
 
                                    /*
                                    .----------------------------------------------------.
                                    | Aggiornamento del valore per l'ultima variabile    |
                                    | memorizzata con stato a 1 (mancanza di valore)     |
                                    '----------------------------------------------------' */
                                    int istack = rPharse.iTabTagXML-1; 
                                    int istackVar = rPharse.rTabTagXML[istack].dimTabVar; 
                                    rPharse.rTabTagXML[istack].rTabVar[istackVar].value = malloc(strlen(buffer)); 
                                    strcpy(rPharse.rTabTagXML[istack].rTabVar[istackVar].value, buffer);
                                    rPharse.rTabTagXML[rPharse.iTabTagXML-1].dimTabVar++;

                                    /*
                                    .----------------------------------------------------------.
                                    | Aggiornamento dello stato dell'ultima variabile inserita |
                                    | al valore di __ZERO__ conclusa definizione.              |
                                    '----------------------------------------------------------' */
                                    rPharse.rTabTagXML[rPharse.iTabTagXML-1].stato_last_variable = 0; 
                                    break;
                             };

                        }; 
                           
                        foundToken++;
                   };


                   /* 
                   .-----------------------------------------------------------.
                   | Controlla però che *SIA* stato già assegnato un nome      |
                   | definitivo al TAG prima di classificare il Token come     |
                   | "Chiusura TAGXML". In questo modo si evita di far passare |
                   | questa casistica <??>, <? ?>.                             |
                   '-----------------------------------------------------------' */
                   if (strcmp(rPharse.rTabTagXML[rPharse.iTabTagXML-1].tag,"*Unknown*")==0) {
                       rPharse.row_error = countRowXMLFile + 1;
                       rPharse.token_unknown = malloc(strlen("\0")); 
                       strcpy(rPharse.token_unknown,"\0");
                       rPharse.return_code = -43;
                       return rPharse; };

                   /*
                   .-----------------------------------------------------------.
                   | Controlla prima di chiudere il TAGXML che la situazione   |
                   | delle variabili sia consolidata stato_last_variabile == 0 |
                   | Se così non fosse è presente una definizione non conclusa |
                   | In questo modo si evitano situazioni di questo tipo:      |
                   | <?xml var?>, <?xml var=?>                                 |
                   '-----------------------------------------------------------' */
                   if (rPharse.rTabTagXML[rPharse.iTabTagXML-1].stato_last_variable != 0) {
                       rPharse.row_error = countRowXMLFile + 1;
                       int istack = rPharse.iTabTagXML-1; 
                       int istackVar = rPharse.rTabTagXML[istack].dimTabVar; 
                       rPharse.token_unknown = malloc(strlen(rPharse.rTabTagXML[istack].rTabVar[istackVar].var)); 
                       strcpy(rPharse.token_unknown,rPharse.rTabTagXML[istack].rTabVar[istackVar].var);
                       rPharse.return_code = -44;
                       return rPharse; };



                   /*
                   .----------------------------------------------------------------.
                   | Modifica lo stato da 0 a 3 per il TAG analizzato perchè chiuso |
                   '----------------------------------------------------------------' */
                   rPharse.rTabTagXML[rPharse.iTabTagXML-1].stato = 3;

                   /*
                   .----------------------------------------------.
                   |  Classificazione e valorizzazione del Token  |
                   '----------------------------------------------' */
                   token[foundToken].type  = malloc(strlen("XMLEnd\0"));
                   strcpy(token[foundToken].type  , "XMLEnd\0");
                   token[foundToken].value = malloc(strlen(__TAG_END_XML__));
                   strcpy(token[foundToken].value , __TAG_END_XML__) ;
                   foundToken++;
              };



              /*
              .-------------------------------------------------------------------------------------.
              |                                                                                     |
              |                                                                                     |
              |                                                                                     |
              |                                  __TAG_END_OPEN__                                   |
              |                                                                                     |
              |                                                                                     |
              |                                                                                     |
              |                                                                                     |
              '-------------------------------------------------------------------------------------' */
              if ( search(buffer,__TAG_END_OPEN__) == __TRUE__ && remark == __OFF__ && foundToken == __FALSE__ ) 
              {
                   /*
                   .---------------------------------------------------------.
                   | Isolamento della stringa alla sinistra del delimitatore |
                   '---------------------------------------------------------' */
                   i = i - strlen(__TAG_END_OPEN__);
                   buffer[i] = __NULL__;

                   /*
                   .----------------------------------------------------.
                   | Prima di un delimitatore iniziale del TAG di       |
                   | chiusura <\tag> non può esserci nessun token       |
                   | conosciuto non ancora analizzato. Infatti prima di |
                   | questo si può trovare il delimitatore finale del   |
                   | tag di chiusura di un figlio (>) o il delimitatore |
                   | finale del corrispondente tag di apertura (>).     |
                   | Entrambi però vengono intercettati ed analizzati   |
                   | in anticipo per il metodo implementato da questo   |
                   | pharsen. Pertanto l'eventuale token alla sinistra  |
                   | del __TAG_END_OPEN è sconosciuto o non supportato  |
                   | Pertanto questa casistica è considerata un errore. |
                   '----------------------------------------------------' */
                   if ( i > __ZERO__ ) {
                        rPharse.row_error = countRowXMLFile + 1;
                        rPharse.token_unknown = malloc(strlen(buffer));
                        strcpy(rPharse.token_unknown,buffer);
                        rPharse.return_code = -18;
                        return rPharse; };

                   /*
                   .--------------------------------------------------.
                   | Sicuramente non esiste nessun TAG di apertura    |
                   | se lo stack è empty. The pharsen go out.         |
                   '--------------------------------------------------' */
                   if ( rPharse.iTabTag == __ZERO__ ) {
                        rPharse.row_error = countRowXMLFile + 1;
                        rPharse.token_unknown = malloc(strlen("\0"));
                        strcpy(rPharse.token_unknown,"\0");
                        rPharse.return_code = -19;
                        return rPharse; };

                   /*
                   .-----------------------------------------------------------------.
                   | Rilegge a ritroso lo stack per cercare il corrispondente TAG    |
                   | di apertura (stato = 1).  Lungo il traggitto se                 |
                   | incontra tag a livello 0 oppure 2 esce con errore perchè        |
                   | significa che un suo fratello, nipote o il padre non è concluso |
                   | o completamente definito. Se legge tutto lo stack senza trovare |
                   | nessuna corrispondenza (cioè in stato 2) quindi solo nodi       |
                   | completamente definiti (cioè in stato 3) esce per errore perchè |
                   | manca il corrispondente TAG di apertura.                        |
                   '-----------------------------------------------------------------' */
                   j = rPharse.iTabTag;
                   while ( j > 0 && rPharse.rTabTag[j-1].stato != 1 )
                   {
                           /*
                           .--------------------------------------------------.
                           | Trovato TAG di TESTA non chiuso quindi quello    |
                           | che si sta analizzando è definito all'interno    |
                           | es. <tag .... <\tag>                             |
                           '--------------------------------------------------' */
                           if ( rPharse.rTabTag[j-1].stato == 0 ) {
                                rPharse.row_error = countRowXMLFile + 1;
                                rPharse.token_unknown = malloc(strlen(rPharse.rTabTag[j-1].tag));
                                strcpy(rPharse.token_unknown,rPharse.rTabTag[j-1].tag);
                                rPharse.return_code = -20;
                                return rPharse; };

                           /*
                           .-------------------------------------------------.
                           | Trovato TAG di coda non chiuso quindi quello    |
                           | che si sta analizzando è definito all'interno   |
                           | es. <tag><\tag ... <\tag>                       |
                           '-------------------------------------------------' */
                           if ( rPharse.rTabTag[j-1].stato == 2 ) {
                                rPharse.row_error = countRowXMLFile + 1;
                                rPharse.token_unknown = malloc(strlen("\0"));
                                strcpy(rPharse.token_unknown,"\0");
                                rPharse.return_code = -21;
                                return rPharse; };

                           j--;
                   };

                   /*
                   .--------------------------------------------------.
                   | Tutti i nodi trovati sono in stato completamente |
                   | descritti ossia in stato 3 (es. <tag> ... <\tag> |
                   | Non trovato corrisponde TAG di apertura.         |
                   | il pharsen esce per errore.                      |
                   '--------------------------------------------------' */
                   if ( rPharse.iTabTag > 0 && rPharse.rTabTag[0].stato == 3 ) {
                        rPharse.row_error = countRowXMLFile + 1;
                        rPharse.token_unknown = malloc(strlen(buffer));
                        strcpy(rPharse.token_unknown,buffer);
                        rPharse.return_code = -22;
                        return rPharse; };

                   /*
                   .----------------------------------------------.
                   |  Classificazione e valorizzazione del Token  |
                   '----------------------------------------------' */
                   token[foundToken].type  = malloc(strlen("TagEndOpen\0"));
                   token[foundToken].value = malloc(strlen(__TAG_END_OPEN__));
                   strcpy(token[foundToken].type  , "TagEndOpen\0");
                   strcpy(token[foundToken].value , __TAG_END_OPEN__) ;
                   foundToken++;

                   /*
                   .---------------------------------------------------.
                   | Varia lo stato del TAG da 1 (Chiuso TAG Apertura) |
                   | a 2 (Aperto TAG Chiusura).                        | 
                   '---------------------------------------------------' */
                   rPharse.rTabTag[j-1].stato = 2;
              };



              /*
              .-------------------------------------------------------------------------------------.
              |                                                                                     |
              |                                                                                     |
              |                                                                                     |
              |                                   __TAG_INI_END__                                   |
              |                                                                                     |
              |                                                                                     |
              |                                                                                     |
              |                                                                                     |
              '-------------------------------------------------------------------------------------' */
              if ( search(buffer,__TAG_INI_END__) == __TRUE__ && remark == __OFF__ && foundToken == __FALSE__ ) 
              {
                   /*
                   .-----------------------------------------------------------------------------------.
                   | Un delimitatore di chiusura presuppone di un TAG aperto, ma prima ancora che      |
                   | esisti l'intestazione del file XML è piu in generale che sia popolato lo stack    |
                   | dei TAG XML. Se tale stack è vuoto significa che non solo non è presente il       |
                   | relativo TAG di apertura, ma che addirittura quello trovato è il primo token      |
                   | trovato nel file.                                                                 | 
                   '-----------------------------------------------------------------------------------' */
                   if (rPharse.iTabTagXML == 0) {  
                       rPharse.row_error = countRowXMLFile + 1;
                       rPharse.token_unknown = malloc(strlen("\0")); 
                       strcpy(rPharse.token_unknown,"\0");
                       rPharse.return_code = -45;
                       return rPharse; };

                   /*
                   .-----------------------------------------------------------------------------------.
                   | Verificato che lo stack XML contiene qualcosa è necessario verificare che lo      |
                   | stack XML sia consolidatoche significa verificare che l'ultimo TAG inserito sia   |
                   | regolarmente chiuso/concluso, cioè che il suo stato sia uguale a 3.               |
                   '-----------------------------------------------------------------------------------' */
                   if (rPharse.rTabTagXML[rPharse.iTabTagXML-1].stato != 3) { 
                       rPharse.row_error = countRowXMLFile + 1;
                       rPharse.token_unknown = malloc(strlen(rPharse.rTabTagXML[rPharse.iTabTagXML-1].tag)); 
                       strcpy(rPharse.token_unknown,rPharse.rTabTagXML[rPharse.iTabTagXML-1].tag);
                       rPharse.return_code = -46;
                       return rPharse; };

                   /*
                   .---------------------------------------------------------.
                   | Isolamento della stringa alla sinistra del delimitatore |
                   '---------------------------------------------------------' */
                   i = i - strlen(__TAG_INI_END__);
                   buffer[i] = __NULL__; 


                   /*
                   .-------------------------------------------------------------------------------------.
                   |                                                                                     | 
                   | Alla sinintra del token trovato è possibile che ci sia qualcosa. Potrebbe esserci : |
                   |                                                                                     | 
                   |                                                                                     |
                   | [1] la "value" di una variabile precedentemente dichiarata. es. <tag var = valore/> |
                   |     Il parser si aspetta di trovare una "value" nel caso in cui il TAG è aperto,    |
                   |     ha un nome assegnato ed una variabile definita ed in attesa di assegnazione del |
                   |     valore (stato_last_variabile == 2.                                              |
                   |                                                                                     |
                   | [2] il nome del TAG nel caso in cui ci sia una spazio tra il delimitatore di        |
                   |     apertura, come per esempio in questo caso < tag/>                               |
                   |                                                                                     |
                   | [3] delimitatore di apertura e nome TAG nel caso non ci siano spazi tra tale        |
                   |     delimitatore e nome TAG, come nell'esempio <tag/>                               |
                   |                                                                                     |
                   |                                                                                     |
                   '-------------------------------------------------------------------------------------' */

                   /*
                   .----------------------------------------------------------------------------.
                   | TROVATO QUALCOSA                                                           |
                   '----------------------------------------------------------------------------' */
                   if ( i  > __ZERO__ ) 
                   {

                        /*
                        .-----------------------------------------------------------.
                        | Se non si tratta di primo TAG e lo stato dell'ultimo TAG  |
                        | è ZERO quindi APERTO il parse                             |
                        | si aspetta di trovare: [1] il nometag se questo non è     |
                        | ancora assegnato assegnato (*Unknown*) [2] value se il    |
                        | nometag è assegnato (!*Unknown*) e lo stato ultima        |
                        | variabile acquisita è ugual a "2" (var + assign).         |
                        '-----------------------------------------------------------' */ 
                        if ( rPharse.rTabTag[rPharse.iTabTag-1].stato == 0 && rPharse.iTabTag > 0 )
                        {
                             if ( strcmp(rPharse.rTabTag[rPharse.iTabTag-1].tag,"*Unknown*") == 0 )
                             {
                                  /*
                                  .----------------------------------------------.
                                  |  Classificazione e valorizzazione del Token  |
                                  '----------------------------------------------' */
                                  token[foundToken].type = malloc(strlen("TagIni\0"));
                                  strcpy(token[foundToken].type, "TagIni\0");
                                  foundToken++;

                                  /*
                                  .----------------------------------------------.
                                  |           Aggiornamento nome TAG             |
                                  '----------------------------------------------' */
                                  rPharse.rTabTag[rPharse.iTabTag-1].tag = malloc(strlen(buffer));
                                  strcpy(rPharse.rTabTag[rPharse.iTabTag-1].tag, buffer);
                             }
                             else
                             {

                                  /*
                                  .-----------------------------------------------------------.
                                  | Lo stato è 0 (TAG di TESTA aperto) ed il nome del TAG     |
                                  | risulta assegnato e quindi alla sinistra del token in     |
                                  | analisi il parser considera che ci sia proprio il valore  |
                                  | di una variabile precedentemente inserita. Pertanto va    |
                                  | prima di tutto a verificare se nello stack ci sia almeno  |
                                  | una variabile inserita e che essa sia in stato 2 (attesa  |
                                  | di un valore). In caso contrario il parse si ferma per    |
                                  | errore.                                                   |
                                  '-----------------------------------------------------------' */

                                  /*
                                  .------------------------------------------------.
                                  | Stack variabili vuoto - errore - token unknown |
                                  '------------------------------------------------' */
                                  if ( rPharse.rTabTag[rPharse.iTabTag-1].dimTabVar == 0 && 
                                       rPharse.rTabTag[rPharse.iTabTag-1].stato_last_variable == 0) {
                                       rPharse.row_error = countRowXMLFile + 1;
                                       rPharse.token_unknown = malloc(strlen(buffer));
                                       strcpy(rPharse.token_unknown,buffer);
                                       rPharse.return_code = -17;
                                       return rPharse;
                                  }
 
                                  /*
                                  .-------------------------------------------------------------.
                                  | Lo stack ha almeno una variabile definita per l'ultimo TAG  |
                                  | inserito. Il parse a questo punto verifica che tale var sia |
                                  | in stato 2 (cioè in attesa di valore ("var ="). In caso     |
                                  | contrario (quindi se è in stato 0|1) si ferma segnalando    |
                                  | l'errore.                                                   |
                                  '-------------------------------------------------------------' */
                                  switch (rPharse.rTabTag[rPharse.iTabTag-1].stato_last_variable)
                                  {
                                          case 0: 
                                                  /*
                                                  .--------------------------------------------------------------.
                                                  | CASO 1: Value (es. <tag UnknownToken/>                       |
                                                  .--------------------------------------------------------------.
                                                  | Il parse si aspettava la value per l'ultima variabile        |
                                                  | presente nello stack, però questa non ha bisogno del valore  |
                                                  | perchè risulta già completa (var=value) e quindi il parse si |
                                                  | ferma e segnala l'errore.                                    |
                                                  |                                                              |
                                                  '--------------------------------------------------------------' */
                                                  rPharse.row_error = countRowXMLFile + 1;
                                                  rPharse.token_unknown = malloc(strlen(buffer));
                                                  strcpy(rPharse.token_unknown,buffer);
                                                  rPharse.return_code = -26;
                                                  return rPharse;
                                                  break;

                                          case 1: 
                                                  /*
                                                  .--------------------------------------------------------------.
                                                  | CASO 2: Value (es. <tag var UnknownToken/>)                  |
                                                  .--------------------------------------------------------------.
                                                  | Il parse si aspettava la value per l'ultima variabile        |
                                                  | presente nello stack, però questa non ha bisogno del valore  |
                                                  | ma dell'assign (var) e quindi il parse si ferma e segnala    |
                                                  | l'errore.                                                    |
                                                  |                                                              |
                                                  '--------------------------------------------------------------' */
                                                  rPharse.row_error = countRowXMLFile + 1;
                                                  rPharse.token_unknown = malloc(strlen(buffer));
                                                  strcpy(rPharse.token_unknown,buffer);
                                                  rPharse.return_code = -39;
                                                  return rPharse;
                                                  break;

                                          case 2: 
                                                  /*
                                                  .--------------------------------------------------------------.
                                                  | CASO 3: Value (es. <tag var = [value]>)                      |
                                                  .--------------------------------------------------------------.
                                                  | Il parse si aspetta la value per l'ultima variabile          |
                                                  | presente nello stack e realmente trova un ultima variabile   |
                                                  | in attesa di esso. Quindi assegna il value e cambia lo stato |
                                                  | della variabile nuovamente a zero.                           |
                                                  |                                                              |
                                                  '--------------------------------------------------------------' */

                                                  /*
                                                  .--------------------------------------------------------.
                                                  | Identificato e classificato la tipologia del token     |
                                                  '--------------------------------------------------------' */
                                                  token[foundToken].type  = malloc(strlen("Value\0"));
                                                  strcpy(token[foundToken].type  , "Value\0");
                                                  token[foundToken].value = malloc(strlen(buffer));
                                                  strcpy(token[foundToken].value , buffer);
                                                  foundToken++;

                                                  /*
                                                  .-------------------------------------------------------.
                                                  | Imposta il valore relativo alla variabile nello stack |
                                                  '-------------------------------------------------------' */
                                                  int istack = rPharse.iTabTag-1 ;
                                                  int istackVar = rPharse.rTabTag[istack].dimTabVar ;
                                                  rPharse.rTabTag[istack].rTabVar[istackVar].value = malloc(strlen(buffer));
                                                  strcpy(rPharse.rTabTag[istack].rTabVar[istackVar].value,buffer);
                                                  rPharse.rTabTag[istack].dimTabVar++;

                                                  rPharse.rTabTag[istack].stato_last_variable = 0 ;
                                                  break;

                                  } // end-if switch
                             }
                        }

                                  
                        /*
                        .-------------------------------------------------------------.
                        | Se si tratta di primo TAG.                                  |
                        |                                                             |
                        | Se lo stato dell'ultimo TAG acqusito è in stato UNO (1)     |
                        | (record di TESTA chiuso) è necessario capire se alla        |
                        | sinistra del token analizzato ci sia relativo delimitatore  |
                        | di sinistra e nome tag. In caso contrario è un errore.      |
                        | <tag-1>                                                     |
                        |       [<tag-2]/>                                            |
                        |                                                             |
                        | Se lo stato dell'utimo TAG è TRE (3) significa che l'ultimo |
                        | TAG trovato è completamente chiuso (es. <tag></tag> oppure  |
                        | <tag /> quindi il parser si aspetta di trovare alla         |
                        | del token analizzato l'apertura ed il nome tag              |
                        | <tag-1> ... </tag-1>                                        |
                        |[<tag-2]/>                                                   |
                        |                                                             |
                        | caso estremo da considerare è :                             |
                        | unknown_token<tag/>                                         |
                        '-------------------------------------------------------------' */ 
                        if (( rPharse.rTabTag[rPharse.iTabTag-1].stato == 0 && rPharse.iTabTag == 0 ) ||
                            ( rPharse.rTabTag[rPharse.iTabTag-1].stato == 1 )                         ||
                            ( rPharse.rTabTag[rPharse.iTabTag-1].stato == 3 ))

                        {
                            /*
                            .--------------------------------------------------------------.
                            | Alla sinistra del token />  ci  può essere un delimitatore   |
                            | iniziale di TAG con il nome del TAG  stesso  (es. "<tag". In |
                            | questo  caso  per  riconoscerlo   è  necessario  cercare  il |
                            | __TAG_INI_OPEN__ (<) nella stringa. Viene ricercato in tutta |
                            | la sottostringa senza considerare  una  specifica  posizione |
                            | questo perchè alla sinistra  del  delimitatore  ci  potrebbe |
                            | essere qualcos'altro (es. "unknown<tag". Se questo avviene è |
                            | un errore ed il parse si ferma  perchè  alla  sinistra  di   |
                            | tale delimitatore (<) non ci può essere nulla  che  non  sia |
                            | stato classificato in precedenza come token.   Nel  caso  di |      
                            | eccezione: "<<tag" sarebbe un errore  andare  avanti  quindi |
                            | bisogna accertarsi che di delimitatori non ce ne sia piu' di |
                            | uno.                                                         |   
                            '--------------------------------------------------------------' */ 
		            if ( search(buffer,__TAG_INI_OPEN__) > __ZERO__ ) 
                            {
                                 /*
                                 .---------------------------------------------------------.
                                 | Il delimitatore di apertura tag (<) è stato trovato si, |
                                 | ma non in prima posizione (es. "unknown<tag ") quindi   |
                                 | il parsen esce con un errore perchè alla sinistra del   |
                                 | delimitatore non è previsto nulla. Qualunque            |
                                 | delimitatore di chiusura precedente sarebbe stato       |
                                 | individuato precedentemente.                            |
                                 '---------------------------------------------------------' */ 
                                 i = 0;
                                 while ( i < strlen(buffer) && buffer[i] != __TAG_INI_OPEN2__ ) i++;
                                 if ( i > 0 ) {
                                      rPharse.row_error = countRowXMLFile + 1;
                                      rPharse.token_unknown = malloc(strlen(buffer));
                                      strcpy(rPharse.token_unknown,buffer);
                                      rPharse.token_unknown[i] = __NULL__ ;
                                      rPharse.return_code = -53;
                                      return rPharse; };   

                                 /*
                                 .-------------------------------------------------------------.
                                 | trovato il delimitatore di apertura tag in prima posizione  |
                                 | ma trovato anche almeno un'altro medesimo delimitatore      |
                                 | (es. "<tag<tag ") Questo significa per il parsen che il     |
                                 | tag precedente (lo considera un tag il token tra i due      |
                                 | delimitatori di apertura) non è stato chiuso e quindi esce  |
                                 | per errore.                                                 |
                                 '-------------------------------------------------------------' */
                                 if ( search(buffer,__TAG_INI_OPEN__) > __ONE__ ) 
                                 {
                                      /*
                                      .-----------------------------------------------------------------.
                                      | Questo TEST serve solo per differenziare il messaggio di errore |
                                      | per una maggiore comprensione: se nelle prime due posizioni     |
                                      | trova il delimitatore "es. <<tag" non si può parlare di         |
                                      | delimitatore di chiusura del tag non trovato bensi di un        |
                                      | delimitatore inserito più di una volta ad esempio per una       |
                                      | distrazione. Mentre nel caso in cui tra i due delimitatori ci   |
                                      | sia qualcosa di diverso "<tag<" in questo caso l'errore         |
                                      | potrebbe essere interpretato come di un delimitatore di         |
                                      | chiusura non trovato.                                           |
                                      '-----------------------------------------------------------------' */
                                      if (buffer[0]==__TAG_INI_OPEN2__ && buffer[1]==__TAG_INI_OPEN2__) 
                                      { 
                                          rPharse.row_error = countRowXMLFile + 1;
                                          rPharse.token_unknown = malloc(3);
                                          rPharse.token_unknown[0]=buffer[0];
                                          rPharse.token_unknown[1]=buffer[1];
                                          rPharse.token_unknown[2]=__NULL__;
                                          rPharse.return_code = -54;
                                          return rPharse; 
                                      }
                                      else
                                      {
                                          rPharse.token_unknown = malloc(strlen(buffer));
                                          i=1; 
                                          while (i<strlen(buffer) && buffer[i]!=__TAG_INI_OPEN2__) {
                                                 rPharse.token_unknown[i-1] = buffer[i];
                                                 i++; };
                                          rPharse.token_unknown[i-1]=__NULL__; 
                                          rPharse.return_code = -55;
                                          return rPharse;                                      
                                      }; 
                                 };

                                 /*
                                 .-----------------------------------------------------------------.
                                 |                                                                 |
                                 |                                                                 |
                                 |                   Inserimento di un nuovo TAG                   |
                                 |                                                                 |
                                 |                                                                 | 
                                 '-----------------------------------------------------------------' */
                                 rPharse.rTabTag[rPharse.iTabTag].tag = malloc(strlen("*Unknown*"));
                                 strcpy(rPharse.rTabTag[rPharse.iTabTag].tag, "*Unknown*");
                                 rPharse.rTabTag[rPharse.iTabTag].stato = 0;

                                 /*
                                 .-----------------------------------------------------------------.
                                 | Ricava il livello : [1] se lo stack è vuoto si tratta di primo  |
                                 | tag quindi il livello è 0 perchè si è in presenta della radice  |
                                 | [2] se lo stack non è vuoto si verifica l'ultimo TAG inserito,  |
                                 | se questo è in stato 2 (chiusa testa) si è in presenza di un    |
                                 | figlio di questo e quindi il livello viene incrementato di 1.   |
                                 | [3] Infice se lo stack non è vuoto e il TAG precedemente        |
                                 | caricato è in stato 3 (completamente chiuso) si è in presenza   |
                                 | di un fratello e quindi il livello è mantenuto                  | 
                                 '-----------------------------------------------------------------' */
                                 if (rPharse.iTabTag == 0) 
                                 {
                                     rPharse.rTabTag[rPharse.iTabTag].level = 0;
                                 }
                                 else 
                                 {
                                     if (rPharse.rTabTag[rPharse.iTabTag-1].stato == 2)
                                         rPharse.rTabTag[rPharse.iTabTag].level = rPharse.rTabTag[j-1].level+1;
                                    
                                     if (rPharse.rTabTag[rPharse.iTabTag-1].stato == 3)
                                         rPharse.rTabTag[rPharse.iTabTag].level = rPharse.rTabTag[j-1].level;
                                 }

                                 rPharse.rTabTag[rPharse.iTabTag].dimTabVar = 0;
                                 rPharse.iTabTag++;

                                 /*
                                 .--------------------------------------------------------.
                                 | identificato il token delimitatore iniziale del tag    |
                                 '--------------------------------------------------------' */       
                                 token[foundToken].type  = malloc(strlen("TagIniOpen\0"));
                                 strcpy(token[foundToken].type  , "TagIniOpen\0");
                                 token[foundToken].value = malloc(strlen(__TAG_INI_OPEN__));
                                 strcpy(token[foundToken].value , __TAG_INI_OPEN__);
                                 foundToken++; 

                                 /* 
                                 .--------------------------------------------------------.
                                 | elimina dal buffer il delimitatore lasciando solo      |
                                 | il nome tag "<tag\0" ==> "tag\0".                      |
                                 | Successivamente verrà catalogato il tag.               |
                                 | Tale codice non è riportato in questo blocco perchè    |
                                 | potrebbe accadere anche che tra delimitatore e tag ci  |
                                 | sia uno spazio di mezzo "< tag". In questo caso i due  |
                                 | token verrebbero letti in due bufferizzazioni          |
                                 | differenti anche se consecutive.                       |
                                 '--------------------------------------------------------' */ 
                                 j = 1;
                                 while (buffer[j]!=__ZERO__) { 
                                        buffer[j-1] = buffer[j];
                                        j++;
                                 };
                                 buffer[j-1] = __ZERO__;

                            };

                            /*
                            .----------------------------------------------.
                            |  Classificazione e valorizzazione del Token  |
                            '----------------------------------------------' */
                            token[foundToken].type = malloc(strlen("TagIni\0"));
                            strcpy(token[foundToken].type, "TagIni\0");
                            foundToken++;

                            /*
                            .----------------------------------------------.
                            |           Aggiornamento nome TAG             |
                            '----------------------------------------------' */
                            rPharse.rTabTag[rPharse.iTabTag-1].tag = malloc(strlen(buffer));
                            strcpy(rPharse.rTabTag[rPharse.iTabTag-1].tag, buffer);
                        }; 


                        /*
                        .---------------------------------------------------------------.
                        | Se lo stato dell'utimo TAG è DUE (2) significa che l'ultimo   |
                        | TAG trovato è di CODA ed è apeto "<tab></tag unknownToken/>". |
                        | Questo è un errore ed il parser si ferma.                     |
                        '---------------------------------------------------------------' */ 
                        if ( rPharse.rTabTag[rPharse.iTabTag-1].stato == 2 ) {
                             rPharse.row_error = countRowXMLFile + 1;
                             rPharse.token_unknown = malloc(strlen(rPharse.rTabTag[rPharse.iTabTag-1].tag));
                             strcpy(rPharse.token_unknown,rPharse.rTabTag[rPharse.iTabTag-1].tag);
                             rPharse.return_code = -49;
                             return rPharse; }; 
                   };



                   /*
                   .----------------------------------------------------------------------------.
                   | TEST X: TAG Aperto                                                         |
                   .----------------------------------------------------------------------------.
                   | Un delimitatore di chiusura presuppone un TAG aperto. Se lo stack è vuoto  |
                   | quindi se non ci sono TAG precedentemente analizzati manca l'apertura e    |
                   | pertanto il parsen si ferma con un errore.                                 |
                   '----------------------------------------------------------------------------' */
                   if ( rPharse.iTabTag == 0 ) 
                   {
                        rPharse.row_error = countRowXMLFile + 1;
                        rPharse.token_unknown = malloc(strlen("\0"));
                        strcpy(rPharse.token_unknown,"\0");
                        rPharse.return_code = -47;
                        return rPharse; 
                   };

                   /*
                   .----------------------------------------------------------------------------.
                   | TEST X: Value (es. <tag>|<tag/>|<tag><\|<tag><\tag|<tag><\tag> />)         |
                   .----------------------------------------------------------------------------.
                   | Medesimo discorso fatto in precedenza. Se lo stack è pieno ma l'ultimo     |
                   | elemento non è un TAG di TESTA aperto è un errore. Il delimitatore di      |
                   | chiusura di un NODO vuoto infatti si aspetta di trovare questo <tag ... /> |
                   | Sullo stack questa volta basta quest'unico controllo a differenza di altre |
                   | casistiche che leggono l'intero stack alla ricerca di padre, in questo     |
                   | il nodo non può avere figli.                                               |
                   '----------------------------------------------------------------------------' */
                   if (rPharse.rTabTag[rPharse.iTabTag-1].stato != 0 ) {
                       rPharse.row_error = countRowXMLFile + 1;
                       rPharse.token_unknown = malloc(strlen("\0"));
                       strcpy(rPharse.token_unknown,"\0");
                       rPharse.return_code = -48;
                       return rPharse; };

                   /*
                   .------------------------------------------------------------------.
                   | TEST X: Value (es. <*Unknown* />)                                |
                   .------------------------------------------------------------------.
                   | Se l'ultimo TAG aperto non ha un nome definitivo assegnato è un  |
                   | errore ed il parser si ferma.                                    |
                   '------------------------------------------------------------------' */
                   if ( strcmp(rPharse.rTabTag[rPharse.iTabTag-1].tag,"*Unknown*") == 0 ) {
                        rPharse.row_error = countRowXMLFile + 1;
                        rPharse.token_unknown = malloc(strlen("\0"));
                        strcpy(rPharse.token_unknown,"\0");
                        rPharse.return_code = -50; 
                        return rPharse; };

                   /*
                   .----------------------------------------------------------.
                   | TEST consolidamento ULTIMA VARIABILE. Se l'ultimo TAG    |
                   | risulta di TESTA ed aperto ed il nome è regolarmente     |
                   | assegnato il parse verifica la sitazione variabili. In   |
                   | particolare l'ultima deve essere in stato 0, cioè        |
                   | completamente definita. (var = value). Se non è così     |
                   | il parse si interrompe segnalando l'errore.              |
                   '----------------------------------------------------------' */
                   if ( rPharse.rTabTag[rPharse.iTabTag-1].stato_last_variable == 1 ) {   
                        /*
                        .--------------------------------------------------------------.
                        | CASO 1: Value (es. <tag var/>)                               |
                        .--------------------------------------------------------------.
                        | In questo caso la variabile è in stato 1. Solamente definita |
                        | ma manca l'ASSIGN ed il VALUE.                               |
                        '--------------------------------------------------------------' */
                        int istack = rPharse.iTabTag-1 ;
                        int istackVar = rPharse.rTabTag[istack].dimTabVar ;
                        rPharse.row_error = countRowXMLFile + 1;
                        rPharse.token_unknown = malloc(strlen(rPharse.rTabTag[istack].rTabVar[istackVar].var));
                        strcpy(rPharse.token_unknown,rPharse.rTabTag[istack].rTabVar[istackVar].var);
                        rPharse.return_code = -51;
                        return rPharse; };


                   if ( rPharse.rTabTag[rPharse.iTabTag-1].stato_last_variable == 2 ) {
                        /*
                        .--------------------------------------------------------------.
                        | CASO 2: Value (es. <tag var = />)                            |
                        .--------------------------------------------------------------.
                        | In questo caso la variabile è in stato 2: definità e con     |
                        | l'assign, manca però la value.                               | 
                        '--------------------------------------------------------------' */
                        int istack = rPharse.iTabTag-1 ;
                        int istackVar = rPharse.rTabTag[istack].dimTabVar ;
                        rPharse.row_error = countRowXMLFile + 1;
                        rPharse.token_unknown = malloc(strlen(rPharse.rTabTag[istack].rTabVar[istackVar].var));
                        strcpy(rPharse.token_unknown,rPharse.rTabTag[istack].rTabVar[istackVar].var);
                        rPharse.return_code = -52;
                        return rPharse; };


                   /*
                   .----------------------------------------------.
                   |  Classificazione e valorizzazione del Token  |
                   '----------------------------------------------' */
                   token[foundToken].type  = malloc(strlen("TagIniEnd\0"));
                   token[foundToken].value = malloc(strlen(__TAG_INI_END__));
                   strcpy(token[foundToken].type  , "TagIniEnd\0"); 
                   strcpy(token[foundToken].value , __TAG_INI_END__) ;
                   foundToken++;

                   /*
                   .----------------------------------------------.
                   |          Cambia lo STATO DEL TAG             |
                   '----------------------------------------------' */
                   rPharse.rTabTag[rPharse.iTabTag-1].stato = 3;
              }; 



              /*
              .-------------------------------------------------------------------------------------.
              |                                                                                     |
              |                                                                                     |
              |                                                                                     |
              |                                     __TAG_END__                                     |
              |                                                                                     |
              |                                                                                     |
              |                                                                                     |
              |                                                                                     |
              '-------------------------------------------------------------------------------------' */
              if ( search(buffer,__TAG_END__) == __TRUE__ && remark == __OFF__ && foundToken == __FALSE__ ) 
              {

                   /*
                   .-----------------------------------------------------------------------------------.
                   | Err 1. Area INTESTAZIONI XML non presente - Stack XML vuoto.                      |
                   .-----------------------------------------------------------------------------------.
                   | Un token di CHIUSURA presuppone che [1] ci sia un TAG aperto [2] che il nometag   |
                   | sia assegnato e [3] che ci sia nello stack una situazione consolidata per le      |
                   | variabili... (nessuna o completamente definite - stato 2). Fatta questa premessa  |
                   | Il primo TEST da implementare è sulla presenta del TAG di TESTA aperto.           |
                   | Prima di questo è necessario verificare che lo stack XML sia consolidato cioè se  |
                   | ci troviamo nell'area dati dell'XML. Se lo stack XML è assente o non ancora       |
                   | concluso è un errore perchè il token analizzato come detto chiude i TAG dati.     |
                   | Quello successivo è il TEST per verificare che nello stack XML ci sia qualcosa.   |
                   | E' inutile prima processare un'eventuale buffer pieno perchè anche se fosse       |
                   | sicuramente non riuscirebbe da solo a sanare un'eventuale mancanza riscontrata    |
                   | nell'attuale stack.                                                               |
                   '-----------------------------------------------------------------------------------' */
                   if (rPharse.iTabTagXML == 0) {  
                       rPharse.row_error = countRowXMLFile + 1;
                       rPharse.token_unknown = malloc(strlen("\0")); 
                       strcpy(rPharse.token_unknown,"\0");
                       rPharse.return_code = -69;
                       return rPharse; 
                   }

                   /*
                   .----------------------------------------------------------------------------.
                   | Err 2. Area INTESTAZIONI XML non conclusa                                  |
                   .----------------------------------------------------------------------------.
                   | Se lo stack XML non è vuoto è necessario (per continuare) che l'ultimo TAG |
                   | inserito sia CHIUSO che quindi la sezione intestazione sia consolidata.    |
                   | Come detto in precedenza il token analizzato può trovarsi solo nella       |
                   | sezione dati. Il buffer non viene verificato se è pieno o meno, perchè     |
                   | anche se fosse pieno di qualcosa non riuscirebbe a sanare completamente    |
                   | un'eventuale mancanza riscontrata.                                         |
                   '----------------------------------------------------------------------------' */
                   if (rPharse.rTabTagXML[rPharse.iTabTagXML-1].stato != 3)
                   {
                       rPharse.row_error = countRowXMLFile + 1;
                       rPharse.token_unknown = malloc(strlen(rPharse.rTabTagXML[rPharse.iTabTagXML-1].tag)); 
                       strcpy(rPharse.token_unknown,rPharse.rTabTagXML[rPharse.iTabTagXML-1].tag);
                       rPharse.return_code = -70;
                       return rPharse;
                   };

                   /*
                   .-----------------------------------------------------------.
                   | Isolamento della stringa alla sinistra del __TAG_END__    |
                   '-----------------------------------------------------------' */
                   i = i - strlen(__TAG_END__);
                   buffer[i] = __NULL__;

                   /*
                   .----------------------------------------------------------------------------.
                   | Err 3. Missing Area DATI                                                   |
                   .----------------------------------------------------------------------------.
                   | Un delimitatore di chiusura presuppone un TAG aperto. Se lo stack è vuoto  |
                   | quindi se non ci sono TAG precedentemente analizzati ed il buffer da       |
                   | analizzare è vuoto (quindi si tratta di una situazione aggiornata)         |
                   | manca un tag aperto pronto a ricevere la chiusura e pertanto il parsen     |
                   | si ferma con questo errore.                                                |
                   '----------------------------------------------------------------------------' */
                   if (rPharse.iTabTag == 0 && i == 0) {
                       rPharse.row_error = countRowXMLFile + 1;
                       rPharse.token_unknown = malloc(strlen("\0")); 
                       strcpy(rPharse.token_unknown,"\0");
                       rPharse.return_code = -35;
                       return rPharse; };

                   /*
                   .---------------------------------------------------------------------.
                   | Se L'albero TAG dati è pieno e aperto i successivi due blocchi di   |
                   | codici verificano che [1] l'albero non sia chiuso, cioè che il      |
                   | primo TAG (radice) non sia in stato 3 (completamente definito) e    |
                   | [2] verificato che sia aperto l'albero il parse si posiziona sul    |
                   | primo TAG (leggendo a ritroso in stato 0, 1 o 2. Questo per poter   |
                   | identificare quale TAG il parse si aspetta.                         |
                   '---------------------------------------------------------------------' */ 
                   if ( rPharse.iTabTag > 0 ) 
                   {

                        /*
                        .----------------------------------------------------------------------------.
                        | Err 4. Close Area DATI                                                     |
                        .----------------------------------------------------------------------------.
                        | Medesimo discorso fatto in precedenza. Se lo stack è pieno ma il nodo      |
                        | radice è consolidato (chiuso, stato a 3) significa che nessun stack        |
                        | risulta aperto (stato 0, stato 2), anzi per dirla ancora meglio che        |
                        | l'intero albero|documento è consolidato e non c'è spazio per nulla altro   |
                        | Il parse quindi si ferma con un errore.                                    |
                        |                                                                            |
                        | ### forse questo TEST bisognerebbe inserirlo in tutte le sezioni come      |
                        | primo dopo le verifiche del caso sullo STACK XML ######################### |
                        '----------------------------------------------------------------------------' */
                        if ( rPharse.rTabTag[0].stato == 3 ) {
                             rPharse.row_error = countRowXMLFile + 1;
                             rPharse.token_unknown = malloc(strlen("\0"));
                             strcpy(rPharse.token_unknown,"\0");
                             rPharse.return_code = -36;
                             return rPharse; };
                   };

                   /*
                   .----------------------------------------------------------------------------.
                   |                                                                            |
                   | Ok se siamo arrivati a questo punto significa che :                        |
                   |                                                                            |
                   | [a] le intestazioni XML son tutte chiuse.                                  |
                   |                                                                            |
                   | [b] E' vera una delle due condizioni :                                     |
                   |                                                                            |
                   |     [b.1] l'albero dati è completamente vuoto ma il buffer è pieno quindi  |
                   |           potrebbe essere che nel buffer è contenuto l'intero TAG di TESTA |
                   |                                                                            |
                   |     [b.2] l'albero dati è pieno e non chiuso.                              |
                   |           Il parse si appresta ad entrarci dentro... uhaaa uhaaa           |
                   |                                                                            |
                   |  A questo punto il parsen legge a ritroso lo stack alla ricerca del primo  |
                   |  TAG in attesa di un delimitatore di chiusura, quindi in stato 0 o stato 2 |
                   |  Se lungo il traggitto incontra un TAG con stato a 1 (testa completa)      |
                   |  termina comunque la ricerca perchè anche questo per il momento è          |
                   |  compatibile (il buffer potrebbe contenere ciò che manca per ottenere un   |
                   |  nuovo tag successivo in STATO 0). Nel caso di albero assente l'indice j   |
                   |  (posizione dell'elemento nello stack) rimane a 0.                         |
                   |                                                                            |
                   '----------------------------------------------------------------------------' */
                   j = rPharse.iTabTag;

                   while ( j > 0 && rPharse.rTabTag[j-1].stato != 0 
                                 && rPharse.rTabTag[j-1].stato != 1 
                                 && rPharse.rTabTag[j-1].stato != 2 )
                                 j--;

                   /*
                   .---------------------------------------------------------------------------.
                   | Individuato il primo TAG (leggendo l'albero a ritroso come si è detto) in |
                   | STATO 0|1|2 vengono effettuati su di esso dei controlli per verificare    |
                   | nel dettaglio la compatibilità con il token di chiusura TAG TESTA|CODA    |
                   | trovato e di cui si sta verificando la correttezza. In particolare si     |
                   | verifica di tale TAG trovato nel caso in cui il buffer è vuoto (quindi    |
                   | nel caso in cui la situazione è aggiornata) la consistenza, cioè se il    |
                   | è in attesa, può ricevere il delimitatore di chiusura. Chiaramente i TEST |
                   | sono possibili solemente se è stato trovato un TAG, quindi se lo stack è  |
                   | pieno altrimenti il blocco di TEST viene saltato. In ogni caso se non     |
                   | esiste buffer da processare lo stack è sempre pieno altrimenti il parse   |
                   | sarebbe uscito ad uno dei test fatti in precedenza.                       |
                   '---------------------------------------------------------------------------' */ 
                   if ( i == 0 ) 
                   {
                        /*
                        .----------------------------------------------------------------------------.
                        | Err 5. Found delimiter close but there isnt a TAG open.                    |
                        .----------------------------------------------------------------------------.
                        | Se il primo TAG trovato è in STATO 1 quindi TESTA CHIUSO (es. <tag>) è     |
                        | necessario che il buffer sia pieno per poter ancora sperare che il tutto   |
                        | sia corretto perchè nel buffer potrebbe esserci un intero TAG figlio       |
                        | ancora non processato che validerebbe il token trovato (>) che il token    |
                        | sta in questo momento analizzando.(es. token:<tagpadre> buffer:<tagfiglio>)|
                        | Se però è stato trovato un TAG in STATO 1 ed il BUFFER è vuoto questo è    |
                        | prova certa che esiste un errore del tipo: chiusura di TAG inesistente,    |
                        | cioè mai aperto. (es. <tag> >, <tagpadre><tagfiglio/>>)                    |
                        '----------------------------------------------------------------------------' */
                        if ( rPharse.rTabTag[j-1].stato == 1 ) { 
                             rPharse.row_error = countRowXMLFile + 1;
                             rPharse.token_unknown = malloc(strlen("\0"));
                             strcpy(rPharse.token_unknown,"\0");
                             rPharse.return_code = -71;
                             return rPharse; };

                        /*
                        .----------------------------------------------------------------------------.
                        | Err 6. The TAG havent name (es. <*Unknown* >)                              |
                        .----------------------------------------------------------------------------.
                        | Il TAG trovato non ha un nome quindi non può essere chiuso. Come detto non | 
                        | c'è nulla nel buffer quindi la situazione è aggiornata con un delimitatore |
                        | di chiusura trovato da validare e da registrare. Chiaramente se è così lo  |
                        | stato del TAG è ZERO condizione omessa perchè ovvia.                       | 
                        '----------------------------------------------------------------------------' */
                        if ( strcmp(rPharse.rTabTag[j-1].tag,"*Unknown*") == 0 ) {
                             rPharse.row_error = countRowXMLFile + 1;
                             rPharse.token_unknown = malloc(strlen("\0"));
                             strcpy(rPharse.token_unknown,"\0");
                             rPharse.return_code = -72;
                             return rPharse; };

                        /*
                        .----------------------------------------------------------------------------.
                        | Err 7. the last variable missing token assign (es. <tag var >)             |
                        .----------------------------------------------------------------------------.
                        | Il TAG trovato non ha una situazione variabili consolidata. Come detto non |
                        | c'è nulla nel buffer quindi la situazione è aggiornata con u delimitatore  |
                        | di chiusura trovato da validare e da registrare. In particolare l'ultima   |
                        | variabile trovata nello stack è attesa di assign e value.                  |
                        '----------------------------------------------------------------------------' */
                        if ( rPharse.rTabTag[j-1].stato_last_variable == 1 ) { 
                             rPharse.row_error = countRowXMLFile + 1;
                             rPharse.token_unknown = malloc(strlen(rPharse.rTabTag[j-1].tag));
                             strcpy(rPharse.token_unknown,rPharse.rTabTag[j-1].tag);
                             rPharse.return_code = -73;
                             return rPharse; };

                        /*
                        .----------------------------------------------------------------------------.
                        | Err 8. the last variable missing token assign (es. <tag var >)             |
                        .----------------------------------------------------------------------------.
                        | Il TAG trovato non ha una situazione variabili consolidata. Come detto non |
                        | c'è nulla nel buffer quindi la situazione è aggiornata con u delimitatore  |
                        | di chiusura trovato da validare e da registrare. In particolare l'ultima   |
                        | variabile trovata nello stack è attesa di assign e value.                  |
                        '----------------------------------------------------------------------------' */
                        if ( rPharse.rTabTag[j-1].stato_last_variable == 2 ) { 
                             rPharse.row_error = countRowXMLFile + 1;
                             rPharse.token_unknown = malloc(strlen(rPharse.rTabTag[j-1].tag));
                             strcpy(rPharse.token_unknown,rPharse.rTabTag[j-1].tag);
                             rPharse.return_code = -74;
                             return rPharse; };
                   };

                   /*
                   .---------------------------------------------------------------------------.
                   | E' necessario a questo punto aggiornare lo stack dati XML e per farlo     |
                   | bisogna analizzare il buffer, se questo risulta pieno. E' pieno se alla   |
                   | sinistra del token che stiamo in questo momento analizzando.              |
                   | Il test successivo serve appunto a capire se esiste qualcosa nel buffer.  |
                   | La variabile "i" contiene la lunghezza della stringa alla sinistra del    |
                   | di tale token, quindi del buffer stesso.                                  |
                   '---------------------------------------------------------------------------' */
                   if ( i > 0 )
                   {

                        /*
                        .------------------------------------------------------------------------------.
                        | Il buffer è pieno perchè la sia lunghezza è maggiore di ZERO. Il parse       |
                        | il TAG su cui si è fermato in precedenza per capire se nel buffer è presente |
                        | realmente la parte mancante che ancora serve prima di poter registrare il    |
                        | token di chiusura. Il parsen considera tutte le casistiche in cui il TAG     |
                        | potrebbe trovarsi, partendo dal suo stato che come detto a questo punto      |
                        | dell'analisi potrebbe trovarsi in stato 0,1 e 2 (aperto tag di testa, chiuso |
                        | tag di testa, aperto tag di coda).                                           |
                        '------------------------------------------------------------------------------' */

                        /*
                        .------------------------------------------------------------------------------.
                        |                                 TAG TESTA APERTO                             |
                        .------------------------------------------------------------------------------.
                        | Si sta considerando solamente la casistica in cui l'albero XML dati risulta  |
                        | pieno di almeno un TAG altrimenti il successivo blocco non è applicabile.    |
                        | Il TAG su cui il parse si è fermato è un TAG di TESTA aperto. Il parsen      |
                        | verifica il suo stato di consolidamente seguendo la sintassi prevista:       |
                        | [1] assegnazione del nome [2] stato variabili. Il parse si aspetta di        |
                        | trovare un TAG senza nome perchè in quel caso il contenuto del buffer viene  |
                        | considerato tale, oppure che ci siano variabili nello stack per quel tag e   |
                        | che l'ultimo sia in attesa di valore perchè in quel caso viene considerato   |
                        | tale il contenuto del buffer. In tutti gli altri casi il parse come si vedrà |
                        | li considererà errori perchè queste sono gli unici due scenari in cui manca  |
                        | per raggiungere la consistena un'unico token.                                |
                        '------------------------------------------------------------------------------' */
                        if ( rPharse.iTabTag > 0 ) {
                        if ( rPharse.rTabTag[j-1].stato == 0 )
                        {
                             /* 
                             .---------------------------------------------------------------------------.
                             | Se il TAG trovato è in STATO 0 quindi di TESTA e APERTO il parse verifica |
                             | che il nome sia stato assegnato. Se non risulta assegnato il parse        |
                             | deduce che nel buffer ci sia proprio il nome del TAG e quindi lo assegna. |
                             | E' stata raggiunta la consistena tale per poterlo chiudere.               |
                             '---------------------------------------------------------------------------' */
                             if ( strcmp(rPharse.rTabTag[j-1].tag,"*Unknown*") == 0 ) 
                             { 
                                  /*
                                  .----------------------------------------------.
                                  |  Classificazione e valorizzazione del Token  |
                                  '----------------------------------------------' */
                                  token[foundToken].type = malloc(strlen("TagIni\0"));
                                  strcpy(token[foundToken].type, "TagIni\0");
                                  token[foundToken].value = malloc(strlen(buffer));
                                  strcpy(token[foundToken].value , buffer);
                                  foundToken++;

                                  /*
                                  .----------------------------------------------.
                                  |           Aggiornamento nome TAG             |
                                  '----------------------------------------------' */
                                  rPharse.rTabTag[j-1].tag = malloc(strlen(buffer));
                                  strcpy(rPharse.rTabTag[j-1].tag, buffer);
                             }
                             else
                             {   
                                  /*
                                  .---------------------------------------------------------------------------.
                                  | Se il TAG trovato è in STATO 0 quindi di TESTA e APERTO. Il nome come     |
                                  | visto è stato assegnato. Il parse passa a verificare come detto in        |
                                  | lo stato delle variabili nella speranza di trovarne almeno una e che      |
                                  | l'ultima sia attesa di un valore. Il parse considera il contenuto del     |
                                  | buffer come valore. Anche in questo caso si raggiunge così per il tag     |
                                  | un punto di consistenza necessario per poterlo chiudere. In tutti gli     |
                                  | casi (nessuna variabile, ultima variabile in attesa di assign) è un       |
                                  | errore perchè in questi casi il contenuto del buffer in nessun caso       |
                                  | riuscirebbe a sanare le mancanze riscontrate.                             |
                                  '---------------------------------------------------------------------------' */
                                  switch ( rPharse.rTabTag[j-1].stato_last_variable )
                                  {                                 

                                           case 0: 

                                                   /*
                                                   .--------------------------------------------------------------.
                                                   | Err 9. CASO 1: Value (es. <tag UnknownToken>                 |
                                                   .--------------------------------------------------------------.
                                                   | Il parse si aspettava la value per l'ultima variabile        |
                                                   | presente nello stack, però questa non ha bisogno del valore  |
                                                   | perchè risulta già completa (var=value) e quindi il parse si |
                                                   | ferma e segnala l'errore. Oppure non sono presenti variabili |
                                                   |                                                              |
                                                   '--------------------------------------------------------------' */
                                                   rPharse.row_error = countRowXMLFile + 1;
                                                   rPharse.token_unknown = malloc(strlen(rPharse.rTabTag[j-1].tag));
                                                   strcpy(rPharse.token_unknown,rPharse.rTabTag[j-1].tag);
                                                   rPharse.return_code = -75;
                                                   return rPharse;
                                                   break;

                                           case 1:
 
                                                   /*
                                                   .--------------------------------------------------------------.
                                                   | Err 10. CASO 2: Value (es. <tag var UnknownToken/>)          |
                                                   .--------------------------------------------------------------.
                                                   | Il parse si aspettava la value per l'ultima variabile        |
                                                   | presente nello stack, però questa non ha bisogno del valore  |
                                                   | ma dell'assign (var) e quindi il parse si ferma e segnala    |
                                                   | l'errore.                                                    |
                                                   |                                                              |
                                                   '--------------------------------------------------------------' */
                                                   rPharse.row_error = countRowXMLFile + 1;
                                                   rPharse.token_unknown = malloc(strlen(rPharse.rTabTag[j-1].tag));
                                                   strcpy(rPharse.token_unknown,rPharse.rTabTag[j-1].tag);
                                                   rPharse.return_code = -76;
                                                   return rPharse;
                                                   break;

                                           case 2:
 
                                                   /*
                                                   .--------------------------------------------------------------.
                                                   | CASO 3: Value (es. <tag var = [value]>)                      |
                                                   .--------------------------------------------------------------.
                                                   | Il parse si aspetta la value per l'ultima variabile          |
                                                   | presente nello stack e realmente trova un ultima variabile   |
                                                   | in attesa di esso. Quindi assegna il value e cambia lo stato |
                                                   | della variabile nuovamente a zero.                           |
                                                   |                                                              |
                                                   '--------------------------------------------------------------' */

                                                   /*
                                                   .--------------------------------------------------------.
                                                   | Identificato e classificato la tipologia del token     |
                                                   '--------------------------------------------------------' */
                                                   token[foundToken].type  = malloc(strlen("Value\0"));
                                                   strcpy(token[foundToken].type  , "Value\0");
                                                   token[foundToken].value = malloc(strlen(buffer));
                                                   strcpy(token[foundToken].value , buffer);
                                                   foundToken++;

                                                   /*
                                                   .-------------------------------------------------------.
                                                   | Imposta il valore relativo alla variabile nello stack |
                                                   '-------------------------------------------------------' */
                                                   int istack = j-1 ;
                                                   int istackVar = rPharse.rTabTag[istack].dimTabVar ;
                                                   rPharse.rTabTag[istack].rTabVar[istackVar].value = malloc(strlen(buffer));
                                                   strcpy(rPharse.rTabTag[istack].rTabVar[istackVar].value,buffer);
                                                   rPharse.rTabTag[istack].dimTabVar++;

                                                   /*
                                                   .-------------------------------------------------------.
                                                   | Imposta l'ultimo stato variabili a zero (consolidato) |
                                                   '-------------------------------------------------------' */
                                                   rPharse.rTabTag[istack].stato_last_variable = 0 ;
                                                   break;
                                  };
                             };

                        }};


                        /*
                        .------------------------------------------------------------------------------.
                        |                                TAG TESTA CHIUSO                              |
                        .------------------------------------------------------------------------------.
                        |                                                                              |
                        |  Se il primo TAG valido ha stato ad 1 (TAG di TESTA chiuso) oppure l'albero  |
                        |  nello STACK è completamente assente. Il entrambi i casi il parse come       |
                        |  tentativo per considerare valido l'XML verifica il contenuto del BUFFER     |
                        |  nel tentativo di trovare non ancora processato un intero TAG di TESTA.      |
                        |  (es. <tag-1><tag-2> con <tag-2> ancora nel buffer. oppure <tag-1> con       |
                        |  <tag-1> ancora nel buffer). Uso un flag (?) son costretto ad usarlo perchè  | 
                        |  test del tipo "rPharse.rTabTag[j-1].stato == 1 || rPharse.iTabTag == 0 mi   |
                        |  potrebbero creare problemi nel caso in cui lo stack è vuoto, in quel caso j |
                        |  vale 0 e quindi j-1 equivale a dire -1.                                     |
                        '------------------------------------------------------------------------------' */
                        int fHeadClose = 0;
                        if ( rPharse.iTabTag > 0 )    
                             if ( rPharse.rTabTag[j-1].stato == 1 )
                                  fHeadClose = 1 ;

                        if ( rPharse.iTabTag == 0 )    
                             fHeadClose = 1 ;

                        if ( fHeadClose == 1 )
                        {
                             /*
                             .--------------------------------------------------------------.
                             | Alla sinistra del token >  ci  può essere un delimitatore    |
                             | iniziale di TAG con il nome del TAG  stesso  (es. "<tag". In |
                             | questo  caso  per  riconoscerlo   è  necessario  cercare  il |
                             | __TAG_INI_OPEN__ (<) nella stringa. Viene ricercato in tutta |
                             | la sottostringa senza considerare  una  specifica  posizione |
                             | questo perchè alla sinistra  del  delimitatore  ci  potrebbe |
                             | essere qualcos'altro (es. "unknown<tag". Se questo avviene è |
                             | un errore ed il parse si ferma  perchè  alla  sinistra  di   |
                             | tale delimitatore (<) non ci può essere nulla  che  non  sia |
                             | stato classificato in precedenza come token.   Nel  caso  di |      
                             | eccezione: "<<tag" sarebbe un errore  andare  avanti  quindi |
                             | bisogna accertarsi che di delimitatori non ce ne sia piu' di |
                             | uno. Nel caso in cui invece mancasse il delimitatore di      |
                             | apertura del tag è un errore perchè in questo caso il parse  |
                             | necessita di trovare nel buffer un TAG di TESTA completo.    |
                             '--------------------------------------------------------------' */

                             /*
                             .--------------------------------------------------------------.
                             | Err 11. Not found TAG left delimiter                         |
                             .--------------------------------------------------------------.
                             | Non è stato trovato nessun delimitatore  di apertura del TAG |
                             | quindi nel buffer non può essere contenuto nessun TAG di     |
                             | TESTA completo e quindi il parse si ferma perchè cercava     |
                             | proprio questo.                                              |
                             '--------------------------------------------------------------' */
		             if ( search(buffer,__TAG_INI_OPEN__) == __ZERO__ ) 
                             {
                                  rPharse.row_error = countRowXMLFile + 1;
                                  rPharse.token_unknown = malloc(strlen(buffer));
                                  strcpy(rPharse.token_unknown,buffer);
                                  rPharse.token_unknown[i] = __NULL__ ;
                                  rPharse.return_code = -77;
                                  return rPharse;    
                             }
                             else
                             {
                                  /*
                                  .---------------------------------------------------------.
                                  | Err 12. *UnknownToken* found left TAG                   |
                                  .---------------------------------------------------------.
                                  | Il delimitatore di apertura tag (<) è stato trovato si, |
                                  | ma non in prima posizione (es. "unknown<tag ") quindi   |
                                  | il parsen esce con un errore perchè alla sinistra del   |
                                  | delimitatore non è previsto nulla. Qualunque            |
                                  | delimitatore di chiusura precedente sarebbe stato       |
                                  | individuato precedentemente.                            |
                                  '---------------------------------------------------------' */ 
                                  i = 0;
                                  while ( i < strlen(buffer) && buffer[i] != __TAG_INI_OPEN2__ ) i++;
                                  if ( i > 0 ) {
                                       rPharse.row_error = countRowXMLFile + 1;
                                       rPharse.token_unknown = malloc(strlen(buffer));
                                       strcpy(rPharse.token_unknown,buffer);
                                       rPharse.token_unknown[i] = __NULL__ ;
                                       rPharse.return_code = -78;
                                       return rPharse; };   

                                  /*
                                  .-------------------------------------------------------------.
                                  | trovato il delimitatore di apertura tag in prima posizione  |
                                  | ma trovato anche almeno un'altro medesimo delimitatore      |
                                  | (es. "<tag<tag ") Questo significa per il parsen che il     |
                                  | tag precedente (lo considera un tag il token tra i due      |
                                  | delimitatori di apertura) non è stato chiuso e quindi esce  |
                                  | per errore.                                                 |
                                  '-------------------------------------------------------------' */
                                  if ( search(buffer,__TAG_INI_OPEN__) > __ONE__ ) 
                                  {
                                       /*
                                       .-----------------------------------------------------------------.
                                       | Questo TEST serve solo per differenziare il messaggio di errore |
                                       | per una maggiore comprensione: se nelle prime due posizioni     |
                                       | trova il delimitatore "es. <<tag" non si può parlare di         |
                                       | delimitatore di chiusura del tag non trovato bensi di un        |
                                       | delimitatore inserito più di una volta ad esempio per una       |
                                       | distrazione. Mentre nel caso in cui tra i due delimitatori ci   |
                                       | sia qualcosa di diverso "<tag<" in questo caso l'errore         |
                                       | potrebbe essere interpretato come di un delimitatore di         |
                                       | chiusura non trovato.                                           |
                                       '-----------------------------------------------------------------' */
                                       if (buffer[0]==__TAG_INI_OPEN2__ && buffer[1]==__TAG_INI_OPEN2__) 
                                       { 
                                           /*
                                           .---------------------------------------.
                                           | Err 13. duplicate delimiter left TAG  |
                                           '---------------------------------------' */
                                           rPharse.row_error = countRowXMLFile + 1;
                                           rPharse.token_unknown = malloc(3);
                                           rPharse.token_unknown[0]=buffer[0];
                                           rPharse.token_unknown[1]=buffer[1];
                                           rPharse.token_unknown[2]=__NULL__;
                                           rPharse.return_code = -79;
                                           return rPharse; 
                                       }
                                       else
                                       {
                                           /*
                                           .-----------------------------------------------.
                                           | Err 14. found two or more delimiter left TAG  |
                                           '-----------------------------------------------' */
                                           rPharse.token_unknown = malloc(strlen(buffer));
                                           i=1; 
                                           while (i<strlen(buffer) && buffer[i]!=__TAG_INI_OPEN2__) {
                                                  rPharse.token_unknown[i-1] = buffer[i];
                                                  i++; };
                                           rPharse.token_unknown[i-1]=__NULL__; 
                                           rPharse.return_code = -80;
                                           return rPharse;                                      
                                       }; 
                                  };

                                  /*
                                  .-----------------------------------------------------------------.
                                  |                                                                 |
                                  |                                                                 |
                                  |                   Inserimento di un nuovo TAG                   |
                                  |                                                                 |
                                  |                                                                 | 
                                  '-----------------------------------------------------------------' */
                                  rPharse.rTabTag[rPharse.iTabTag].tag = malloc(strlen("*Unknown*"));
                                  strcpy(rPharse.rTabTag[rPharse.iTabTag].tag, "*Unknown*");
                                  rPharse.rTabTag[rPharse.iTabTag].stato = 0;

                                  /*
                                  .-----------------------------------------------------------------.
                                  | Ricava il livello : [1] se lo stack è vuoto si tratta di primo  |
                                  | tag quindi il livello è 0 perchè si è in presenza della radice  |
                                  | [2] se lo stack non è vuoto si incrementa di 1 il livello del   |
                                  | tag trovato in stato 1 che è considerato suo padre.             |
                                  '-----------------------------------------------------------------' */
                                  if (rPharse.iTabTag == 0) 
                                  {
                                      rPharse.rTabTag[rPharse.iTabTag].level = 0;
                                  }
                                  else 
                                  {
                                      rPharse.rTabTag[rPharse.iTabTag].level = rPharse.rTabTag[j-1].level+1;
                                  }

                                  rPharse.rTabTag[rPharse.iTabTag].dimTabVar = 0;
                                  rPharse.iTabTag++;

                                  /*
                                  .--------------------------------------------------------.
                                  | identificato il token delimitatore iniziale del tag    |
                                  '--------------------------------------------------------' */       
                                  token[foundToken].type  = malloc(strlen("TagIniOpen\0"));
                                  strcpy(token[foundToken].type  , "TagIniOpen\0");
                                  token[foundToken].value = malloc(strlen(__TAG_INI_OPEN__));
                                  strcpy(token[foundToken].value , __TAG_INI_OPEN__);
                                  foundToken++; 

                                  /* 
                                  .--------------------------------------------------------.
                                  | elimina dal buffer il delimitatore lasciando solo      |
                                  | il nome tag "<tag\0" ==> "tag\0".                      |
                                  | Successivamente verrà catalogato il tag.               |
                                  | Tale codice non è riportato in questo blocco perchè    |
                                  | potrebbe accadere anche che tra delimitatore e tag ci  |
                                  | sia uno spazio di mezzo "< tag". In questo caso i due  |
                                  | token verrebbero letti in due bufferizzazioni          |
                                  | differenti anche se consecutive.                       |
                                  '--------------------------------------------------------' */ 
                                  int j2 = 1;
                                  while (buffer[j2]!=__ZERO__) { 
                                         buffer[j2-1] = buffer[j2];
                                         j2++;
                                  };
                                  buffer[j2-1] = __ZERO__;

                             };

                             /*
                             .----------------------------------------------.
                             |  Classificazione e valorizzazione del Token  |
                             '----------------------------------------------' */
                             token[foundToken].type = malloc(strlen("TagIni\0"));
                             strcpy(token[foundToken].type, "TagIni\0");
                             foundToken++;

                             /*
                             .----------------------------------------------.
                             |           Aggiornamento nome TAG             |
                             '----------------------------------------------' */
                             rPharse.rTabTag[rPharse.iTabTag-1].tag = malloc(strlen(buffer));
                             strcpy(rPharse.rTabTag[rPharse.iTabTag-1].tag, buffer);
                             
                        }; 

                     

                        /*
                        .------------------------------------------------------------------------------.
                        |                                TAG CODA APERTO                               |
                        .------------------------------------------------------------------------------.
                        |                                                                              |
                        |                                                                              |
                        |                                                                              |
                        |                                                                              |
                        |                                                                              |
                        |                                                                              |
                        |                                                                              |
                        |                                                                              |
                        '------------------------------------------------------------------------------' */
                        if ( rPharse.iTabTag > 0 ) {
                        if ( rPharse.rTabTag[j-1].stato == 2 )
                        {
                             // .---------------------------------------------------.      
                             // *****************************************************     
                             // ****** PROBLEMA IL TAG DI CHIUSURA COME SI FA *******     
                             // ****** A CAPIRE SE GIA' E' ASSEGNATO ???????? *******     
                             // *****************************************************      
                             // '---------------------------------------------------'     

                             /*
                             .--------------------------------------------------------.
                             | Identificato e classificato la tipologia del token     |
                             '--------------------------------------------------------' */
                             token[foundToken].type  = malloc(strlen("TagEnd\0"));
                             strcpy(token[foundToken].type  , "TagEnd\0");

                             /*
                             .---------------------------------------------------------.
                             | Registra Token trovato alla sinistra                    |
                             '---------------------------------------------------------' */
                             token[foundToken].value = malloc(strlen(buffer));
                             strcpy(token[foundToken].value  , buffer);
                             foundToken++;
                        };};

                   };

                   /*
                   .----------------------------------------------------------------------------.
                   | In questo punto si ricongiungono i due rami: buffer vuoto, buffer pieno.   |
                   | Quindi da questo punto in poi lo stack dei TAG è aggiornato. In presenza   |
                   | di errori il parse li ha già rilevati ed è uscito. Quindi a questo punto   |
                   | Tutto è OK e si appresta a registrare il Token di chiusura trovato. E' a   |
                   | questo punto che il parse ha necessità di capire se il tiken di chiusura è |
                   | legato ad un TAG di TESTA oppure ad un TAG di CODA.                        |
                   '----------------------------------------------------------------------------' */


                   /*
                   .----------------------------------------------------------------------------.
                   |                                                                            | 
                   |  Chiude il TAG di TESTA                                                    | 
                   |                                                                            | 
                   '----------------------------------------------------------------------------' */
                   if ( rPharse.rTabTag[j-1].stato == 0 ) { 

                        /*
                        .--------------------------------------------------------.
                        | identificato il token delimitatore iniziale del tag    |
                        '--------------------------------------------------------' */       
                        token[foundToken].type  = malloc(strlen("TagIniClose\0"));
                        strcpy(token[foundToken].type  , "TagIniClose\0");
                        token[foundToken].value = malloc(strlen(__TAG_END__));
                        strcpy(token[foundToken].value , __TAG_END__);
                        foundToken++; 

                        /*
                        .-------------------------------------------------------------.
                        | passa lo stato del TAG da 0 (TESTA OPEN) a 1 (TESTA CLOSE). |
                        '-------------------------------------------------------------' */       
                        rPharse.rTabTag[j-1].stato = 1;
                   };

                   /*
                   .----------------------------------------------------------------------------.
                   |                                                                            | 
                   |  Chiude il TAG di TESTA figlio del TAG in STATO 1 trovato (padre)          | 
                   |                                                                            | 
                   '----------------------------------------------------------------------------' */
                   if ( rPharse.rTabTag[j-1].stato == 1 ) {                         

                        /*
                        .--------------------------------------------------------.
                        | identificato il token delimitatore iniziale del tag    |
                        '--------------------------------------------------------' */       
                        token[foundToken].type  = malloc(strlen("TagIniClose\0"));
                        strcpy(token[foundToken].type  , "TagIniClose\0");
                        token[foundToken].value = malloc(strlen(__TAG_END__));
                        strcpy(token[foundToken].value , __TAG_END__);
                        foundToken++; 

                        /*
                        .-------------------------------------------------------------.
                        | passa lo stato del TAG da 0 (TESTA OPEN) a 1 (TESTA CLOSE). |
                        '-------------------------------------------------------------' */       
                        rPharse.rTabTag[rPharse.iTabTag-1].stato = 1;
                   };

                   /*
                   .----------------------------------------------------------------------------.
                   |                                                                            | 
                   |  Chiude il TAG di CODA                                                     | 
                   |                                                                            | 
                   '----------------------------------------------------------------------------' */
                   if ( rPharse.rTabTag[j-1].stato == 2 ) { 

                        /*
                        .--------------------------------------------------------.
                        | identificato il token delimitatore iniziale del tag    |
                        '--------------------------------------------------------' */       
                        token[foundToken].type  = malloc(strlen("TagEndClose\0"));
                        strcpy(token[foundToken].type  , "TagEndClose\0");
                        token[foundToken].value = malloc(strlen(__TAG_END__));
                        strcpy(token[foundToken].value , __TAG_END__);
                        foundToken++; 

                        /*
                        .-----------------------------------------------------------.
                        | passa lo stato del TAG da 2 (CODA OPEN) a 3 (CODA CLOSE). |
                        '-----------------------------------------------------------' */       
                        rPharse.rTabTag[j-1].stato = 3;
                   };


              };



              /*
              .-------------------------------------------------------------------------------------.
              |                                                                                     |
              |                                                                                     |
              |                                                                                     |
              |                                  __TAG_SEPARATORE__                                 |
              |                                                                                     |
              |                                                                                     |
              |                                                                                     |
              |                                                                                     |
              '-------------------------------------------------------------------------------------' */
              if ( search(buffer,__TAG_SEPARATORE__) == __TRUE__ && remark == __OFF__ && foundToken == __FALSE__ ) 
              {
                   /*
                   .---------------------------------------------------------.
                   | Isolamento della stringa alla sinistra del delimitatore |
                   '---------------------------------------------------------' */
                   i = i - strlen(__TAG_SEPARATORE__);
                   buffer[i] = __NULL__ ;


                   if ( i > __ZERO__ ) 
                   {
                        /*
                        .--------------------------------------------------------------.
                        | CASO 1: Delimitatore + TAG (es. <tag)                        |
                        .--------------------------------------------------------------.
                        | Alla sinistra del separatore  ci  può essere un delimitatore |
                        | iniziale di TAG con il nome del TAG  stesso  (es. "<tag". In |
                        | questo  caso  per  riconoscerlo   è  necessario  cercare  il |
                        | __TAG_INI_OPEN__ (<) nella stringa. Viene ricercato in tutta |
                        | la sottostringa senza considerare  una  specifica  posizione |
                        | questo perchè alla sinistra  del  delimitatore  ci  potrebbe |
                        | essere qualcos'altro (es. "unknown<tag". Se questo avviene è |
                        | un errore ed il pharsen si ferma  perchè  alla  sinistra  di |
                        | tale delimitatore (<) non ci può essere nulla  che  non  sia |
                        | stato classificato in precedenza come token.   Nel  caso  di |      
                        | eccezione: "<<tag" sarebbe un errore  andare  avanti  quindi |
                        | bisogna accertarsi che di delimitatori non ce ne sia piu' di |
                        | uno.                                                         |   
                        '--------------------------------------------------------------' */ 
		        if ( search(buffer,__TAG_INI_OPEN__) > __ZERO__ ) 
                        {
                             /*
                             .---------------------------------------------------------.
                             | Il delimitatore di apertura tag (<) è stato trovato si, |
                             | ma non in prima posizione (es. "unknown<tag ") quindi   |
                             | il parsen esce con un errore perchè alla sinistra del   |
                             | delimitatore non è previsto nulla. Qualunque            |
                             | delimitatore di chiusura precedente sarebbe stato       |
                             | individuato precedentemente.                            |
                             '---------------------------------------------------------' */
                             i=0;
                             while ( i < strlen(buffer) && buffer[i] != __TAG_INI_OPEN2__ ) i++;
                             if ( i > 0 ) {
                                  rPharse.row_error = countRowXMLFile + 1;
                                  rPharse.token_unknown = malloc(strlen(buffer));
                                  strcpy(rPharse.token_unknown,buffer);
                                  rPharse.token_unknown[i] = __NULL__ ;
                                  rPharse.return_code = -8;
                                  return rPharse; };

                             /*
                             .-------------------------------------------------------------.
                             | trovato il delimitatore di apertura tag in prima posizione  |
                             | ma trovato anche almeno un'altro medesimo delimitatore      |
                             | (es. "<tag<tag ") Questo significa per il parsen che il     |
                             | tag precedente (lo considera un tag il token tra i due      |
                             | delimitatori di apertura) non è stato chiuso e quindi esce  |
                             | per errore.                                                 |
                             '-------------------------------------------------------------' */
                             if ( search(buffer,__TAG_INI_OPEN__) > __ONE__ ) 
                             {
                                  /*
                                  .-----------------------------------------------------------------.
                                  | Questo TEST serve solo per differenziare il messaggio di errore |
                                  | per una maggiore comprensione: se nelle prime due posizioni     |
                                  | trova il delimitatore "es. <<tag" non si può parlare di         |
                                  | delimitatore di chiusura del tag non trovato bensi di un        |
                                  | delimitatore inserito più di una volta ad esempio per una       |
                                  | distrazione. Mentre nel caso in cui tra i due delimitatori ci   |
                                  | sia qualcosa di diverso "<tag<" in questo caso l'errore         |
                                  | potrebbe essere interpretato come di un delimitatore di         |
                                  | chiusura non trovato.                                           |
                                  '-----------------------------------------------------------------' */
                                  if (buffer[0]==__TAG_INI_OPEN2__ && buffer[1]==__TAG_INI_OPEN2__) 
                                  { 
                                      rPharse.row_error = countRowXMLFile + 1;
                                      rPharse.token_unknown = malloc(3);
                                      rPharse.token_unknown[0]=buffer[0];
                                      rPharse.token_unknown[1]=buffer[1];
                                      rPharse.token_unknown[2]=__NULL__;
                                      rPharse.return_code = -9;
                                      return rPharse; 
                                  }
                                  else
                                  {
                                      rPharse.token_unknown = malloc(strlen(buffer));
                                      i=1; 
                                      while (i<strlen(buffer) && buffer[i]!=__TAG_INI_OPEN2__) {
                                             rPharse.token_unknown[i-1] = buffer[i];
                                             i++; };
                                      rPharse.token_unknown[i-1]=__NULL__; 
                                      rPharse.return_code = -10;
                                      return rPharse;                                      
                                  }; 
                             };

                             /*
                             .---------------------------------------------------.
                             |                                                   |
                             |                                                   |
                             |                                                   |
                             |                                                   |
                             '---------------------------------------------------' */
                             if ( rPharse.iTabTagXML == 0 ) {
                                  rPharse.row_error = countRowXMLFile + 1;
                                  rPharse.token_unknown = malloc(strlen(buffer));
                                  strcpy(rPharse.token_unknown,buffer);
                                  rPharse.return_code = -40;
                                  return rPharse; 
                             };

                             /*
                             .---------------------------------------------------.
                             |                                                   |
                             |                                                   |
                             |                                                   |
                             |                                                   |
                             '---------------------------------------------------' */
                             if ( rPharse.iTabTagXML > 0 ) {
                                  if ( rPharse.rTabTagXML[rPharse.iTabTagXML-1].stato != 3 ) {
                                       rPharse.row_error = countRowXMLFile + 1;
                                       rPharse.token_unknown = malloc(strlen(rPharse.rTabTagXML[rPharse.iTabTagXML-1].tag));
                                       strcpy(rPharse.token_unknown,rPharse.rTabTagXML[rPharse.iTabTagXML-1].tag);
                                       rPharse.return_code = -999;
                                       return rPharse;
                                  };
                             };

                             /*
                             .-----------------------------------------------------------------.
                             | Rilegge a ritroso lo stack per cercare il padre, da lui ricava  |
                             | il proprio livello che corrisponde a quello del padre + 1.  Il  |
                             | padre è il primo che trova con stato uguale ad 1. Se lo stack   |
                             | vuoto unico caso in cui un nodo può non avere per XML padre il  |
                             | livello attribuito al nodo è __ZERO__. Lungo il traggitto se    |
                             | incontra tag a livello 0 oppure 2 esce con errore perchè        |
                             | significa che un suo fratello, nipote o il padre non è concluso |
                             | o completamente definito. Se legge tutto lo stack senza trovare |
                             | nessun nodo padre (cioè in stato 2) quindi solo nodi            |
                             | completamente definiti (cioè in stato 3) esce per errore perchè |
                             | non sono ammessi due nodi radice in XML.                        |
                             '-----------------------------------------------------------------' */
                             j = rPharse.iTabTag;
                             while ( j > 0 && rPharse.rTabTag[j-1].stato != 1 )
                             {
                                     /*
                                     .--------------------------------------------------.
                                     | Trovato TAG di TESTA non chiuso quindi quello    |
                                     | che si sta analizzando è definito all'interno    |
                                     | es. <tag .... <tag_analizzato                    |
                                     '--------------------------------------------------' */
                                     if ( rPharse.rTabTag[j-1].stato == 0 ) {
                                          rPharse.row_error = countRowXMLFile + 1;
                                          rPharse.token_unknown = malloc(strlen(rPharse.rTabTag[j-1].tag));
                                          strcpy(rPharse.token_unknown,rPharse.rTabTag[j-1].tag);
                                          rPharse.return_code = -11;
                                          return rPharse; };
                                     /*
                                     .-------------------------------------------------.
                                     | Trovato TAG di coda non chiuso quindi quello    |
                                     | che si sta analizzando è definito all'interno   |
                                     | es. <tag><\tag ... <tag_analizzato              |
                                     '-------------------------------------------------' */
                                     if ( rPharse.rTabTag[j-1].stato == 2 ) {
                                          rPharse.row_error = countRowXMLFile + 1;
                                          rPharse.token_unknown = malloc(strlen(rPharse.rTabTag[j-1].tag));
                                          strcpy(rPharse.token_unknown,rPharse.rTabTag[j-1].tag);
                                          rPharse.return_code = -12;
                                          return rPharse; };

                                     j--;
                             };

                             /*
                             .--------------------------------------------------.
                             | Tutti i nodi trovati sono in stato completamente |
                             | descritti ossia in stato 3 (es. <tag> ... <\tag> |
                             | il tag analizzato quindi è un nodo radice ma     |
                             | XML non permette piu' di un nodo radine quindi   |
                             | il pharsen esce per errore.                      |
                             '--------------------------------------------------' */
                             if ( rPharse.iTabTag > 0 && rPharse.rTabTag[0].stato == 3 ) {
                                  rPharse.row_error = countRowXMLFile + 1;
                                  rPharse.token_unknown = malloc(strlen(buffer));
                                  strcpy(rPharse.token_unknown,buffer);
                                  rPharse.return_code = -13;
                                  return rPharse; };

                             /*
                             .----------------------------------------------.
                             |          Inserimento di un nuovo TAG         |
                             '----------------------------------------------' */
                             rPharse.rTabTag[rPharse.iTabTag].tag = malloc(strlen("*Unknown*"));
                             strcpy(rPharse.rTabTag[rPharse.iTabTag].tag, "*Unknown*");
                             rPharse.rTabTag[rPharse.iTabTag].stato = 0;
                             rPharse.rTabTag[rPharse.iTabTag].level = 0;
                             if (rPharse.iTabTag > 0)
                                 rPharse.rTabTag[rPharse.iTabTag].level = rPharse.rTabTag[j-1].level+1;
                             rPharse.rTabTag[rPharse.iTabTag].dimTabVar = 0;
                             rPharse.iTabTag++;

                             /*
                             .--------------------------------------------------------.
                             | identificato il token delimitatore iniziale del tag    |
                             '--------------------------------------------------------' */       
                             token[foundToken].type  = malloc(strlen("TagIniOpen\0"));
                             strcpy(token[foundToken].type  , "TagIniOpen\0");
                             token[foundToken].value = malloc(strlen(__TAG_INI_OPEN__));
                             strcpy(token[foundToken].value , __TAG_INI_OPEN__);
                             foundToken++;

                             /* 
                             .--------------------------------------------------------.
                             | elimina dal buffer il delimitatore lasciando solo      |
                             | il nome tag "<tag\0" ==> "tag\0".                      |
                             | Successivamente verrà catalogato il tag.               |
                             | Tale codice non è riportato in questo blocco perchè    |
                             | potrebbe accadere anche che tra delimitatore e tag ci  |
                             | sia uno spazio di mezzo "< tag". In questo caso i due  |
                             | token verrebbero letti in due bufferizzazioni          |
                             | differenti anche se consecutive.                       |
                             '--------------------------------------------------------' */ 
                             j = 1;
                             while (buffer[j]!=__ZERO__) { 
                                    buffer[j-1] = buffer[j];
                                    j++;
                             };
                             buffer[j-1] = __ZERO__;
                             i = strlen(buffer);

                        };
                   };


                   if ( i > __ZERO__ ) 
                   {                    
                        /*
                        .--------------------------------------------------------------.
                        |                                                              |
                        .--------------------------------------------------------------.
                        |                                                              |
                        |                                                              |
                        |                                                              |
                        |                                                              |
                        |                                                              |
                        '--------------------------------------------------------------' */
                        if ( rPharse.iTabTag > 0 && rPharse.rTabTag[rPharse.iTabTag-1].stato == 0 ) 
                        {
                             /*
                             .--------------------------------------------------------------.
                             | CASO 2: Tag (es. < [tag])                                    |
                             .--------------------------------------------------------------.
                             |                                                              |
                             |                                                              |
                             |                                                              |
                             |                                                              |
                             |                                                              |
                             '--------------------------------------------------------------' */
                             if ( strcmp(rPharse.rTabTag[rPharse.iTabTag-1].tag,"*Unknown*")==0 ) 
                             {
                                  /*
                                  .----------------------------------------------.
                                  |  Classificazione e valorizzazione del Token  |
                                  '----------------------------------------------' */
                                  token[foundToken].type = malloc(strlen("TagIni\0"));
                                  strcpy(token[foundToken].type, "TagIni\0");

                                  /*
                                  .----------------------------------------------.
                                  |           Aggiornamento nome TAG             |
                                  '----------------------------------------------' */
                                  rPharse.rTabTag[rPharse.iTabTag-1].tag = malloc(strlen(buffer));
                                  strcpy(rPharse.rTabTag[rPharse.iTabTag-1].tag, buffer);
                             }
                             else
                             {
                                  switch ( rPharse.rTabTag[rPharse.iTabTag-1].stato_last_variable ) 
                                  { 
                                       case 0: 
                                 
                                           /*
                                           .--------------------------------------------------------------.
                                           | CASO 3: Variable (es. <tag [var] )                           |
                                           .--------------------------------------------------------------.
                                           |                                                              |
                                           |                                                              |
                                           |                                                              |
                                           |                                                              |
                                           |                                                              |
                                           '--------------------------------------------------------------' */

                                           /*
                                           .----------------------------------------------.
                                           |  Classificazione e valorizzazione del Token  |
                                           '----------------------------------------------' */
                                           token[foundToken].type  = malloc(strlen("Variable\0"));
                                           strcpy(token[foundToken].type  , "Variable\0");

                                           /*
                                           .---------------------------------------------------.
                                           | Inserisce la nuova variabile nello stack|stackXML |
                                           '---------------------------------------------------' */
                                           int itoken = rPharse.iTabTag-1 ;
                                           int itokenVar = rPharse.rTabTag[itoken].dimTabVar ;
                                           rPharse.rTabTag[itoken].rTabVar[itokenVar].var = malloc(strlen(buffer));
                                           strcpy(rPharse.rTabTag[itoken].rTabVar[itokenVar].var, buffer);

                                           /*
                                           .----------------------------------------------------------.
                                           | Aggiornamento dello stato dell'ultima variabile inserita |
                                           | al valore di __UNO__ aperta definizione.                 |
                                           '----------------------------------------------------------' */
                                           rPharse.rTabTag[rPharse.iTabTag-1].stato_last_variable = 1;
                                           break;


                                       case 1:

                                           /*
                                           .--------------------------------------------------------------.
                                           | CASO 4: Manca ASSIGN (es. <tag var [word])                   |
                                           .--------------------------------------------------------------.
                                           |                                                              |
                                           |                                                              |
                                           |                                                              |
                                           |                                                              |
                                           |                                                              |
                                           '--------------------------------------------------------------' */
                                           rPharse.row_error = countRowXMLFile + 1;
                                           rPharse.token_unknown = malloc(strlen(buffer));
                                           strcpy(rPharse.token_unknown,buffer);
                                           rPharse.return_code = -37;
                                           return rPharse; 
                                           break;


                                       case 2:

                                           /*
                                           .--------------------------------------------------------------.
                                           | CASO 5: Value (es. <tag var = [value])                       |
                                           .--------------------------------------------------------------.
                                           |                                                              |
                                           |                                                              |
                                           |                                                              |
                                           |                                                              |
                                           |                                                              |
                                           '--------------------------------------------------------------' */

                                           /*
                                           .----------------------------------------------.
                                           |  Classificazione e valorizzazione del Token  |
                                           '----------------------------------------------' */
                                           token[foundToken].type  = malloc(strlen("Value\0"));
                                           strcpy(token[foundToken].type  , "Value\0");

                                           /*
                                           .----------------------------------------------------.
                                           | Aggiornamento del valore per l'ultima variabile    |
                                           | memorizzata con stato a 1 (mancanza di valore)     |
                                           '----------------------------------------------------' */
                                           int istack = rPharse.iTabTag-1;
                                           int istackVar = rPharse.rTabTag[istack].dimTabVar;
                                           rPharse.rTabTag[istack].rTabVar[istackVar].value = malloc(strlen(buffer));
                                           strcpy(rPharse.rTabTag[istack].rTabVar[istackVar].value, buffer);
                                           rPharse.rTabTag[rPharse.iTabTag-1].dimTabVar++;

                                           /*
                                           .----------------------------------------------------------.
                                           | Aggiornamento dello stato dell'ultima variabile inserita |
                                           | al valore di __ZERO__ conclusa definizione.              |
                                           '----------------------------------------------------------' */
                                           rPharse.rTabTag[rPharse.iTabTag-1].stato_last_variable = 0;
                                           break;
                                  };
                             };
                        };


                        /*
                        .--------------------------------------------------------------.
                        |                                                              |
                        .--------------------------------------------------------------.
                        |                                                              |
                        |                                                              |
                        |                                                              |
                        |                                                              |
                        |                                                              |
                        '--------------------------------------------------------------' */
                        if ( rPharse.iTabTagXML > 0 && rPharse.rTabTagXML[rPharse.iTabTagXML-1].stato == 0 ) 
                        {
                             /*
                             .--------------------------------------------------------------.
                             | CASO 6: Tag (es. < [tagXML])                                 |
                             .--------------------------------------------------------------.
                             |                                                              |
                             |                                                              |
                             |                                                              |
                             |                                                              |
                             |                                                              |
                             '--------------------------------------------------------------' */
                             if ( strcmp(rPharse.rTabTagXML[rPharse.iTabTagXML-1].tag,"*Unknown*") == 0 )  
                             {
                                  /*
                                  .----------------------------------------------.
                                  |  Classificazione e valorizzazione del Token  |
                                  '----------------------------------------------' */
                                  token[foundToken].type  = malloc(strlen("XML\0"));
                                  strcpy(token[foundToken].type  , "XML\0");

                                  /*
                                  .----------------------------------------------.
                                  |          Aggiornamento nome TAG XML          |
                                  '----------------------------------------------' */
                                  rPharse.rTabTagXML[rPharse.iTabTagXML-1].tag = malloc(strlen(buffer));
                                  strcpy(rPharse.rTabTagXML[rPharse.iTabTagXML-1].tag, buffer);
                             }
                             else
                             {
                                  switch ( rPharse.rTabTagXML[rPharse.iTabTagXML-1].stato_last_variable )
                                  {
                                       case 0:

                                           /*
                                           .--------------------------------------------------------------.
                                           | CASO 7: Variable (es. <tagXML [var] )                        |
                                           .--------------------------------------------------------------.
                                           |                                                              |
                                           |                                                              |
                                           |                                                              |
                                           |                                                              |
                                           |                                                              |
                                           '--------------------------------------------------------------' */

                                           /*
                                           .----------------------------------------------.
                                           |  Classificazione e valorizzazione del Token  |
                                           '----------------------------------------------' */
                                           token[foundToken].type  = malloc(strlen("Variable\0"));
                                           strcpy(token[foundToken].type  , "Variable\0");

                                           /*
                                           .---------------------------------------------------.
                                           | Inserisce la nuova variabile nello stack|stackXML |
                                           '---------------------------------------------------' */
                                           int itoken = rPharse.iTabTagXML-1 ;
                                           int itokenVar = rPharse.rTabTagXML[itoken].dimTabVar ;
                                           rPharse.rTabTagXML[itoken].rTabVar[itokenVar].var = malloc(strlen(buffer));
                                           strcpy(rPharse.rTabTagXML[itoken].rTabVar[itokenVar].var, buffer);

                                           /*
                                           .----------------------------------------------------------.
                                           | Aggiornamento dello stato dell'ultima variabile inserita |
                                           | al valore di __UNO__ aperta definizione.                 |
                                           '----------------------------------------------------------' */
                                           rPharse.rTabTagXML[rPharse.iTabTagXML-1].stato_last_variable = 1;
                                           break;


                                       case 1:

                                           /*
                                           .--------------------------------------------------------------.
                                           | CASO 7: Manca ASSIGN (es. <tagXML var [word])                |
                                           .--------------------------------------------------------------.
                                           |                                                              |
                                           |                                                              |
                                           |                                                              |
                                           |                                                              |
                                           |                                                              |
                                           '--------------------------------------------------------------' */
                                           rPharse.row_error = countRowXMLFile + 1;
                                           rPharse.token_unknown = malloc(strlen(buffer));
                                           strcpy(rPharse.token_unknown,buffer);
                                           rPharse.return_code = -37;
                                           return rPharse;
                                           break;


                                       case 2:

                                           /*
                                           .--------------------------------------------------------------.
                                           | CASO 8: Value (es. <tagXML var = [value])                    |
                                           .--------------------------------------------------------------.
                                           |                                                              |
                                           |                                                              |
                                           |                                                              |
                                           |                                                              |
                                           |                                                              |
                                           '--------------------------------------------------------------' */

                                           /*
                                           .----------------------------------------------.
                                           |  Classificazione e valorizzazione del Token  |
                                           '----------------------------------------------' */
                                           token[foundToken].type  = malloc(strlen("Value\0"));
                                           strcpy(token[foundToken].type  , "Value\0");

                                           /*
                                           .----------------------------------------------------.
                                           | Aggiornamento del valore per l'ultima variabile    |
                                           | memorizzata con stato a 1 (mancanza di valore)     |
                                           '----------------------------------------------------' */
                                           int istack = rPharse.iTabTagXML-1;
                                           int istackVar = rPharse.rTabTagXML[istack].dimTabVar;
                                           rPharse.rTabTagXML[istack].rTabVar[istackVar].value = malloc(strlen(buffer));
                                           strcpy(rPharse.rTabTagXML[istack].rTabVar[istackVar].value, buffer);
                                           rPharse.rTabTagXML[rPharse.iTabTagXML-1].dimTabVar++;

                                           /*
                                           .----------------------------------------------------------.
                                           | Aggiornamento dello stato dell'ultima variabile inserita |
                                           | al valore di __ZERO__ conclusa definizione.              |
                                           '----------------------------------------------------------' */
                                           rPharse.rTabTagXML[rPharse.iTabTagXML-1].stato_last_variable = 0;
                                           break;
                                  };
                             };
                        };

                        token[foundToken].value  = malloc(strlen(buffer));
                        strcpy(token[foundToken].value , buffer);
                        foundToken++;                      
                   };

                   token[foundToken].type  = malloc(strlen("Separate\0"));
                   token[foundToken].value  = malloc(strlen(__TAG_SEPARATORE__));
                   strcpy(token[foundToken].type  , "Separate\0");
                   strcpy(token[foundToken].value , __TAG_SEPARATORE__);
                   foundToken++;
              };



              /*
              .-------------------------------------------------------------------------------------.
              |                                                                                     |
              |                                                                                     |
              |                                                                                     |
              |                                    __TAG_ASSIGN__                                   |
              |                                                                                     |
              |                                                                                     |
              |                                                                                     |
              |                                                                                     |
              '-------------------------------------------------------------------------------------' */
              if ( search(buffer,__TAG_ASSIGN__) == __TRUE__ && remark == __OFF__ && foundToken == __FALSE__ ) 
              {
                   /*
                   .-----------------------------------------------------------.
                   | Isolamento della stringa alla sinistra del __TAG_ASSIGN__ |
                   '-----------------------------------------------------------' */
                   i = i - strlen(__TAG_ASSIGN__);
                   buffer[i] = __NULL__;

                   /*
                   .-----------------------------------------------------------------------------------.
                   | Err 1. Stack XML vuoto.                                                           |
                   .-----------------------------------------------------------------------------------.
                   | Un token di ASSIGN presuppone che [1] ci sia un TAG aperto [2] che il nometag     |
                   | sia assegnato e [3] che ci sia nello stack almeno una variabile in attesa di      |
                   | valore... (stato 1). Fatta questa premessa... Il primo TEST da implementare è     |
                   | sulla presenta del TAG di TESTA aperto. Se lo stack XML è vuoto non c'è scampo, è |
                   | un errore, perchè anche se ci fosse qualcosa alla sinistra del token analizzato   |
                   | questo non potrebbe essere un TAG XML "<?xml nometag" perchè il TOKEN interrompe  |
                   | la bufferizzazione e quindi questo non può trovarsi nel buffer ma risulterebbe    |
                   | già processato ed il TAG già inserito nello stack.                                | 
                   '-----------------------------------------------------------------------------------' */
                   if (rPharse.iTabTagXML == 0) {  
                       rPharse.row_error = countRowXMLFile + 1;
                       rPharse.token_unknown = malloc(strlen("\0")); 
                       strcpy(rPharse.token_unknown,"\0");
                       rPharse.return_code = -56;
                       return rPharse; 
                   }
                   else
                   {
                       /*
                       .----------------------------------------------------------------------------.
                       | Err 2. TAG XML Open not found                                              |
                       .----------------------------------------------------------------------------.
                       | Se lo stack XML non è vuoto è necessario (per continuare) che l'ultimo TAG |
                       | inserito sia APERTO e di TESTA (il TAG XML può avere stato 0 o 3 perchè    |
                       | ha solo la TESTA). Quindi se non è ZERO non è ritenuto al momento un       |
                       | errore perchè significhe che tale TAG è in stato 3 quindi consolidato e    |
                       | l'ASSIGN potrebbe far riferimento ad un TAG DATI. Per essere errore allo   |
                       | stato 3 (consolidato) del TAG XML deve corrispondere anche lo stack dati   |
                       | vuoto. Questo vuol dire che l'assign è successivo alla chiusura di un TAG  |
                       | XML. (es. <?xml version = "1.0"?>=|<?xml version = "1.0"?>unknown=)        |
                       '----------------------------------------------------------------------------' */
                       if (rPharse.rTabTagXML[rPharse.iTabTagXML-1].stato == 3 && rPharse.iTabTag == 0 )
                       {
                           rPharse.row_error = countRowXMLFile + 1;
                           rPharse.token_unknown = malloc(strlen("\0")); 
                           strcpy(rPharse.token_unknown,"\0");
                           rPharse.return_code = -57;
                           return rPharse;
                       };

 
                       /*
                       .----------------------------------------------------------------------------.
                       | Ok competenza STACK XML                                                    |
                       .----------------------------------------------------------------------------.
                       | Se lo stack XML è pieno e l'ultimo TAG è in stato 0 quindi aperto.         |
                       | L'assign quindi è posizionato nel contesto TAG XML.                        |
                       '----------------------------------------------------------------------------' */
                       if (rPharse.rTabTagXML[rPharse.iTabTagXML-1].stato == 0 )
                       {
                           /*
                           .------------------------------------------------------------------.
                           | Err. 3. TAG havent a name.                                       |
                           .------------------------------------------------------------------.
                           | Il TAG trovato aperto non ha ancora un nome assegnato. Questo è  |
                           | un errore perchè anche se ci fosse qualcosa alla sinistra del    |
                           | token analizzato non potrebbe essere contestualmente nometag e   |
                           | nome variabile che per essere validi devono essere separati da   |
                           | uno spazio e questo avrebbe interrotto la bufferizzazione.       |
                           '------------------------------------------------------------------' */
                           if ( strcmp(rPharse.rTabTagXML[rPharse.iTabTagXML-1].tag,"*Unknown*") == 0 ) {
                                rPharse.row_error = countRowXMLFile + 1;
                                rPharse.token_unknown = malloc(strlen("\0"));
                                strcpy(rPharse.token_unknown,"\0");
                                rPharse.return_code = -58; 
                                return rPharse; 
                           };

                           /*
                           .-------------------------------------------------------------------.
                           | Ok. Verifica stato variabili.                                     |
                           .-------------------------------------------------------------------.
                           | Il TAG è stato trovato aperto, il nome risulta assegnato...       |
                           | Adesso il parse verifica lo stato delle variabili. Per essere     |
                           | valido l'assign il parse deve trovare una variabile (l'ultima)    |
                           | senza assign... ma bisogan tener contro sempre del buffer non     |
                           | ancora processato naturalmente.                                   |
                           '-------------------------------------------------------------------' */
                           switch ( rPharse.rTabTagXML[rPharse.iTabTagXML-1].stato_last_variable )
                           {

                                case 0 :
                                         /*
                                         .-------------------------------------------------------------------.
                                         | err 4. lo stack variabili è vuoto oppure ultima variabile consol. |
                                         .-------------------------------------------------------------------.
                                         | lo stack delle variabili è vuoto oppure l'ultima variabile        |
                                         | inserità è consistente (definizione completata). Quindi siamo di  |
                                         | fronte ad una possibile nuova definizione di variabile, che       |
                                         | potrebbe risultare alla sinistra del token che stiamo analizzando |
                                         | Se però alla sinistra non c'è nulla (i è la lunghezza della       |
                                         | stringa alla sinistra) questo è chiaramente la prova che manca    |
                                         | il nome della variabile e quindi è un errore.                     |
                                         '-------------------------------------------------------------------' */
                                         if ( i == 0 ) {
                                              rPharse.row_error = countRowXMLFile + 1;
                                              rPharse.token_unknown = malloc(strlen("\0"));
                                              strcpy(rPharse.token_unknown,"\0");
                                              rPharse.return_code = -59; 
                                              return rPharse; 
                                         }; 
 
                                         /*
                                         .-------------------------------------------------------------------.
                                         | Ok. identificato nuova variabile nel buffer                       |
                                         .-------------------------------------------------------------------.
                                         | Ricapitolando: TAG Aperto, il nome è assegnato. Nello stack la    |
                                         | delle variabili è consolidata ma alla sinistra dell'assign c'è    |
                                         | un token che viene considerato come variabile. E' necessario      |
                                         | registrare il token variabile, inserire nello stack la nuova      |
                                         | variabile e cambiato lo stato dell'ultima variabile acquisita da  |
                                         | zero ad uno (inserimento nuova variabile in attesa di assign).    |
                                         '-------------------------------------------------------------------' */

                                         /*
                                         .-----------------------------------------.
                                         | Identificato un token Variable          |
                                         '-----------------------------------------' */
                                         token[foundToken].type  = malloc(strlen("Variable\0"));
                                         token[foundToken].value = malloc(strlen(buffer));
                                         strcpy(token[foundToken].type  , "Variable\0");
                                         strcpy(token[foundToken].value , buffer);
                                         foundToken++;

                                         /*
                                         .---------------------------------------------------.
                                         | Inserisce la nuova variabile nello stack          |
                                         '---------------------------------------------------' */
                                         int itoken = rPharse.iTabTagXML-1 ;
                                         int itokenVar = rPharse.rTabTagXML[itoken].dimTabVar ;
                                         rPharse.rTabTagXML[itoken].rTabVar[itokenVar].var = malloc(strlen(buffer));
                                         strcpy(rPharse.rTabTagXML[itoken].rTabVar[itokenVar].var, buffer); 

                                         /*
                                         .---------------------------------------------------.
                                         | Passa lo stato da ZERO a UNO nello stack          |
                                         '---------------------------------------------------' */
                                         rPharse.rTabTagXML[itoken].stato_last_variable = 1;                            

                                         break;

                 
                                case 1 :
                                         /*
                                         .-------------------------------------------------------------------.
                                         | err 5. Unknown token in buffer                                    |
                                         .-------------------------------------------------------------------.
                                         | Se il TAG è aperto il nome è assegnato ma l'ultima variabile      |
                                         | inserita nello stack è in stato 1 (quindi in attesa di assign)    |
                                         | ma alla sinistra del token analizzato (assign) è presente         |
                                         | qualcosa di sconosciuto (a questo punto) e di non classificabile  |
                                         | pertanto il parse si ferma con questo errore.                     |
                                         '-------------------------------------------------------------------' */
                                         if ( i > 0) {
                                              rPharse.row_error = countRowXMLFile + 1;
                                              rPharse.token_unknown = malloc(strlen(buffer));
                                              strcpy(rPharse.token_unknown,buffer);
                                              rPharse.return_code = -60; 
                                              return rPharse; 
                                         };
                                            
                                         break;

                                case 2 :
                                         /*
                                         .-------------------------------------------------------------------.
                                         | err 6. con variabile ed assign errore                             |
                                         .-------------------------------------------------------------------.
                                         | Se il TAG è aperto il nome è assegnato ma l'ultima variabile      |
                                         | inserita nello stack è in stato 2 (quindi in attesa del value)    |
                                         | ma il token analizzato è un assign (doppio a questo punto) e di   |
                                         | non classificabile pertanto il parse si ferma con questo errore.  |
                                         '-------------------------------------------------------------------' */
                                         itoken = rPharse.iTabTagXML-1 ;
                                         itokenVar = rPharse.rTabTagXML[itoken].dimTabVar ;
                                         
                                         rPharse.row_error = countRowXMLFile + 1;
                                         rPharse.token_unknown = malloc(strlen(rPharse.rTabTagXML[itoken].rTabVar[itokenVar].var));
                                         strcpy(rPharse.token_unknown,rPharse.rTabTagXML[itoken].rTabVar[itokenVar].var);
                                         rPharse.return_code = -61; 
                                         return rPharse; 

                                         break; 
                           };

                           /*
                           .---------------------------------------------------.
                           | Passa lo stato da ZERO a UNO nello stack          |
                           '---------------------------------------------------' */
                           rPharse.rTabTagXML[rPharse.iTabTagXML-1].stato_last_variable = 2;

                           /*
                           .-----------------------------------------.
                           | Identificato un token Assign            |
                           '-----------------------------------------' */
                           token[foundToken].type  = malloc(strlen("Assign\0"));
                           token[foundToken].value = malloc(strlen(__TAG_ASSIGN__));
                           strcpy(token[foundToken].type  , "Assign\0"); 
                           strcpy(token[foundToken].value , __TAG_ASSIGN__) ;
                           foundToken++;
                       };

                   };


                   /*
                   .----------------------------------------------------------------------------.
                   | Ok. Assign di competenza dello stack dati utente.                          |
                   .----------------------------------------------------------------------------.
                   | Se lo stack XML è risultato consolidato (non vuoto e ultimo TAG            |
                   | consolidato - stato 3 - ) e lo stack dati utente è pieno di qualcosa       |
                   | fa riferimento ad un TAG di competenza di quest'ultimo. Attenzione nel     |
                   | in cui è di pertinenza invece dello stack XML in questo ramo nemmeno ci    |
                   | entra perchè tale stack è vuoto se i controlli precedenti funzionano a     |
                   | dovere perchè ad ogni token analizzato si effettua il controllo che prima  |
                   | di iniziare a scrivere qua' sia consolidato lo stack XML.                  | 
                   '----------------------------------------------------------------------------' */
                   if (rPharse.iTabTag > 0) 
                   {  
                       /*
                       .----------------------------------------------------------------------------.
                       | Ok competenza STACK Dati Utente                                            |
                       .----------------------------------------------------------------------------.
                       | Se lo stack è pieno e l'ultimo TAG è in stato 0 quindi aperto.             |
                       | L'assign quindi è posizionato nel contesto TAG dati Utente.                |
                       '----------------------------------------------------------------------------' */
                       if (rPharse.rTabTag[rPharse.iTabTag-1].stato == 0 )
                       {
                           /*
                           .------------------------------------------------------------------.
                           | Err. 7. TAG havent a name.                                       |
                           .------------------------------------------------------------------.
                           | Il TAG trovato aperto non ha ancora un nome assegnato. Questo è  |
                           | un errore perchè anche se ci fosse qualcosa alla sinistra del    |
                           | token analizzato non potrebbe essere contestualmente nometag e   |
                           | nome variabile che per essere validi devono essere separati da   |
                           | uno spazio e questo avrebbe interrotto la bufferizzazione.       |
                           '------------------------------------------------------------------' */
                           if ( strcmp(rPharse.rTabTag[rPharse.iTabTag-1].tag,"*Unknown*") == 0 ) {
                                rPharse.row_error = countRowXMLFile + 1;
                                rPharse.token_unknown = malloc(strlen("\0"));
                                strcpy(rPharse.token_unknown,"\0");
                                rPharse.return_code = -62; 
                                return rPharse; 
                           };


                           /*
                           .-------------------------------------------------------------------.
                           | Ok. Verifica stato variabili.                                     |
                           .-------------------------------------------------------------------.
                           | Il TAG è stato trovato aperto, il nome risulta assegnato...       |
                           | Adesso il parse verifica lo stato delle variabili. Per essere     |
                           | valido l'assign il parse deve trovare una variabile (l'ultima)    |
                           | senza assign... ma bisogan tener contro sempre del buffer non     |
                           | ancora processato naturalmente.                                   |
                           '-------------------------------------------------------------------' */
                           switch ( rPharse.rTabTag[rPharse.iTabTag-1].stato_last_variable )
                           {

                                case 0 :
                                         /*
                                         .-------------------------------------------------------------------.
                                         | err 8. lo stack variabili è vuoto oppure ultima variabile consol. |
                                         .-------------------------------------------------------------------.
                                         | lo stack delle variabili è vuoto oppure l'ultima variabile        |
                                         | inserità è consistente (definizione completata). Quindi siamo di  |
                                         | fronte ad una possibile nuova definizione di variabile, che       |
                                         | potrebbe risultare alla sinistra del token che stiamo analizzando |
                                         | Se però alla sinistra non c'è nulla (i è la lunghezza della       |
                                         | stringa alla sinistra) questo è chiaramente la prova che manca    |
                                         | il nome della variabile e quindi è un errore.                     |
                                         '-------------------------------------------------------------------' */
                                         if ( i == 0 ) {
                                              rPharse.row_error = countRowXMLFile + 1;
                                              rPharse.token_unknown = malloc(strlen("\0"));
                                              strcpy(rPharse.token_unknown,"\0");
                                              rPharse.return_code = -63; 
                                              return rPharse; 
                                         }; 
 
                                         /*
                                         .-------------------------------------------------------------------.
                                         | Ok. identificato nuova variabile nel buffer                       |
                                         .-------------------------------------------------------------------.
                                         | Ricapitolando: TAG Aperto, il nome è assegnato. Nello stack la    |
                                         | delle variabili è consolidata ma alla sinistra dell'assign c'è    |
                                         | un token che viene considerato come variabile. E' necessario      |
                                         | registrare il token variabile, inserire nello stack la nuova      |
                                         | variabile e cambiato lo stato dell'ultima variabile acquisita da  |
                                         | zero ad uno (inserimento nuova variabile in attesa di assign).    |
                                         '-------------------------------------------------------------------' */

                                         /*
                                         .-----------------------------------------.
                                         | Identificato un token Variable          |
                                         '-----------------------------------------' */
                                         token[foundToken].type  = malloc(strlen("Variable\0"));
                                         token[foundToken].value = malloc(strlen(buffer));
                                         strcpy(token[foundToken].type  , "Variable\0");
                                         strcpy(token[foundToken].value , buffer);
                                         foundToken++;

                                         /*
                                         .---------------------------------------------------.
                                         | Inserisce la nuova variabile nello stack          |
                                         '---------------------------------------------------' */
                                         int itoken = rPharse.iTabTag-1 ;
                                         int itokenVar = rPharse.rTabTag[itoken].dimTabVar ;
                                         rPharse.rTabTag[itoken].rTabVar[itokenVar].var = malloc(strlen(buffer));
                                         strcpy(rPharse.rTabTag[itoken].rTabVar[itokenVar].var, buffer); 

                                         /*
                                         .---------------------------------------------------.
                                         | Passa lo stato da ZERO a UNO nello stack          |
                                         '---------------------------------------------------' */
                                         rPharse.rTabTag[itoken].stato_last_variable = 1;                            

                                         break;

                 
                                case 1 :
                                         /*
                                         .-------------------------------------------------------------------.
                                         | err 9. Unknown token in buffer                                    |
                                         .-------------------------------------------------------------------.
                                         | Se il TAG è aperto il nome è assegnato ma l'ultima variabile      |
                                         | inserita nello stack è in stato 1 (quindi in attesa di assign)    |
                                         | ma alla sinistra del token analizzato (assign) è presente         |
                                         | qualcosa di sconosciuto (a questo punto) e di non classificabile  |
                                         | pertanto il parse si ferma con questo errore.                     |
                                         '-------------------------------------------------------------------' */
                                         if ( i > 0) {
                                              rPharse.row_error = countRowXMLFile + 1;
                                              rPharse.token_unknown = malloc(strlen(buffer));
                                              strcpy(rPharse.token_unknown,buffer);
                                              rPharse.return_code = -64; 
                                              return rPharse; 
                                         };
                                            
                                         break;

                                case 2 :
                                         /*
                                         .-------------------------------------------------------------------.
                                         | err 10. Con variabile ed assign errore                            |
                                         .-------------------------------------------------------------------.
                                         | Se il TAG è aperto il nome è assegnato ma l'ultima variabile      |
                                         | inserita nello stack è in stato 2 (quindi in attesa del value)    |
                                         | ma il token analizzato è un assign (doppio a questo punto) e di   |
                                         | non classificabile pertanto il parse si ferma con questo errore.  |
                                         '-------------------------------------------------------------------' */
                                         itoken = rPharse.iTabTagXML-1 ;
                                         itokenVar = rPharse.rTabTagXML[itoken].dimTabVar ;
                                         
                                         rPharse.row_error = countRowXMLFile + 1;
                                         rPharse.token_unknown = malloc(strlen(rPharse.rTabTagXML[itoken].rTabVar[itokenVar].var));
                                         strcpy(rPharse.token_unknown,rPharse.rTabTagXML[itoken].rTabVar[itokenVar].var);
                                         rPharse.return_code = -65; 
                                         return rPharse; 

                                         break; 
                           };

                       };


                       /*
                       .----------------------------------------------------------------------------.
                       |  err 11. Nessun TAG TESTA aperto. (ultimo TAG testa chiuso)                |
                       .----------------------------------------------------------------------------.
                       |  L'ultimo TAG è una TESTA chiusa e quindi il parse si ferma.               |
                       |  es.1: <tag> =                                                             |
                       |  es.2: <tag> *Unknown*=                                                    |
                       '----------------------------------------------------------------------------' */
                       if (rPharse.rTabTag[rPharse.iTabTag-1].stato == 1 )
                       {
                           rPharse.row_error = countRowXMLFile + 1;
                           rPharse.token_unknown = malloc(strlen("\0"));
                           strcpy(rPharse.token_unknown,"\0");
                           rPharse.return_code = -66;
                           return rPharse; 
                       };

                       /*
                       .----------------------------------------------------------------------------.
                       | err 12. Nessun TAG TESTA aperto. (ultimo TAG coda aperto)                  |
                       .----------------------------------------------------------------------------.
                       |  L'ultimo TAG è una CODA aperta e quindi il parse si ferma.                |
                       |  es.1: <tag></tag =                                                        |
                       |  es.2: <tag></tag *Unknown*=                                               |
                       '----------------------------------------------------------------------------' */
                       if (rPharse.rTabTag[rPharse.iTabTag-1].stato == 2 )
                       {
                           rPharse.row_error = countRowXMLFile + 1;
                           rPharse.token_unknown = malloc(strlen("\0"));
                           strcpy(rPharse.token_unknown,"\0");
                           rPharse.return_code = -67;
                           return rPharse; 
                       };

                       /*
                       .----------------------------------------------------------------------------.
                       | err 13. Nessun TAG TESTA aperto. (ultimo TAG coda chiuso - consolidato)    |
                       .----------------------------------------------------------------------------.
                       |  L'ultimo TAG è una CODA chiusa e quindi il parse si ferma.                |
                       |  es.1: <tag></tag> =                                                       |
                       |  es.2: <tag></tag> *Unknown*=                                              |
                       '----------------------------------------------------------------------------' */
                       if (rPharse.rTabTag[rPharse.iTabTag-1].stato == 3 )
                       {
                           rPharse.row_error = countRowXMLFile + 1;
                           rPharse.token_unknown = malloc(strlen("\0"));
                           strcpy(rPharse.token_unknown,"\0");
                           rPharse.return_code = -68; 
                           return rPharse; 
                       };


                       /*
                       .---------------------------------------------------.
                       | Passa lo stato da ZERO a UNO nello stack          |
                       '---------------------------------------------------' */
                       rPharse.rTabTag[rPharse.iTabTag-1].stato_last_variable = 2;

                       /*
                       .-----------------------------------------.
                       | Identificato un token Assign            |
                       '-----------------------------------------' */
                       token[foundToken].type  = malloc(strlen("Assign\0"));
                       token[foundToken].value = malloc(strlen(__TAG_ASSIGN__));
                       strcpy(token[foundToken].type  , "Assign\0"); 
                       strcpy(token[foundToken].value , __TAG_ASSIGN__) ;
                       foundToken++;

                   };

              };

          }; 


          /*
          .-------------------------------------------------------------------------------------.
          |                                                                                     |
          |                                                                                     |
          |                                                                                     |
          |                           PRINT TOKEN FOUND IN THE BUFFER                           |
          |                                                                                     |
          |                                                                                     |
          |                                                                                     |
          |                                                                                     |
          '-------------------------------------------------------------------------------------' */
          if ( foundToken > __ZERO__ && ignore == __OFF__ ) 
          {
               j = 0;
               while ( j < foundToken )
               {
                       switch (debug_level)
                       {
                           case 5 : printf ("\n[DEBUG] [%15s] | [%80s]", token[j].type, token[j].value); 
                                    break; 

                           case 4 : if (strcmp(token[j].type,"RemarkIni"  )!=0 &&
                                        strcmp(token[j].type,"RemarkEnd"  )!=0 &&
                                        strcmp(token[j].type,"Remark"     )!=0 ) 
                                        printf ("\n[DEBUG] [%15s] | [%80s]", token[j].type, token[j].value); 
                                        break;   

                           case 3 : if (strcmp(token[j].type,"RemarkIni"  )!=0 &&
                                        strcmp(token[j].type,"RemarkEnd"  )!=0 &&
                                        strcmp(token[j].type,"Remark"     )!=0 &&
                                        strcmp(token[j].type,"Assign"     )!=0 &&
                                        strcmp(token[j].type,"Separate"   )!=0)  
                                        printf ("\n[DEBUG] [%15s] | [%80s]", token[j].type, token[j].value); 
                                        break;   
                           
                           case 2 : if (strcmp(token[j].type,"RemarkIni"  )!=0 &&
                                        strcmp(token[j].type,"RemarkEnd"  )!=0 &&
                                        strcmp(token[j].type,"Remark"     )!=0 &&
                                        strcmp(token[j].type,"Assign"     )!=0 &&
                                        strcmp(token[j].type,"Separate"   )!=0 &&
                                        strcmp(token[j].type,"TagIniOpen" )!=0 &&
                                        strcmp(token[j].type,"TagEndOpen" )!=0 &&
                                        strcmp(token[j].type,"TagEndClose")!=0) 
                                        printf ("\n[DEBUG] [%15s] | [%80s]", token[j].type, token[j].value); 
                                        break;  

                           case 1 : break; 
                           case 0 : break; 
                           default: break; 
                       };
 
                       //????free(token[j].type); 
                       //????free(token[j].value); 

                       j++;
               };

               buffer[0]     = __NULL__  ;
               i             = __ZERO__  ;
               foundToken    = __FALSE__ ;
          }; 

          /*
          .--------------------------------------.
          | Terminato il cliclo chiude il remark |
          '--------------------------------------' */
          if ( remark == __ON_TO_OFF__ ) remark = __OFF__;
    };
    if (debug_level > 0) printf("\n");



    /*
    .-----------------------------------------------------------------.
    |                                                                 |
    |                        Controlli Finali                         |
    |                                                                 |
    '-----------------------------------------------------------------' */

    /*
    .----------------.
    | File XML Empty | 
    '----------------' */
    if (rPharse.iTabTagXML == __ZERO__ && rPharse.iTabTag == __ZERO__ && strlen(buffer) == __ZERO__) { 
        rPharse.row_error = countRowXMLFile + 1;
        rPharse.token_unknown = malloc(strlen("\0"));
        strcpy(rPharse.token_unknown,"\0"); 
        rPharse.return_code = -30;
        return rPharse; };

    /*
    .------------------------------------.
    | It isnt XML File - Not found token |
    '------------------------------------' */
    if (rPharse.iTabTagXML == __ZERO__ && rPharse.iTabTag == __ZERO__ && strlen(buffer)  > __ZERO__) {
        rPharse.row_error = countRowXMLFile + 1;
        rPharse.token_unknown = malloc(strlen(buffer));
        strcpy(rPharse.token_unknown, buffer);
        rPharse.return_code = -31;
        return rPharse; };

    /*
    .-----------------------------------------.
    | XML File incomplete - TAG XML not Close |
    '-----------------------------------------' */
    if (rPharse.iTabTagXML > 0) 
    { 
        j = rPharse.iTabTagXML;
        while ( j > 0 && rPharse.rTabTagXML[j-1].stato == 3 ) j--; 
        if (j > 0) 
        {   
            rPharse.row_error = countRowXMLFile + 1;
            rPharse.token_unknown = malloc(strlen(rPharse.rTabTagXML[rPharse.iTabTagXML-1].tag));
            strcpy(rPharse.token_unknown, rPharse.rTabTagXML[rPharse.iTabTagXML-1].tag);
            rPharse.return_code = -32;
            return rPharse; 
        };
    };

    /*
    .-----------------------------------------.
    | XML File incomplete - TAG not Close     |
    '-----------------------------------------' */
    if (rPharse.iTabTag > 0)
    {
        j = rPharse.iTabTag;
        while ( j > 0 && rPharse.rTabTag[j-1].stato == 3 ) j--; 
        if (j > 0) 
        {
            rPharse.row_error = countRowXMLFile + 1;
            rPharse.token_unknown = malloc(strlen(rPharse.rTabTag[rPharse.iTabTag-1].tag));
            strcpy(rPharse.token_unknown, rPharse.rTabTag[rPharse.iTabTag-1].tag);
            rPharse.return_code = -33;
            return rPharse; 
        };
    };
    



    /*
    .----------------.
    | chiude il file |
    '----------------' */ 
    fclose(fp);

    /*
    .---------------.
    | ritorna esito |
    '---------------' */
    rPharse.return_code = 0;
    return rPharse;
};




/*
.------------------------------------------------------------.
|                                                            |
|                                                            |
|                                                            |
|                                                            |
|                                                            |
|                                                            |
|                                                            |
|                                                            |
|                                                            |
|                                                            |
`------------------------------------------------------------' */
int search ( char str[1024], char key[32] ) 
{
    int  i     = 0;
    int  j     = 0;
    int  freq  = 0;

    while (i < 1024 && str[i] != 0)
    {
           if (str[i]==key[j]) { 
               j++ ;
           }
           else
           {
               if (j > 0) i--;
               j=0;
           };

           if (key[j]==0) { 
               j=0;
               freq++;
           };
           i++;
    };
    return freq;
};    
