#!/usr/bin/python -O
# -*- coding: iso-8859-15 -*-

##    Copyright 2012, Momme Winkelnkemper <specmate@posteo.de>
##
##    This file is part of SpecMate.
##
##    SpecMate is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    Specmate is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with SpecMate.  If not, see <http://www.gnu.org/licenses/>.

"""
This Module implements the the backend of the "SpecMate" application.
"""

#----------------------------------------------------------------------------
# packages to import
#----------------------------------------------------------------------------
import sys,os
import copy
import time
import cPickle
import warnings

try:
	import wx
except:
	print "Please install wxPython!"
	sys.exit(1)
try:
	import numpy as na
except:
	print "Please install numpy!"
	sys.exit(1)
try:
	from scipy.optimize import fmin,fmin_cg,fmin_bfgs,fmin_l_bfgs_b,fmin_tnc
except:
	print "Please install scipy!"
	sys.exit(1)


from specmate_datasets import specmate_singleDataset
from specmate_datasets import specmate_refDataset
from specmate_datasets import specmate_peakDataset
from specmate_datasets import specmate_functionDataset
#from specmate_menubar import specmate_menubar
#from specmate_globals import *
from specmate_globals import appName,DEFAULT_REF_COEFF,DEFAULT_NAMES,DEFAULT_LABELS,labelStr
#from min5 import ga_minimize
#----------------------------------------------------------------------------
#----------------------------------------------------------------------------




#-----------------------------------------------------------------------------------------
# begin class "specmate_backend_basicDataAndIO"
#----------------------------------------------------------------------------------------
class specmate_backend_basicDataAndIO():
	"""
	This class implements the basic data structure of SpecMate. Any methods that is capable of creating or removing a whole dataset is found here. Dataset are either spectra, represented by an instance of "specmate_dataset" (or a derived class), or dictionaries containing in most cases non-numeric data.

	@ivar frontend: pointer to the frontend-instance that is to be used (Default: "None")
	@type frontend: pointer to a specmate_frontend-instance (or a string: "None")
	@ivar specData: "spectrum" data.
	@type specData: specmate_singleDataset
	@ivar diffData: "residual" data.
	@type diffData: specmate_singleDataset
	@ivar background: "baseline" data.
	@type background: specmate_singleDataset
	@ivar refSpectra: list of reference spectra datasets.
	@type refSpectra: list of specmate_refDataset
	@ivar peakSpectra: list of peak datasets.
	@type peakSpectra: list of specmate_peakDataset
	@ivar functionSpectra: list of function datasets.
	@type functionSpectra: list of specmate_functionDataset
	@ivar namesAndLabels: dictionary with label strings for the plot.
	@type namesAndLabels: dict
	@ivar generalFitParam: dictionary with general fitting parameters that do not belong to a particular peak or refSpectrum
	@type generalFitParam: dict

	"""
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def interpolateAll(self):
		"""
		Interpolate all spectra, peaks, and functions to the X-Axis of the current spectrum
		"""
		if len(self.refSpectra)>0:
			for eachSpectrum in self.refSpectra:
				eachSpectrum.interpolate(self.specData.data[:,0])
		if len(self.peakSpectra)>0:
			for eachSpectrum in self.peakSpectra:
				eachSpectrum.interpolate(self.specData.data[:,0])
		if len(self.functionSpectra)>0:
			for eachSpectrum in self.functionSpectra:
				eachSpectrum.interpolate(self.specData.data[:,0])
		if self.background.hasData:
			self.background.interpolate(self.specData.data[:,0])
		self._calcDiff()
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def readSpectrum(self,filename):
		"""
		Open a txt-based spectrum-file and store it in self.specData. Ensure that all other datasets are interpolated to this X-Axis.

		@param filename: file to open
		@type filename: string
		@return: success(0)/fail(1)
		@rtype: int
		"""
		if self.specData.read(filename)==0:
			self.interpolateAll()
			return(0)
		else:
			return(1)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def shiftSpectrum(self,shift):
		"""
		shift X-axis of the spectrum
		@param shift: shift x-axis by this value
		@type shift: float
		@return: success(0)/fail(1)
		@rtype: int
		"""
		if not self.specData.hasData:
			return(1)
		self.specData.shiftData(shift)
		self.interpolateAll()
		return(0)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def clearBackground(self):
		"""
		Clear background data.
		"""
		self.background.clear()
		self._calcDiff()
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def readBackground(self,filename):
		"""
		Open a txt-based background-file and store it in self.background. 

		@param filename: file to open
		@type filename: string
		@return: success(0)/fail(1)
		@rtype: int
		"""
		if self.background.read(filename)==0:
			if self.specData.hasData:
				self.background.interpolate(self.specData.data[:,0])
			self._calcDiff()
			return(0)
		else:
			return(1)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def setBackground(self,points,interpScheme='linear'):
		"""
		Add a background dataset based on point data. Existing background data will be overwritten. Data will be interpolated to the X-Axis of self.specData if possible.
		
		@param points: point data for new background
		@type points: numpy-array(2D)
		@param interpScheme: Interpolation scheme to be used 
		@type interpScheme: string ('linear' / 'cubic splines')

		"""
		if len(points)>0:
			self.background.data=copy.deepcopy(points)
			self.background.hasData=True
			self.background.interpType=interpScheme
			if self.specData.hasData:
				self.background.interpolate(self.specData.data[:,0])
		else:
			self.background.clear()
		self._calcDiff()
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def addReference(self,filename,calc=True):
		"""
		Open a txt-based reference-spectrum file and append the dataset to self.refSpectra. Create interpolated data if possible. Set coeff,maxCoeff based on the dataset and self.specData  
		
		@param filename: file to open
		@type filename: string
		@param calc: whether or not to call self._calcDiff() in the end.
		@param calc: boolean
		@return: success(0)/fail(1)
		@rtype: int
		@default calc: True
		"""
		try:
			self.refSpectra.append(specmate_refDataset(filename,DEFAULT_REF_COEFF))
		except:
			#print "Couldn't append Reference Spectrum: "+filename
			warnings.warn("Couldn't append reference spectrum: "+filename,UserWarning)
			return(1)
		if self.specData.hasData:
			self.refSpectra[-1].interpolate(self.specData.data[:,0])
			self.refSpectra[-1].maxCoeff=self.specData.data[:,1].max()/self.refSpectra[-1].data[:,1].max()
			self.refSpectra[-1].coeff=0.5*self.refSpectra[-1].maxCoeff
		if calc:
			self._calcDiff()
		return(0)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def addReferences(self,filenames):
		"""
		Open a number of txt-based reference-spectra files and append the datasets to self.refSpectra. Create interpolated data if possible. Set coeff,maxCoeff based on the dataset and self.specData  
		@param filenames: files to open
		@type filenames: list of string
		@return: number of files that couldn't be opened
		@rtype: int
		"""
		nNotOpened=0
		for filename in filenames:
			nNotOpened+=self.addReference(filename,calc=False)
		self._calcDiff()
		return(nNotOpened)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------


	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def updateReference(self,which,coeff,maxCoeff='None',minCoeff='None',name='None',calc=True):
		"""
		Update one particular refSpectra-dataset.

		@param which: index of the spectrum to be updated
		@type which: int
		@param coeff: new coefficient for this reference spectrum
		@type coeff: float
		@param maxCoeff: new maximum Coefficient for this reference spectrum
		@type maxCoeff: float
		@default maxCoeff: "None" (No changes)
		@param minCoeff: new minimum Coefficient for this reference spectrum
		@type minCoeff: float
		@default minCoeff: "None" (No changes)
		@param name: new name 
		@type name: string
		@default name: "None" (No changes)
		@param calc: Whether or not to self._calcDiff() in the end, i.e. update all data. 
		@type calc: Bool
		@default calc: True

		@return: success(0)/fail(1)
		@rtype: int
		"""
		if which>len(self.refSpectra)-1:
			warnings.warn("Reference Spectrum No. %d does not exist!" % (which),UserWarning)
			#print "Reference Spectrum No. %d does not exist!" % (which)
			return(1)
		self.refSpectra[which].coeff=coeff
		if maxCoeff!='None':
			self.refSpectra[which].maxCoeff=maxCoeff
		if minCoeff!='None':
			self.refSpectra[which].minCoeff=minCoeff
		if name!='None':
			self.refSpectra[which].name=name
		if calc:
			self._calcDiff()
		return(0)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def updateRefNames(self,names):
		"""
		Change the names of the reference spectra.
		@param names: list with the new names for the reference spectra.
		@type names: list of strings
		@return: success(0)/fail(1)
		@rtype: int
		"""
		if len(self.refSpectra)!=len(names):
			return(1)
		for i in xrange(0,len(self.refSpectra)):
			self.refSpectra[i].name=names[i]
		self._createPlotData()
		return(0)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def updateReferences(self,coeff,maxCoeff,minCoeff,name):
		"""
		Update all refSpectra-datasets at once.

		@param coeff: new coefficients for the reference spectra
		@type coeff: list of floats
		@param maxCoeff: new maximum coefficients for the reference spectra
		@type maxCoeff: list of float
		@param minCoeff: new minimum coefficients for the reference spectra
		@type minCoeff: list of float
		@param name: new names for the reference spectra 
		@type name: list of strings
		"""
		for i in xrange(0,len(self.refSpectra)):
			self.updateReference(i,coeff[i],maxCoeff[i],minCoeff[i],name[i],False)
		self._calcDiff()
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def deleteReference(self,i):
		"""
		delete the i-th reference spectrum from self.refSpectra

		@param i: index of reference spectrum to delete
		@type filename: int 
		@return: success(0)/fail(1)
		@rtype: int
		"""
		if len(self.refSpectra)>i:
			self.refSpectra.pop(i)
			self._calcDiff()
			return(0)
		return(1)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------


	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def addPeak(self,peakType='gaussian',x0=0.0,y0=1.0,FWHM=10.0,shape=0.0):
		"""
		Append a peak-dataset to self.peakSpectra
		@param peakType: Peak type, "gaussian", "lorentzian", or "voigt"
		@type peakType: string ("gaussian"/"lorentzian"/"voigt")
		@param x0: Peak position
		@type x0: float
		@param y0: Peak amplitude
		@type y0: float
		@param FWHM: FWHM of the peak
		@type FWHM: float
		@param shape: shape factor (use for voigt peaks only). See doc-string of function "voigt" for details.
		@type shape: float(0..1)
		@return: success(0)/fail(1)
		@rtype: int
		"""
		if not self.specData.hasData:
			return(1)
		if self.frontend!='None':
			xlim,ylim=self.frontend.getPlotLimits()
		else:
			xlim=(self.specData.data[0,0],self.specData.data[-1,0])
			ylim=(self.specData.data[:,1].min(),self.specData.data[:,1].max())
		if self.specData.hasData:
			x0=(xlim[0]+xlim[1])*0.5
			y0=ylim[1]*0.9
		if len(self.peakSpectra)>0:
			FWHM=self.peakSpectra[-1].FWHM
		else:
			FWHM=(xlim[1]-xlim[0])*0.1

		self.peakSpectra.append(specmate_peakDataset(self.specData.data[:,0],peakType,x0,y0,FWHM,shape))
		self.peakSpectra[-1].name="Peak %d" % (len(self.peakSpectra))
		self._calcDiff()
		return(0)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def deletePeak(self,i):
		"""
		delete the i-th peak spectrum from self.peakSpectra

		@param i: index of peak to delete
		@type filename: int 
		@return: success(0)/fail(1)
		@rtype: int
		"""
		if len(self.peakSpectra)>i:
			self.peakSpectra.pop(i)
			self._calcDiff()
			return(0)
		return(1)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def modifyPeak(self,i,x0,y0,FWHM,shape,peakType,name,calc=True):
		"""
		modify the parameters of the i-th peak spectrum in self.peakSpectra

		@param i: index of peak to delete
		@type filename: int 
		@param peakType: Peak type, "gaussian", "lorentzian", or "voigt"
		@type peakType: string ("gaussian"/"lorentzian"/"voigt")
		@param x0: Peak position
		@type x0: float
		@param y0: Peak amplitude
		@type y0: float
		@param FWHM: FWHM of the peak
		@type FWHM: float
		@param shape: shape factor (use for voigt peaks only). See doc-string of function "voigt" for details.
		@type shape: float(0..1)
		@param name: Peak name
		@type name: string
		@return: success(0)/fail(1)
		@rtype: int
		"""
		if len(self.peakSpectra)>i:
			if FWHM<0.0:
				FWHM=0.0
			self.peakSpectra[i].FWHM=FWHM
			if shape<0.0:
				shape=0.0
			if shape>1.0:
				shape=1.0
			self.peakSpectra[i].x0=x0
			self.peakSpectra[i].y0=y0
			self.peakSpectra[i].shape=shape
			self.peakSpectra[i].peakType=peakType
			self.peakSpectra[i].name=name
			self.peakSpectra[i].calculate()
			if calc:
				self._calcDiff()
			return(0)
		return(1)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def addFunction(self,filename,calc=False):
		"""
		Append a function-dataset to self.functionSpectra
		@param functionString: Code to compile to be the user functions
		@type functionString: string
		@param a: parameters for the function 
		@type a: numpy.zeros((10),'Float64')
		@return: success(0)/fail(1)
		@rtype: int
		"""
		if not self.specData.hasData:
			return(1)
		#if self.frontend!='None':
		#	xlim,ylim=self.frontend.getPlotLimits()
		#else:
		#	xlim=(self.specData.data[0,0],self.specData.data[-1,0])
		#	ylim=(self.specData.data[:,1].min(),self.specData.data[:,1].max())
		#if self.specData.hasData:
		#	x0=(xlim[0]+xlim[1])*0.5
		#	y0=ylim[1]*0.9
		#if len(self.peakSpectra)>0:
		#	FWHM=self.peakSpectra[-1].FWHM
		#else:
		#	FWHM=(xlim[1]-xlim[0])*0.1
		self.functionSpectra.append(specmate_functionDataset(self.specData.data[:,0]))
		if not self.functionSpectra[-1].read(filename):
			self.deleteFunction(-1)
			return(1)
		self.functionSpectra[-1].name="Function %d" % (len(self.functionSpectra))
		if calc:
			self._calcDiff()
		return(0)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def deleteFunctionk(self,i):
		"""
		delete the i-th function spectrum from self.functionSpectra

		@param i: index of function to delete
		@type filename: int 
		@return: success(0)/fail(1)
		@rtype: int
		"""
		if len(self.functionSpectra)>i:
			self.functionSpectra.pop(i)
			self._calcDiff()
			return(0)
		return(1)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def modifyFunction(self,i,a,name="None",calc=True):
		"""
		modify the parameters of the i-th function spectrum in self.peakSpectra

		@param i: index of function to modify
		@type filename: int 
		@param a: list of parameters for the function
		@type a: numpy.array(10,Float64)
		@return: success(0)/fail(1)
		@rtype: int
		"""
		if len(self.functionSpectra)>i:
			self.functionSpectra[i].a=a
			self.functionSpectra[i].calculate()
			if name!="None":
				self.functionSpectra[i].name=name
			if calc:
				self._calcDiff()
			return(0)
		return(1)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def _calcDiff(self,autoscale=False):
		"""
		Calculate the residual spectrum from all other datasets, store it in self.diffData. Call self._createPlotData() in the end.
		"""
		self.diffData.hasData=False
		self.diffData.hasInterpData=False
		#self.diffData.data[:,:]=self.specData.data[:,:]
		self.diffData.data=copy.deepcopy(self.specData.data)
		if self.specData.hasData and len(self.refSpectra)>0:
			for i in xrange(0,len(self.refSpectra)):
				self.diffData.data[:,1]-=self.refSpectra[i].Y()[:]
			self.diffData.hasData=True
		if self.specData.hasData and len(self.peakSpectra)>0:
			for i in xrange(0,len(self.peakSpectra)):
				self.diffData.data[:,1]-=self.peakSpectra[i].Y()[:]
			self.diffData.hasData=True
		if self.specData.hasData and len(self.functionSpectra)>0:
			for i in xrange(0,len(self.functionSpectra)):
				self.diffData.data[:,1]-=self.functionSpectra[i].Y()[:]
			self.diffData.hasData=True
		if self.specData.hasData and self.background.hasData:
			#if self.diffData.hasInterpData==False:
			#	self.diffData.data[:,:]=self.specData.data[:,:]
			self.diffData.data[:,1]-=self.background.Y()[:]	
			self.diffData.hasData=True
		self._createPlotData(autoscale)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def _createPlotData(self,autoscale=False):
		"""
		Create a dataset that contains the all data for the plot and (if self.frontend!="None") tell the frontend to plot it (via self.frontend.replot()).
		"""
		self.plotData=[]
		if self.specData.hasData:
			self.plotData.append([self.specData.X()[:],self.specData.data[:,1],self.specData.name,2.0,'r-'])
		for i in xrange(0,len(self.refSpectra)):
			if self.refSpectra[i].hasInterpData:
				self.plotData.append([self.refSpectra[i].X()[:],self.refSpectra[i].Y()[:], self.refSpectra[i].name,1.0,'-'])
			elif self.refSpectra[i].hasData:
				self.plotData.append([self.refSpectra[i].X()[:],self.refSpectra[i].Y()[:], self.refSpectra[i].name,1.0,'-'],)
		for i in xrange(0,len(self.peakSpectra)):
			if self.peakSpectra[i].hasInterpData:
				self.plotData.append([self.peakSpectra[i].X()[:],self.peakSpectra[i].Y()[:], self.peakSpectra[i].name,1.0,'-'])
		for i in xrange(0,len(self.functionSpectra)):
			if self.functionSpectra[i].hasInterpData:
				self.plotData.append([self.functionSpectra[i].X()[:],self.functionSpectra[i].Y()[:], self.functionSpectra[i].name,1.0,'-'])
		if self.diffData.hasData:
			self.plotData.append([self.diffData.X()[:],self.diffData.data[:,1], self.diffData.name,1.0,'k-'])
		if self.frontend!="None":
			self.frontend.replot(autoscale)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def initEmptyProject(self):
		"""
		Create an empty project / Clear all data.
		"""
		self.namesAndLabels=DEFAULT_LABELS
		self.specData=specmate_singleDataset()
		self.background=specmate_singleDataset()
		self.diffData=specmate_singleDataset()#copy.deepcopy(self.specData)
		self.refSpectra=[]
		self.peakSpectra=[]
		self.functionSpectra=[]
		self.diffData.name=DEFAULT_NAMES[labelStr["residualName"]]
		self.background.name=DEFAULT_NAMES[labelStr["baselineName"]]
		self.generalFitParam={}
		self._calcDiff()
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#Constructor
	#--------------------------------------------------------------------------------
	def __init__(self,argv=['None']):
		"""
		Constructor: ... 
		@param argv: list of filenames to open: first entry is used as spectrum, all following names as reference spectra.
		@type argv: list of strings
		"""
		self.frontend="None"
		self.initEmptyProject()
		if len(argv)>1:
			self.readSpectrum(argv[1])
			if len(argv)>2:
				for i in xrange(2,len(argv)):
					self.addReference(argv[i])
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
#end class "specmate_backend_basicDataAndIO"
#-----------------------------------------------------------------------------------------

#-----------------------------------------------------------------------------------------
# begin class "specmate_backend_projectIO"
#----------------------------------------------------------------------------------------
class specmate_backend_projectIO(specmate_backend_basicDataAndIO):
	"""
	This class adds all methods that deal with Project-IO, i.e. load/save project and export datasets. 
	"""
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def project2Dict(self):
		"""
		Create a dictionary that contains all project data. (To be used by self.saveProject())
		@return: dictionary with all project data
		@rtype: dict
		"""
		data={'nRefSpectra': len(self.refSpectra),'nPeakSpectra': len(self.peakSpectra),'nFunctionSpectra': len(self.functionSpectra),'spectrum':self.specData.exportDataAsDict(),'background':self.background.exportDataAsDict(),'namesAndLabels':self.namesAndLabels,'generalFitParam':self.generalFitParam}	
		for i in xrange(0,len(self.refSpectra)):
			name='refSpectrum%d' % (i)
			data[name]=self.refSpectra[i].exportDataAsDict()
		for i in xrange(0,len(self.peakSpectra)):
			name='peakSpectrum%d' % (i)
			data[name]=self.peakSpectra[i].exportDataAsDict()
		for i in xrange(0,len(self.functionSpectra)):
			name='functionSpectrum%d' % (i)
			data[name]=self.functionSpectra[i].exportDataAsDict()
		return(data)

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def dict2Project(self,data):
		"""
		Read project data from a dictionary, store it in the instance's variables (To be used by self.openProject())
		@param data: dictionary with the project data
		@type data: dict
		"""
		self.initEmptyProject()
		if 'spectrum' in data.keys():
			self.specData.importDataFromDict(data['spectrum'])
		if 'background' in data.keys():
			self.background.importDataFromDict(data['background'])
		if 'namesAndLabels' in data.keys():
			for key in self.namesAndLabels.keys():
				if key in data['namesAndLabels'].keys():
					self.namesAndLabels[key]=data['namesAndLabels'][key]
		if 'generalFitParam' in data.keys():
			self.generalFitParam=data['generalFitParam']
		if 'nRefSpectra' in data.keys():
			for i in xrange(0,data['nRefSpectra']):
				name='refSpectrum%d' % (i)
				self.refSpectra.append(specmate_refDataset())
				self.refSpectra[i].importDataFromDict(data[name])
		if 'nPeakSpectra' in data.keys():
			for i in xrange(0,data['nPeakSpectra']):
				name='peakSpectrum%d' % (i)
				self.peakSpectra.append(specmate_peakDataset())
				self.peakSpectra[i].importDataFromDict(data[name])
		if 'nFunctionSpectra' in data.keys():
			for i in xrange(0,data['nFunctionSpectra']):
				name='functionSpectrum%d' % (i)
				self.functionSpectra.append(specmate_functionDataset())
				self.functionSpectra[i].importDataFromDict(data[name])
		self._calcDiff(autoscale=True)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def saveProject(self,filename):
		"""
		Save the whole Project. (based on "cPickle")
		@param filename: filename 
		@type filename: string
		@return: success(0)/fail(1)
		@rtype: int
		"""
		data=self.project2Dict()
		try:
			f=open(filename,'wb')
			cPickle.dump(data,f)
			f.close()
			return(0)
		except:
			warnings.warn("Couldn't write file: "+filename,UserWarning)
			return(1)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def openProject(self,filename):
		"""
		Open a project previously save using self.saveProject.
		@param filename: filename to open
		@type filename: string
		@return: success(0)/fail(1)
		@rtype: int
		"""
		try:
			f=open(filename,'rb')
			data=cPickle.load(f)
			f.close()
		except:
			print "Couldn't unPickle file: "+filename
			return(1)
		self.dict2Project(data)
		return(0)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def _selectDataset(self,which='spectrum',number=-1,name='None'):
		"""
		Helper function to select a dataset of the current project.
		@param which: which dataset type
		@type which: string ("spectrum","background","reference","peak","function")
		@param number: index of reference or peak or function
		@type number: int
		@param name: name of reference or peak or function
		@type name: string
		@return: the selected dataset
		@rtype: specmate_singleDataset (or direved class) (Default: None)
		"""

		dataset=None
		if which=='spectrum':
			dataset=self.specData
		elif which=='background':
			dataset=self.background
		elif which=='residual':
			dataset=self.diffData
		elif which=='reference':
			if (number==-1 and name=='None'):
				if len(self.refSpectra)>1:
					return(1)
				else:
					dataset=self.refSpectra[0]
			elif number>-1:
				dataset=self.refSpectra[number]
			else:
				for each in self.refSpectra:
					if each.name==name:
						dataset=each
						break
		elif which=='peak':
			if (number==-1 and name=='None'):
				if len(self.peakSpectra)>1:
					return(1)
				else:
					dataset=self.peakSpectra[0]
			elif number>-1:
				dataset=self.peakSpectra[number]
			else:
				for each in self.peakSpectra:
					if each.name==name:
						dataset=each
						break
		elif which=='function':
			if (number==-1 and name=='None'):
				if len(self.functionSpectra)>1:
					return(1)
				else:
					dataset=self.functionSpectra[0]
			elif number>-1:
				dataset=self.functionSpectra[number]
			else:
				for each in self.functionSpectra:
					if each.name==name:
						dataset=each
						break
		return dataset
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def saveSingleSet(self,filename,which='spectrum',number=-1,name='None',which_data='interpolated'):
		"""
		Save a single dataset as "txt" 
		@param which: which dataset type is to be saved
		@type which: string ("spectrum","background","reference","peak","function")
		@param number: index of reference or peak or function
		@type number: int
		@param name: name of reference or peak or function
		@type name: string
		@param filename: filename to write
		@type filename: string
		@return: success(0)/fail(1)
		@rtype: int
		"""
		dataset=self._selectDataset(which,number,name)
		if dataset!=None:
			try:
				dataset.write(filename,which_data)
			except:
				warnings.warn("Couldn't write file: "+filename,UserWarning)
				return(1)
			return(0)
		else:
			return(1)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def saveAllSets(self,filename):
		"""
		Save all datasets in a single "txt"-file. Additionally a file "..._info.txt" is written containing information about the data. 
		
		@param filename: filename to write
		@type filename: string
		@return: success(0)/fail(1)
		@rtype: int
		"""

		t=time.localtime()
		header=	"# data generated with %s on %d-%d-%d, %d:%d:%d\n" % (appName,t[0],t[1],t[2],t[3],t[4],t[5])
		header+="# The following data has been written to the file: "+filename+":\n"
		header+="# Col. 1: common X-Axis, as read from the spectrum dataset\n"
		header+="# Col. 2: spectrum, name: "+self.specData.name+", filename: "+self.specData.filename+", X-Axis shift: "+str(self.specData.xShift)+"\n"
		if self.specData.hasData==True:
			save_data=[self.specData.X(),self.specData.Y()]
		else:
			return(1)
		colCount=3
		for eachSpectrum in self.refSpectra:
			if eachSpectrum.hasInterpData:
				save_data.append(eachSpectrum.Y())
				header+="# Col. %d: reference spectrum, name: %s, filename: %s, coeff.: %e\n" % (colCount,eachSpectrum.name,eachSpectrum.filename,eachSpectrum.coeff)
				colCount+=1
		for eachSpectrum in self.peakSpectra:
			if eachSpectrum.hasInterpData:
				save_data.append(eachSpectrum.Y())
				header+="# Col. %d: peak, name: %s, filename: %s, peak type: %s, x0: %e, y0: %e, FWHM: %e, shape: %e\n" \
					% (colCount,eachSpectrum.name,eachSpectrum.filename,eachSpectrum.peakType,eachSpectrum.x0,eachSpectrum.y0,eachSpectrum.FWHM,eachSpectrum.shape)
				colCount+=1
		for eachSpectrum in self.functionSpectra:
			if eachSpectrum.hasInterpData:
				save_data.append(eachSpectrum.Y())
				header+="# Col. %d: function, name: %s, filename: %s, function type: %s, a0: %e, a1: %e, a2: %e, a3: %e\n" \
					% (colCount,eachSpectrum.name,eachSpectrum.filename,eachSpectrum.functionName,eachSpectrum.a[0],eachSpectrum.a[1],eachSpectrum.a[2],eachSpectrum.a[3])
				colCount+=1
		if self.background.hasData:
			save_data.append(self.background.Y())
			header+="# Col. %d: baseline, name: %s, filename: %s\n" % (colCount,self.background.name,self.background.filename)
			colCount+=1
		if self.diffData.hasData:
			save_data.append(self.diffData.Y())
			colCount+=1
			header+="# Col. %d: residual, name: %s, filename: %s\n" % (colCount,self.diffData.name,self.diffData.filename)

		try:
			na.savetxt(filename,na.transpose(na.array(save_data)),fmt='%.5e')#,header=saveHeader)
		except:
			warnings.warn("Warning: Couldn't write file "+filename,UserWarning)
			return(1)
		infoFilename=filename.split('.')[0]+'_info.txt'
		infoFile=open(infoFilename,'w')
		infoFile.write(header)
		infoFile.close()
		return(0)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#Constructor
	#--------------------------------------------------------------------------------
	def __init__(self,argv=['None']):
		"""
		Constructor: Will try to open argv[1] as project if possible or as spectrum otherwise. If argv[2 ...] exist, these will be used as additional references. 
		@param argv: list of filenames to open. 
		@type argv: list of strings
		"""
		self.frontend="None"
		self.initEmptyProject()
		if len(argv)>1:
			if self.openProject(argv[1])!=0:
				self.readSpectrum(argv[1])
				if len(argv)>2:
					for i in xrange(2,len(argv)):
						self.addReference(argv[i])
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
# end class "specmate_backend_projectIO"
#----------------------------------------------------------------------------------------


#-----------------------------------------------------------------------------------------
# begin class "specmate_backend_dataLocking"
#----------------------------------------------------------------------------------------
class specmate_backend_dataLocking(specmate_backend_projectIO):
	"""
	Add functionality "lock" individual datasets.

	@ivar _isLocked: dictionary of boolean to store wehter a dataset is locked (valid keys are "spectrum","references","namesAndLabels","background","peaks")  
	@type _isLocked: dict 	
	"""
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def lock(self,key):
		"""
		set the value for "key" to True ("locked")

		@param key: which dataset  to locked
		@type key: string (valid key)
		@return: successfully locked(0)/ was locked already(1)
		@rtype: int
		"""
		if self._isLocked[key]==False:
			self._isLocked[key]=True
			return(0)
		return(1)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def unlock(self,key):
		"""
		set the value for "key" to False ("unlocked")

		@param key: which dataset  to unlocked
		@type key: string (valid key)
		@return: successfully unlocked(0)/ was unlocked already(1)
		@rtype: int
		"""
		if self._isLocked[key]==True:
			self._isLocked[key]=False
			return(0)
		return(1)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def isLocked(self,key):
		"""
		check wether dataset "key" is locked

		@param key: which dataset  to unlocked
		@type key: string (valid key)
		@return: True/False
		@rtype: bool
		"""
		if self._isLocked[key]==True:
			return(True)
		return(False)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def initEmptyProject(self):
		"""
		Add to parent's "initEmptyProject": self._isLocked

		"""
		self._isLocked={'spectrum':False,'references':False,'peaks':False,'namesAndLabels':False,'background':False,'functions':False}
		specmate_backend_projectIO.initEmptyProject(self)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
# end class "specmate_backend_dataLocking"
#----------------------------------------------------------------------------------------

#-----------------------------------------------------------------------------------------
# begin class "specmate_backend_dataBackup"
#----------------------------------------------------------------------------------------
class specmate_backend_dataBackup(specmate_backend_dataLocking):
	"""
	Add functionality to "backup" project data in a ring buffer. To be used by cancel buttons or a (future) "undo" function. 
	
	@ivar backupData: ring buffer for the backups
	@type backupData: list of lists.
	"""
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def dataBackup(self,what,num=-1):
		"""
		Create a new backup, put it as last item in the ring buffer.
		@param what: what is to be backuped ("spectrum","background","references","peaks")
		@type what: list of strings
		@param num: ...
		@type num: int
		"""
		newDataset=[]
		for each in what:
			if each=='spectrum':
				newDataset.append(['spectrum',-1,copy.deepcopy(self.specData)])
			elif each=='background':
				newDataset.append(['background',-1,copy.deepcopy(self.background)])
			elif each=='references':
				newDataset.append(['references',-1,copy.deepcopy(self.refSpectra)])
			elif each=='reference':
				if len(self.refSpectra)>num:
					newDataset.append(['reference',num,copy.deepcopy(self.refSpectra[num])])
			elif each=='peaks':
				newDataset.append(['peaks',-1,copy.deepcopy(self.peakSpectra)])
			elif each=='peak':
				if len(self.refSpectra)>num:
					newDataset.append(['peak',num,copy.deepcopy(self.peakSpectra[num])])
			elif each=='functions':
				newDataset.append(['functions',-1,copy.deepcopy(self.functionSpectra)])
			elif each=='function':
				if len(self.refSpectra)>num:
					newDataset.append(['function',num,copy.deepcopy(self.functionSpectra[num])])
			if len(newDataset)>0:
				self.backupData.pop(0)
				self.backupData.append(newDataset)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def restoreDataBackup(self):
		"""
		Restore the last item from the ring buffer to the instance's variables. 

		"""
		if len(self.backupData[-1])>0:
			for each in self.backupData[-1]:
				if each[0]=='spectrum':
					self.specData=each[2]
				elif each[0]=='background':
					self.background=each[2]
				elif each[0]=='reference':
					if len(self.refSpectra)>each[1]:
						self.refSpectra[each[1]]=each[2]
				elif each[0]=='references':
					self.refSpectra=each[2]
				elif each[0]=='peak':
					if len(self.peakSpectra)>each[1]:
						self.peakSpectra[each[1]]=each[2]
				elif each[0]=='peaks':
					self.peakSpectra=each[2]
				elif each[0]=='function':
					if len(self.functionSpectra)>each[1]:
						self.functionSpectra[each[1]]=each[2]
				elif each[0]=='functions':
					self.functionSpectra=each[2]
			self.backupData.pop(-1)
			self.backupData.insert(0,[])
			self._calcDiff()
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def initEmptyProject(self):
		"""
		Add to parent's "initEmptyProject": create self.backupData.

		"""
		self.backupData=[[],[],[]]
		specmate_backend_dataLocking.initEmptyProject(self)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
# end class "specmate_backend_dataBackup"
#----------------------------------------------------------------------------------------

#-----------------------------------------------------------------------------------------
# begin class "specmate_backend_fittting"
#----------------------------------------------------------------------------------------
class specmate_backend_fitting(specmate_backend_dataBackup):
	"""
	Add fitting functionality.
	@ivar generalFitParam: All fit parameters that are not stored in the individual datasets.
	@type generalFitParam: dict
	@ivar _breakFit: whether or not the "break" button has been pressed.
	@type _breakFit: bool
	@ivar fitModels: Supported fit models ('L_BFGS_B (constrained)', 'TNC (constrained)', 'BFGS (unconstrained)', 'Downhill Simplex (unconstrained)', 'CG (unconstrained)')
	@type fitModels: list of strings
	@ivar fitModel: fit model currently selected
	@type fitmodel: string
	@ivar fitRunning: whether or not a fit is running
	@type fitRunning: bool
	@ivar fitFrontendFeedback: function to announce the current status to the frontend.
	@type fitFrontendFeedback: python function
	@default fitFrontend: None
	@ivar fitness: fitness (average residual squared per point) of this instance of specmate_backend.
	@type fitness: float
	@default fitness: 0.0
	"""

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def initFitParams(self):
		"""
		set initial fitting parameters, if they are not set, yet.
		"""
		if ((self.specData.hasData!=True) or ((len(self.refSpectra)==0) and (len(self.peakSpectra)==0))):
			return(1)
		xMin=self.specData.X().min()
		xMax=self.specData.X().max()

		if not 'fitMin' in self.generalFitParam.keys():
			self.generalFitParam['fitMin']=xMin
		if not 'fitMax' in self.generalFitParam.keys():
			self.generalFitParam['fitMax']=xMax
		for eachSpec in self.refSpectra:
			if not 'include' in eachSpec.fitParam.keys():
				eachSpec.fitParam['include']=False
			if not 'coeff' in eachSpec.fitParam.keys():
				eachSpec.fitParam['coeff']={}
			if not 'min' in eachSpec.fitParam['coeff'].keys(): 
				eachSpec.fitParam['coeff']['min']=eachSpec.coeff*0.8
			if not 'max' in eachSpec.fitParam['coeff'].keys(): 
				eachSpec.fitParam['coeff']['max']=eachSpec.coeff*1.2
			if not 'include' in eachSpec.fitParam['coeff'].keys(): 
				eachSpec.fitParam['coeff']['include']=False
		for eachSpec in self.peakSpectra:
			if not 'include' in eachSpec.fitParam.keys():
				eachSpec.fitParam['include']=False
			if not 'x0' in eachSpec.fitParam.keys():
				eachSpec.fitParam['x0']={}
				eachSpec.fitParam['x0']['min']=eachSpec.x0-0.1*(xMax-xMin)
				eachSpec.fitParam['x0']['max']=eachSpec.x0+0.1*(xMax-xMin)
				eachSpec.fitParam['x0']['include']=False
			if not 'y0' in eachSpec.fitParam.keys():
				eachSpec.fitParam['y0']={}
				eachSpec.fitParam['y0']['min']=eachSpec.y0*0.8
				eachSpec.fitParam['y0']['max']=eachSpec.y0*1.2
				eachSpec.fitParam['y0']['include']=False
			if not 'FWHM' in eachSpec.fitParam.keys():
				eachSpec.fitParam['FWHM']={}
				eachSpec.fitParam['FWHM']['min']=eachSpec.FWHM*0.8
				eachSpec.fitParam['FWHM']['max']=eachSpec.FWHM*1.2
				eachSpec.fitParam['FWHM']['include']=False
			if not 'shape' in eachSpec.fitParam.keys():
				eachSpec.fitParam['shape']={}
				eachSpec.fitParam['shape']['min']=0.0
				eachSpec.fitParam['shape']['max']=1.0
				eachSpec.fitParam['shape']['include']=False
		for eachSpec in self.functionSpectra:
			if not 'include' in eachSpec.fitParam.keys():
				eachSpec.fitParam['include']=False
			if not 'a' in eachSpec.fitParam.keys():
				eachSpec.fitParam['a']={}
				eachSpec.fitParam['a']['min']=copy.deepcopy(eachSpec.a)-na.abs(copy.deepcopy(eachSpec.a)) 
				eachSpec.fitParam['a']['max']=copy.deepcopy(eachSpec.a)+na.abs(copy.deepcopy(eachSpec.a))
				eachSpec.fitParam['a']['include']=[False,False,False,False,False,False,False,False,False,False]
		return(0)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def decryptFitParam(self,params):
		"""
		Decrypt the parameter "params" (which is just a list) which is returned by the fitting routine. Store the result in the instance's variables.
		@param params: list of parameters, returned by the fit routine
		@type params: list of float
		"""
		i=0
		j=0
		for eachRef in self.refSpectra:
			if eachRef.fitParam['include']:
				if eachRef.fitParam['coeff']['include']:
					self.updateReference(j,params[i],calc=False)
					i+=1
			j+=1
		j=0
		for eachPeak in self.peakSpectra:
			if eachPeak.fitParam['include']:
				if eachPeak.fitParam['x0']['include']:
					x0=params[i]
					i+=1
				else:
					x0=eachPeak.x0
				if eachPeak.fitParam['y0']['include']:
					y0=params[i]
					i+=1
				else:
					y0=eachPeak.y0
				if eachPeak.fitParam['FWHM']['include']:
					FWHM=params[i]
					i+=1
				else:
					FWHM=eachPeak.FWHM
				if eachPeak.fitParam['shape']['include']:
					shape=params[i]
					i+=1
				else:
					shape=eachPeak.shape
				self.modifyPeak(j,x0,y0,FWHM,shape,eachPeak.peakType,eachPeak.name,False)
			j+=1
		j=0
		for eachFunction in self.functionSpectra:
			if eachFunction.fitParam['include']:
				for k in xrange(0,10):
					if eachFunction.fitParam['a']['include'][k]:
						a[k]=params[i]
						i+=1
				self.modifyFunction(j,a,name=eachFunction.name,calc=False)
			j+=1
		self._calcDiff()
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def encryptFitParam(self):
		"""
		Encrypt the parameter from the project data for the fit routine.
		@return: list of parameters
		@rtype: list of float
		"""
		params=[]
		for eachRef in self.refSpectra:
			if eachRef.fitParam['include']:
				if eachRef.fitParam['coeff']['include']:
					params.append(eachRef.coeff)
		for eachPeak in self.peakSpectra:
			if eachPeak.fitParam['include']:
				if eachPeak.fitParam['x0']['include']:
					params.append(eachPeak.x0)
				if eachPeak.fitParam['y0']['include']:
					params.append(eachPeak.y0)
				if eachPeak.fitParam['FWHM']['include']:
					params.append(eachPeak.FWHM)
				if eachPeak.fitParam['shape']['include']:
					params.append(eachPeak.shape)
		for eachFunction in self.functionSpectra:
			if eachFunction.fitParam['include']:
				for j in xrange(0,10):
					if eachFunction.fitParam['a']['include'][j]:
						params.append(eachFunction.a[j])
		return(na.array(params))
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def createFitConstraints(self):
		"""
		Create a set of constraints for the fit.
		@return: set of contraints
		@rtype: field of float (N x 2)
		"""
		constraints=[]
		i=0
		for eachRef in self.refSpectra:
			if eachRef.fitParam['include']:
				if eachRef.fitParam['coeff']['include']:
					constraints.append([i,eachRef.fitParam['coeff']['min'],eachRef.fitParam['coeff']['max']])
					i+=1
		for eachPeak in self.peakSpectra:
			if eachPeak.fitParam['include']:
				if eachPeak.fitParam['x0']['include']:
					constraints.append([i,eachPeak.fitParam['x0']['min'],eachPeak.fitParam['x0']['max']])
					i+=1
				if eachPeak.fitParam['y0']['include']:
					constraints.append([i,eachPeak.fitParam['y0']['min'],eachPeak.fitParam['y0']['max']])
					i+=1
				if eachPeak.fitParam['FWHM']['include']:
					constraints.append([i,eachPeak.fitParam['FWHM']['min'],eachPeak.fitParam['FWHM']['max']])
					i+=1
				if eachPeak.fitParam['shape']['include']:
					constraints.append([i,eachPeak.fitParam['shape']['min'],eachPeak.fitParam['shape']['max']])
					i+=1
		for eachFunction in self.functionSpectra:
			if eachFunction.fitParam['include']:
				for j in xrange(0,10):
					if eachFunction.fitParam['a']['include'][j]:
						constraints.append([i,eachFunction.fitParam['a']['min'][j],eachFunction.fitParam['a']['max'][j]])
						i+=1
		return(constraints)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def fitFeedback(self,params):
		"""
		A callback function to be passed to the fit routine (if it allows it) to report the current status of the fit.	

		Can only be called when startFit() has been called before, otherwise it will lead to an error.

		@return: self._breakFit
		@rtype: bool
		"""
		if self.frontend!='None':
			wx.Yield()
		self.decryptFitParam(params)
		#self.fitness=self.fitModel.fitnessFunction(params)
		self.fitness=self.fitnessFunction2(params)
		if self.fitFrontendFeedback!=None:
			self.fitFrontendFeedback('Running',self.fitness)
		else:
			print "Fit residual (squared, per Point): ",self.fitness
		if self._breakFit==True:
			return(1)
		else:
			return(0)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def fitnessFunction(self,params):
		"""
		Calculate your own fitness.
		@param params: Parameter set that is first decrypted and stored.
		@type params: list of float
		@return: fitness (average residual squared per point) of the current project with the parameters from "params"
		@rtype: float
		"""
		self.decryptFitParam(params)
		return(na.square(self.diffData.Y()).sum()/float(len(self.diffData.X())))
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def startFit(self,fitFrontendFeedback=None):
		"""
		...
		"""
		self.fitFrontendFeedback=fitFrontendFeedback
		par=self.encryptFitParam()
		if len(par)==0:
			if self.fitFrontendFeedback!=None:
				self.fitFrontendFeedback('Nothing to fit!',0.0)
			return
		self.constraints=self.createFitConstraints()
		##-------------------------------------------
		##fitModel=copy.deepcopy(self) didn't work for some reason, so here's the long story
		##-------------------------------------------
		fitModel=specmate_backend_fitting()
		fitModel.specData=copy.deepcopy(self.specData)
		fitModel.diffData=copy.deepcopy(self.diffData)
		fitModel.refSpectra=copy.deepcopy(self.refSpectra)
		fitModel.peakSpectra=copy.deepcopy(self.peakSpectra)
		fitModel.functionSpectra=copy.deepcopy(self.functionSpectra)
		fitModel.background=copy.deepcopy(self.background)
		fitModel.constraints=copy.deepcopy(self.constraints)
		fitModel.frontend='None'
		fitModel.specData.clipData(self.generalFitParam['fitMin'],self.generalFitParam['fitMax'])
		fitModel.diffData.clipData(self.generalFitParam['fitMin'],self.generalFitParam['fitMax'])
		x=fitModel.specData.X()
		for eachRef in fitModel.refSpectra:
			eachRef.interpolate(x)
		for eachPeak in fitModel.peakSpectra:
			eachPeak.interpolate(x)
		for eachFunction in fitModel.functionSpectra:
			eachFunction.interpolate(x)
		if fitModel.background.hasData:
			fitModel.background.interpolate(x)
		##-------------------------------------------
		##-------------------------------------------
		self.fitRunning=True
		self.fitnessFunction2=fitModel.fitnessFunction
		if self.fitFrontendFeedback!=None:
			self.fitFrontendFeedback('Running',0.0)
		try:
			#if self.fitModel=='Mommes Weird Algorithm':
			#	ga_minimize(fitModel.fitnessFunction,par,self.constraints,self.fitFeedback)
			if self.fitModel=='Downhill Simplex (unconstrained)':
				par=fmin(fitModel.fitnessFunction,par,mxiter=200,callback=self.fitFeedback)
			elif self.fitModel=='CG (unconstrained)':
				par=fmin_cg(fitModel.fitnessFunction,par,maxiter=200,callback=self.fitFeedback)
			elif self.fitModel=='BFGS (unconstrained)':
				par=fmin_bfgs(fitModel.fitnessFunction,par,maxiter=200,callback=self.fitFeedback)
			elif self.fitModel=='L_BFGS_B (constrained)':
				par=fmin_l_bfgs_b(fitModel.fitnessFunction,par,approx_grad=True,bounds=na.array(self.constraints)[:,1:3])[0]
			elif self.fitModel=='TNC (constrained)':
				par=fmin_tnc(fitModel.fitnessFunction,par,approx_grad=True,bounds=na.array(self.constraints)[:,1:3])[0]
			self.fitFeedback(par)
			if self.fitFrontendFeedback!=None:
				self.fitFrontendFeedback('Done',self.fitness)
		except:
			if self.fitFrontendFeedback!=None:
				self.fitFrontendFeedback('Aborted',0.0)
		self.fitRunning=False
		self._breakFit=False
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def breakFit(self):
		"""
		set self._breakFit=True
		"""
		self._breakFit=True
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def setFitModel(self,model):
		"""
		Set the fit model to be used in the fit
		@param model: fit model
		@type model: string
		"""
		if model in self.fitModels:
			self.fitModel=model
			return(0)
		else:
			return(1)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def initEmptyProject(self):
		"""
		Add to parent's "initEmptyProject": create self.generalFitParam.

		"""
		self.fitModels=['L_BFGS_B (constrained)', 'TNC (constrained)', 'BFGS (unconstrained)', 'Downhill Simplex (unconstrained)', 'CG (unconstrained)']
		self.fitModel=self.fitModels[0]
		self.fitRunning=False
		self._breakFit=False
		self.fitFrontendFeedback=None
		self.fitness=0.0
		specmate_backend_dataBackup.initEmptyProject(self)
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
# end class "specmate_backend_fitting"
#----------------------------------------------------------------------------------------

#-----------------------------------------------------------------------------------------
# begin class "specmate_backend"
#----------------------------------------------------------------------------------------
class specmate_backend(specmate_backend_fitting):
	"""
	This class is the backend alltogether. At the moment it contains all methods that I didn't know where to put.
	"""
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def substractFromSpectrum(self,which,number=-1,name='None'):
		"""
		Substract one of the datasets from the spectrum.
		@param which: which dataset type
		@type which: string ("spectrum","background","reference","peak","function")
		@param number: index of reference or peak
		@type number: int
		@param name: name of reference or peak
		@type name: string
		@return: success(0)/fail(1)
		@rtype: int
		"""
		if self.specData.hasData==False:
			return(1)
		dataset=self._selectDataset(which,number,name)
		if dataset==None:
			return(1)
		if dataset.hasData==False:
			return(1)
		self.specData.data[:,1]-=dataset.Y()[:]
		self._calcDiff()
		return(0)		
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------

	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	def updateNamesAndLabels(self,names):
		"""
		Change names (of the spectrum and the residual) and labels in the plot.
		@param names: list of names and labels.
		@type names: list of strings
		"""
		if len(names)>0:
			self.specData.name=names[0][1]
			if len(names)>1:
				self.diffData.name=names[1][1]
				if len(names)>2:
					for i in xrange(2,len(names)):
						self.namesAndLabels[names[i][0]]=names[i][1]
			self._createPlotData()
	#--------------------------------------------------------------------------------
	#--------------------------------------------------------------------------------
	
#-----------------------------------------------------------------------------------------
# end class "specmate_backend"
#----------------------------------------------------------------------------------------
