;; Intermediate structure between SpcialK and Scheme

;; Copyright (C) 2004  Sylvain Beucler
;; Copyright (C) 2004  Julien Charles
;; Copyright (C) 2004  Pierre Chtel
;; Copyright (C) 2004  Cyril Rodas

;; This file is part of SpcialK.

;; SpcialK is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.

;; SpcialK is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SpcialK; if not, write to the Free Software Foundation,
;; Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

(module special-k-structs 
  mzscheme
  (require "l10n.ss")
  (provide (all-defined))
  ; Definitions de la structure intermdiaire Analyse-Compilation


  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ;         Magic Numbers          ;
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  
  (define magic_number_start 10000)
  
  (define FONC_MAGIC (+ magic_number_start 2))
  (define CLAUSE_MAGIC (+ magic_number_start 3))
  (define ARBRE_MAGIC (+ magic_number_start 4))
  (define EXPR_T_MAGIC (+ magic_number_start 5))
  (define CODE_SCHEME_MAGIC (+ magic_number_start 6))
  
  
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ;       dictionnaire       ;
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ; Un dictionnaire est une liste de fonctions et d'expressions
  
  
  ; Constructeur de dict
  (define (new_dict) '())
  
  
  
  ; Modifieur
  
  ; ajoute une fonction dans le dictionnaire
  ; On suppose que list_clause est deja verifie
  (define (ajouter_fonc dict nom list_clause)
    (append dict (list (new_fonction nom list_clause))))
  
  ; ajoute une clause
  ; nom : le nom de la fonction 
  ; args : liste d'expression (arguments partie gauche)
  ; garde : expression 
  ; partie_droite : expression
  ;(define (ajouter_clause dict nom pos args garde partie_droite) 
  ;   (cond ((null? dict) (list (new_fonction nom (new_clause pos args garde partie_droite))))
  ;        
  ;         ((and (equal? nom (get_nom (car dict)))
  ;               (= (length args) (get_nb_args (car dict))))
  ;          (cons (add_clause (car dict) (new_clause pos args garde partie_droite)) (cdr dict)))
  
  ;         (else (cons (car dict) (ajouter_clause (cdr dict) nom pos args garde partie_droite)))))
  
  
  
  ; Accesseur
  
  ; renvoi la fonction nom(nb_args) ou #f si elle n'existe pas
  (define (chercher_fonc dict nom nb_args)
    (cond ((null? dict) #f)
          
          ((and (equal? nom (get_nom (car dict)))
                (= nb_args (get_nb_args (car dict))))
           (car dict))
          
          (else (chercher_fonc (cdr dict) nom nb_args))))
  
  
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ;   Expression top-level   ;
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  
  (define (new_expression_top start_pos end_pos expr)
    (list EXPR_T_MAGIC start_pos end_pos expr))
  
  (define (get_start_pos_expr_t expr)
    (cadr expr))
  
  (define (get_end_pos_expr_t expr)
    (caddr expr))
  
  (define (get_expr expr)
    (cadddr expr))
  
  (define (expr_top? expr)
    (and (pair? expr) (= (car expr) EXPR_T_MAGIC)))
  
  (define (ajouter_expr dict expr start_pos end_pos)
    (append dict (list (new_expression_top start_pos end_pos expr))))
  
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ;       code scheme       ;
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;
  
  (define (new_code_scheme code)
    (list CODE_SCHEME_MAGIC code))
  
  (define (get_code code)
    (cadr code))
  
  (define (code_scheme? code)
    (and (pair? code) (= (car code) CODE_SCHEME_MAGIC)))
  
  (define (ajouter_code_scheme dict code)
    (append dict (list (new_code_scheme code))))
  
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ;        fonctions        ;
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;
  
  ;constructeur de fonction
  (define (new_fonction nom clauses)
    (list FONC_MAGIC nom (length (get_args (car clauses))) clauses))
  
  ; Accesseurs sur les fonctions
  (define (get_nom fonc) (cadr fonc))
  (define (get_nb_args fonc) (caddr fonc))
  (define (get_clauses fonc) (cadddr fonc))
  
  
  ; Modifieur
  (define (add_clause fonc clause)
    (list FONC_MAGIC (get_nom fonc) (get_nb_args fonc) (append (get_clauses fonc) (list clause))))
  
  ;Predicat
  (define (fonction? fonc)
    (and (number? (car fonc)) (= (car fonc) FONC_MAGIC)))
  
  
  ;;;;;;;;;;;;;;;;;;;;;;;;
  ;        clause        ;
  ;;;;;;;;;;;;;;;;;;;;;;;;
  
  ;Constructeur de clause
  (define (new_clause start_pos end_pos args garde partie_droite) 
    (list CLAUSE_MAGIC start_pos end_pos args garde partie_droite))
  
  ;Accesseurs sur les clauses
  (define (get_start_pos_clause clause) (car (cdr clause)))
  (define (get_end_pos_clause clause) (cadr (cdr clause)))
  (define (get_args clause) (caddr (cdr clause)))
  (define (get_garde clause) (cadddr (cdr clause)))
  (define (get_partie_droite clause) (cadddr (cddr clause)))
  
  ; Predicat?
  (define (clause? clause) (and (number? (car clause)) (= CLAUSE_MAGIC (car clause))))
  
  
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ;      arbres d'expressions     ;
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  
  ; Message d'erreur sur les types
  (define error_str "")
  
  ; Fonction de formatage des messages d'erreurs
  ; Modifie error_str
  (define (error_mess op g d)
    (set! error_str (string-append (localized-message '_operator_) "\"" (symbol->string op) "\" : "))
    
    (cond ((or (num_op? op) (un_num_op? op))
           (set! error_str (string-append error_str (localized-message 'num_op_error))))
          ((int_num_op? op) 
           (set! error_str (string-append error_str (localized-message 'int_num_op_error))))
          ((comparaison_op? op) 
           (set! error_str (string-append error_str (localized-message 'comp_op_error))))
          ((or (un_bool_op? op) (bin_bool_op? op)) 
           (set! error_str (string-append error_str (localized-message 'bool_op_error))))
          ((or (bin_tab_op? op) (ter_tab_op? op)) 
           (set! error_str (string-append error_str (localized-message 'tab_op_error))))
          (else (set! error_str (string-append error_str (localized-message 'internal_error)))))
    (set! error_str (string-append error_str (localized-message '_given_) (type->string (get_type g))))
    (if (not (equal? d 'no_use)) 
        (set! error_str (string-append error_str ", " (type->string (get_type d))))))
  
  ; Types 
  (define INT 0)
  (define REAL 1)
  (define BOOL 2)
  (define LISTE 3)
  (define VAR 4)
  (define NUMERIC 5)
  (define TAB 6)
  (define K-STRING 7)

  (define (type->string type)
    (cond ((= type 0) (localized-message 'type-integer))
          ((= type 1) (localized-message 'type-real))
          ((= type 2) (localized-message 'type-boolean))
          ((= type 3) (localized-message 'type-list))
          ((= type 4) (localized-message 'type-variable))
          ((= type 5) (localized-message 'type-numerical))
          ((= type 6) (localized-message 'type-array))
          ((= type 7) (localized-message 'type-string))
          (else (localized-message 'type-unknown))))
  
  ; Categories d'operateurs
  
  ; Operateurs numeriques entier
  ; Ne s'applique qu'a des objets de type INT ou NUMERIC
  (define (int_num_op? op)
    (or (equal? op 'mod)
        (equal? op 'div)))
  
  ; Operateurs numeriques
  ; Ne s'applique qu'a des objets de type INT ou REAL NUMERIC
  (define (num_op? op)
    (or (equal? op '+)
        (equal? op '-)
        (equal? op '*)
        (equal? op '/)))
  
  (define (un_num_op? op)
    (or (equal? op '-)))
  
  ; Operateurs de comparaison
  ; Ne s'applique qu'a 2 objets de meme type
  (define (comparaison_op? op)
    (or (equal? op '==)
        (equal? op '<)
        (equal? op '>)
        (equal? op '<=)
        (equal? op '>=)
        (equal? op '<>)))
  
  ; Operateur booleen binaire 
  ; Ne s'applique qu'a des objets de type booleen
  (define (bin_bool_op? op)
    (or (equal? op '&)
        (equal? op '\|)))
  
  ; Operateur booleen binaire 
  ; Ne s'applique qu'a un objet de type booleen
  (define (un_bool_op? op)
    (or (equal? op '!)))
  
  ; Autres operateurs 
  ; Peut s'appliquer a n'importe quel type d'objet
  (define (list_op? op)
    (or (equal? op ':)))
  
  (define (call_op? op)
    (equal? op 'call))
  
  (define (affect_op? op)
    (equal? op ':=))
  
  (define (bin_tab_op? op)
    (or (equal? op '\[\]) (equal? op '\{\})))
  
  (define (ter_tab_op? op)
    (or (equal? op '<-) (equal? op '<->)))
  
  ;;;;;;;; Construction d'arbres ;;;;;;;;
  
  ; construit un arbre en determinant le type 
  ; Ex: (arbre '+ a b) -> (('+ NUMERIC) a b)
  ; Renvoi: 
  ; - un arbre si pas d'erreur.
  ; - faux sinon. error_str contient une chaine indiquant la nature de l'erreur.
  ; - 'err_expr si l'erreur venait de plus bas
  ; Les operateurs possibles sont:
  ; > < == <> <= >= : | & + - * / ! <-
  
  
  
  (define (arbre_bin op g d)
    (cond ((or (equal? g 'err_expr) (equal? d 'err_expr)) 'err_expr)
          
          ((int_num_op? op) (cond ((and (or (var? g) (numeric? g) (int? g)) (or (var? d) (numeric? d) (int? d)))   
                                   (tree op INT g d))
                                  (else (begin (error_mess 'nimp g d) #f))))
           
          ((num_op? op) (cond ((or (bool? g) (bool? d) (k_list? g) (k_list? d) (k_string? g) (k_string? d))
                               (begin (error_mess op g d) #f))
                              ((or (k_real? g) (k_real? d)) 
                               (tree op REAL g d))
                              ((or (var? g) (var? d) (numeric? g) (numeric? d))
                               (tree op NUMERIC g d))
                              ((and (int? g) (int? d))
                               (tree op INT g d))
                              (else (begin (error_mess 'nimp g d) #f))))
          
          
          ((comparaison_op? op) (cond ((and (or (numeric? g) (int? g) (real? g) (var? g))
                                            (or (numeric? d) (int? d) (real? d) (var? d)))
                                       (tree op BOOL g d))
                                      ((and (or (list? g) (var? g)) (or (list? d) (var? d)))
                                       (if (and (not (equal? op '==)) (not (equal? op'<>)))
                                           (begin (set! error_str (localized-message 'list_comp_op_error)) #f)
                                           (tree op BOOL g d)))
                                      ((and (or (tab? g) (var? g)) (or (tab? d) (var? d)))
                                       (if (and (not (equal? op '==)) (not (equal? op'<>)))
                                           (begin (set! error_str (localized-message 'tab_comp_op_error)) #f)
                                           (tree op BOOL g d)))
                                      ((and (or (bool? g) (var? g)) (or (bool? d) (var? d)))
                                       (if (and (not (equal? op '==)) (not (equal? op'<>)))
                                           (begin (set! error_str (localized-message 'bool_comp_op_error)) #f)
                                           (tree op BOOL g d)))
                                      
                                      (else (begin (error_mess op g d) #f))))
          
          
          ((bin_bool_op? op) (cond ((and (or (bool? g) (var? g)) (or (bool? d) (var? d)))
                                    (tree op BOOL g d))
                                   (else (begin (error_mess op g d) #f))))
          
          ((list_op? op) (tree op LISTE g d))
          
          ((or (call_op? op))  (tree op VAR g d))
          
          ((and (bin_tab_op? op) (equal? op '\[\])) (if (or (var? g) (tab? g)) 
                                                        (tree op VAR g d)
                                                        (begin (error_mess op g 'no_use) #f)))
          
          ((and (bin_tab_op? op) (equal? op '\{\})) (tree op TAB g d))
          
          ((affect_op? op) (tree op (get_type g) g d))
          
          (else (begin (error_mess 'nimp g d) #f))))
  
  (define (arbre_un op g)
    (cond ((equal? g 'err_expr) 'err_expr)
          
          ((un_bool_op? op)
           (if (or (bool? g) (var? g))
               (tree_unary op BOOL g)
               (begin
                 (error_mess op g 'no_use) #f)))
          
          ((un_num_op? op)
           (cond ((or (numeric? g) (var? g))
                  (tree_unary op NUMERIC g))
                 ((int? g) (tree_unary op INT g))
                 ((k_real? g) (tree_unary op REAL g))
                 (else (begin (error_mess op g 'no_use) #f))))
          
          (else (begin (error_mess 'nimp g 'no_use) #f))))
  
  (define (arbre_ter op g d dd)
    (cond ((or (equal? g 'err_expr) (equal? d 'err_expr) (equal? dd 'err_expr)) 'err_expr)
          ((ter_tab_op? op) (if (or (tab? g) (var? g)) 
                                (tree_ter op TAB g d dd)
                                (begin (error_mess op g 'no_use) #f)))
          (else (begin (error_mess 'nimp g 'no_use) #f))))
  
  (define (feuille content type)
    (new_feuille (new_element content type)))
  
  (define (tree r type g d)
    (list ARBRE_MAGIC (new_element r type) g d))
  
  (define (tree_unary r type g)
    (list ARBRE_MAGIC (new_element r type) g))
  
  (define (tree_ter r type g d dd)
    (list ARBRE_MAGIC (new_element r type) g d dd))
  
  ; VAR contient tous les autres types
  ; NUMERIC contient les types INT et REAL
  
  ; Renvoi le type d'un arbre
  (define (get_type a)
    (cadar (cdr a)))
  
  (define (get_fg a)
    (cadr (cdr a)))
  
  (define (get_fd a)
    (caddr (cdr a)))
  
  (define (get_fdd a)
    (cadddr (cdr a)))
  
  (define (get_op a)
    (caar (cdr a)))
  
  ; Predicats
  
  (define (arbre? a)
    (and (pair? a) (number? (car a)) (= ARBRE_MAGIC (car a))))
  
  (define (unaire? a)
    (and (arbre? a) (null? (cdddr a))))
  
  (define (binaire? a)
    (and (not (unaire? a)) (null? (cdddr (cdr a)))))
  
  (define (ternaire? a)
    (and (not (unaire? a)) (not (binaire? a))))
  
  (define (feuille? a)
    (null? (cddr a)))
  
  ; Predicats sur les types
  (define (int? a)
    (= (get_type a) INT))
  
  (define (k_real? a)
    (= (get_type a) REAL))
  
  (define (numeric? a)
    (= (get_type a) NUMERIC))
  
  (define (bool? a)
    (= (get_type a) BOOL))
  
  (define (var? a)
    (= (get_type a) VAR))
  
  (define (k_list? a)
    (= (get_type a) LISTE))
  
  (define (tab? a)
    (= (get_type a) TAB))
  
  (define (k_string? a)
    (= (get_type a) K-STRING))
  
  ; elements des arbres
  (define (new_element elem type)
    (list elem type))
  
  ;construit une feuille
  (define (new_feuille elem)
    (list ARBRE_MAGIC elem)))



