// $Id: spotlist.cpp,v 1.3 2005/02/25 12:22:16 grosskur Exp $
//
// Copyright (C) 2003 Alan Grosskurth
//
// This file is part of Spatter.
//
// Spatter is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Spatter is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Spatter; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

#include "spotlist.hpp"
#include "dsvfile.hpp"
#include "stringparser.hpp"
#include "hexconverter.hpp"

typedef std::vector< std::string > vector_string;

//
// Spot
//

Spot::Spot(const lti::ipoint& origin, const lti::imatrix& mask)
    : m_origin(origin), m_mask(mask)
{   
}

const lti::ipoint&
Spot::origin() const
{
    return m_origin;
}

const lti::imatrix&
Spot::mask() const
{
    return m_mask;
}

const lti::ipoint
Spot::size() const
{
    return lti::ipoint(m_mask.rows(), m_mask.columns());
}

const lti::ipoint
Spot::center() const
{
    return m_origin + size()/2;
}

//
// SpotList
//

SpotList::SpotList()
{
}

SpotList::SpotList(std::istream& in)
{
    load(in);
}

void
SpotList::load(std::istream& in)
{
    DsvFileReader tab_file(in, ':');

    vector_string data;

    // Read each spot
    while (tab_file.read_line(data) != -1) {
        lti::ipoint origin(spatter::parse_int(data[0]),
                          spatter::parse_int(data[1]));

        lti::ipoint size(spatter::parse_int(data[2]),
                        spatter::parse_int(data[3]));

        lti::imatrix mask(spatter::parse_matrix(
          spatter::hex_to_binary(data[4]).substr(4 - size[0]*size[1] % 4),
          size[0], size[1]));

        data.clear();

        spots.push_back(Spot(origin, mask));
    }
}

void
SpotList::save(std::ostream& out)
{
    DsvFileWriter tab_file(out, ':');

    // Write each spot
    for (unsigned int i = 0; i < spots.size(); ++i) {
        vector_string data;

        data.push_back(spatter::unparse_int(spots[i].origin()[0]));
        data.push_back(spatter::unparse_int(spots[i].origin()[1]));

        data.push_back(spatter::unparse_int(spots[i].mask().rows()));
        data.push_back(spatter::unparse_int(spots[i].mask().columns()));

        data.push_back(spatter::binary_to_hex(
                           spatter::unparse_matrix(spots[i].mask())));

        tab_file.write_line(data);
    }
}
