// $Id: gridder-adjustoneoff.cpp,v 1.3 2005/02/25 12:22:16 grosskur Exp $
//
// Copyright (C) 2003 Alan Grosskurth
//
// This file is part of Spatter.
//
// Spatter is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Spatter is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Spatter; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

#include "grid.hpp"
#include "gridder.hpp"

#include <ltiSplitImageTorgI.h>
#include <limits>


static lti::channel8
get_subgrid_image(const Grid& grid)
{
    // Draw one subgrid
    lti::channel8 subgrid_mat(grid.subgrid_pixel_size()[0],
                              grid.subgrid_pixel_size()[1], (lti::ubyte)0);

    // draw the row lines
    for (int i = 0; i < grid.subgrid_size()[0]; ++i) {
        for (int j = 0; j < grid.subgrid_pixel_size()[1]; ++j) {
            subgrid_mat[(int)(i*grid.subgrid_delta()[0])][j] = 1;
        }
    }

    // draw the column lines
    for (int i = 0; i < grid.subgrid_pixel_size()[0]; ++i) {
        for (int j = 0; j < grid.subgrid_size()[1]; ++j) {
            subgrid_mat[i][(int)(j*grid.subgrid_delta()[1])] = 1;
        }
    }

    return subgrid_mat;
}


static int
get_area(lti::channel8& im,
         lti::channel8& mat,
         int row,
         int col)
{
    mat.fill(im,
             0,
             0,
             mat.rows(),
             mat.columns(),
             row,
             col);

    return mat.sumOfElements();
}


static int
find_shift(int total,
           int l0, int l1, int l2,
           int r0, int r1, int r2)
{
    //int q1 = std::abs(a - b);
    //int q2 = std::abs(c - d);
    //int q3 = std::abs(a - d);
    //int q4 = std::abs(b - c);

    //int move = q1 + q2;
    //int stay = q3 + q4;

    //std::cerr << total << " : "
    //          << a << "," << b << "," << c << "," << d << std::endl;

    int result = 0;
#if 1
    if (l1 <= r1 && l1 < r2) {
        result = 1;
    } else if (r1 <= l1 && r1 < l0) {
        result = -1;
    }
#else
    if (r2 > l0 && /*r2 > l1 &&*/ l2 > l0 /*&& l2 > l1*/) {
        result = 1;
    } else if (l0 > r1 && /*l0 > r2 &&*/ r0 > r1 /*&& r0 > r2*/) {
        result = -1;
    }
#endif
    return result;
}


void
subgrid_adjust_oneoff(Grid& grid,
                      int i,
                      int j,
                      lti::channel8& im8,
                      lti::channel8& subgrid_mat,
                      int num_iter)
{
    // Adjust for "one-spot-off" in either direction

    for (int k = 0; k < num_iter; ++k) {
        lti::ipoint o(grid.subgrid_origin(i, j));

        lti::channel8 mat(grid.subgrid_pixel_size()[0],
                          grid.subgrid_pixel_size()[1],
                          (lti::ubyte)0);

        lti::channel8 col_mat(grid.subgrid_pixel_size()[0],
                              (int)grid.subgrid_delta()[1],
                              (lti::ubyte)0);

        bool can_move_up    = (o[0] >= grid.subgrid_delta()[0]);
        bool can_move_left  = (o[1] >= grid.subgrid_delta()[1]);
        bool can_move_down  = (o[0]
                              + grid.subgrid_size()[0]
                                *grid.subgrid_delta()[0]
                              < im8.rows());
        bool can_move_right = (o[1]
                              + grid.subgrid_size()[1]
                                *grid.subgrid_delta()[1]
                              < im8.columns());
#if 0
        std::cerr << "can_move_up: " << can_move_up << std::endl;
        std::cerr << "can_move_left: " << can_move_left << std::endl;
        std::cerr << "can_move_down: " << can_move_down << std::endl;
        std::cerr << "can_move_right: " << can_move_right << std::endl;
#endif
        int left_outside = 0;
        if (can_move_left) {
            left_outside = get_area(im8,
                                    col_mat,
                                    o[0],
                                    (int)(o[1] - grid.subgrid_delta()[1]));
        }

        int left_inside = get_area(im8,
                                   col_mat,
                                   o[0],
                                   o[1]);

        int left_more_inside = get_area(im8,
                                        col_mat,
                                        o[0],
                                        (int)(o[1] + grid.subgrid_delta()[1]));

        int right_more_inside = get_area(im8,
                                         col_mat,
                                         o[0],
                                         (int)(o[1]
                                               + (grid.subgrid_size()[1] - 2)
                                               *grid.subgrid_delta()[1]));

        int right_inside = get_area(im8,
                                    col_mat,
                                    o[0],
                                    (int)(o[1]
                                          + (grid.subgrid_size()[1] - 1)
                                            *grid.subgrid_delta()[1]));

        int right_outside = 0;
        if (can_move_right) {
            right_outside = get_area(im8,
                                     col_mat,
                                     o[0],
                                     (int)(o[1]
                                           + grid.subgrid_size()[1]
                                             *grid.subgrid_delta()[1]));
        }

        int row_total = get_area(im8, mat, o[0], o[1])
            / grid.subgrid_size()[0];

        lti::channel8 row_mat((int)grid.subgrid_delta()[0],
                              grid.subgrid_pixel_size()[1],
                              (lti::ubyte)0);

        int top_outside = 0;
        if (can_move_up) {
            top_outside = get_area(im8,
                                   row_mat,
                                   (int)(o[0] - grid.subgrid_delta()[0]),
                                   (int)(o[1]));
        }

        int top_inside = get_area(im8,
                                  row_mat,
                                  o[0],
                                  o[1]);

        int top_more_inside = get_area(im8,
                                       row_mat,
                                       (int)(o[0] + grid.subgrid_delta()[0]),
                                       o[1]);

        int bottom_more_inside = get_area(im8,
                                          row_mat,
                                          (int)(o[0]
                                                + (grid.subgrid_size()[0] - 2)
                                                *grid.subgrid_delta()[0]),
                                          (int)(o[1]));

        int bottom_inside = get_area(im8,
                                     row_mat,
                                     (int)(o[0]
                                           + (grid.subgrid_size()[0] - 1)
                                             *grid.subgrid_delta()[0]),
                                     (int)(o[1]));

        int bottom_outside = 0;
        if (can_move_down) {
            bottom_outside = get_area(im8,
                                      row_mat,
                                      (int)(o[0]
                                            + grid.subgrid_size()[0]
                                              *grid.subgrid_delta()[0]),
                                      (int)(o[1]));
        }

        int col_total = get_area(im8, mat, o[0], o[1])
            / grid.subgrid_size()[1];
#if 0
        std::cerr << "(" << top_outside << "," << top_inside
                  << "," << bottom_inside << "," << bottom_outside
                  << ")" << std::endl;
#endif
        lti::ipoint dir(find_shift(row_total,
                                  top_outside,
                                  top_inside,
                                  top_more_inside,
                                  bottom_more_inside,
                                  bottom_inside,
                                  bottom_outside),
                       find_shift(col_total,
                                  left_outside,
                                  left_inside,
                                  left_more_inside,
                                  right_more_inside,
                                  right_inside,
                                  right_outside));

        if (dir == lti::ipoint(0, 0)) {
            break;
        }

        lti::ipoint shift(grid.subgrid_delta()*lti::dpoint(dir));

        grid.subgrid_move(i, j, shift);
#if 0
        std::cerr << "  Jump: " << dir << std::endl;
#endif
    }
}


namespace spatter {

void
grid_adjust_oneoff(const lti::image& im,
                   Grid& grid)
{
    lti::channel8 im8;    
    lti::splitImageTorgI splitter;
    splitter.getIntensity(im, im8);

    lti::channel8 subgrid_mat = get_subgrid_image(grid);
    //lti::channel8 subgrid_mat = get_reverse_grid_spot_image(grid);

    std::cerr << "One-off-tuning subgrid origins..." << std::endl;

    for (int i = 0; i < grid.size()[0]; ++i) {
        for (int j = 0; j < grid.size()[1]; ++j) {
//            std::cerr << "Subgrid: " << lti::ipoint(i, j) << std::endl;
//            std::cerr << "  Origin: " << grid.subgrid_origin(i, j)
//                      << std::endl;
            subgrid_adjust_oneoff(grid, i, j, im8, subgrid_mat, 3);
            std::cerr << i*grid.size()[1] + j + 1
                      << " of "
                      << grid.size()[0]*grid.size()[1]
                      << " subgrids found"
                      << std::endl;
            //std::cerr << "  Final: " << grid.subgrid_origin(i, j)
            //          << std::endl;
        }
    }
}

} // namespace spatter
